"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const targets = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const rds = require("../lib");
module.exports = {
    'create a DB instance'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            licenseModel: rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.MEDIUM),
            multiAz: true,
            storageType: rds.StorageType.IO1,
            masterUsername: 'syscdk',
            vpc,
            databaseName: 'ORCL',
            storageEncrypted: true,
            backupRetention: cdk.Duration.days(7),
            monitoringInterval: cdk.Duration.minutes(1),
            enablePerformanceInsights: true,
            cloudwatchLogsExports: [
                'trace',
                'audit',
                'alert',
                'listener'
            ],
            cloudwatchLogsRetention: logs.RetentionDays.ONE_MONTH,
            autoMinorVersionUpgrade: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            Properties: {
                DBInstanceClass: 'db.t2.medium',
                AllocatedStorage: '100',
                AutoMinorVersionUpgrade: false,
                BackupRetentionPeriod: 7,
                CopyTagsToSnapshot: true,
                DBName: 'ORCL',
                DBSubnetGroupName: {
                    Ref: 'InstanceSubnetGroupF2CBA54F'
                },
                DeletionProtection: true,
                EnableCloudwatchLogsExports: [
                    'trace',
                    'audit',
                    'alert',
                    'listener'
                ],
                EnablePerformanceInsights: true,
                Engine: 'oracle-se1',
                Iops: 1000,
                LicenseModel: 'bring-your-own-license',
                MasterUsername: {
                    'Fn::Join': [
                        '',
                        [
                            '{{resolve:secretsmanager:',
                            {
                                Ref: 'InstanceSecret478E0A47'
                            },
                            ':SecretString:username::}}'
                        ]
                    ]
                },
                MasterUserPassword: {
                    'Fn::Join': [
                        '',
                        [
                            '{{resolve:secretsmanager:',
                            {
                                Ref: 'InstanceSecret478E0A47'
                            },
                            ':SecretString:password::}}'
                        ]
                    ]
                },
                MonitoringInterval: 60,
                MonitoringRoleArn: {
                    'Fn::GetAtt': [
                        'InstanceMonitoringRole3E2B4286',
                        'Arn'
                    ]
                },
                MultiAZ: true,
                PerformanceInsightsRetentionPeriod: 7,
                StorageEncrypted: true,
                StorageType: 'io1',
                VPCSecurityGroups: [
                    {
                        'Fn::GetAtt': [
                            'InstanceSecurityGroupB4E5FA83',
                            'GroupId'
                        ]
                    }
                ]
            },
            DeletionPolicy: 'Retain',
            UpdateReplacePolicy: 'Retain'
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DeletionPolicy: 'Retain',
            UpdateReplacePolicy: 'Retain'
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBSubnetGroup', {
            DBSubnetGroupDescription: 'Subnet group for Instance database',
            SubnetIds: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0'
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A'
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: 'Security group for Instance database',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'monitoring.rds.amazonaws.com'
                        }
                    }
                ],
                Version: '2012-10-17'
            },
            ManagedPolicyArns: [
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition'
                            },
                            ':iam::aws:policy/service-role/AmazonRDSEnhancedMonitoringRole'
                        ]
                    ]
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            Description: {
                'Fn::Join': [
                    '',
                    [
                        'Generated by the CDK for stack: ',
                        {
                            Ref: 'AWS::StackName'
                        }
                    ]
                ]
            },
            GenerateSecretString: {
                ExcludeCharacters: '\"@/\\',
                GenerateStringKey: 'password',
                PasswordLength: 30,
                SecretStringTemplate: '{"username":"syscdk"}'
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::SecretTargetAttachment', {
            SecretId: {
                Ref: 'InstanceSecret478E0A47'
            },
            TargetId: {
                Ref: 'InstanceC1063A87'
            },
            TargetType: 'AWS::RDS::DBInstance'
        }));
        assert_1.expect(stack).to(assert_1.countResources('Custom::LogRetention', 4));
        test.done();
    },
    'instance with option and parameter group'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const optionGroup = new rds.OptionGroup(stack, 'OptionGroup', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            majorEngineVersion: '11.2',
            configurations: [
                {
                    name: 'XMLDB'
                }
            ]
        });
        const parameterGroup = new rds.ParameterGroup(stack, 'ParameterGroup', {
            family: 'hello',
            description: 'desc',
            parameters: {
                key: 'value'
            }
        });
        // WHEN
        new rds.DatabaseInstance(stack, 'Database', {
            engine: rds.DatabaseInstanceEngine.SQL_SERVER_EE,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'syscdk',
            masterUserPassword: cdk.SecretValue.plainText('tooshort'),
            vpc,
            optionGroup,
            parameterGroup
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DBParameterGroupName: {
                Ref: 'ParameterGroup5E32DECB'
            },
            OptionGroupName: {
                Ref: 'OptionGroupACA43DC1'
            }
        }));
        test.done();
    },
    'create an instance from snapshot'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new rds.DatabaseInstanceFromSnapshot(stack, 'Instance', {
            snapshotIdentifier: 'my-snapshot',
            engine: rds.DatabaseInstanceEngine.POSTGRES,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DBSnapshotIdentifier: 'my-snapshot'
        }));
        test.done();
    },
    'throws when trying to generate a new password from snapshot without username'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // THEN
        test.throws(() => new rds.DatabaseInstanceFromSnapshot(stack, 'Instance', {
            snapshotIdentifier: 'my-snapshot',
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc,
            generateMasterUserPassword: true,
        }), '`masterUsername` must be specified when `generateMasterUserPassword` is set to true.');
        test.done();
    },
    'throws when specifying user name without asking to generate a new password'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // THEN
        test.throws(() => new rds.DatabaseInstanceFromSnapshot(stack, 'Instance', {
            snapshotIdentifier: 'my-snapshot',
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc,
            masterUsername: 'superadmin'
        }), 'Cannot specify `masterUsername` when `generateMasterUserPassword` is set to false.');
        test.done();
    },
    'throws when password and generate password ar both specified'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // THEN
        test.throws(() => new rds.DatabaseInstanceFromSnapshot(stack, 'Instance', {
            snapshotIdentifier: 'my-snapshot',
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc,
            masterUserPassword: cdk.SecretValue.plainText('supersecret'),
            generateMasterUserPassword: true,
        }), 'Cannot specify `masterUserPassword` when `generateMasterUserPassword` is set to true.');
        test.done();
    },
    'create a read replica in the same region - with the subnet group name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const sourceInstance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc
        });
        // WHEN
        new rds.DatabaseInstanceReadReplica(stack, 'ReadReplica', {
            sourceDatabaseInstance: sourceInstance,
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            SourceDBInstanceIdentifier: {
                "Fn::Join": ["", [
                        "arn:",
                        { Ref: "AWS::Partition" },
                        ":rds:",
                        { Ref: "AWS::Region" },
                        ":",
                        { Ref: "AWS::AccountId" },
                        ":db:",
                        { Ref: "InstanceC1063A87" },
                    ]],
            },
            DBSubnetGroupName: {
                Ref: 'ReadReplicaSubnetGroup680C605C',
            },
        }));
        test.done();
    },
    'on event'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc
        });
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('dummy'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X
        });
        // WHEN
        instance.onEvent('InstanceEvent', { target: new targets.LambdaFunction(fn) });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            EventPattern: {
                source: [
                    'aws.rds'
                ],
                resources: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition'
                                },
                                ':rds:',
                                {
                                    Ref: 'AWS::Region'
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId'
                                },
                                ':db:',
                                {
                                    Ref: 'InstanceC1063A87'
                                }
                            ]
                        ]
                    }
                ]
            },
            Targets: [
                {
                    Arn: {
                        'Fn::GetAtt': [
                            'Function76856677',
                            'Arn'
                        ],
                    },
                    Id: 'Target0'
                }
            ]
        }));
        test.done();
    },
    'on event without target'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc
        });
        // WHEN
        instance.onEvent('InstanceEvent');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            EventPattern: {
                source: [
                    'aws.rds'
                ],
                resources: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition'
                                },
                                ':rds:',
                                {
                                    Ref: 'AWS::Region'
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId'
                                },
                                ':db:',
                                {
                                    Ref: 'InstanceC1063A87'
                                }
                            ]
                        ]
                    }
                ]
            }
        }));
        test.done();
    },
    'can use metricCPUUtilization'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc
        });
        // THEN
        test.deepEqual(stack.resolve(instance.metricCPUUtilization()), {
            dimensions: { DBInstanceIdentifier: { Ref: 'InstanceC1063A87' } },
            namespace: 'AWS/RDS',
            metricName: 'CPUUtilization',
            period: cdk.Duration.minutes(5),
            statistic: 'Average'
        });
        test.done();
    },
    'can resolve endpoint port and socket address'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc
        });
        test.deepEqual(stack.resolve(instance.instanceEndpoint.port), {
            'Fn::GetAtt': ['InstanceC1063A87', 'Endpoint.Port']
        });
        test.deepEqual(stack.resolve(instance.instanceEndpoint.socketAddress), {
            'Fn::Join': [
                '',
                [
                    { 'Fn::GetAtt': ['InstanceC1063A87', 'Endpoint.Address'] },
                    ':',
                    { 'Fn::GetAtt': ['InstanceC1063A87', 'Endpoint.Port'] },
                ]
            ]
        });
        test.done();
    },
    'can deactivate backup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc,
            backupRetention: cdk.Duration.seconds(0),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            BackupRetentionPeriod: 0
        }));
        test.done();
    },
    'imported instance with imported security group with allowAllOutbound set to false'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const instance = rds.DatabaseInstance.fromDatabaseInstanceAttributes(stack, 'Database', {
            instanceEndpointAddress: 'address',
            instanceIdentifier: 'identifier',
            port: 3306,
            securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                    allowAllOutbound: false
                })],
        });
        // WHEN
        instance.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'sg-123456789',
        }));
        test.done();
    },
    'create an instance with imported monitoring role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const monitoringRole = new aws_iam_1.Role(stack, "MonitoringRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("monitoring.rds.amazonaws.com"),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')
            ]
        });
        // WHEN
        new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc,
            monitoringInterval: cdk.Duration.minutes(1),
            monitoringRole
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::RDS::DBInstance", {
            MonitoringInterval: 60,
            MonitoringRoleArn: {
                "Fn::GetAtt": ["MonitoringRole90457BF9", "Arn"]
            }
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'create an instance with an existing security group'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const securityGroup = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
            allowAllOutbound: false
        });
        // WHEN
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc,
            securityGroups: [securityGroup],
        });
        instance.connections.allowDefaultPortFromAnyIpv4();
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            VPCSecurityGroups: ['sg-123456789']
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            FromPort: {
                'Fn::GetAtt': [
                    'InstanceC1063A87',
                    'Endpoint.Port'
                ]
            },
            GroupId: 'sg-123456789',
            ToPort: {
                'Fn::GetAtt': [
                    'InstanceC1063A87',
                    'Endpoint.Port'
                ]
            }
        }));
        test.done();
    },
    'throws when trying to add rotation to an instance without secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const instance = new rds.DatabaseInstance(stack, 'Database', {
            engine: rds.DatabaseInstanceEngine.SQL_SERVER_EE,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'syscdk',
            masterUserPassword: cdk.SecretValue.plainText('tooshort'),
            vpc
        });
        // THEN
        test.throws(() => instance.addRotationSingleUser(), /without secret/);
        test.done();
    },
    'throws when trying to add single user rotation multiple times'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const instance = new rds.DatabaseInstance(stack, 'Database', {
            engine: rds.DatabaseInstanceEngine.SQL_SERVER_EE,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'syscdk',
            vpc
        });
        // WHEN
        instance.addRotationSingleUser();
        // THEN
        test.throws(() => instance.addRotationSingleUser(), /A single user rotation was already added to this instance/);
        test.done();
    },
    'throws when timezone is set for non-sqlserver database engine'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'vpc');
        const tzSupportedEngines = [rds.DatabaseInstanceEngine.SQL_SERVER_EE, rds.DatabaseInstanceEngine.SQL_SERVER_EX,
            rds.DatabaseInstanceEngine.SQL_SERVER_SE, rds.DatabaseInstanceEngine.SQL_SERVER_WEB];
        const tzUnsupportedEngines = [rds.DatabaseInstanceEngine.MYSQL, rds.DatabaseInstanceEngine.POSTGRES,
            rds.DatabaseInstanceEngine.ORACLE_EE, rds.DatabaseInstanceEngine.MARIADB];
        // THEN
        tzSupportedEngines.forEach((engine) => {
            test.ok(new rds.DatabaseInstance(stack, `${engine.name}-db`, {
                engine,
                instanceClass: ec2.InstanceType.of(ec2.InstanceClass.C5, ec2.InstanceSize.SMALL),
                masterUsername: 'master',
                timezone: 'Europe/Zurich',
                vpc,
            }));
        });
        tzUnsupportedEngines.forEach((engine) => {
            test.throws(() => new rds.DatabaseInstance(stack, `${engine.name}-db`, {
                engine,
                instanceClass: ec2.InstanceType.of(ec2.InstanceClass.C5, ec2.InstanceSize.SMALL),
                masterUsername: 'master',
                timezone: 'Europe/Zurich',
                vpc,
            }), /timezone property can be configured only for Microsoft SQL Server/);
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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