# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['deepl']

package_data = \
{'': ['*']}

install_requires = \
['requests>=2,<3']

entry_points = \
{'console_scripts': ['deepl = deepl.__main__:main']}

setup_kwargs = {
    'name': 'deepl',
    'version': '1.5.1',
    'description': 'Python library for the DeepL API.',
    'long_description': '# DeepL Python Library\n\n[![PyPI version](https://img.shields.io/pypi/v/deepl.svg)](https://pypi.org/project/deepl/)\n[![Supported Python versions](https://img.shields.io/pypi/pyversions/deepl.svg)](https://pypi.org/project/deepl/)\n[![License: MIT](https://img.shields.io/badge/license-MIT-blueviolet.svg)](https://github.com/DeepLcom/deepl-python/blob/main/LICENSE)\n\nThe [DeepL API](https://www.deepl.com/docs-api?utm_source=github&utm_medium=github-python-readme) is a language\ntranslation API that allows other computer programs to send texts and documents to DeepL\'s servers and receive\nhigh-quality translations. This opens a whole universe of opportunities for developers: any translation product you can\nimagine can now be built on top of DeepL\'s best-in-class translation technology.\n\nThe DeepL Python library offers a convenient way for applications written in Python to interact with the DeepL API. We\nintend to support all API functions with the library, though support for new features may be added to the library after\nthey’re added to the API.\n\n\n## Getting an authentication key \n\nTo use the DeepL Python Library, you\'ll need an API authentication key. To get a key, [please create an account here](https://www.deepl.com/pro?utm_source=github&utm_medium=github-python-readme#developer). You can translate up to 500,000 characters/month for free. \n\nAfter you have created an account, you can find your API authentication key on your [DeepL Pro Account](https://www.deepl.com/pro-account/?utm_source=github&utm_medium=github-python-readme).\n\n## Installation\nThe library can be installed from [PyPI](https://pypi.org/project/deepl/) using pip:\n```shell\npip install --upgrade deepl\n```\n\nIf you need to modify this source code, install the dependencies using poetry:\n```shell\npoetry install\n```\n\n### Requirements\nThe library is tested with Python versions 3.6 to 3.10. \n\nThe `requests` module is used to perform HTTP requests; the minimum is version 2.0.\n\n## Usage\n\n```python\nimport deepl\n\n# Create a Translator object providing your DeepL API authentication key.\n# Be careful not to expose your key, for example when sharing source code.\nauth_key = "f63c02c5-f056-..."  # Replace with your key\ntranslator = deepl.Translator(auth_key)\n# This example is for demonstration purposes only. In production code, the\n# authentication key should not be hard-coded, but instead fetched from a\n# configuration file or environment variable.\n\n# Translate text into a target language, in this case, French\nresult = translator.translate_text("Hello, world!", target_lang="FR")\nprint(result)  # "Bonjour, le monde !"\n# Note: printing or converting the result to a string uses the output text\n\n# Translate multiple texts into British English\nresult = translator.translate_text(["お元気ですか？", "¿Cómo estás?"], target_lang="EN-GB")\nprint(result[0].text)  # "How are you?"\nprint(result[0].detected_source_lang)  # "JA"\nprint(result[1].text)  # "How are you?"\nprint(result[1].detected_source_lang)  # "ES"\n\n# Translate a formal document from English to German \ntry:\n    translator.translate_document_from_filepath(\n        "Instruction Manual.docx",\n        "Bedienungsanleitung.docx",\n        target_lang="DE",\n        formality="more"\n    )\nexcept deepl.DocumentTranslationException as error:\n    # If an error occurs during translate_document_from_filepath() or\n    # translate_document() and after the document was already uploaded, a \n    # DocumentTranslationException is raised. The document_handle property\n    # contains the document handle to later retrieve the document or contact\n    # DeepL support.\n    doc_id = error.document_handle.id\n    doc_key = error.document_handle.key\n    print(f"Error after uploading document ${error}, id: ${doc_id} key: ${doc_key}")\nexcept deepl.DeepLException as error:\n    # Errors during upload raise a DeepLException\n    print(error)\n\n# Glossaries allow you to customize your translations\nglossary_en_to_de = translator.create_glossary(\n    "My glossary",\n    source_lang="EN",\n    target_lang="DE",\n    entries={"artist": "Maler", "prize": "Gewinn"},\n)\n\nwith_glossary = translator.translate_text_with_glossary(\n    "The artist was awarded a prize.", glossary_en_to_de\n)\nprint(with_glossary)  # "Der Maler wurde mit einem Gewinn ausgezeichnet."\n\nwithout_glossary = translator.translate_text(\n    "The artist was awarded a prize.", target_lang="DE"\n)\nprint(without_glossary)  # "Der Künstler wurde mit einem Preis ausgezeichnet."\n\n\n# Check account usage\nusage = translator.get_usage()\nif usage.character.limit_exceeded:\n    print("Character limit exceeded.")\nelse:\n    print(f"Character usage: {usage.character.count} of {usage.character.limit}")\n\n# Source and target languages\nprint("Source languages:")\nfor language in translator.get_source_languages():\n    print(f"{language.code} ({language.name})")  # Example: "DE (German)"\n\nprint("Target languages:")\nfor language in translator.get_target_languages():\n    if language.supports_formality:\n        print(f"{language.code} ({language.name}) supports formality")\n    else:\n        print(f"{language.code} ({language.name})")\n```\n\n### Exceptions\nAll module functions may raise `deepl.DeepLException` or one of its subclasses.\nIf invalid arguments are provided, they may raise the standard exceptions `ValueError` and `TypeError`.\n\n### Configuration\n\n#### Logging\nLogging can be enabled to see the HTTP-requests sent and responses received by the library. Enable and control logging\nusing Python\'s logging module, for example:\n```python\nimport logging\nlogging.basicConfig()\nlogging.getLogger(\'deepl\').setLevel(logging.DEBUG)\n```\n\n#### Proxy configuration\nYou can configure a proxy by specifying the `proxy` argument when creating a `deepl.Translator`:\n```python\nproxy = "http://user:pass@10.10.1.10:3128"\ntranslator = deepl.Translator(..., proxy=proxy)\n```\n\nThe proxy argument is passed to the underlying `requests` session,\n[see the documentation here](https://docs.python-requests.org/en/latest/user/advanced/#proxies); a dictionary of schemes\nto proxy URLs is also accepted.\n\n## Command Line Interface\nThe library can be run on the command line supporting all API functions. Use the `--help` option for \nusage information:\n```shell\npython3 -m deepl --help\n```\nThe CLI requires your DeepL authentication key specified either as the `DEEPL_AUTH_KEY` environment variable, or using\nthe `--auth-key` option, for example:\n```shell\npython3 -m deepl --auth-key=YOUR_AUTH_KEY usage\n```\nNote that the `--auth-key` argument must appear *before* the command argument. The recognized commands are:\n\n| Command   | Description                                            |\n| :-------- | :----------------------------------------------------- |\n| text      | translate text(s)                                      |\n| document  | translate document(s)                                  |\n| usage     | print usage information for the current billing period |\n| languages | print available languages                              |\n| glossary  | create, list, and remove glossaries                    |\n\nFor example, to translate text:\n```shell\npython3 -m deepl --auth-key=YOUR_AUTH_KEY text --to=DE "Text to be translated."\n```\nWrap text arguments in quotes to prevent the shell from splitting sentences into words.\n\n## Issues\n\nIf you experience problems using the library, or would like to request a new feature, please open an\n[issue](https://www.github.com/DeepLcom/deepl-python/issues). \n\n## Development\n\nWe are currently unable to accept Pull Requests. If you would like to suggest changes, please open an issue instead.\n\n### Tests \n\nExecute the tests using `pytest`. The tests communicate with the DeepL API using the auth key defined by the\n`DEEPL_AUTH_KEY` environment variable.\n\nBe aware that the tests make DeepL API requests that contribute toward your API usage.\n\nThe test suite may instead be configured to communicate with the mock-server provided by\n[deepl-mock](https://www.github.com/DeepLcom/deepl-mock). Although most test cases work for either, some test cases work\nonly with the DeepL API or the mock-server and will be otherwise skipped.  The test cases that require the mock-server\ntrigger server errors and test the client error-handling. To execute the tests using deepl-mock, run it in another\nterminal while executing the tests. Execute the tests using `pytest` with the `DEEPL_MOCK_SERVER_PORT` and\n`DEEPL_SERVER_URL` environment variables defined referring to the mock-server.\n',
    'author': 'DeepL SE',
    'author_email': 'python-api@deepl.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/DeepLcom/deepl-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6.2,<4',
}


setup(**setup_kwargs)
