"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const events = require("@aws-cdk/aws-events");
const cdk = require("@aws-cdk/core");
const targets = require("../../lib");
test('Can use EC2 taskdef as EventRule target', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    cluster.addCapacity('DefaultAutoScalingGroup', {
        instanceType: new ec2.InstanceType('t2.micro'),
    });
    const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
        memoryLimitMiB: 256,
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    // WHEN
    rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        taskCount: 1,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', events.EventField.fromPath('$.detail.event')],
            }],
    }));
    // THEN
    expect(stack).toHaveResourceLike('AWS::Events::Rule', {
        Targets: [
            {
                Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                EcsParameters: {
                    TaskCount: 1,
                    TaskDefinitionArn: { Ref: 'TaskDef54694570' },
                },
                InputTransformer: {
                    InputPathsMap: {
                        'detail-event': '$.detail.event',
                    },
                    InputTemplate: '{"containerOverrides":[{"name":"TheContainer","command":["echo",<detail-event>]}]}',
                },
                RoleArn: { 'Fn::GetAtt': ['TaskDefEventsRoleFB3B67B8', 'Arn'] },
                Id: 'Target0',
            },
        ],
    });
});
test('Can use Fargate taskdef as EventRule target', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    // WHEN
    rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        taskCount: 1,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', events.EventField.fromPath('$.detail.event')],
            }],
    }));
    // THEN
    expect(stack).toHaveResourceLike('AWS::Events::Rule', {
        Targets: [
            {
                Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                EcsParameters: {
                    TaskCount: 1,
                    TaskDefinitionArn: { Ref: 'TaskDef54694570' },
                    LaunchType: 'FARGATE',
                    NetworkConfiguration: {
                        AwsVpcConfiguration: {
                            Subnets: [
                                {
                                    Ref: 'VpcPrivateSubnet1Subnet536B997A',
                                },
                            ],
                            AssignPublicIp: 'DISABLED',
                            SecurityGroups: [
                                {
                                    'Fn::GetAtt': [
                                        'TaskDefSecurityGroupD50E7CF0',
                                        'GroupId',
                                    ],
                                },
                            ],
                        },
                    },
                },
                InputTransformer: {
                    InputPathsMap: {
                        'detail-event': '$.detail.event',
                    },
                    InputTemplate: '{"containerOverrides":[{"name":"TheContainer","command":["echo",<detail-event>]}]}',
                },
                RoleArn: { 'Fn::GetAtt': ['TaskDefEventsRoleFB3B67B8', 'Arn'] },
                Id: 'Target0',
            },
        ],
    });
});
test('Can use same fargate taskdef with multiple rules', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const scheduledRule = new events.Rule(stack, 'ScheduleRule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    const patternRule = new events.Rule(stack, 'PatternRule', {
        eventPattern: {
            detail: ['test'],
        },
    });
    scheduledRule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
    }));
    expect(() => patternRule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
    }))).not.toThrow();
});
test('Can use same fargate taskdef multiple times in a rule', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const rule = new events.Rule(stack, 'ScheduleRule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', events.EventField.fromPath('$.detail.a')],
            }],
    }));
    expect(() => rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', events.EventField.fromPath('$.detail.b')],
            }],
    }))).not.toThrow();
});
test('Isolated subnet does not have AssignPublicIp=true', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', {
        maxAzs: 1,
        subnetConfiguration: [{
                subnetType: ec2.SubnetType.ISOLATED,
                name: 'Isolated',
            }],
    });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    // WHEN
    rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        taskCount: 1,
        subnetSelection: { subnetType: ec2.SubnetType.ISOLATED },
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', 'yay'],
            }],
    }));
    // THEN
    expect(stack).toHaveResourceLike('AWS::Events::Rule', {
        Targets: [
            {
                Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                EcsParameters: {
                    TaskCount: 1,
                    TaskDefinitionArn: { Ref: 'TaskDef54694570' },
                    LaunchType: 'FARGATE',
                    NetworkConfiguration: {
                        AwsVpcConfiguration: {
                            Subnets: [
                                {
                                    Ref: 'VpcIsolatedSubnet1SubnetE48C5737',
                                },
                            ],
                            AssignPublicIp: 'DISABLED',
                            SecurityGroups: [
                                {
                                    'Fn::GetAtt': [
                                        'TaskDefSecurityGroupD50E7CF0',
                                        'GroupId',
                                    ],
                                },
                            ],
                        },
                    },
                },
                Input: '{"containerOverrides":[{"name":"TheContainer","command":["echo","yay"]}]}',
                RoleArn: { 'Fn::GetAtt': ['TaskDefEventsRoleFB3B67B8', 'Arn'] },
                Id: 'Target0',
            },
        ],
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXZlbnQtcnVsZS10YXJnZXQudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImV2ZW50LXJ1bGUtdGFyZ2V0LnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxnQ0FBOEI7QUFDOUIsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyw4Q0FBOEM7QUFDOUMscUNBQXFDO0FBQ3JDLHFDQUFxQztBQUVyQyxJQUFJLENBQUMseUNBQXlDLEVBQUUsR0FBRyxFQUFFO0lBQ25ELFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQ3JELE1BQU0sT0FBTyxHQUFHLElBQUksR0FBRyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztJQUM5RCxPQUFPLENBQUMsV0FBVyxDQUFDLHlCQUF5QixFQUFFO1FBQzdDLFlBQVksRUFBRSxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDO0tBQy9DLENBQUMsQ0FBQztJQUVILE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztJQUNuRSxjQUFjLENBQUMsWUFBWSxDQUFDLGNBQWMsRUFBRTtRQUMxQyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO1FBQzlDLGNBQWMsRUFBRSxHQUFHO0tBQ3BCLENBQUMsQ0FBQztJQUVILE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1FBQzFDLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUM7S0FDcEQsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDO1FBQ2pDLE9BQU87UUFDUCxjQUFjO1FBQ2QsU0FBUyxFQUFFLENBQUM7UUFDWixrQkFBa0IsRUFBRSxDQUFDO2dCQUNuQixhQUFhLEVBQUUsY0FBYztnQkFDN0IsT0FBTyxFQUFFLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLENBQUM7YUFDaEUsQ0FBQztLQUNILENBQUMsQ0FBQyxDQUFDO0lBRUosT0FBTztJQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUIsRUFBRTtRQUNwRCxPQUFPLEVBQUU7WUFDUDtnQkFDRSxHQUFHLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxvQkFBb0IsRUFBRSxLQUFLLENBQUMsRUFBRTtnQkFDcEQsYUFBYSxFQUFFO29CQUNiLFNBQVMsRUFBRSxDQUFDO29CQUNaLGlCQUFpQixFQUFFLEVBQUUsR0FBRyxFQUFFLGlCQUFpQixFQUFFO2lCQUM5QztnQkFDRCxnQkFBZ0IsRUFBRTtvQkFDaEIsYUFBYSxFQUFFO3dCQUNiLGNBQWMsRUFBRSxnQkFBZ0I7cUJBQ2pDO29CQUNELGFBQWEsRUFBRSxvRkFBb0Y7aUJBQ3BHO2dCQUNELE9BQU8sRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFDLDJCQUEyQixFQUFFLEtBQUssQ0FBQyxFQUFFO2dCQUMvRCxFQUFFLEVBQUUsU0FBUzthQUNkO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2Q0FBNkMsRUFBRSxHQUFHLEVBQUU7SUFDdkQsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEVBQUUsTUFBTSxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDckQsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO0lBRTlELE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztJQUN2RSxjQUFjLENBQUMsWUFBWSxDQUFDLGNBQWMsRUFBRTtRQUMxQyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO0tBQy9DLENBQUMsQ0FBQztJQUVILE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1FBQzFDLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUM7S0FDcEQsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDO1FBQ2pDLE9BQU87UUFDUCxjQUFjO1FBQ2QsU0FBUyxFQUFFLENBQUM7UUFDWixrQkFBa0IsRUFBRSxDQUFDO2dCQUNuQixhQUFhLEVBQUUsY0FBYztnQkFDN0IsT0FBTyxFQUFFLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLENBQUM7YUFDaEUsQ0FBQztLQUNILENBQUMsQ0FBQyxDQUFDO0lBRUosT0FBTztJQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUIsRUFBRTtRQUNwRCxPQUFPLEVBQUU7WUFDUDtnQkFDRSxHQUFHLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxvQkFBb0IsRUFBRSxLQUFLLENBQUMsRUFBRTtnQkFDcEQsYUFBYSxFQUFFO29CQUNiLFNBQVMsRUFBRSxDQUFDO29CQUNaLGlCQUFpQixFQUFFLEVBQUUsR0FBRyxFQUFFLGlCQUFpQixFQUFFO29CQUM3QyxVQUFVLEVBQUUsU0FBUztvQkFDckIsb0JBQW9CLEVBQUU7d0JBQ3BCLG1CQUFtQixFQUFFOzRCQUNuQixPQUFPLEVBQUU7Z0NBQ1A7b0NBQ0UsR0FBRyxFQUFFLGlDQUFpQztpQ0FDdkM7NkJBQ0Y7NEJBQ0QsY0FBYyxFQUFFLFVBQVU7NEJBQzFCLGNBQWMsRUFBRTtnQ0FDZDtvQ0FDRSxZQUFZLEVBQUU7d0NBQ1osOEJBQThCO3dDQUM5QixTQUFTO3FDQUNWO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2dCQUNELGdCQUFnQixFQUFFO29CQUNoQixhQUFhLEVBQUU7d0JBQ2IsY0FBYyxFQUFFLGdCQUFnQjtxQkFDakM7b0JBQ0QsYUFBYSxFQUFFLG9GQUFvRjtpQkFDcEc7Z0JBQ0QsT0FBTyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMsMkJBQTJCLEVBQUUsS0FBSyxDQUFDLEVBQUU7Z0JBQy9ELEVBQUUsRUFBRSxTQUFTO2FBQ2Q7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtEQUFrRCxFQUFFLEdBQUcsRUFBRTtJQUM1RCxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUNyRCxNQUFNLE9BQU8sR0FBRyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7SUFFOUQsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQ3ZFLGNBQWMsQ0FBQyxZQUFZLENBQUMsY0FBYyxFQUFFO1FBQzFDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7S0FDL0MsQ0FBQyxDQUFDO0lBRUgsTUFBTSxhQUFhLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7UUFDM0QsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQztLQUNwRCxDQUFDLENBQUM7SUFFSCxNQUFNLFdBQVcsR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtRQUN4RCxZQUFZLEVBQUU7WUFDWixNQUFNLEVBQUUsQ0FBQyxNQUFNLENBQUM7U0FDakI7S0FDRixDQUFDLENBQUM7SUFFSCxhQUFhLENBQUMsU0FBUyxDQUFDLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQztRQUMxQyxPQUFPO1FBQ1AsY0FBYztLQUNmLENBQUMsQ0FBQyxDQUFDO0lBRUosTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDO1FBQ3JELE9BQU87UUFDUCxjQUFjO0tBQ2YsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsT0FBTyxFQUFFLENBQUM7QUFDckIsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsdURBQXVELEVBQUUsR0FBRyxFQUFFO0lBQ2pFLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQ3JELE1BQU0sT0FBTyxHQUFHLElBQUksR0FBRyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztJQUU5RCxNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDdkUsY0FBYyxDQUFDLFlBQVksQ0FBQyxjQUFjLEVBQUU7UUFDMUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQztLQUMvQyxDQUFDLENBQUM7SUFFSCxNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtRQUNsRCxRQUFRLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDO0tBQ3BELENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDO1FBQ2pDLE9BQU87UUFDUCxjQUFjO1FBQ2Qsa0JBQWtCLEVBQUUsQ0FBQztnQkFDbkIsYUFBYSxFQUFFLGNBQWM7Z0JBQzdCLE9BQU8sRUFBRSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQzthQUM1RCxDQUFDO0tBQ0gsQ0FBQyxDQUFDLENBQUM7SUFFSixNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUM7UUFDOUMsT0FBTztRQUNQLGNBQWM7UUFDZCxrQkFBa0IsRUFBRSxDQUFDO2dCQUNuQixhQUFhLEVBQUUsY0FBYztnQkFDN0IsT0FBTyxFQUFFLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDO2FBQzVELENBQUM7S0FDSCxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsQ0FBQztBQUNyQixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxtREFBbUQsRUFBRSxHQUFHLEVBQUU7SUFDN0QsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1FBQ3BDLE1BQU0sRUFBRSxDQUFDO1FBQ1QsbUJBQW1CLEVBQUUsQ0FBQztnQkFDcEIsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsUUFBUTtnQkFDbkMsSUFBSSxFQUFFLFVBQVU7YUFDakIsQ0FBQztLQUNILENBQUMsQ0FBQztJQUNILE1BQU0sT0FBTyxHQUFHLElBQUksR0FBRyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztJQUU5RCxNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDdkUsY0FBYyxDQUFDLFlBQVksQ0FBQyxjQUFjLEVBQUU7UUFDMUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQztLQUMvQyxDQUFDLENBQUM7SUFFSCxNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUMxQyxRQUFRLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDO0tBQ3BELENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQztRQUNqQyxPQUFPO1FBQ1AsY0FBYztRQUNkLFNBQVMsRUFBRSxDQUFDO1FBQ1osZUFBZSxFQUFFLEVBQUUsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsUUFBUSxFQUFFO1FBQ3hELGtCQUFrQixFQUFFLENBQUM7Z0JBQ25CLGFBQWEsRUFBRSxjQUFjO2dCQUM3QixPQUFPLEVBQUUsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDO2FBQ3pCLENBQUM7S0FDSCxDQUFDLENBQUMsQ0FBQztJQUVKLE9BQU87SUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CLEVBQUU7UUFDcEQsT0FBTyxFQUFFO1lBQ1A7Z0JBQ0UsR0FBRyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLEVBQUU7Z0JBQ3BELGFBQWEsRUFBRTtvQkFDYixTQUFTLEVBQUUsQ0FBQztvQkFDWixpQkFBaUIsRUFBRSxFQUFFLEdBQUcsRUFBRSxpQkFBaUIsRUFBRTtvQkFDN0MsVUFBVSxFQUFFLFNBQVM7b0JBQ3JCLG9CQUFvQixFQUFFO3dCQUNwQixtQkFBbUIsRUFBRTs0QkFDbkIsT0FBTyxFQUFFO2dDQUNQO29DQUNFLEdBQUcsRUFBRSxrQ0FBa0M7aUNBQ3hDOzZCQUNGOzRCQUNELGNBQWMsRUFBRSxVQUFVOzRCQUMxQixjQUFjLEVBQUU7Z0NBQ2Q7b0NBQ0UsWUFBWSxFQUFFO3dDQUNaLDhCQUE4Qjt3Q0FDOUIsU0FBUztxQ0FDVjtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjtnQkFDRCxLQUFLLEVBQUUsMkVBQTJFO2dCQUNsRixPQUFPLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQywyQkFBMkIsRUFBRSxLQUFLLENBQUMsRUFBRTtnQkFDL0QsRUFBRSxFQUFFLFNBQVM7YUFDZDtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGVjcyBmcm9tICdAYXdzLWNkay9hd3MtZWNzJztcbmltcG9ydCAqIGFzIGV2ZW50cyBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIHRhcmdldHMgZnJvbSAnLi4vLi4vbGliJztcblxudGVzdCgnQ2FuIHVzZSBFQzIgdGFza2RlZiBhcyBFdmVudFJ1bGUgdGFyZ2V0JywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgdnBjID0gbmV3IGVjMi5WcGMoc3RhY2ssICdWcGMnLCB7IG1heEF6czogMSB9KTtcbiAgY29uc3QgY2x1c3RlciA9IG5ldyBlY3MuQ2x1c3RlcihzdGFjaywgJ0Vjc0NsdXN0ZXInLCB7IHZwYyB9KTtcbiAgY2x1c3Rlci5hZGRDYXBhY2l0eSgnRGVmYXVsdEF1dG9TY2FsaW5nR3JvdXAnLCB7XG4gICAgaW5zdGFuY2VUeXBlOiBuZXcgZWMyLkluc3RhbmNlVHlwZSgndDIubWljcm8nKSxcbiAgfSk7XG5cbiAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnVGFza0RlZicpO1xuICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ1RoZUNvbnRhaW5lcicsIHtcbiAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnaGVuaycpLFxuICAgIG1lbW9yeUxpbWl0TWlCOiAyNTYsXG4gIH0pO1xuXG4gIGNvbnN0IHJ1bGUgPSBuZXcgZXZlbnRzLlJ1bGUoc3RhY2ssICdSdWxlJywge1xuICAgIHNjaGVkdWxlOiBldmVudHMuU2NoZWR1bGUuZXhwcmVzc2lvbigncmF0ZSgxIG1pbiknKSxcbiAgfSk7XG5cbiAgLy8gV0hFTlxuICBydWxlLmFkZFRhcmdldChuZXcgdGFyZ2V0cy5FY3NUYXNrKHtcbiAgICBjbHVzdGVyLFxuICAgIHRhc2tEZWZpbml0aW9uLFxuICAgIHRhc2tDb3VudDogMSxcbiAgICBjb250YWluZXJPdmVycmlkZXM6IFt7XG4gICAgICBjb250YWluZXJOYW1lOiAnVGhlQ29udGFpbmVyJyxcbiAgICAgIGNvbW1hbmQ6IFsnZWNobycsIGV2ZW50cy5FdmVudEZpZWxkLmZyb21QYXRoKCckLmRldGFpbC5ldmVudCcpXSxcbiAgICB9XSxcbiAgfSkpO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RXZlbnRzOjpSdWxlJywge1xuICAgIFRhcmdldHM6IFtcbiAgICAgIHtcbiAgICAgICAgQXJuOiB7ICdGbjo6R2V0QXR0JzogWydFY3NDbHVzdGVyOTcyNDJCODQnLCAnQXJuJ10gfSxcbiAgICAgICAgRWNzUGFyYW1ldGVyczoge1xuICAgICAgICAgIFRhc2tDb3VudDogMSxcbiAgICAgICAgICBUYXNrRGVmaW5pdGlvbkFybjogeyBSZWY6ICdUYXNrRGVmNTQ2OTQ1NzAnIH0sXG4gICAgICAgIH0sXG4gICAgICAgIElucHV0VHJhbnNmb3JtZXI6IHtcbiAgICAgICAgICBJbnB1dFBhdGhzTWFwOiB7XG4gICAgICAgICAgICAnZGV0YWlsLWV2ZW50JzogJyQuZGV0YWlsLmV2ZW50JyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIElucHV0VGVtcGxhdGU6ICd7XCJjb250YWluZXJPdmVycmlkZXNcIjpbe1wibmFtZVwiOlwiVGhlQ29udGFpbmVyXCIsXCJjb21tYW5kXCI6W1wiZWNob1wiLDxkZXRhaWwtZXZlbnQ+XX1dfScsXG4gICAgICAgIH0sXG4gICAgICAgIFJvbGVBcm46IHsgJ0ZuOjpHZXRBdHQnOiBbJ1Rhc2tEZWZFdmVudHNSb2xlRkIzQjY3QjgnLCAnQXJuJ10gfSxcbiAgICAgICAgSWQ6ICdUYXJnZXQwJyxcbiAgICAgIH0sXG4gICAgXSxcbiAgfSk7XG59KTtcblxudGVzdCgnQ2FuIHVzZSBGYXJnYXRlIHRhc2tkZWYgYXMgRXZlbnRSdWxlIHRhcmdldCcsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHZwYyA9IG5ldyBlYzIuVnBjKHN0YWNrLCAnVnBjJywgeyBtYXhBenM6IDEgfSk7XG4gIGNvbnN0IGNsdXN0ZXIgPSBuZXcgZWNzLkNsdXN0ZXIoc3RhY2ssICdFY3NDbHVzdGVyJywgeyB2cGMgfSk7XG5cbiAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkZhcmdhdGVUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnKTtcbiAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdUaGVDb250YWluZXInLCB7XG4gICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ2hlbmsnKSxcbiAgfSk7XG5cbiAgY29uc3QgcnVsZSA9IG5ldyBldmVudHMuUnVsZShzdGFjaywgJ1J1bGUnLCB7XG4gICAgc2NoZWR1bGU6IGV2ZW50cy5TY2hlZHVsZS5leHByZXNzaW9uKCdyYXRlKDEgbWluKScpLFxuICB9KTtcblxuICAvLyBXSEVOXG4gIHJ1bGUuYWRkVGFyZ2V0KG5ldyB0YXJnZXRzLkVjc1Rhc2soe1xuICAgIGNsdXN0ZXIsXG4gICAgdGFza0RlZmluaXRpb24sXG4gICAgdGFza0NvdW50OiAxLFxuICAgIGNvbnRhaW5lck92ZXJyaWRlczogW3tcbiAgICAgIGNvbnRhaW5lck5hbWU6ICdUaGVDb250YWluZXInLFxuICAgICAgY29tbWFuZDogWydlY2hvJywgZXZlbnRzLkV2ZW50RmllbGQuZnJvbVBhdGgoJyQuZGV0YWlsLmV2ZW50JyldLFxuICAgIH1dLFxuICB9KSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFdmVudHM6OlJ1bGUnLCB7XG4gICAgVGFyZ2V0czogW1xuICAgICAge1xuICAgICAgICBBcm46IHsgJ0ZuOjpHZXRBdHQnOiBbJ0Vjc0NsdXN0ZXI5NzI0MkI4NCcsICdBcm4nXSB9LFxuICAgICAgICBFY3NQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgVGFza0NvdW50OiAxLFxuICAgICAgICAgIFRhc2tEZWZpbml0aW9uQXJuOiB7IFJlZjogJ1Rhc2tEZWY1NDY5NDU3MCcgfSxcbiAgICAgICAgICBMYXVuY2hUeXBlOiAnRkFSR0FURScsXG4gICAgICAgICAgTmV0d29ya0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgIEF3c1ZwY0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgICAgU3VibmV0czogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIFJlZjogJ1ZwY1ByaXZhdGVTdWJuZXQxU3VibmV0NTM2Qjk5N0EnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIEFzc2lnblB1YmxpY0lwOiAnRElTQUJMRUQnLFxuICAgICAgICAgICAgICBTZWN1cml0eUdyb3VwczogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICAgICAnVGFza0RlZlNlY3VyaXR5R3JvdXBENTBFN0NGMCcsXG4gICAgICAgICAgICAgICAgICAgICdHcm91cElkJyxcbiAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgSW5wdXRUcmFuc2Zvcm1lcjoge1xuICAgICAgICAgIElucHV0UGF0aHNNYXA6IHtcbiAgICAgICAgICAgICdkZXRhaWwtZXZlbnQnOiAnJC5kZXRhaWwuZXZlbnQnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgSW5wdXRUZW1wbGF0ZTogJ3tcImNvbnRhaW5lck92ZXJyaWRlc1wiOlt7XCJuYW1lXCI6XCJUaGVDb250YWluZXJcIixcImNvbW1hbmRcIjpbXCJlY2hvXCIsPGRldGFpbC1ldmVudD5dfV19JyxcbiAgICAgICAgfSxcbiAgICAgICAgUm9sZUFybjogeyAnRm46OkdldEF0dCc6IFsnVGFza0RlZkV2ZW50c1JvbGVGQjNCNjdCOCcsICdBcm4nXSB9LFxuICAgICAgICBJZDogJ1RhcmdldDAnLFxuICAgICAgfSxcbiAgICBdLFxuICB9KTtcbn0pO1xuXG50ZXN0KCdDYW4gdXNlIHNhbWUgZmFyZ2F0ZSB0YXNrZGVmIHdpdGggbXVsdGlwbGUgcnVsZXMnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCB2cGMgPSBuZXcgZWMyLlZwYyhzdGFjaywgJ1ZwYycsIHsgbWF4QXpzOiAxIH0pO1xuICBjb25zdCBjbHVzdGVyID0gbmV3IGVjcy5DbHVzdGVyKHN0YWNrLCAnRWNzQ2x1c3RlcicsIHsgdnBjIH0pO1xuXG4gIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5GYXJnYXRlVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG4gIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignVGhlQ29udGFpbmVyJywge1xuICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCdoZW5rJyksXG4gIH0pO1xuXG4gIGNvbnN0IHNjaGVkdWxlZFJ1bGUgPSBuZXcgZXZlbnRzLlJ1bGUoc3RhY2ssICdTY2hlZHVsZVJ1bGUnLCB7XG4gICAgc2NoZWR1bGU6IGV2ZW50cy5TY2hlZHVsZS5leHByZXNzaW9uKCdyYXRlKDEgbWluKScpLFxuICB9KTtcblxuICBjb25zdCBwYXR0ZXJuUnVsZSA9IG5ldyBldmVudHMuUnVsZShzdGFjaywgJ1BhdHRlcm5SdWxlJywge1xuICAgIGV2ZW50UGF0dGVybjoge1xuICAgICAgZGV0YWlsOiBbJ3Rlc3QnXSxcbiAgICB9LFxuICB9KTtcblxuICBzY2hlZHVsZWRSdWxlLmFkZFRhcmdldChuZXcgdGFyZ2V0cy5FY3NUYXNrKHtcbiAgICBjbHVzdGVyLFxuICAgIHRhc2tEZWZpbml0aW9uLFxuICB9KSk7XG5cbiAgZXhwZWN0KCgpID0+IHBhdHRlcm5SdWxlLmFkZFRhcmdldChuZXcgdGFyZ2V0cy5FY3NUYXNrKHtcbiAgICBjbHVzdGVyLFxuICAgIHRhc2tEZWZpbml0aW9uLFxuICB9KSkpLm5vdC50b1Rocm93KCk7XG59KTtcblxudGVzdCgnQ2FuIHVzZSBzYW1lIGZhcmdhdGUgdGFza2RlZiBtdWx0aXBsZSB0aW1lcyBpbiBhIHJ1bGUnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCB2cGMgPSBuZXcgZWMyLlZwYyhzdGFjaywgJ1ZwYycsIHsgbWF4QXpzOiAxIH0pO1xuICBjb25zdCBjbHVzdGVyID0gbmV3IGVjcy5DbHVzdGVyKHN0YWNrLCAnRWNzQ2x1c3RlcicsIHsgdnBjIH0pO1xuXG4gIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5GYXJnYXRlVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG4gIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignVGhlQ29udGFpbmVyJywge1xuICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCdoZW5rJyksXG4gIH0pO1xuXG4gIGNvbnN0IHJ1bGUgPSBuZXcgZXZlbnRzLlJ1bGUoc3RhY2ssICdTY2hlZHVsZVJ1bGUnLCB7XG4gICAgc2NoZWR1bGU6IGV2ZW50cy5TY2hlZHVsZS5leHByZXNzaW9uKCdyYXRlKDEgbWluKScpLFxuICB9KTtcblxuICBydWxlLmFkZFRhcmdldChuZXcgdGFyZ2V0cy5FY3NUYXNrKHtcbiAgICBjbHVzdGVyLFxuICAgIHRhc2tEZWZpbml0aW9uLFxuICAgIGNvbnRhaW5lck92ZXJyaWRlczogW3tcbiAgICAgIGNvbnRhaW5lck5hbWU6ICdUaGVDb250YWluZXInLFxuICAgICAgY29tbWFuZDogWydlY2hvJywgZXZlbnRzLkV2ZW50RmllbGQuZnJvbVBhdGgoJyQuZGV0YWlsLmEnKV0sXG4gICAgfV0sXG4gIH0pKTtcblxuICBleHBlY3QoKCkgPT4gcnVsZS5hZGRUYXJnZXQobmV3IHRhcmdldHMuRWNzVGFzayh7XG4gICAgY2x1c3RlcixcbiAgICB0YXNrRGVmaW5pdGlvbixcbiAgICBjb250YWluZXJPdmVycmlkZXM6IFt7XG4gICAgICBjb250YWluZXJOYW1lOiAnVGhlQ29udGFpbmVyJyxcbiAgICAgIGNvbW1hbmQ6IFsnZWNobycsIGV2ZW50cy5FdmVudEZpZWxkLmZyb21QYXRoKCckLmRldGFpbC5iJyldLFxuICAgIH1dLFxuICB9KSkpLm5vdC50b1Rocm93KCk7XG59KTtcblxudGVzdCgnSXNvbGF0ZWQgc3VibmV0IGRvZXMgbm90IGhhdmUgQXNzaWduUHVibGljSXA9dHJ1ZScsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHZwYyA9IG5ldyBlYzIuVnBjKHN0YWNrLCAnVnBjJywge1xuICAgIG1heEF6czogMSxcbiAgICBzdWJuZXRDb25maWd1cmF0aW9uOiBbe1xuICAgICAgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuSVNPTEFURUQsXG4gICAgICBuYW1lOiAnSXNvbGF0ZWQnLFxuICAgIH1dLFxuICB9KTtcbiAgY29uc3QgY2x1c3RlciA9IG5ldyBlY3MuQ2x1c3RlcihzdGFjaywgJ0Vjc0NsdXN0ZXInLCB7IHZwYyB9KTtcblxuICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRmFyZ2F0ZVRhc2tEZWZpbml0aW9uKHN0YWNrLCAnVGFza0RlZicpO1xuICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ1RoZUNvbnRhaW5lcicsIHtcbiAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnaGVuaycpLFxuICB9KTtcblxuICBjb25zdCBydWxlID0gbmV3IGV2ZW50cy5SdWxlKHN0YWNrLCAnUnVsZScsIHtcbiAgICBzY2hlZHVsZTogZXZlbnRzLlNjaGVkdWxlLmV4cHJlc3Npb24oJ3JhdGUoMSBtaW4pJyksXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgcnVsZS5hZGRUYXJnZXQobmV3IHRhcmdldHMuRWNzVGFzayh7XG4gICAgY2x1c3RlcixcbiAgICB0YXNrRGVmaW5pdGlvbixcbiAgICB0YXNrQ291bnQ6IDEsXG4gICAgc3VibmV0U2VsZWN0aW9uOiB7IHN1Ym5ldFR5cGU6IGVjMi5TdWJuZXRUeXBlLklTT0xBVEVEIH0sXG4gICAgY29udGFpbmVyT3ZlcnJpZGVzOiBbe1xuICAgICAgY29udGFpbmVyTmFtZTogJ1RoZUNvbnRhaW5lcicsXG4gICAgICBjb21tYW5kOiBbJ2VjaG8nLCAneWF5J10sXG4gICAgfV0sXG4gIH0pKTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkV2ZW50czo6UnVsZScsIHtcbiAgICBUYXJnZXRzOiBbXG4gICAgICB7XG4gICAgICAgIEFybjogeyAnRm46OkdldEF0dCc6IFsnRWNzQ2x1c3Rlcjk3MjQyQjg0JywgJ0FybiddIH0sXG4gICAgICAgIEVjc1BhcmFtZXRlcnM6IHtcbiAgICAgICAgICBUYXNrQ291bnQ6IDEsXG4gICAgICAgICAgVGFza0RlZmluaXRpb25Bcm46IHsgUmVmOiAnVGFza0RlZjU0Njk0NTcwJyB9LFxuICAgICAgICAgIExhdW5jaFR5cGU6ICdGQVJHQVRFJyxcbiAgICAgICAgICBOZXR3b3JrQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgQXdzVnBjQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgICBTdWJuZXRzOiBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgUmVmOiAnVnBjSXNvbGF0ZWRTdWJuZXQxU3VibmV0RTQ4QzU3MzcnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIEFzc2lnblB1YmxpY0lwOiAnRElTQUJMRUQnLFxuICAgICAgICAgICAgICBTZWN1cml0eUdyb3VwczogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICAgICAnVGFza0RlZlNlY3VyaXR5R3JvdXBENTBFN0NGMCcsXG4gICAgICAgICAgICAgICAgICAgICdHcm91cElkJyxcbiAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgSW5wdXQ6ICd7XCJjb250YWluZXJPdmVycmlkZXNcIjpbe1wibmFtZVwiOlwiVGhlQ29udGFpbmVyXCIsXCJjb21tYW5kXCI6W1wiZWNob1wiLFwieWF5XCJdfV19JyxcbiAgICAgICAgUm9sZUFybjogeyAnRm46OkdldEF0dCc6IFsnVGFza0RlZkV2ZW50c1JvbGVGQjNCNjdCOCcsICdBcm4nXSB9LFxuICAgICAgICBJZDogJ1RhcmdldDAnLFxuICAgICAgfSxcbiAgICBdLFxuICB9KTtcbn0pO1xuIl19