"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
/**
 * Use an SQS Queue as a target for AWS CloudWatch event rules.
 *
 * @example
 *
 *    // publish to an SQS queue every time code is committed
 *    // to a CodeCommit repository
 *    repository.onCommit(new targets.SqsQueue(queue));
 *
 */
class SqsQueue {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        if (props.messageGroupId !== undefined && !queue.fifo) {
            throw new Error('messageGroupId cannot be specified for non-FIFO queues');
        }
    }
    /**
     * Returns a RuleTarget that can be used to trigger this SQS queue as a
     * result from a CloudWatch event.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/resource-based-policies-cwe.html#sqs-permissions
     */
    bind(rule, _id) {
        // deduplicated automatically
        this.queue.grantSendMessages(new iam.ServicePrincipal('events.amazonaws.com', {
            conditions: {
                ArnEquals: { 'aws:SourceArn': rule.ruleArn },
            },
        }));
        return {
            id: '',
            arn: this.queue.queueArn,
            input: this.props.message,
            targetResource: this.queue,
            sqsParameters: this.props.messageGroupId ? { messageGroupId: this.props.messageGroupId } : undefined,
        };
    }
}
exports.SqsQueue = SqsQueue;
//# sourceMappingURL=data:application/json;base64,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