"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InputValidator = void 0;
const cdk = require("@aws-cdk/core");
/**
 * Class to validate that inputs match requirements.
 */
class InputValidator {
    /**
     * Validates length is between allowed min and max lengths.
     */
    static validateLength(resourceName, inputName, minLength, maxLength, inputString) {
        if (!cdk.Token.isUnresolved(inputString) && inputString !== undefined && (inputString.length < minLength || inputString.length > maxLength)) {
            throw new Error(`Invalid ${inputName} for resource ${resourceName}, must have length between ${minLength} and ${maxLength}, got: '${this.truncateString(inputString, 100)}'`);
        }
    }
    /**
     * Validates string matches the allowed regex pattern.
     */
    static validateRegex(resourceName, inputName, regexp, inputString) {
        if (!cdk.Token.isUnresolved(inputString) && inputString !== undefined && !regexp.test(inputString)) {
            throw new Error(`Invalid ${inputName} for resource ${resourceName}, must match regex pattern ${regexp}, got: '${this.truncateString(inputString, 100)}'`);
        }
    }
    /**
     * Validates string matches the valid URL regex pattern.
     */
    static validateUrl(resourceName, inputName, inputString) {
        this.validateRegex(resourceName, inputName, /^https?:\/\/.*/, inputString);
    }
    /**
    * Validates string matches the valid email regex pattern.
    */
    static validateEmail(resourceName, inputName, inputString) {
        this.validateRegex(resourceName, inputName, /^[\w\d.%+\-]+@[a-z\d.\-]+\.[a-z]{2,4}$/i, inputString);
    }
    /**
    * Validates that a role being used as a local launch role has the role name set
    */
    static validateRoleNameSetForLocalLaunchRole(role) {
        if (role.node.defaultChild) {
            if (cdk.Token.isUnresolved(role.node.defaultChild.roleName)) {
                throw new Error(`Role ${role.node.id} used for Local Launch Role must have roleName explicitly set`);
            }
        }
    }
    static truncateString(string, maxLength) {
        if (string.length > maxLength) {
            return string.substring(0, maxLength) + '[truncated]';
        }
        return string;
    }
}
exports.InputValidator = InputValidator;
//# sourceMappingURL=data:application/json;base64,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