"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const codebuild_1 = require("./rules/codebuild");
const dms_1 = require("./rules/dms");
const dynamodb_1 = require("./rules/dynamodb");
const ec2_1 = require("./rules/ec2");
const efs_1 = require("./rules/efs");
const elasticache_1 = require("./rules/elasticache");
const elasticsearch_1 = require("./rules/elasticsearch");
const elb_1 = require("./rules/elb");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const lambda_1 = require("./rules/lambda");
const rds_1 = require("./rules/rds");
const redshift_1 = require("./rules/redshift");
const s3_1 = require("./rules/s3");
const sagemaker_1 = require("./rules/sagemaker");
const sns_1 = require("./rules/sns");
/**
 * Check for NIST 800-53 compliance.
 *
 * Based on the NIST 800-53 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_4.html
 *
 * @stability stable
 */
class NIST80053Checks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkAPIGW(node, ignores);
            this.checkAutoScaling(node, ignores);
            this.checkCloudTrail(node, ignores);
            this.checkCloudWatch(node, ignores);
            this.checkCodeBuild(node, ignores);
            this.checkDMS(node, ignores);
            this.checkDynamoDB(node, ignores);
            this.checkEC2(node, ignores);
            this.checkEFS(node, ignores);
            this.checkElastiCache(node, ignores);
            this.checkElasticsearch(node, ignores);
            this.checkELB(node, ignores);
            this.checkEMR(node, ignores);
            this.checkIAM(node, ignores);
            this.checkLambda(node, ignores);
            this.checkRDS(node, ignores);
            this.checkRedshift(node, ignores);
            this.checkS3(node, ignores);
            this.checkSageMaker(node, ignores);
            this.checkSNS(node, ignores);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-APIGWCacheEnabledAndEncrypted') &&
            !apigw_1.nist80053APIGWCacheEnabledAndEncrypted(node)) {
            const ruleId = 'NIST.800.53-APIGWCacheEnabledAndEncrypted';
            const info = 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: SC-13, SC-28).';
            const explanation = "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-APIGWExecutionLoggingEnabled') &&
            !apigw_1.nist80053APIGWExecutionLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-APIGWExecutionLoggingEnabled';
            const info = 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c)).';
            const explanation = 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Auto Scaling Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-AutoScalingHealthChecks') &&
            !autoscaling_1.nist80053AutoScalingHealthChecks(node)) {
            const ruleId = 'NIST.800.53-AutoScalingHealthChecks';
            const info = 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control IDs: SC-5).';
            const explanation = 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CloudTrail Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudTrailCloudWatchLogsEnabled') &&
            !cloudtrail_1.nist80053CloudTrailCloudWatchLogsEnabled(node)) {
            const ruleId = 'NIST.800.53-CloudTrailCloudWatchLogsEnabled';
            const info = 'The trail does not have CloudWatch logs enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-6(1)(3), AU-7(1), AU-12(a)(c), CA-7(a)(b), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).';
            const explanation = 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudTrailEncryptionEnabled') &&
            !cloudtrail_1.nist80053CloudTrailEncryptionEnabled(node)) {
            const ruleId = 'NIST.800.53-CloudTrailEncryptionEnabled';
            const info = 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: AU-9).';
            const explanation = 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudTrailLogFileValidationEnabled') &&
            !cloudtrail_1.nist80053CloudTrailLogFileValidationEnabled(node)) {
            const ruleId = 'NIST.800.53-CloudTrailLogFileValidationEnabled';
            const info = 'The trail does not have log file validation enabled - (Control ID: AC-6).';
            const explanation = 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CloudWatch Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudWatchAlarmAction') &&
            !cloudwatch_1.nist80053CloudWatchAlarmAction(node)) {
            const ruleId = 'NIST.800.53-CloudWatchAlarmAction';
            const info = 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control IDs: AC-2(4), AU-6(1)(3), AU-7(1), CA-7(a)(b), IR-4(1), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).';
            const explanation = 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudWatchLogGroupEncrypted') &&
            !cloudwatch_1.nist80053CloudWatchLogGroupEncrypted(node)) {
            const ruleId = 'NIST.800.53-CloudWatchLogGroupEncrypted';
            const info = 'The CloudWatch Log Group is not encrypted with a KMS Customer Master Key (CMK) - (Control IDs: AU-9, SC-13, SC-28).';
            const explanation = 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CodeBuild Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-CodeBuildCheckEnvVars') &&
            !codebuild_1.nist80053CodeBuildCheckEnvVars(node)) {
            const ruleId = 'NIST.800.53-CodeBuildCheckEnvVars';
            const info = 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: AC-6, IA-5(7), SA-3(a)).';
            const explanation = 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CodeBuildURLCheck') &&
            !codebuild_1.nist80053CodeBuildURLCheck(node)) {
            const ruleId = 'NIST.800.53-CodeBuildURLCheck';
            const info = 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control IDs: SA-3(a).';
            const explanation = 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check DMS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDMS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-DMSReplicationNotPublic') &&
            !dms_1.nist80053DMSReplicationNotPublic(node)) {
            const ruleId = 'NIST.800.53-DMSReplicationNotPublic';
            const info = 'The DMS replication instance is public - (Control IDs: AC-3).';
            const explanation = 'DMS replication instances can contain sensitive information and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check DynamoDB Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-DynamoDBPITREnabled') &&
            !dynamodb_1.nist80053DynamoDBPITREnabled(node)) {
            const ruleId = 'NIST.800.53-DynamoDBPITREnabled';
            const info = 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: CP-9(b), CP-10, SI-12).';
            const explanation = 'The recovery maintains continuous backups of your table for the last 35 days.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EC2 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckDefaultSecurityGroupClosed') &&
            !ec2_1.nist80053EC2CheckDefaultSecurityGroupClosed(node)) {
            const ruleId = 'NIST.800.53-EC2CheckDefaultSecurityGroupClosed';
            const info = "The VPC's default security group allows inbound or outbound traffic - (Control IDs: AC-4, SC-7, SC-7(3)).";
            const explanation = 'When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckDetailedMonitoring') &&
            !ec2_1.nist80053EC2CheckDetailedMonitoring(node)) {
            const ruleId = 'NIST.800.53-EC2CheckDetailedMonitoring';
            const info = 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: CA-7(a)(b), SI-4(2), SI-4(a)(b)(c)).';
            const explanation = 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckInsideVPC') &&
            !ec2_1.nist80053EC2CheckInsideVPC(node)) {
            const ruleId = 'NIST.800.53-EC2CheckInsideVPC';
            const info = 'The EC2 instance is not within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckNoPublicIPs') &&
            !ec2_1.nist80053EC2CheckNoPublicIPs(node)) {
            const ruleId = 'NIST.800.53-EC2CheckNoPublicIPs';
            const info = 'The EC2 instance is associated with a public IP address - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)). ';
            const explanation = 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckSSHRestricted') &&
            !ec2_1.nist80053EC2CheckSSHRestricted(node)) {
            const ruleId = 'NIST.800.53-EC2CheckSSHRestricted';
            const info = 'The Security Group allows unrestricted SSH access - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckCommonPortsRestricted') &&
            !ec2_1.nist80053EC2CheckCommonPortsRestricted(node)) {
            const ruleId = 'NIST.800.53-EC2CheckCommonPortsRestricted';
            const info = 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on common ports (20, 21, 3389, 3306, 4333) - (Control IDs: AC-4, CM-2, SC-7, SC-7(3)).';
            const explanation = 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EFS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEFS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EFSEncrypted') &&
            !efs_1.nist80053EFSEncrypted(node)) {
            const ruleId = 'NIST.800.53-EFSEncrypted';
            const info = 'The EFS does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check ElastiCache Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-ElastiCacheRedisClusterAutomaticBackup') &&
            !elasticache_1.nist80053ElastiCacheRedisClusterAutomaticBackup(node)) {
            const ruleId = 'NIST.800.53-ElastiCacheRedisClusterAutomaticBackup';
            const info = 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: CP-9(b), CP-10, SI-12).';
            const explanation = 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Elasticsearch Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkElasticsearch(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-ElasticsearchEncryptedAtRest') &&
            !elasticsearch_1.nist80053ElasticsearchEncryptedAtRest(node)) {
            const ruleId = 'NIST.800.53-ElasticsearchEncryptedAtRest';
            const info = 'The Elasticsearch domain does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elasticsearch Service (Amazon ES) domains.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ElasticsearchNodeToNodeEncrypted') &&
            !elasticsearch_1.nist80053ElasticsearchNodeToNodeEncrypted(node)) {
            const ruleId = 'NIST.800.53-ElasticsearchNodeToNodeEncrypted';
            const info = 'The Elasticsearch domain does not have node-to-node encryption enabled - (Control IDs: SC-7, SC-8, SC-8(1)).';
            const explanation = 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon Elasticsearch Service (Amazon ES) domains.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ElasticsearchRunningWithinVPC') &&
            !elasticsearch_1.nist80053ElasticsearchRunningWithinVPC(node)) {
            const ruleId = 'NIST.800.53-ElasticsearchRunningWithinVPC';
            const info = 'The Elasticsearch domain is not running within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'VPCs help secure your AWS resources and provide an extra layer of protection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkELB(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-ALBHttpDropInvalidHeaderEnabled') &&
            !elb_1.nist80053ALBHttpDropInvalidHeaderEnabled(node)) {
            const ruleId = 'NIST.800.53-ALBHttpDropInvalidHeaderEnabled';
            const info = 'The ALB does not have invalid http header dropping enabled - (Control ID: AC-17(2)).';
            const explanation = 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ALBHttpToHttpsRedirection') &&
            !elb_1.nist80053ALBHttpToHttpsRedirection(node)) {
            const ruleId = 'NIST.800.53-ALBHttpToHttpsRedirection';
            const info = 'The ALB\'s Http listeners are not configured to redirect to https - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13, SC-23).';
            const explanation = 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBCrossZoneBalancing') &&
            !elb_1.nist80053ELBCrossZoneBalancing(node)) {
            const ruleId = 'NIST.800.53-ELBCrossZoneBalancing';
            const info = 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: SC-5, CP-10).';
            const explanation = 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBDeletionProtectionEnabled') &&
            !elb_1.nist80053ELBDeletionProtectionEnabled(node)) {
            const ruleId = 'NIST.800.53-ELBDeletionProtectionEnabled';
            const info = 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: CM-2, CP-10).';
            const explanation = 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBListenersUseSSLOrHTTPS') &&
            !elb_1.nist80053ELBListenersUseSSLOrHTTPS(node)) {
            const ruleId = 'NIST.800.53-ELBListenersUseSSLOrHTTPS';
            const info = 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-23).';
            const explanation = 'Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBLoggingEnabled') &&
            !elb_1.nist80053ELBLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-ELBLoggingEnabled';
            const info = 'The ELB does not have logging enabled - (Control ID: AU-2(a)(d)).';
            const explanation = "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBUseACMCerts') &&
            !elb_1.nist80053ELBUseACMCerts(node)) {
            const ruleId = 'NIST.800.53-ELBUseACMCerts';
            const info = 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).';
            const explanation = 'Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EMR Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEMR(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EMRKerberosEnabled') &&
            !emr_1.nist80053EMRKerberosEnabled(node)) {
            const ruleId = 'NIST.800.53-EMRKerberosEnabled';
            const info = 'The EMR cluster does not have Kerberos enabled - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).';
            const explanation = 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check IAM Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkIAM(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMGroupMembershipCheck') &&
            !iam_1.nist80053IamGroupMembership(node)) {
            const ruleId = 'NIST.800.53-IAMGroupMembershipCheck';
            const info = 'The IAM user does not belong to any group(s) - (Control IDs: AC-2(1), AC-2(j), AC-3, and AC-6).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMNoInlinePolicyCheck') &&
            !iam_1.nist80053IamNoInlinePolicy(node)) {
            const ruleId = 'NIST.800.53-IAMNoInlinePolicyCheck';
            const info = 'The IAM Group, User, or Role contains an inline policy - (Control ID: AC-6).';
            const explanation = 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess') &&
            !iam_1.nist80053IamPolicyNoStatementsWithAdminAccess(node)) {
            const ruleId = 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess';
            const info = 'The IAM policy grants admin access - (Control IDs AC-2(1), AC-2(j), AC-3, AC-6).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMUserNoPoliciesCheck') &&
            !iam_1.nist80053IamUserNoPolicies(node)) {
            const ruleId = 'NIST.800.53-IAMUserNoPoliciesCheck';
            const info = 'The IAM policy is attached at the user level - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).';
            const explanation = 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Lambda Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkLambda(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-LambdaFunctionsInsideVPC') &&
            !lambda_1.nist80053LambdaFunctionsInsideVPC(node)) {
            const ruleId = 'NIST.800.53-LambdaFunctionsInsideVPC';
            const info = 'The Lambda function is not defined within within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon RDS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkRDS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSEnhancedMonitoringEnabled') &&
            !rds_1.nist80053RDSEnhancedMonitoringEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSEnhancedMonitoringEnabled';
            const info = 'The RDS DB Instance does not enhanced monitoring enabled - (Control ID: CA-7(a)(b)).';
            const explanation = 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstanceBackupEnabled') &&
            !rds_1.nist80053RDSInstanceBackupEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSInstanceBackupEnabled';
            const info = 'The RDS DB Instance does not have backups enabled - (Control IDs: CP-9(b), CP-10, SI-12).';
            const explanation = 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstanceDeletionProtectionEnabled') &&
            !rds_1.nist80053RDSInstanceDeletionProtectionEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSInstanceDeletionProtectionEnabled';
            const info = 'The RDS DB Instance or Aurora Cluster does not have deletion protection enabled - (Control ID: SC-5).';
            const explanation = 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstanceMultiAzSupport') &&
            !rds_1.nist80053RDSInstanceMultiAZSupport(node)) {
            const ruleId = 'NIST.800.53-RDSInstanceMultiAzSupport';
            const info = 'The RDS DB Instance does not have multi-AZ support - (Control IDs: CP-10, SC-5, SC-36).';
            const explanation = 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstancePublicAccess') &&
            !rds_1.nist80053RDSInstancePublicAccess(node)) {
            const ruleId = 'NIST.800.53-RDSInstancePublicAccess';
            const info = 'The RDS DB Instance allows public access - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSLoggingEnabled') &&
            !rds_1.nist80053RDSLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSLoggingEnabled';
            const info = 'The RDS DB Instance does not have all CloudWatch log types exported - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).';
            const explanation = 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSStorageEncrypted') &&
            !rds_1.nist80053RDSStorageEncrypted(node)) {
            const ruleId = 'NIST.800.53-RDSStorageEncrypted';
            const info = 'The RDS DB Instance or Aurora Cluster does not have storage encrypted - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in Amazon RDS instances and clusters, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Redshift Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-RedshiftClusterConfiguration') &&
            !redshift_1.nist80053RedshiftClusterConfiguration(node)) {
            const ruleId = 'NIST.800.53-RedshiftClusterConfiguration';
            const info = 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c), SC-13).';
            const explanation = 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RedshiftClusterPublicAccess') &&
            !redshift_1.nist80053RedshiftClusterPublicAccess(node)) {
            const ruleId = 'NIST.800.53-RedshiftClusterPublicAccess';
            const info = 'The Redshift cluster allows public access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon S3 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkS3(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-S3BucketLoggingEnabled') &&
            !s3_1.nist80053S3BucketLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-S3BucketLoggingEnabled';
            const info = 'The S3 Bucket does not have server access logs enabled - (Control IDs: AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).';
            const explanation = 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check SageMaker Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-SageMakerEndpointKMS') &&
            !sagemaker_1.nist80053SageMakerEndpointKMS(node)) {
            const ruleId = 'NIST.800.53-SageMakerEndpointKMS';
            const info = 'The SageMaker endpoint is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-SageMakerNotebookDirectInternetAccessDisabled') &&
            !sagemaker_1.nist80053SageMakerNotebookDirectInternetAccessDisabled(node)) {
            const ruleId = 'NIST.800.53-SageMakerNotebookDirectInternetAccessDisbabled';
            const info = 'The SageMaker notebook does not disable direct internet access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-SageMakerNotebookKMS') &&
            !sagemaker_1.nist80053SageMakerNotebookKMS(node)) {
            const ruleId = 'NIST.800.53-SageMakerNotebookKMS';
            const info = 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon SNS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSNS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-SNSEncryptedKMS') &&
            !sns_1.nist80053SNSEncryptedKMS(node)) {
            const ruleId = 'NIST.800.53-SNSEncryptedKMS';
            const info = 'The SNS topic does not have KMS encryption enabled - (Control ID: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.NIST80053Checks = NIST80053Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053Checks[_a] = { fqn: "monocdk-nag.NIST80053Checks", version: "0.0.56" };
//# sourceMappingURL=data:application/json;base64,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