# coding: utf-8

# (C) Copyright IBM Corp. 2020.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
The IBM Cloud Virtual Private Cloud (VPC) API can be used to programmatically provision
and manage infrastructure resources, including virtual server instances, subnets, volumes,
and load balancers.
"""

from datetime import datetime
from enum import Enum
from typing import Dict, List
import base64
import json
import sys

from ibm_cloud_sdk_core import BaseService, DetailedResponse
from ibm_cloud_sdk_core.authenticators.authenticator import Authenticator
from ibm_cloud_sdk_core.get_authenticator import get_authenticator_from_environment
from ibm_cloud_sdk_core.utils import convert_model, datetime_to_string, string_to_datetime

from .common import get_sdk_headers

##############################################################################
# Service
##############################################################################


class VpcClassicV1(BaseService):
    """The vpc_classic V1 service."""

    DEFAULT_SERVICE_URL = 'https://us-south.iaas.cloud.ibm.com/v1'
    DEFAULT_SERVICE_NAME = 'vpc_classic'

    @classmethod
    def new_instance(
        cls,
        version: str = '2020-06-02',
        service_name: str = DEFAULT_SERVICE_NAME,
        generation: int = 1,
    ) -> 'VpcClassicV1':
        """
        Return a new client for the vpc_classic service using the specified
               parameters and external configuration.

        :param str version: Requests the version of the API as of a date in the
               format `YYYY-MM-DD`. Any date up to the current date may be provided.
               Specify the current date to request the latest version.
        """

        authenticator = get_authenticator_from_environment(service_name)
        service = cls(
            version,
            authenticator,
            generation,
        )
        service.configure_service(service_name)
        return service

    def __init__(
        self,
        version: str = '2020-06-02',
        authenticator: Authenticator = None,
        generation: int = 1,
    ) -> None:
        """
        Construct a new client for the vpc_classic service.

        :param str version: Requests the version of the API as of a date in the
               format `YYYY-MM-DD`. Any date up to the current date may be provided.
               Specify the current date to request the latest version.

        :param Authenticator authenticator: The authenticator specifies the authentication mechanism.
               Get up to date information from https://github.com/IBM/python-sdk-core/blob/master/README.md
               about initializing the authenticator of your choice.
        """
        if version is None:
            raise ValueError('version must be provided')

        BaseService.__init__(self,
                             service_url=self.DEFAULT_SERVICE_URL,
                             authenticator=authenticator)
        self.version = version
        self.generation = generation

    #########################
    # floatingIPs
    #########################

    def list_floating_ips(self,
                          *,
                          start: str = None,
                          limit: int = None,
                          **kwargs) -> DetailedResponse:
        """
        List all floating IPs.

        This request retrieves all floating IPs in the region. Floating IPs allow inbound
        and outbound traffic from the Internet to an instance.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIPCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_floating_ips')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/floating_ips'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_floating_ip(self, floating_ip_prototype: 'FloatingIPPrototype',
                           **kwargs) -> DetailedResponse:
        """
        Reserve a floating IP.

        This request reserves a new floating IP.

        :param FloatingIPPrototype floating_ip_prototype: The floating IP prototype
               object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if floating_ip_prototype is None:
            raise ValueError('floating_ip_prototype must be provided')
        if isinstance(floating_ip_prototype, FloatingIPPrototype):
            floating_ip_prototype = convert_model(floating_ip_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_floating_ip')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(floating_ip_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/floating_ips'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_floating_ip(self, id: str, **kwargs) -> DetailedResponse:
        """
        Release the specified floating IP.

        This request disassociates (if associated) and releases a floating IP. This
        operation cannot be reversed. For this request to succeed, the floating IP must
        not be required by another resource, such as a public gateway.

        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_floating_ip')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/floating_ips/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_floating_ip(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve the specified floating IP.

        This request retrieves a single floating IP specified by the identifier in the
        URL.

        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_floating_ip')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/floating_ips/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_floating_ip(self,
                           id: str,
                           *,
                           name: str = None,
                           target: 'NetworkInterfaceIdentity' = None,
                           **kwargs) -> DetailedResponse:
        """
        Update the specified floating IP.

        This request updates a floating IP's name and/or target.

        :param str id: The floating IP identifier.
        :param str name: (optional) The unique user-defined name for this floating
               IP.
        :param NetworkInterfaceIdentity target: (optional) A new target to bind
               this floating IP with, replacing any existing binding.
               For this request to succeed, the existing floating IP must not be required
               by another
               resource, such as a public gateway.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if id is None:
            raise ValueError('id must be provided')
        if target is not None:
            target = convert_model(target)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_floating_ip')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name, 'target': target}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/floating_ips/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    #########################
    # geography
    #########################

    def list_regions(self, **kwargs) -> DetailedResponse:
        """
        List all regions.

        This request lists all regions. Each region is a separate geographic area that
        contains multiple isolated zones. Resources can be provisioned into a one or more
        zones in a region. Each zone is isolated, but connected to other zones in the same
        region with low-latency and high-bandwidth links. Regions represent the top-level
        of fault isolation available. Resources deployed within a single region also
        benefit from the low latency afforded by geographic proximity.

        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RegionCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_regions')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/regions'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_region(self, name: str, **kwargs) -> DetailedResponse:
        """
        Retrieve a region.

        This request retrieves a single region specified by the name in the URL.

        :param str name: The region name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Region` object
        """

        if name is None:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_region')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/regions/{0}'.format(*self.encode_path_vars(name))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_region_zones(self, region_name: str, **kwargs) -> DetailedResponse:
        """
        List all zones in a region.

        This request lists all zones in a region. Zones represent logically-isolated data
        centers with high-bandwidth and low-latency interconnects to other zones in the
        same region. Faults in a zone do not affect other zones.

        :param str region_name: The region name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ZoneCollection` object
        """

        if region_name is None:
            raise ValueError('region_name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_region_zones')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/regions/{0}/zones'.format(*self.encode_path_vars(region_name))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_region_zone(self, region_name: str, zone_name: str,
                        **kwargs) -> DetailedResponse:
        """
        Retrieve a zone.

        This request retrieves a single zone specified by the region and zone names in the
        URL.

        :param str region_name: The region name.
        :param str zone_name: The zone name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Zone` object
        """

        if region_name is None:
            raise ValueError('region_name must be provided')
        if zone_name is None:
            raise ValueError('zone_name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_region_zone')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/regions/{0}/zones/{1}'.format(
            *self.encode_path_vars(region_name, zone_name))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    #########################
    # images
    #########################

    def list_images(self,
                    *,
                    start: str = None,
                    limit: int = None,
                    resource_group_id: str = None,
                    name: str = None,
                    visibility: str = None,
                    **kwargs) -> DetailedResponse:
        """
        List all images.

        This request lists all images available in the region. An image provides source
        data for a volume. Images are either system-provided, or created from another
        source, such as importing from object storage.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources within one of the resource groups identified in a comma-separated
               list of resource group identifiers.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str visibility: (optional) Filters the collection to images with the
               specified visibility.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ImageCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_images')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'name': name,
            'visibility': visibility
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/images'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_image(self, image_prototype: 'ImagePrototype',
                     **kwargs) -> DetailedResponse:
        """
        Create an image.

        This request creates a new image from an image prototype object. The prototype
        object is structured in the same way as a retrieved image, and contains the
        information necessary to create the new image. A URL to the image file on object
        storage must be provided.

        :param ImagePrototype image_prototype: The image prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Image` object
        """

        if image_prototype is None:
            raise ValueError('image_prototype must be provided')
        if isinstance(image_prototype, ImagePrototype):
            image_prototype = convert_model(image_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_image')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(image_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/images'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_image(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete specified image.

        This request deletes an image. This operation cannot be reversed. System-provided
        images are not allowed to be deleted. An image with a `status` of `pending`,
        `tentative`, or `deleting` cannot be deleted.

        :param str id: The image identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_image')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/images/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_image(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve the specified image.

        This request retrieves a single image specified by the identifier in the URL.

        :param str id: The image identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Image` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_image')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/images/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_image(self,
                     id: str,
                     *,
                     name: str = None,
                     **kwargs) -> DetailedResponse:
        """
        Update specified image.

        This request updates an image with the information in a provided image patch. The
        image patch object is structured in the same way as a retrieved image and contains
        only the information to be updated. System-provided images are not allowed to be
        updated. An image with a `status` of `deleting` cannot be updated.

        :param str id: The image identifier.
        :param str name: (optional) The unique user-defined name for this image.
               Names starting with "ibm-" are not allowed.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Image` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_image')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/images/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_operating_systems(self,
                               *,
                               start: str = None,
                               limit: int = None,
                               **kwargs) -> DetailedResponse:
        """
        Retrieves all operating systems.

        This request retrieves all operating systems.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `OperatingSystemCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_operating_systems')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/operating_systems'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_operating_system(self, name: str, **kwargs) -> DetailedResponse:
        """
        Retrieves an operating system.

        This request retrieves a single operating system specified by the name in the URL.

        :param str name: The operating system name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `OperatingSystem` object
        """

        if name is None:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_operating_system')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/operating_systems/{0}'.format(*self.encode_path_vars(name))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    #########################
    # instances
    #########################

    def list_instance_profiles(self,
                               *,
                               start: str = None,
                               limit: int = None,
                               **kwargs) -> DetailedResponse:
        """
        List all instance profiles.

        This request lists all instance profiles available in the region. An instance
        profile specifies the performance characteristics and pricing model for an
        instance.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceProfileCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_profiles')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instance/profiles'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_instance_profile(self, name: str, **kwargs) -> DetailedResponse:
        """
        Retrieve specified instance profile.

        This request retrieves a single instance profile specified by the name in the URL.

        :param str name: The instance profile name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceProfile` object
        """

        if name is None:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_profile')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instance/profiles/{0}'.format(*self.encode_path_vars(name))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_instances(self,
                       *,
                       start: str = None,
                       limit: int = None,
                       network_interfaces_subnet_id: str = None,
                       network_interfaces_subnet_crn: str = None,
                       network_interfaces_subnet_name: str = None,
                       **kwargs) -> DetailedResponse:
        """
        List all instances.

        This request lists all instances in the region.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str network_interfaces_subnet_id: (optional) Filters the collection
               to instances on the subnet of the specified identifier.
        :param str network_interfaces_subnet_crn: (optional) Filters the collection
               to instances on the subnet of the specified CRN.
        :param str network_interfaces_subnet_name: (optional) Filters the
               collection to instances on the subnet of the specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instances')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'network_interfaces.subnet.id': network_interfaces_subnet_id,
            'network_interfaces.subnet.crn': network_interfaces_subnet_crn,
            'network_interfaces.subnet.name': network_interfaces_subnet_name
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_instance(self, instance_prototype: 'InstancePrototype',
                        **kwargs) -> DetailedResponse:
        """
        Create an instance.

        This request provisions a new instance from an instance prototype object. The
        prototype object is structured in the same way as a retrieved instance, and
        contains the information necessary to provision the new instance. The instance is
        automatically started.

        :param InstancePrototype instance_prototype: The instance prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Instance` object
        """

        if instance_prototype is None:
            raise ValueError('instance_prototype must be provided')
        if isinstance(instance_prototype, InstancePrototype):
            instance_prototype = convert_model(instance_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(instance_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_instance(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete specified instance.

        This request deletes an instance. This operation cannot be reversed. Any floating
        IPs associated with the instance's network interfaces are implicitly
        disassociated.

        :param str id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_instance(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve an instance.

        This request retrieves a single instance specified by the identifier in the URL.

        :param str id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Instance` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_instance(self,
                        id: str,
                        *,
                        name: str = None,
                        **kwargs) -> DetailedResponse:
        """
        Update specified instance.

        This request updates an instance with the information in a provided instance
        patch. The instance patch object is structured in the same way as a retrieved
        instance and contains only the information to be updated.

        :param str id: The instance identifier.
        :param str name: (optional) The user-defined name for this virtual server
               instance (and default system hostname).
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Instance` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def get_instance_initialization(self, id: str,
                                    **kwargs) -> DetailedResponse:
        """
        Retrieve configuration used to initialize the instance.

        This request retrieves configuration variables used to initialize the instance,
        such as SSH keys and the Windows administrator password.

        :param str id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceInitialization` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='get_instance_initialization')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/initialization'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_instance_action(self, instance_id: str, type: str,
                               **kwargs) -> DetailedResponse:
        """
        Create an instance action.

        This request creates a new action which will be queued up to run as soon as any
        pending or running actions have completed.

        :param str instance_id: The instance identifier.
        :param str type: The type of action.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceAction` object
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        if type is None:
            raise ValueError('type must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance_action')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'type': type}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/actions'.format(
            *self.encode_path_vars(instance_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_instance_network_interfaces(self, instance_id: str,
                                         **kwargs) -> DetailedResponse:
        """
        List all network interfaces on an instance.

        This request lists all network interfaces on an instance. A network interface is
        an abstract representation of a network interface card and connects an instance to
        a subnet. While each network interface can attach to only one subnet, multiple
        network interfaces can be created to attach to multiple subnets. Multiple
        interfaces may also attach to the same subnet.

        :param str instance_id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkInterfaceCollection` object
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_instance_network_interfaces')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/network_interfaces'.format(
            *self.encode_path_vars(instance_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_instance_network_interface(self, instance_id: str, id: str,
                                       **kwargs) -> DetailedResponse:
        """
        Retrieve specified network interface.

        This request retrieves a single network interface specified by the identifier in
        the URL.

        :param str instance_id: The instance identifier.
        :param str id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkInterface` object
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='get_instance_network_interface')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/network_interfaces/{1}'.format(
            *self.encode_path_vars(instance_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_instance_network_interface_floating_ips(
            self, instance_id: str, network_interface_id: str,
            **kwargs) -> DetailedResponse:
        """
        List all floating IPs associated with a network interface.

        This request lists all floating IPs associated with a network interface.

        :param str instance_id: The instance identifier.
        :param str network_interface_id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIPUnpaginatedCollection` object
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        if network_interface_id is None:
            raise ValueError('network_interface_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_instance_network_interface_floating_ips')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/network_interfaces/{1}/floating_ips'.format(
            *self.encode_path_vars(instance_id, network_interface_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def remove_instance_network_interface_floating_ip(
            self, instance_id: str, network_interface_id: str, id: str,
            **kwargs) -> DetailedResponse:
        """
        Disassociate specified floating IP.

        This request disassociates the specified floating IP from the specified network
        interface.

        :param str instance_id: The instance identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        if network_interface_id is None:
            raise ValueError('network_interface_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='remove_instance_network_interface_floating_ip')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/network_interfaces/{1}/floating_ips/{2}'.format(
            *self.encode_path_vars(instance_id, network_interface_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_instance_network_interface_floating_ip(
            self, instance_id: str, network_interface_id: str, id: str,
            **kwargs) -> DetailedResponse:
        """
        Retrieve associated floating IP.

        This request a retrieves a specified floating IP address if it is associated with
        the network interface and instance specified in the URL.

        :param str instance_id: The instance identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        if network_interface_id is None:
            raise ValueError('network_interface_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='get_instance_network_interface_floating_ip')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/network_interfaces/{1}/floating_ips/{2}'.format(
            *self.encode_path_vars(instance_id, network_interface_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def add_instance_network_interface_floating_ip(
            self, instance_id: str, network_interface_id: str, id: str,
            **kwargs) -> DetailedResponse:
        """
        Associate a floating IP with a network interface.

        This request associates the specified floating IP with the specified network
        interface, replacing any existing association. For this request to succeed, the
        existing floating IP must not be required by another resource, such as a public
        gateway. A request body is not required, and if supplied, is ignored.

        :param str instance_id: The instance identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        if network_interface_id is None:
            raise ValueError('network_interface_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='add_instance_network_interface_floating_ip')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/network_interfaces/{1}/floating_ips/{2}'.format(
            *self.encode_path_vars(instance_id, network_interface_id, id))
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_instance_volume_attachments(self, instance_id: str,
                                         **kwargs) -> DetailedResponse:
        """
        List all volumes attached to an instance.

        This request lists all volume attachments for an instance. A volume attachment
        connects a volume to an instance. Each instance may have many volume attachments
        but each volume attachment connects exactly one instance to exactly one volume.

        :param str instance_id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeAttachmentCollection` object
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_instance_volume_attachments')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/volume_attachments'.format(
            *self.encode_path_vars(instance_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_instance_volume_attachment(
            self,
            instance_id: str,
            volume: 'VolumeIdentity',
            *,
            delete_volume_on_instance_delete: bool = None,
            name: str = None,
            **kwargs) -> DetailedResponse:
        """
        Create a volume attachment, connecting a volume to an instance.

        This request creates a new volume attachment from a volume attachment prototype
        object. The prototype object is structured in the same way as a retrieved volume
        attachment, and contains the information necessary to create the new volume
        attachment. The creation of a new volume attachment connects a volume to an
        instance.

        :param str instance_id: The instance identifier.
        :param VolumeIdentity volume: The identity of the volume to attach to the
               instance.
        :param bool delete_volume_on_instance_delete: (optional) If set to true,
               when deleting the instance the volume will also be deleted.
        :param str name: (optional) The user-defined name for this volume
               attachment. If unspecified, the name will be a hyphenated list of
               randomly-selected words.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeAttachment` object
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        if volume is None:
            raise ValueError('volume must be provided')
        volume = convert_model(volume)
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='create_instance_volume_attachment')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'volume':
                volume,
            'delete_volume_on_instance_delete':
                delete_volume_on_instance_delete,
            'name':
                name
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/volume_attachments'.format(
            *self.encode_path_vars(instance_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_instance_volume_attachment(self, instance_id: str, id: str,
                                          **kwargs) -> DetailedResponse:
        """
        Delete a volume attachment, detaching a volume from an instance.

        This request deletes a volume attachment. The deletion of a volume attachment
        detaches a volume from an instance.

        :param str instance_id: The instance identifier.
        :param str id: The volume attachment identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='delete_instance_volume_attachment')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/volume_attachments/{1}'.format(
            *self.encode_path_vars(instance_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_instance_volume_attachment(self, instance_id: str, id: str,
                                       **kwargs) -> DetailedResponse:
        """
        Retrieve specified volume attachment.

        This request retrieves a single volume attachment specified by the identifier in
        the URL.

        :param str instance_id: The instance identifier.
        :param str id: The volume attachment identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeAttachment` object
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='get_instance_volume_attachment')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/volume_attachments/{1}'.format(
            *self.encode_path_vars(instance_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_instance_volume_attachment(
            self,
            instance_id: str,
            id: str,
            *,
            delete_volume_on_instance_delete: bool = None,
            name: str = None,
            **kwargs) -> DetailedResponse:
        """
        Update a volume attachment.

        This request updates a volume attachment with the information in a provided volume
        attachment patch. The volume attachment patch object is structured in the same way
        as a retrieved volume attachment and can contain an updated name.

        :param str instance_id: The instance identifier.
        :param str id: The volume attachment identifier.
        :param bool delete_volume_on_instance_delete: (optional) If set to true,
               when deleting the instance the volume will also be deleted.
        :param str name: (optional) The user-defined name for this volume
               attachment.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeAttachment` object
        """

        if instance_id is None:
            raise ValueError('instance_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='update_instance_volume_attachment')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'delete_volume_on_instance_delete':
                delete_volume_on_instance_delete,
            'name':
                name
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/instances/{0}/volume_attachments/{1}'.format(
            *self.encode_path_vars(instance_id, id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    #########################
    # loadBalancers
    #########################

    def list_load_balancers(self, **kwargs) -> DetailedResponse:
        """
        List all load balancers.

        This request retrieves a paginated list of all load balancers that belong to this
        account.

        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_load_balancers')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_load_balancer(
            self,
            is_public: bool,
            subnets: List['SubnetIdentity'],
            *,
            listeners: List[
                'LoadBalancerListenerPrototypeLoadBalancerContext'] = None,
            name: str = None,
            pools: List['LoadBalancerPoolPrototype'] = None,
            resource_group: 'ResourceGroupIdentity' = None,
            **kwargs) -> DetailedResponse:
        """
        Create and provision a load balancer.

        This request creates and provisions a new load balancer.

        :param bool is_public: The type of this load balancer, public or private.
        :param List[SubnetIdentity] subnets: The subnets to provision this load
               balancer.
        :param List[LoadBalancerListenerPrototypeLoadBalancerContext] listeners:
               (optional) The listeners of this load balancer.
        :param str name: (optional) The user-defined name for this load balancer.
               If unspecified, the name will be a hyphenated list of randomly-selected
               words.
        :param List[LoadBalancerPoolPrototype] pools: (optional) The pools of this
               load balancer.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               for this load balancer.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancer` object
        """

        if is_public is None:
            raise ValueError('is_public must be provided')
        if subnets is None:
            raise ValueError('subnets must be provided')
        subnets = [convert_model(x) for x in subnets]
        if listeners is not None:
            listeners = [convert_model(x) for x in listeners]
        if pools is not None:
            pools = [convert_model(x) for x in pools]
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_load_balancer')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'is_public': is_public,
            'subnets': subnets,
            'listeners': listeners,
            'name': name,
            'pools': pools,
            'resource_group': resource_group
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_load_balancer(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete a load balancer.

        This request deletes a load balancer. This operation cannot be reversed.

        :param str id: The load balancer identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_load_balancer')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_load_balancer(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve a load balancer.

        This request retrieves a single load balancer specified by the identifier in the
        URL path.

        :param str id: The load balancer identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancer` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_load_balancer(self,
                             id: str,
                             *,
                             name: str = None,
                             **kwargs) -> DetailedResponse:
        """
        Update a load balancer.

        This request updates a load balancer.

        :param str id: The load balancer identifier.
        :param str name: (optional) The unique user-defined name for this load
               balancer.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancer` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_load_balancer')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def get_load_balancer_statistics(self, id: str,
                                     **kwargs) -> DetailedResponse:
        """
        List statistics of a load balancer.

        This request lists statistics of a load balancer specified by the identifier in
        the URL path.

        :param str id: The load balancer identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerStatistics` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='get_load_balancer_statistics')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/statistics'.format(
            *self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_load_balancer_listeners(self, load_balancer_id: str,
                                     **kwargs) -> DetailedResponse:
        """
        List all listeners of the load balancer.

        This request retrieves a list of all listeners that belong to the load balancer.

        :param str load_balancer_id: The load balancer identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerCollection` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_load_balancer_listeners')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners'.format(
            *self.encode_path_vars(load_balancer_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_load_balancer_listener(
            self,
            load_balancer_id: str,
            port: int,
            protocol: str,
            *,
            certificate_instance: 'CertificateInstanceIdentity' = None,
            connection_limit: int = None,
            default_pool: 'LoadBalancerPoolIdentity' = None,
            policies: List['LoadBalancerListenerPolicyPrototype'] = None,
            **kwargs) -> DetailedResponse:
        """
        Create a listener.

        This request creates a new listener to the load balancer.

        :param str load_balancer_id: The load balancer identifier.
        :param int port: The listener port number.
        :param str protocol: The listener protocol.
        :param CertificateInstanceIdentity certificate_instance: (optional) The
               certificate instance used for SSL termination. It is applicable only to
               `https`
               protocol.
        :param int connection_limit: (optional) The connection limit of the
               listener.
        :param LoadBalancerPoolIdentity default_pool: (optional) The default pool
               associated with the listener.
        :param List[LoadBalancerListenerPolicyPrototype] policies: (optional) The
               list of policies of this listener.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListener` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if port is None:
            raise ValueError('port must be provided')
        if protocol is None:
            raise ValueError('protocol must be provided')
        if certificate_instance is not None:
            certificate_instance = convert_model(certificate_instance)
        if default_pool is not None:
            default_pool = convert_model(default_pool)
        if policies is not None:
            policies = [convert_model(x) for x in policies]
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='create_load_balancer_listener')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'port': port,
            'protocol': protocol,
            'certificate_instance': certificate_instance,
            'connection_limit': connection_limit,
            'default_pool': default_pool,
            'policies': policies
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners'.format(
            *self.encode_path_vars(load_balancer_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_load_balancer_listener(self, load_balancer_id: str, id: str,
                                      **kwargs) -> DetailedResponse:
        """
        Delete a listener.

        This request deletes a load balancer listener. This operation cannot be reversed.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The listener identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='delete_load_balancer_listener')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}'.format(
            *self.encode_path_vars(load_balancer_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_load_balancer_listener(self, load_balancer_id: str, id: str,
                                   **kwargs) -> DetailedResponse:
        """
        Retrieve a listener.

        This request retrieves a single listener specified by the identifier in the URL
        path.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The listener identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListener` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer_listener')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}'.format(
            *self.encode_path_vars(load_balancer_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_load_balancer_listener(
            self,
            load_balancer_id: str,
            id: str,
            *,
            certificate_instance: 'CertificateInstanceIdentity' = None,
            connection_limit: int = None,
            default_pool: 'LoadBalancerPoolIdentity' = None,
            port: int = None,
            protocol: str = None,
            **kwargs) -> DetailedResponse:
        """
        Update a listener.

        This request updates a load balancer listener from a listener patch.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The listener identifier.
        :param CertificateInstanceIdentity certificate_instance: (optional) The
               certificate instance used for SSL termination. It is applicable only to
               `https`
               protocol.
        :param int connection_limit: (optional) The connection limit of the
               listener.
        :param LoadBalancerPoolIdentity default_pool: (optional) The default pool
               associated with the listener.
        :param int port: (optional) The listener port number.
        :param str protocol: (optional) The listener protocol.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListener` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if certificate_instance is not None:
            certificate_instance = convert_model(certificate_instance)
        if default_pool is not None:
            default_pool = convert_model(default_pool)
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='update_load_balancer_listener')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'certificate_instance': certificate_instance,
            'connection_limit': connection_limit,
            'default_pool': default_pool,
            'port': port,
            'protocol': protocol
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}'.format(
            *self.encode_path_vars(load_balancer_id, id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_load_balancer_listener_policies(self, load_balancer_id: str,
                                             listener_id: str,
                                             **kwargs) -> DetailedResponse:
        """
        List all policies of the load balancer listener.

        Retrieves a list of all policies belonging to the load balancer listener.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicyCollection` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if listener_id is None:
            raise ValueError('listener_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_load_balancer_listener_policies')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}/policies'.format(
            *self.encode_path_vars(load_balancer_id, listener_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_load_balancer_listener_policy(
            self,
            load_balancer_id: str,
            listener_id: str,
            action: str,
            priority: int,
            *,
            name: str = None,
            rules: List['LoadBalancerListenerPolicyRulePrototype'] = None,
            target: 'LoadBalancerListenerPolicyPrototypeTarget' = None,
            **kwargs) -> DetailedResponse:
        """
        Create a policy for the load balancer listener.

        Creates a new policy to the load balancer listener.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str action: The policy action.
        :param int priority: Priority of the policy. Lower value indicates higher
               priority.
        :param str name: (optional) The user-defined name for this policy. Names
               must be unique within the load balancer listener the policy resides in.
        :param List[LoadBalancerListenerPolicyRulePrototype] rules: (optional) The
               list of rules of this policy.
        :param LoadBalancerListenerPolicyPrototypeTarget target: (optional) When
               `action` is `forward`, `LoadBalancerPoolIdentity` is required to specify
               which
               pool the load balancer forwards the traffic to. When `action` is
               `redirect`,
               `LoadBalancerListenerPolicyRedirectURLPrototype` is required to specify the
               url and
               http status code used in the redirect response.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicy` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if listener_id is None:
            raise ValueError('listener_id must be provided')
        if action is None:
            raise ValueError('action must be provided')
        if priority is None:
            raise ValueError('priority must be provided')
        if rules is not None:
            rules = [convert_model(x) for x in rules]
        if target is not None:
            target = convert_model(target)
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='create_load_balancer_listener_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'action': action,
            'priority': priority,
            'name': name,
            'rules': rules,
            'target': target
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}/policies'.format(
            *self.encode_path_vars(load_balancer_id, listener_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_load_balancer_listener_policy(self, load_balancer_id: str,
                                             listener_id: str, id: str,
                                             **kwargs) -> DetailedResponse:
        """
        Delete a policy of the load balancer listener.

        Deletes a policy of the load balancer listener. This operation cannot be reversed.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str id: The policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if listener_id is None:
            raise ValueError('listener_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='delete_load_balancer_listener_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}/policies/{2}'.format(
            *self.encode_path_vars(load_balancer_id, listener_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_load_balancer_listener_policy(self, load_balancer_id: str,
                                          listener_id: str, id: str,
                                          **kwargs) -> DetailedResponse:
        """
        Retrieve a policy of the load balancer listener.

        Retrieve a single policy specified by the identifier in the URL path.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str id: The policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicy` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if listener_id is None:
            raise ValueError('listener_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='get_load_balancer_listener_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}/policies/{2}'.format(
            *self.encode_path_vars(load_balancer_id, listener_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_load_balancer_listener_policy(
            self,
            load_balancer_id: str,
            listener_id: str,
            id: str,
            *,
            name: str = None,
            priority: int = None,
            target: 'LoadBalancerListenerPolicyPatchTarget' = None,
            **kwargs) -> DetailedResponse:
        """
        Update a policy of the load balancer listener.

        Updates a policy from a policy patch.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str id: The policy identifier.
        :param str name: (optional) The user-defined name for this policy. Names
               must be unique within the load balancer listener the policy resides in.
        :param int priority: (optional) Priority of the policy. Lower value
               indicates higher priority.
        :param LoadBalancerListenerPolicyPatchTarget target: (optional) When
               `action` is `forward`, `LoadBalancerPoolIdentity` specifies which pool the
               load
               balancer forwards the traffic to. When `action` is `redirect`,
               `LoadBalancerListenerPolicyRedirectURLPatch` specifies the url and http
               status code used in the redirect response.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicy` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if listener_id is None:
            raise ValueError('listener_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if target is not None:
            target = convert_model(target)
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='update_load_balancer_listener_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name, 'priority': priority, 'target': target}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}/policies/{2}'.format(
            *self.encode_path_vars(load_balancer_id, listener_id, id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_load_balancer_listener_policy_rules(self, load_balancer_id: str,
                                                 listener_id: str,
                                                 policy_id: str,
                                                 **kwargs) -> DetailedResponse:
        """
        List all rules of the load balancer listener policy.

        Retrieves a list of all rules belonging to the load balancer listener policy.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str policy_id: The policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicyRuleCollection` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if listener_id is None:
            raise ValueError('listener_id must be provided')
        if policy_id is None:
            raise ValueError('policy_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_load_balancer_listener_policy_rules')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}/policies/{2}/rules'.format(
            *self.encode_path_vars(load_balancer_id, listener_id, policy_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_load_balancer_listener_policy_rule(self,
                                                  load_balancer_id: str,
                                                  listener_id: str,
                                                  policy_id: str,
                                                  condition: str,
                                                  type: str,
                                                  value: str,
                                                  *,
                                                  field: str = None,
                                                  **kwargs) -> DetailedResponse:
        """
        Create a rule for the load balancer listener policy.

        Creates a new rule for the load balancer listener policy.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str policy_id: The policy identifier.
        :param str condition: The condition of the rule.
        :param str type: The type of the rule.
        :param str value: Value to be matched for rule condition.
        :param str field: (optional) HTTP header field. This is only applicable to
               "header" rule type.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicyRule` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if listener_id is None:
            raise ValueError('listener_id must be provided')
        if policy_id is None:
            raise ValueError('policy_id must be provided')
        if condition is None:
            raise ValueError('condition must be provided')
        if type is None:
            raise ValueError('type must be provided')
        if value is None:
            raise ValueError('value must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='create_load_balancer_listener_policy_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'condition': condition,
            'type': type,
            'value': value,
            'field': field
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}/policies/{2}/rules'.format(
            *self.encode_path_vars(load_balancer_id, listener_id, policy_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_load_balancer_listener_policy_rule(self, load_balancer_id: str,
                                                  listener_id: str,
                                                  policy_id: str, id: str,
                                                  **kwargs) -> DetailedResponse:
        """
        Delete a rule from the load balancer listener policy.

        Deletes a rule from the load balancer listener policy. This operation cannot be
        reversed.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str policy_id: The policy identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if listener_id is None:
            raise ValueError('listener_id must be provided')
        if policy_id is None:
            raise ValueError('policy_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='delete_load_balancer_listener_policy_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}/policies/{2}/rules/{3}'.format(
            *self.encode_path_vars(load_balancer_id, listener_id, policy_id,
                                   id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_load_balancer_listener_policy_rule(self, load_balancer_id: str,
                                               listener_id: str, policy_id: str,
                                               id: str,
                                               **kwargs) -> DetailedResponse:
        """
        Retrieve a rule of the load balancer listener policy.

        Retrieves a single rule specified by the identifier in the URL path.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str policy_id: The policy identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicyRule` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if listener_id is None:
            raise ValueError('listener_id must be provided')
        if policy_id is None:
            raise ValueError('policy_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='get_load_balancer_listener_policy_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}/policies/{2}/rules/{3}'.format(
            *self.encode_path_vars(load_balancer_id, listener_id, policy_id,
                                   id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_load_balancer_listener_policy_rule(self,
                                                  load_balancer_id: str,
                                                  listener_id: str,
                                                  policy_id: str,
                                                  id: str,
                                                  *,
                                                  condition: str = None,
                                                  field: str = None,
                                                  type: str = None,
                                                  value: str = None,
                                                  **kwargs) -> DetailedResponse:
        """
        Update a rule of the load balancer listener policy.

        Updates a rule of the load balancer listener policy.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str policy_id: The policy identifier.
        :param str id: The rule identifier.
        :param str condition: (optional) The condition of the rule.
        :param str field: (optional) HTTP header field. This is only applicable to
               "header" rule type.
        :param str type: (optional) The type of the rule.
        :param str value: (optional) Value to be matched for rule condition.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicyRule` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if listener_id is None:
            raise ValueError('listener_id must be provided')
        if policy_id is None:
            raise ValueError('policy_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='update_load_balancer_listener_policy_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'condition': condition,
            'field': field,
            'type': type,
            'value': value
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/listeners/{1}/policies/{2}/rules/{3}'.format(
            *self.encode_path_vars(load_balancer_id, listener_id, policy_id,
                                   id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_load_balancer_pools(self, load_balancer_id: str,
                                 **kwargs) -> DetailedResponse:
        """
        List all pools of the load balancer.

        This request lists all pools that belong to the load balancer.

        :param str load_balancer_id: The load balancer identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolCollection` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_load_balancer_pools')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools'.format(
            *self.encode_path_vars(load_balancer_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_load_balancer_pool(
            self,
            load_balancer_id: str,
            algorithm: str,
            health_monitor: 'LoadBalancerPoolHealthMonitorPrototype',
            protocol: str,
            *,
            members: List['LoadBalancerPoolMemberPrototype'] = None,
            name: str = None,
            session_persistence:
        'LoadBalancerPoolSessionPersistencePrototype' = None,
            **kwargs) -> DetailedResponse:
        """
        Create a load balancer pool.

        This request creates a new pool from a pool prototype object.

        :param str load_balancer_id: The load balancer identifier.
        :param str algorithm: The load balancing algorithm.
        :param LoadBalancerPoolHealthMonitorPrototype health_monitor: The health
               monitor of this pool.
        :param str protocol: The protocol used for this load balancer pool.
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the pool on
               which the unexpected property value was encountered.
        :param List[LoadBalancerPoolMemberPrototype] members: (optional) The
               members for this load balancer pool. For load balancers in the `network`
               family, the same `port` and `target` tuple cannot be shared by a member of
               any other load balancer.
        :param str name: (optional) The user-defined name for this load balancer
               pool. If unspecified, the name will be a hyphenated list of
               randomly-selected words.
        :param LoadBalancerPoolSessionPersistencePrototype session_persistence:
               (optional) The session persistence of this pool.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPool` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if algorithm is None:
            raise ValueError('algorithm must be provided')
        if health_monitor is None:
            raise ValueError('health_monitor must be provided')
        if protocol is None:
            raise ValueError('protocol must be provided')
        health_monitor = convert_model(health_monitor)
        if members is not None:
            members = [convert_model(x) for x in members]
        if session_persistence is not None:
            session_persistence = convert_model(session_persistence)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_load_balancer_pool')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'algorithm': algorithm,
            'health_monitor': health_monitor,
            'protocol': protocol,
            'members': members,
            'name': name,
            'session_persistence': session_persistence
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools'.format(
            *self.encode_path_vars(load_balancer_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_load_balancer_pool(self, load_balancer_id: str, id: str,
                                  **kwargs) -> DetailedResponse:
        """
        Delete a pool.

        This request deletes a load balancer pool. This operation cannot be reversed.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The pool identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_load_balancer_pool')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools/{1}'.format(
            *self.encode_path_vars(load_balancer_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_load_balancer_pool(self, load_balancer_id: str, id: str,
                               **kwargs) -> DetailedResponse:
        """
        Retrieve a load balancer pool.

        This request retrieves a single pool specified by the identifier in the URL path.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The pool identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPool` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer_pool')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools/{1}'.format(
            *self.encode_path_vars(load_balancer_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_load_balancer_pool(
            self,
            load_balancer_id: str,
            id: str,
            *,
            algorithm: str = None,
            health_monitor: 'LoadBalancerPoolHealthMonitorPatch' = None,
            name: str = None,
            protocol: str = None,
            session_persistence:
        'LoadBalancerPoolSessionPersistencePatch' = None,
            **kwargs) -> DetailedResponse:
        """
        Update a load balancer pool.

        This request updates a load balancer pool from a pool patch.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The pool identifier.
        :param str algorithm: (optional) The load balancing algorithm.
        :param LoadBalancerPoolHealthMonitorPatch health_monitor: (optional) The
               health monitor of this pool.
        :param str name: (optional) The user-defined name for this load balancer
               pool.
        :param str protocol: (optional) The protocol used for this load balancer
               pool.
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the pool on
               which the unexpected property value was encountered.
        :param LoadBalancerPoolSessionPersistencePatch session_persistence:
               (optional) The session persistence of this pool.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPool` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if health_monitor is not None:
            health_monitor = convert_model(health_monitor)
        if session_persistence is not None:
            session_persistence = convert_model(session_persistence)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_load_balancer_pool')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'algorithm': algorithm,
            'health_monitor': health_monitor,
            'name': name,
            'protocol': protocol,
            'session_persistence': session_persistence
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools/{1}'.format(
            *self.encode_path_vars(load_balancer_id, id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_load_balancer_pool_members(self, load_balancer_id: str,
                                        pool_id: str,
                                        **kwargs) -> DetailedResponse:
        """
        List all members of the load balancer pool.

        This request retrieves a paginated list of all members that belong to the pool.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolMemberCollection` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if pool_id is None:
            raise ValueError('pool_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_load_balancer_pool_members')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools/{1}/members'.format(
            *self.encode_path_vars(load_balancer_id, pool_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_load_balancer_pool_member(
            self,
            load_balancer_id: str,
            pool_id: str,
            port: int,
            target: 'LoadBalancerPoolMemberTargetPrototype',
            *,
            weight: int = None,
            **kwargs) -> DetailedResponse:
        """
        Create a member in the load balancer pool.

        This request creates a new member and adds the member to the pool.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param int port: The port number of the application running in the server
               member.
        :param LoadBalancerPoolMemberTargetPrototype target: The pool member
               target.
        :param int weight: (optional) Weight of the server member. This takes
               effect only when the load balancing algorithm of its belonging pool is
               `weighted_round_robin`.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolMember` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if pool_id is None:
            raise ValueError('pool_id must be provided')
        if port is None:
            raise ValueError('port must be provided')
        if target is None:
            raise ValueError('target must be provided')
        target = convert_model(target)
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='create_load_balancer_pool_member')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'port': port, 'target': target, 'weight': weight}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools/{1}/members'.format(
            *self.encode_path_vars(load_balancer_id, pool_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def replace_load_balancer_pool_members(
            self, load_balancer_id: str, pool_id: str,
            members: List['LoadBalancerPoolMemberPrototype'],
            **kwargs) -> DetailedResponse:
        """
        Update members of the load balancer pool.

        This request updates members of the load balancer pool from a collection of member
        prototype objects.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param List[LoadBalancerPoolMemberPrototype] members: Array of pool member
               prototype objects.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolMemberCollection` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if pool_id is None:
            raise ValueError('pool_id must be provided')
        if members is None:
            raise ValueError('members must be provided')
        members = [convert_model(x) for x in members]
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='replace_load_balancer_pool_members')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'members': members}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools/{1}/members'.format(
            *self.encode_path_vars(load_balancer_id, pool_id))
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_load_balancer_pool_member(self, load_balancer_id: str,
                                         pool_id: str, id: str,
                                         **kwargs) -> DetailedResponse:
        """
        Delete a member from the load balancer pool.

        This request deletes a member from the pool. This operation cannot be reversed.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param str id: The member identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if pool_id is None:
            raise ValueError('pool_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='delete_load_balancer_pool_member')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools/{1}/members/{2}'.format(
            *self.encode_path_vars(load_balancer_id, pool_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_load_balancer_pool_member(self, load_balancer_id: str, pool_id: str,
                                      id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve a member in the load balancer pool.

        This request retrieves a single member specified by the identifier in the URL
        path.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param str id: The member identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolMember` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if pool_id is None:
            raise ValueError('pool_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='get_load_balancer_pool_member')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools/{1}/members/{2}'.format(
            *self.encode_path_vars(load_balancer_id, pool_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_load_balancer_pool_member(
            self,
            load_balancer_id: str,
            pool_id: str,
            id: str,
            *,
            port: int = None,
            target: 'LoadBalancerPoolMemberTargetPrototype' = None,
            weight: int = None,
            **kwargs) -> DetailedResponse:
        """
        Update a member in the load balancer pool.

        This request updates an existing member from a member patch.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param str id: The member identifier.
        :param int port: (optional) The port number of the application running in
               the server member.
        :param LoadBalancerPoolMemberTargetPrototype target: (optional) The pool
               member target.
        :param int weight: (optional) Weight of the server member. This takes
               effect only when the load balancing algorithm of its belonging pool is
               `weighted_round_robin`.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolMember` object
        """

        if load_balancer_id is None:
            raise ValueError('load_balancer_id must be provided')
        if pool_id is None:
            raise ValueError('pool_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if target is not None:
            target = convert_model(target)
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='update_load_balancer_pool_member')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'port': port, 'target': target, 'weight': weight}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/load_balancers/{0}/pools/{1}/members/{2}'.format(
            *self.encode_path_vars(load_balancer_id, pool_id, id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    #########################
    # networkACLs
    #########################

    def list_network_acls(self,
                          *,
                          start: str = None,
                          limit: int = None,
                          **kwargs) -> DetailedResponse:
        """
        List all network ACLs.

        This request lists all network ACLs in the region. A network ACL defines a set of
        packet filtering (5-tuple) rules for all traffic in and out of a subnet. Both
        allow and deny rules can be defined, and rules are stateless such that reverse
        traffic in response to allowed traffic is not automatically permitted.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACLCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_network_acls')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/network_acls'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_network_acl(self,
                           *,
                           network_acl_prototype: 'NetworkACLPrototype' = None,
                           **kwargs) -> DetailedResponse:
        """
        Create a network ACL.

        This request creates a new network ACL from a network ACL prototype object. The
        prototype object is structured in the same way as a retrieved network ACL, and
        contains the information necessary to create the new network ACL.

        :param NetworkACLPrototype network_acl_prototype: (optional) The network
               ACL prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACL` object
        """

        if network_acl_prototype is not None and isinstance(
                network_acl_prototype, NetworkACLPrototype):
            network_acl_prototype = convert_model(network_acl_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_network_acl')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(network_acl_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/network_acls'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_network_acl(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete specified network ACL.

        This request deletes a network ACL. This operation cannot be reversed. For this
        request to succeed, the network ACL must not be the default network ACL for any
        VPCs, and the network ACL must not be attached to any subnets.

        :param str id: The network ACL identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_network_acl')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/network_acls/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_network_acl(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve specified network ACL.

        This request retrieves a single network ACL specified by the identifier in the
        URL.

        :param str id: The network ACL identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACL` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_network_acl')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/network_acls/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_network_acl(self,
                           id: str,
                           *,
                           name: str = None,
                           **kwargs) -> DetailedResponse:
        """
        Update a network ACL.

        This request updates a network ACL's name.

        :param str id: The network ACL identifier.
        :param str name: (optional) The unique user-defined name for this network
               ACL.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACL` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_network_acl')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/network_acls/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_network_acl_rules(self,
                               network_acl_id: str,
                               *,
                               start: str = None,
                               limit: int = None,
                               direction: str = None,
                               **kwargs) -> DetailedResponse:
        """
        List all rules for a network ACL.

        This request lists all rules for a network ACL. These rules can allow or deny
        traffic between a source CIDR block and a destination CIDR block over a particular
        protocol and port range.

        :param str network_acl_id: The network ACL identifier.
        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str direction: (optional) Filters the collection to rules with the
               specified direction.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACLRuleCollection` object
        """

        if network_acl_id is None:
            raise ValueError('network_acl_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_network_acl_rules')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'direction': direction
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/network_acls/{0}/rules'.format(
            *self.encode_path_vars(network_acl_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_network_acl_rule(
            self, network_acl_id: str,
            network_acl_rule_prototype: 'NetworkACLRulePrototype',
            **kwargs) -> DetailedResponse:
        """
        Create a rule.

        This request creates a new rule from a network ACL rule prototype object. The
        prototype object is structured in the same way as a retrieved rule, and contains
        the information necessary to create the new rule.

        :param str network_acl_id: The network ACL identifier.
        :param NetworkACLRulePrototype network_acl_rule_prototype: The network ACL
               rule prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACLRule` object
        """

        if network_acl_id is None:
            raise ValueError('network_acl_id must be provided')
        if network_acl_rule_prototype is None:
            raise ValueError('network_acl_rule_prototype must be provided')
        if isinstance(network_acl_rule_prototype, NetworkACLRulePrototype):
            network_acl_rule_prototype = convert_model(
                network_acl_rule_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_network_acl_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(network_acl_rule_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/network_acls/{0}/rules'.format(
            *self.encode_path_vars(network_acl_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_network_acl_rule(self, network_acl_id: str, id: str,
                                **kwargs) -> DetailedResponse:
        """
        Delete specified rule.

        This request deletes a rule. This operation cannot be reversed.

        :param str network_acl_id: The network ACL identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if network_acl_id is None:
            raise ValueError('network_acl_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_network_acl_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/network_acls/{0}/rules/{1}'.format(
            *self.encode_path_vars(network_acl_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_network_acl_rule(self, network_acl_id: str, id: str,
                             **kwargs) -> DetailedResponse:
        """
        Retrieve specified rule.

        This request retrieves a single rule specified by the identifier in the URL.

        :param str network_acl_id: The network ACL identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACLRule` object
        """

        if network_acl_id is None:
            raise ValueError('network_acl_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_network_acl_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/network_acls/{0}/rules/{1}'.format(
            *self.encode_path_vars(network_acl_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_network_acl_rule(self, network_acl_id: str, id: str,
                                network_acl_rule_patch: 'NetworkACLRulePatch',
                                **kwargs) -> DetailedResponse:
        """
        Update a rule.

        This request updates a rule with the information in a provided rule patch. The
        rule patch object is structured in the same way as a retrieved rule and contains
        only the information to be updated.

        :param str network_acl_id: The network ACL identifier.
        :param str id: The rule identifier.
        :param NetworkACLRulePatch network_acl_rule_patch: The rule patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACLRule` object
        """

        if network_acl_id is None:
            raise ValueError('network_acl_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if network_acl_rule_patch is None:
            raise ValueError('network_acl_rule_patch must be provided')
        if isinstance(network_acl_rule_patch, NetworkACLRulePatch):
            network_acl_rule_patch = convert_model(network_acl_rule_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_network_acl_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(network_acl_rule_patch)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/network_acls/{0}/rules/{1}'.format(
            *self.encode_path_vars(network_acl_id, id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    #########################
    # publicGateways
    #########################

    def list_public_gateways(self,
                             *,
                             start: str = None,
                             limit: int = None,
                             **kwargs) -> DetailedResponse:
        """
        List all public gateways.

        This request lists all public gateways. A public gateway is a virtual network
        device associated with a VPC, which allows access to the Internet. A public
        gateway resides in a zone and can be connected to subnets in the same zone only.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGatewayCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_public_gateways')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/public_gateways'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_public_gateway(
            self,
            vpc: 'VPCIdentity',
            zone: 'ZoneIdentity',
            *,
            floating_ip: 'PublicGatewayPrototypeFloatingIp' = None,
            name: str = None,
            **kwargs) -> DetailedResponse:
        """
        Create a public gateway.

        This request creates a new public gateway from a public gateway prototype object.
        If a floating IP is provided, it must be unbound. If a floating IP is not
        provided, one will be created and bound to the public gateway. Once a public
        gateway has been created, its floating IP cannot be unbound. A public gateway must
        be explicitly attached to each subnet it will provide connectivity for.

        :param VPCIdentity vpc: The VPC this public gateway will serve.
        :param ZoneIdentity zone: The zone where this public gateway will be
               created.
        :param PublicGatewayPrototypeFloatingIp floating_ip: (optional)
        :param str name: (optional) The user-defined name for this public gateway.
               Names must be unique within the VPC the public gateway resides in. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGateway` object
        """

        if vpc is None:
            raise ValueError('vpc must be provided')
        if zone is None:
            raise ValueError('zone must be provided')
        vpc = convert_model(vpc)
        zone = convert_model(zone)
        if floating_ip is not None:
            floating_ip = convert_model(floating_ip)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_public_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'vpc': vpc,
            'zone': zone,
            'floating_ip': floating_ip,
            'name': name
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/public_gateways'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_public_gateway(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete specified public gateway.

        This request deletes a public gateway. This operation cannot be reversed. For this
        request to succeed, the public gateway must not be attached to any subnets. The
        public gateway's floating IP will be automatically unbound. If the floating IP was
        created when the public gateway was created, it will be deleted.

        :param str id: The public gateway identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_public_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/public_gateways/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_public_gateway(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve specified public gateway.

        This request retrieves a single public gateway specified by the identifier in the
        URL.

        :param str id: The public gateway identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGateway` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_public_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/public_gateways/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_public_gateway(self,
                              id: str,
                              *,
                              name: str = None,
                              **kwargs) -> DetailedResponse:
        """
        Update a public gateway's name.

        This request updates a public gateway's name.

        :param str id: The public gateway identifier.
        :param str name: (optional) The user-defined name for this public gateway.
               Names must be unique within the VPC the public gateway resides in.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGateway` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_public_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/public_gateways/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    #########################
    # sSHKeys
    #########################

    def list_keys(self,
                  *,
                  start: str = None,
                  limit: int = None,
                  **kwargs) -> DetailedResponse:
        """
        List all keys.

        This request lists all keys. A key contains a public SSH key which may be
        installed on instances when they are created. Private keys are not stored.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `KeyCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_keys')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/keys'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_key(self,
                   public_key: str,
                   *,
                   name: str = None,
                   resource_group: 'ResourceGroupIdentity' = None,
                   type: str = None,
                   **kwargs) -> DetailedResponse:
        """
        Create a key.

        This request creates a new SSH key from an key prototype object. The prototype
        object is structured in the same way as a retrieved key, and contains the
        information necessary to create the new key. The public key value must be
        provided.

        :param str public_key: A unique public SSH key to import, encoded in PEM
               format. The key (prior to encoding) must be either 2048 or 4096 bits long.
        :param str name: (optional) The user-defined name for this key.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param str type: (optional) The cryptosystem used by this key.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Key` object
        """

        if public_key is None:
            raise ValueError('public_key must be provided')
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_key')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'public_key': public_key,
            'name': name,
            'resource_group': resource_group,
            'type': type
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/keys'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_key(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete specified key.

        This request deletes a key. This operation cannot be reversed.

        :param str id: The key identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_key')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/keys/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_key(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve specified key.

        This request retrieves a single key specified by the identifier in the URL.

        :param str id: The key identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Key` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_key')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/keys/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_key(self,
                   id: str,
                   *,
                   name: str = None,
                   **kwargs) -> DetailedResponse:
        """
        Update specified key.

        This request updates a key's name.

        :param str id: The key identifier.
        :param str name: (optional) The user-defined name for this key.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Key` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_key')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/keys/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    #########################
    # securityGroups
    #########################

    def list_security_groups(self,
                             *,
                             vpc_id: str = None,
                             vpc_crn: str = None,
                             vpc_name: str = None,
                             **kwargs) -> DetailedResponse:
        """
        List all security groups.

        This request lists all existing security groups. Security groups provide a
        convenient way to apply IP filtering rules to instances in the associated VPC.
        With security groups, all traffic is denied by default, and rules added to
        security groups define which traffic the security group permits. Security group
        rules are stateful such that reverse traffic in response to allowed traffic is
        automatically permitted.

        :param str vpc_id: (optional) Filters the collection to resources in the
               VPC with the specified identifier.
        :param str vpc_crn: (optional) Filters the collection to resources in the
               VPC with the specified CRN.
        :param str vpc_name: (optional) Filters the collection to resources in the
               VPC with the exact specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_security_groups')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'vpc.id': vpc_id,
            'vpc.crn': vpc_crn,
            'vpc.name': vpc_name
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_security_group(self,
                              vpc: 'VPCIdentity',
                              *,
                              name: str = None,
                              resource_group: 'ResourceGroupIdentity' = None,
                              rules: List['SecurityGroupRulePrototype'] = None,
                              **kwargs) -> DetailedResponse:
        """
        Create a security group.

        This request creates a new security group from a security group prototype object.
        The prototype object is structured in the same way as a retrieved security group,
        and contains the information necessary to create the new security group. If
        security group rules are included in the protoype object, those rules will be
        added to the security group. Each security group is scoped to one VPC. Only
        network interfaces on instances in that VPC can be added to the security group.

        :param VPCIdentity vpc: The VPC this security group is to be a part of.
        :param str name: (optional) The user-defined name for this security group.
               If unspecified, the name will be a hyphenated list of randomly-selected
               words. Security group names must be unique, within the scope of an account.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param List[SecurityGroupRulePrototype] rules: (optional) Array of rule
               prototype objects for rules to be created for this security group. If
               unspecified, no rules will be created, resulting in all traffic being
               denied.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroup` object
        """

        if vpc is None:
            raise ValueError('vpc must be provided')
        vpc = convert_model(vpc)
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        if rules is not None:
            rules = [convert_model(x) for x in rules]
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_security_group')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'vpc': vpc,
            'name': name,
            'resource_group': resource_group,
            'rules': rules
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_security_group(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete a security group.

        This request deletes a security group. A security group cannot be deleted if it is
        referenced by any network interfaces or other security group rules. Additionally,
        a VPC's default security group cannot be deleted. This operation cannot be
        reversed.

        :param str id: The security group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_security_group')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_security_group(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve a security group.

        This request retrieves a single security group specified by the identifier in the
        URL path.

        :param str id: The security group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroup` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_security_group')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_security_group(self,
                              id: str,
                              *,
                              name: str = None,
                              **kwargs) -> DetailedResponse:
        """
        Update a security group.

        This request updates a security group with the information provided in a security
        group patch object. The security group patch object is structured in the same way
        as a retrieved security group and contains only the information to be updated.

        :param str id: The security group identifier.
        :param str name: (optional) The user-defined name for this security group.
               Security group names must be unique, within the scope of an account.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroup` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_security_group')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_security_group_network_interfaces(self, security_group_id: str,
                                               **kwargs) -> DetailedResponse:
        """
        List a security group's network interfaces.

        This request lists all network interfaces associated with the security group, to
        which the rules in the security group are applied.

        :param str security_group_id: The security group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkInterfaceCollection` object
        """

        if security_group_id is None:
            raise ValueError('security_group_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_security_group_network_interfaces')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}/network_interfaces'.format(
            *self.encode_path_vars(security_group_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def remove_security_group_network_interface(self, security_group_id: str,
                                                id: str,
                                                **kwargs) -> DetailedResponse:
        """
        Remove a network interface from a security group.

        This request removes a network interface from a security group. Security groups
        are stateful, so any changes to a network interface's security groups are applied
        to new connections. Existing connections are not affected. If the network
        interface being removed has no other security groups, it will be attached to the
        VPC's default security group.

        :param str security_group_id: The security group identifier.
        :param str id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if security_group_id is None:
            raise ValueError('security_group_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='remove_security_group_network_interface')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}/network_interfaces/{1}'.format(
            *self.encode_path_vars(security_group_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_security_group_network_interface(self, security_group_id: str,
                                             id: str,
                                             **kwargs) -> DetailedResponse:
        """
        Retrieve a network interface in a security group.

        This request retrieves a single network interface specified by the identifier in
        the URL path. The network interface must be an existing member of the security
        group.

        :param str security_group_id: The security group identifier.
        :param str id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkInterface` object
        """

        if security_group_id is None:
            raise ValueError('security_group_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='get_security_group_network_interface')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}/network_interfaces/{1}'.format(
            *self.encode_path_vars(security_group_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def add_security_group_network_interface(self, security_group_id: str,
                                             id: str,
                                             **kwargs) -> DetailedResponse:
        """
        Add a network interface to a security group.

        This request adds an existing network interface to an existing security group.
        When a network interface is added to a security group, the security group rules
        are applied to the network interface. A request body is not required, and if
        supplied, is ignored.

        :param str security_group_id: The security group identifier.
        :param str id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkInterface` object
        """

        if security_group_id is None:
            raise ValueError('security_group_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='add_security_group_network_interface')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}/network_interfaces/{1}'.format(
            *self.encode_path_vars(security_group_id, id))
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_security_group_rules(self, security_group_id: str,
                                  **kwargs) -> DetailedResponse:
        """
        List all the rules of a security group.

        This request lists all the security group rules for a particular security group.
        These rules define what traffic the security group permits. Security group rules
        are stateful, such that reverse traffic in response to allowed traffic is
        automatically permitted.

        :param str security_group_id: The security group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupRuleCollection` object
        """

        if security_group_id is None:
            raise ValueError('security_group_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_security_group_rules')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}/rules'.format(
            *self.encode_path_vars(security_group_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_security_group_rule(
            self, security_group_id: str,
            security_group_rule_prototype: 'SecurityGroupRulePrototype',
            **kwargs) -> DetailedResponse:
        """
        Create a security group rule.

        This request creates a new security group rule from a security group rule
        prototype object. The prototype object is structured in the same way as a
        retrieved security group rule and contains the information necessary to create the
        rule. As part of creating a new rule in a security group, the rule is applied to
        all the networking interfaces in the security group. Rules specify which IP
        traffic a security group should allow. Security group rules are stateful, such
        that reverse traffic in response to allowed traffic is automatically permitted. A
        rule allowing inbound TCP traffic on port 80 also allows outbound TCP traffic on
        port 80 without the need for an additional rule.

        :param str security_group_id: The security group identifier.
        :param SecurityGroupRulePrototype security_group_rule_prototype: The
               properties of the security group rule to be created.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupRule` object
        """

        if security_group_id is None:
            raise ValueError('security_group_id must be provided')
        if security_group_rule_prototype is None:
            raise ValueError('security_group_rule_prototype must be provided')
        if isinstance(security_group_rule_prototype,
                      SecurityGroupRulePrototype):
            security_group_rule_prototype = convert_model(
                security_group_rule_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_security_group_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(security_group_rule_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}/rules'.format(
            *self.encode_path_vars(security_group_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_security_group_rule(self, security_group_id: str, id: str,
                                   **kwargs) -> DetailedResponse:
        """
        Delete a security group rule.

        This request deletes a security group rule. This operation cannot be reversed.
        Removing a security group rule will not end existing connections allowed by that
        rule.

        :param str security_group_id: The security group identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if security_group_id is None:
            raise ValueError('security_group_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_security_group_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}/rules/{1}'.format(
            *self.encode_path_vars(security_group_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_security_group_rule(self, security_group_id: str, id: str,
                                **kwargs) -> DetailedResponse:
        """
        Retrieve a security group rule.

        This request retrieves a single security group rule specified by the identifier in
        the URL path.

        :param str security_group_id: The security group identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupRule` object
        """

        if security_group_id is None:
            raise ValueError('security_group_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_security_group_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}/rules/{1}'.format(
            *self.encode_path_vars(security_group_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_security_group_rule(
            self, security_group_id: str, id: str,
            security_group_rule_patch: 'SecurityGroupRulePatch',
            **kwargs) -> DetailedResponse:
        """
        Update a security group rule.

        This request updates a security group rule with the information provided in a rule
        patch object. The patch object is structured in the same way as a retrieved
        security group rule and needs to contain only the information to be updated.

        :param str security_group_id: The security group identifier.
        :param str id: The rule identifier.
        :param SecurityGroupRulePatch security_group_rule_patch: The security group
               rule patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupRule` object
        """

        if security_group_id is None:
            raise ValueError('security_group_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if security_group_rule_patch is None:
            raise ValueError('security_group_rule_patch must be provided')
        if isinstance(security_group_rule_patch, SecurityGroupRulePatch):
            security_group_rule_patch = convert_model(security_group_rule_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_security_group_rule')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(security_group_rule_patch)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/security_groups/{0}/rules/{1}'.format(
            *self.encode_path_vars(security_group_id, id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    #########################
    # subnets
    #########################

    def list_subnets(self,
                     *,
                     start: str = None,
                     limit: int = None,
                     **kwargs) -> DetailedResponse:
        """
        List all subnets.

        This request lists all subnets in the region. Subnets are contiguous ranges of IP
        addresses specified in CIDR block notation. Each subnet is within a particular
        zone and cannot span multiple zones or regions.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SubnetCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_subnets')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/subnets'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_subnet(self, subnet_prototype: 'SubnetPrototype',
                      **kwargs) -> DetailedResponse:
        """
        Create a subnet.

        This request creates a new subnet from a subnet prototype object. The prototype
        object is structured in the same way as a retrieved subnet, and contains the
        information necessary to create the new subnet. For this request to succeed, the
        prototype's CIDR block must not overlap with an existing subnet in the VPC.

        :param SubnetPrototype subnet_prototype: The subnet prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Subnet` object
        """

        if subnet_prototype is None:
            raise ValueError('subnet_prototype must be provided')
        if isinstance(subnet_prototype, SubnetPrototype):
            subnet_prototype = convert_model(subnet_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_subnet')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(subnet_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/subnets'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_subnet(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete specified subnet.

        This request deletes a subnet. This operation cannot be reversed. For this request
        to succeed, the subnet must not be referenced by any network interfaces, VPN
        gateways, or load balancers. A delete operation automatically detaches the subnet
        from any network ACLs and public gateways.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_subnet')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/subnets/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_subnet(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve specified subnet.

        This request retrieves a single subnet specified by the identifier in the URL.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Subnet` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_subnet')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/subnets/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_subnet(self,
                      id: str,
                      *,
                      name: str = None,
                      network_acl: 'NetworkACLIdentity' = None,
                      public_gateway: 'PublicGatewayIdentity' = None,
                      **kwargs) -> DetailedResponse:
        """
        Update specified subnet.

        This request updates a subnet with the information in a provided subnet patch. The
        subnet patch object is structured in the same way as a retrieved subnet and
        contains only the information to be updated.

        :param str id: The subnet identifier.
        :param str name: (optional) The user-defined name for this subnet. Names
               must be unique within the VPC the subnet resides in.
        :param NetworkACLIdentity network_acl: (optional) The network ACL to use
               for this subnet.
        :param PublicGatewayIdentity public_gateway: (optional) The public gateway
               to handle internet bound traffic for this subnet.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Subnet` object
        """

        if id is None:
            raise ValueError('id must be provided')
        if network_acl is not None:
            network_acl = convert_model(network_acl)
        if public_gateway is not None:
            public_gateway = convert_model(public_gateway)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_subnet')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'name': name,
            'network_acl': network_acl,
            'public_gateway': public_gateway
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/subnets/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def get_subnet_network_acl(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve a subnet's attached network ACL.

        This request retrieves the network ACL attached to the subnet specified by the
        identifier in the URL.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACL` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_subnet_network_acl')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/subnets/{0}/network_acl'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def replace_subnet_network_acl(self, id: str,
                                   network_acl_identity: 'NetworkACLIdentity',
                                   **kwargs) -> DetailedResponse:
        """
        Attach a network ACL to a subnet.

        This request attaches the network ACL, specified in the request body, to the
        subnet specified by the subnet identifier in the URL. This replaces the existing
        network ACL on the subnet.

        :param str id: The subnet identifier.
        :param NetworkACLIdentity network_acl_identity: The network ACL identity.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACL` object
        """

        if id is None:
            raise ValueError('id must be provided')
        if network_acl_identity is None:
            raise ValueError('network_acl_identity must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='replace_subnet_network_acl')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(network_acl_identity)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/subnets/{0}/network_acl'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def unset_subnet_public_gateway(self, id: str,
                                    **kwargs) -> DetailedResponse:
        """
        Detach a public gateway from a subnet.

        This request detaches the public gateway from the subnet specified by the subnet
        identifier in the URL.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='unset_subnet_public_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/subnets/{0}/public_gateway'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_subnet_public_gateway(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve a subnet's attached public gateway.

        This request retrieves the public gateway attached to the subnet specified by the
        identifier in the URL.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGateway` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_subnet_public_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/subnets/{0}/public_gateway'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def set_subnet_public_gateway(
            self, id: str, public_gateway_identity: 'PublicGatewayIdentity',
            **kwargs) -> DetailedResponse:
        """
        Attach a public gateway to a subnet.

        This request attaches the public gateway, specified in the request body, to the
        subnet specified by the subnet identifier in the URL. The public gateway must have
        the same VPC and zone as the subnet.

        :param str id: The subnet identifier.
        :param PublicGatewayIdentity public_gateway_identity: The public gateway
               identity.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGateway` object
        """

        if id is None:
            raise ValueError('id must be provided')
        if public_gateway_identity is None:
            raise ValueError('public_gateway_identity must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='set_subnet_public_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(public_gateway_identity)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/subnets/{0}/public_gateway'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    #########################
    # vPCs
    #########################

    def list_vpcs(self,
                  *,
                  start: str = None,
                  limit: int = None,
                  classic_access: bool = None,
                  **kwargs) -> DetailedResponse:
        """
        List all VPCs.

        This request lists all VPCs. A VPC is a virtual network that belongs to an account
        and provides logical isolation from other networks. A VPC is made up of resources
        in one or more zones. VPCs are regional, and each VPC can contain resources in
        multiple zones in a region.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param bool classic_access: (optional) The `classic_access` parameter
               filters the returned collection by the supplied field. If the supplied
               field is `true`, only Classic Access VPCs will be returned. If the supplied
               field is `false`, only VPCs without Classic Access will be returned.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPCCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpcs')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'classic_access': classic_access
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_vpc(self,
                   *,
                   address_prefix_management: str = None,
                   classic_access: bool = None,
                   name: str = None,
                   resource_group: 'ResourceGroupIdentity' = None,
                   **kwargs) -> DetailedResponse:
        """
        Create a VPC.

        This request creates a new VPC from a VPC prototype object. The prototype object
        is structured in the same way as a retrieved VPC, and contains the information
        necessary to create the new VPC.

        :param str address_prefix_management: (optional) Indicates whether a
               default address prefix should be automatically created for each zone in
               this VPC. If `manual`, this VPC will be created with no default address
               prefixes.
        :param bool classic_access: (optional) Indicates whether this VPC should be
               connected to Classic Infrastructure. If true, this VPC's resources will
               have private network connectivity to the account's Classic Infrastructure
               resources. Only one VPC, per region, may be connected in this way. This
               value is set at creation and subsequently immutable.
        :param str name: (optional) The unique user-defined name for this VPC. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPC` object
        """

        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpc')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'address_prefix_management': address_prefix_management,
            'classic_access': classic_access,
            'name': name,
            'resource_group': resource_group
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_vpc(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete specified VPC.

        This request deletes a VPC. This operation cannot be reversed. For this request to
        succeed, the VPC must not contain any instances, subnets, or public gateways. All
        security groups associated with the VPC are automatically deleted. If the default
        network ACL was automatically created, it is automatically deleted.

        :param str id: The VPC identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpc')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_vpc(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve specified VPC.

        This request retrieves a single VPC specified by the identifier in the URL.

        :param str id: The VPC identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPC` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_vpc(self,
                   id: str,
                   *,
                   name: str = None,
                   **kwargs) -> DetailedResponse:
        """
        Update specified VPC.

        This request updates a VPC's name.

        :param str id: The VPC identifier.
        :param str name: (optional) The unique user-defined name for this VPC.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPC` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpc')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def get_vpc_default_security_group(self, id: str,
                                       **kwargs) -> DetailedResponse:
        """
        Retrieve a VPC's default security group.

        This request retrieves the default security group for the VPC specified by the
        identifier in the URL. The default security group is applied to any new network
        interfaces in the VPC that do not specify a security group.

        :param str id: The VPC identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DefaultSecurityGroup` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='get_vpc_default_security_group')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/default_security_group'.format(
            *self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_vpc_address_prefixes(self,
                                  vpc_id: str,
                                  *,
                                  start: str = None,
                                  limit: int = None,
                                  **kwargs) -> DetailedResponse:
        """
        List all address pool prefixes for a VPC.

        This request lists all address pool prefixes for a VPC.

        :param str vpc_id: The VPC identifier.
        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `AddressPrefixCollection` object
        """

        if vpc_id is None:
            raise ValueError('vpc_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpc_address_prefixes')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/address_prefixes'.format(
            *self.encode_path_vars(vpc_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_vpc_address_prefix(self,
                                  vpc_id: str,
                                  cidr: str,
                                  zone: 'ZoneIdentity',
                                  *,
                                  is_default: bool = None,
                                  name: str = None,
                                  **kwargs) -> DetailedResponse:
        """
        Create an address pool prefix.

        This request creates a new prefix from a prefix prototype object. The prototype
        object is structured in the same way as a retrieved prefix, and contains the
        information necessary to create the new prefix.

        :param str vpc_id: The VPC identifier.
        :param str cidr: The CIDR block for this address prefix. The request must
               not overlap with any existing address prefixes in the VPC, or the reserved
               CIDR blocks 169.254.0.0/16 and 161.26.0.0/16.
        :param ZoneIdentity zone: The zone this address prefix is to belong to.
        :param bool is_default: (optional) Indicates whether this is the default
               prefix for this zone in this VPC. If true, this prefix will become the
               default prefix for this zone in this VPC. This fails if the VPC currently
               has a default address prefix for this zone.
        :param str name: (optional) The user-defined name for this address prefix.
               Names must be unique within the VPC the address prefix resides in. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `AddressPrefix` object
        """

        if vpc_id is None:
            raise ValueError('vpc_id must be provided')
        if cidr is None:
            raise ValueError('cidr must be provided')
        if zone is None:
            raise ValueError('zone must be provided')
        zone = convert_model(zone)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpc_address_prefix')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'cidr': cidr,
            'zone': zone,
            'is_default': is_default,
            'name': name
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/address_prefixes'.format(
            *self.encode_path_vars(vpc_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_vpc_address_prefix(self, vpc_id: str, id: str,
                                  **kwargs) -> DetailedResponse:
        """
        Delete specified address pool prefix.

        This request deletes a prefix. This operation cannot be reversed. The request will
        fail if any subnets use addresses from this prefix.

        :param str vpc_id: The VPC identifier.
        :param str id: The prefix identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if vpc_id is None:
            raise ValueError('vpc_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpc_address_prefix')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/address_prefixes/{1}'.format(
            *self.encode_path_vars(vpc_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_vpc_address_prefix(self, vpc_id: str, id: str,
                               **kwargs) -> DetailedResponse:
        """
        Retrieve specified address pool prefix.

        This request retrieves a single prefix specified by the identifier in the URL.

        :param str vpc_id: The VPC identifier.
        :param str id: The prefix identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `AddressPrefix` object
        """

        if vpc_id is None:
            raise ValueError('vpc_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc_address_prefix')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/address_prefixes/{1}'.format(
            *self.encode_path_vars(vpc_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_vpc_address_prefix(self,
                                  vpc_id: str,
                                  id: str,
                                  *,
                                  is_default: bool = None,
                                  name: str = None,
                                  **kwargs) -> DetailedResponse:
        """
        Update an address pool prefix.

        This request updates a prefix with the information in a provided prefix patch. The
        prefix patch object is structured in the same way as a retrieved prefix and
        contains only the information to be updated.

        :param str vpc_id: The VPC identifier.
        :param str id: The prefix identifier.
        :param bool is_default: (optional) Indicates whether this is the default
               prefix for this zone in this VPC. Updating to true makes this prefix the
               default prefix for this zone in this VPC, provided the VPC currently has no
               default address prefix for this zone. Updating to false removes the default
               prefix for this zone in this VPC.
        :param str name: (optional) The user-defined name for this address prefix.
               Names must be unique within the VPC the address prefix resides in.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `AddressPrefix` object
        """

        if vpc_id is None:
            raise ValueError('vpc_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpc_address_prefix')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'is_default': is_default, 'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/address_prefixes/{1}'.format(
            *self.encode_path_vars(vpc_id, id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_vpc_routes(self,
                        vpc_id: str,
                        *,
                        zone_name: str = None,
                        **kwargs) -> DetailedResponse:
        """
        List all routes in the VPC's default routing table.

        This request retrieves routes in the VPC's default routing table. For
        compatibility, routes with `action` values other than `deliver` are omitted. Each
        route is zone-specific and directs any packets matching its destination CIDR block
        to a `next_hop` IP address. The most specific route matching a packet's
        destination will be used. If multiple equally-specific routes exist, traffic will
        be distributed across them.

        :param str vpc_id: The VPC identifier.
        :param str zone_name: (optional) Filters the collection to resources in the
               zone with the exact specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RouteCollection` object
        """

        if vpc_id is None:
            raise ValueError('vpc_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpc_routes')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'zone.name': zone_name
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/routes'.format(*self.encode_path_vars(vpc_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_vpc_route(self,
                         vpc_id: str,
                         destination: str,
                         zone: 'ZoneIdentity',
                         *,
                         name: str = None,
                         next_hop: 'RouteNextHopPrototype' = None,
                         **kwargs) -> DetailedResponse:
        """
        Create a route in the VPC's default routing table.

        This request creates a new route in the VPC's default routing table. The route
        prototype object is structured in the same way as a retrieved route, and contains
        the information necessary to create the new route. The request will fail if the
        new route will cause a loop.

        :param str vpc_id: The VPC identifier.
        :param str destination: The destination of the route. At most two routes
               per `zone` in a table can have the same destination, and only if both
               routes have an `action` of `deliver`.
        :param ZoneIdentity zone: The zone to apply the route to. (Traffic from
               subnets in this zone will be
               subject to this route.).
        :param str name: (optional) The user-defined name for this route. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
               Names must be unique within the VPC routing table the route resides in.
        :param RouteNextHopPrototype next_hop: (optional) The next hop that packets
               will be delivered to.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Route` object
        """

        if vpc_id is None:
            raise ValueError('vpc_id must be provided')
        if destination is None:
            raise ValueError('destination must be provided')
        if zone is None:
            raise ValueError('zone must be provided')
        zone = convert_model(zone)
        if next_hop is not None:
            next_hop = convert_model(next_hop)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpc_route')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'destination': destination,
            'zone': zone,
            'name': name,
            'next_hop': next_hop
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/routes'.format(*self.encode_path_vars(vpc_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_vpc_route(self, vpc_id: str, id: str,
                         **kwargs) -> DetailedResponse:
        """
        Delete the specified route in the VPC's default routing table.

        This request deletes a route. This operation cannot be reversed.

        :param str vpc_id: The VPC identifier.
        :param str id: The route identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if vpc_id is None:
            raise ValueError('vpc_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpc_route')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/routes/{1}'.format(*self.encode_path_vars(vpc_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_vpc_route(self, vpc_id: str, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve the specified route in the VPC's default routing table.

        This request retrieves a single route specified by the identifier in the URL.

        :param str vpc_id: The VPC identifier.
        :param str id: The route identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Route` object
        """

        if vpc_id is None:
            raise ValueError('vpc_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc_route')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/routes/{1}'.format(*self.encode_path_vars(vpc_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_vpc_route(self,
                         vpc_id: str,
                         id: str,
                         *,
                         name: str = None,
                         **kwargs) -> DetailedResponse:
        """
        Update the specified route in the VPC's default routing table.

        This request updates a route with the information in a provided route patch. The
        route patch object is structured in the same way as a retrieved route and contains
        only the information to be updated.

        :param str vpc_id: The VPC identifier.
        :param str id: The route identifier.
        :param str name: (optional) The user-defined name for this route. Names
               must be unique within the VPC routing table the route resides in.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Route` object
        """

        if vpc_id is None:
            raise ValueError('vpc_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpc_route')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpcs/{0}/routes/{1}'.format(*self.encode_path_vars(vpc_id, id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    #########################
    # vPN
    #########################

    def list_ike_policies(self,
                          *,
                          start: str = None,
                          limit: int = None,
                          **kwargs) -> DetailedResponse:
        """
        List all IKE policies.

        This request retrieves a paginated list of all IKE policies that belong to this
        account.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IKEPolicyCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_ike_policies')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ike_policies'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_ike_policy(self,
                          authentication_algorithm: str,
                          dh_group: int,
                          encryption_algorithm: str,
                          ike_version: int,
                          *,
                          key_lifetime: int = None,
                          name: str = None,
                          resource_group: 'ResourceGroupIdentity' = None,
                          **kwargs) -> DetailedResponse:
        """
        Create an IKE policy.

        This request creates a new IKE policy.

        :param str authentication_algorithm: The authentication algorithm.
        :param int dh_group: The Diffie-Hellman group.
        :param str encryption_algorithm: The encryption algorithm.
        :param int ike_version: The IKE protocol version.
        :param int key_lifetime: (optional) The key lifetime in seconds.
        :param str name: (optional) The user-defined name for this IKE policy.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IKEPolicy` object
        """

        if authentication_algorithm is None:
            raise ValueError('authentication_algorithm must be provided')
        if dh_group is None:
            raise ValueError('dh_group must be provided')
        if encryption_algorithm is None:
            raise ValueError('encryption_algorithm must be provided')
        if ike_version is None:
            raise ValueError('ike_version must be provided')
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_ike_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'authentication_algorithm': authentication_algorithm,
            'dh_group': dh_group,
            'encryption_algorithm': encryption_algorithm,
            'ike_version': ike_version,
            'key_lifetime': key_lifetime,
            'name': name,
            'resource_group': resource_group
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ike_policies'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_ike_policy(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete an IKE policy.

        This request deletes an IKE policy. This operation cannot be reversed.

        :param str id: The IKE policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_ike_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ike_policies/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_ike_policy(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve the specified IKE policy.

        This request retrieves a single IKE policy specified by the identifier in the URL.

        :param str id: The IKE policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IKEPolicy` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_ike_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ike_policies/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_ike_policy(self,
                          id: str,
                          *,
                          authentication_algorithm: str = None,
                          dh_group: int = None,
                          encryption_algorithm: str = None,
                          ike_version: int = None,
                          key_lifetime: int = None,
                          name: str = None,
                          **kwargs) -> DetailedResponse:
        """
        Update an IKE policy.

        This request updates the properties of an existing IKE policy.

        :param str id: The IKE policy identifier.
        :param str authentication_algorithm: (optional) The authentication
               algorithm.
        :param int dh_group: (optional) The Diffie-Hellman group.
        :param str encryption_algorithm: (optional) The encryption algorithm.
        :param int ike_version: (optional) The IKE protocol version.
        :param int key_lifetime: (optional) The key lifetime in seconds.
        :param str name: (optional) The user-defined name for this IKE policy.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IKEPolicy` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_ike_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'authentication_algorithm': authentication_algorithm,
            'dh_group': dh_group,
            'encryption_algorithm': encryption_algorithm,
            'ike_version': ike_version,
            'key_lifetime': key_lifetime,
            'name': name
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ike_policies/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_ike_policy_connections(self, id: str,
                                    **kwargs) -> DetailedResponse:
        """
        List all connections that use the specified IKE policy.

        This request lists all the connections that use the specified policy.

        :param str id: The IKE policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnectionCollection` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_ike_policy_connections')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ike_policies/{0}/connections'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_ipsec_policies(self,
                            *,
                            start: str = None,
                            limit: int = None,
                            **kwargs) -> DetailedResponse:
        """
        List all IPsec policies.

        This request retrieves a paginated list of all IPsec policies that belong to this
        account.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IPsecPolicyCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_ipsec_policies')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ipsec_policies'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_ipsec_policy(self,
                            authentication_algorithm: str,
                            encryption_algorithm: str,
                            pfs: str,
                            *,
                            key_lifetime: int = None,
                            name: str = None,
                            resource_group: 'ResourceGroupIdentity' = None,
                            **kwargs) -> DetailedResponse:
        """
        Create an IPsec policy.

        This request creates a new IPsec policy.

        :param str authentication_algorithm: The authentication algorithm.
        :param str encryption_algorithm: The encryption algorithm.
        :param str pfs: Perfect Forward Secrecy.
        :param int key_lifetime: (optional) The key lifetime in seconds.
        :param str name: (optional) The user-defined name for this IPsec policy.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IPsecPolicy` object
        """

        if authentication_algorithm is None:
            raise ValueError('authentication_algorithm must be provided')
        if encryption_algorithm is None:
            raise ValueError('encryption_algorithm must be provided')
        if pfs is None:
            raise ValueError('pfs must be provided')
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_ipsec_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'authentication_algorithm': authentication_algorithm,
            'encryption_algorithm': encryption_algorithm,
            'pfs': pfs,
            'key_lifetime': key_lifetime,
            'name': name,
            'resource_group': resource_group
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ipsec_policies'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_ipsec_policy(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete an IPsec policy.

        This request deletes an IPsec policy. This operation cannot be reversed.

        :param str id: The IPsec policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_ipsec_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ipsec_policies/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_ipsec_policy(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve the specified IPsec policy.

        This request retrieves a single IPsec policy specified by the identifier in the
        URL.

        :param str id: The IPsec policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IPsecPolicy` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_ipsec_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ipsec_policies/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_ipsec_policy(self,
                            id: str,
                            *,
                            authentication_algorithm: str = None,
                            encryption_algorithm: str = None,
                            key_lifetime: int = None,
                            name: str = None,
                            pfs: str = None,
                            **kwargs) -> DetailedResponse:
        """
        Update an IPsec policy.

        This request updates the properties of an existing IPsec policy.

        :param str id: The IPsec policy identifier.
        :param str authentication_algorithm: (optional) The authentication
               algorithm.
        :param str encryption_algorithm: (optional) The encryption algorithm.
        :param int key_lifetime: (optional) The key lifetime in seconds.
        :param str name: (optional) The user-defined name for this IPsec policy.
        :param str pfs: (optional) Perfect Forward Secrecy.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IPsecPolicy` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_ipsec_policy')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'authentication_algorithm': authentication_algorithm,
            'encryption_algorithm': encryption_algorithm,
            'key_lifetime': key_lifetime,
            'name': name,
            'pfs': pfs
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ipsec_policies/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_ipsec_policy_connections(self, id: str,
                                      **kwargs) -> DetailedResponse:
        """
        List all connections that use the specified IPsec policy.

        This request lists all the connections that use the specified policy.

        :param str id: The IPsec policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnectionCollection` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_ipsec_policy_connections')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/ipsec_policies/{0}/connections'.format(
            *self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_vpn_gateways(self,
                          *,
                          start: str = None,
                          limit: int = None,
                          resource_group_id: str = None,
                          **kwargs) -> DetailedResponse:
        """
        List all VPN gateways.

        This request retrieves a paginated list of all VPN gateways that belong to this
        account.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources within one of the resource groups identified in a comma-separated
               list of resource group identifiers.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpn_gateways')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_vpn_gateway(self,
                           subnet: 'SubnetIdentity',
                           *,
                           name: str = None,
                           resource_group: 'ResourceGroupIdentity' = None,
                           **kwargs) -> DetailedResponse:
        """
        Create a VPN gateway.

        This request creates a new VPN gateway.

        :param SubnetIdentity subnet: Identifies a subnet by a unique property.
        :param str name: (optional) The user-defined name for this VPN gateway.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGateway` object
        """

        if subnet is None:
            raise ValueError('subnet must be provided')
        subnet = convert_model(subnet)
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpn_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'subnet': subnet,
            'name': name,
            'resource_group': resource_group
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_vpn_gateway(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete a VPN gateway.

        This request deletes a VPN gateway. A VPN gateway with a `status` of `pending`
        cannot be deleted. This operation deletes all VPN connections associated with this
        VPN gateway.  This operation cannot be reversed.

        :param str id: The VPN gateway identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpn_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_vpn_gateway(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve the specified VPN gateway.

        This request retrieves a single VPN gateway specified by the identifier in the
        URL.

        :param str id: The VPN gateway identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGateway` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpn_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_vpn_gateway(self,
                           id: str,
                           *,
                           name: str = None,
                           **kwargs) -> DetailedResponse:
        """
        Update a VPN gateway.

        This request updates the properties of an existing VPN gateway.

        :param str id: The VPN gateway identifier.
        :param str name: (optional) The user-defined name for this VPN gateway.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGateway` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpn_gateway')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_vpn_gateway_connections(self,
                                     vpn_gateway_id: str,
                                     *,
                                     status: str = None,
                                     **kwargs) -> DetailedResponse:
        """
        List all connections of a VPN gateway.

        This request lists all the connections of a particular VPN gateway.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str status: (optional) Filters the collection to VPN connections
               with the specified status.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnectionCollection` object
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_vpn_gateway_connections')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'status': status
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections'.format(
            *self.encode_path_vars(vpn_gateway_id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_vpn_gateway_connection(
            self,
            vpn_gateway_id: str,
            peer_address: str,
            psk: str,
            *,
            admin_state_up: bool = None,
            dead_peer_detection: 'VPNGatewayConnectionDPDPrototype' = None,
            ike_policy: 'IKEPolicyIdentity' = None,
            ipsec_policy: 'IPsecPolicyIdentity' = None,
            local_cidrs: List[str] = None,
            name: str = None,
            peer_cidrs: List[str] = None,
            **kwargs) -> DetailedResponse:
        """
        Create a VPN connection.

        This request creates a new VPN connection.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str peer_address: The IP address of the peer VPN gateway.
        :param str psk: The preshared key.
        :param bool admin_state_up: (optional) If set to false, the VPN connection
               is shut down.
        :param VPNGatewayConnectionDPDPrototype dead_peer_detection: (optional) The
               Dead Peer Detection settings.
        :param IKEPolicyIdentity ike_policy: (optional) Optional IKE policy
               configuration. The absence of a policy indicates autonegotiation.
        :param IPsecPolicyIdentity ipsec_policy: (optional) Optional IPsec policy
               configuration. The absence of a policy indicates
               autonegotiation.
        :param List[str] local_cidrs: (optional) A collection of local CIDRs for
               this resource.
        :param str name: (optional) The user-defined name for this VPN connection.
        :param List[str] peer_cidrs: (optional) A collection of peer CIDRs for this
               resource.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnection` object
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if peer_address is None:
            raise ValueError('peer_address must be provided')
        if psk is None:
            raise ValueError('psk must be provided')
        if dead_peer_detection is not None:
            dead_peer_detection = convert_model(dead_peer_detection)
        if ike_policy is not None:
            ike_policy = convert_model(ike_policy)
        if ipsec_policy is not None:
            ipsec_policy = convert_model(ipsec_policy)
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='create_vpn_gateway_connection')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'peer_address': peer_address,
            'psk': psk,
            'admin_state_up': admin_state_up,
            'dead_peer_detection': dead_peer_detection,
            'ike_policy': ike_policy,
            'ipsec_policy': ipsec_policy,
            'local_cidrs': local_cidrs,
            'name': name,
            'peer_cidrs': peer_cidrs
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections'.format(
            *self.encode_path_vars(vpn_gateway_id))
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_vpn_gateway_connection(self, vpn_gateway_id: str, id: str,
                                      **kwargs) -> DetailedResponse:
        """
        Delete a VPN connection.

        This request deletes a VPN connection. This operation cannot be reversed.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='delete_vpn_gateway_connection')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}'.format(
            *self.encode_path_vars(vpn_gateway_id, id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_vpn_gateway_connection(self, vpn_gateway_id: str, id: str,
                                   **kwargs) -> DetailedResponse:
        """
        Retrieve the specified VPN connection.

        This request retrieves a single VPN connection specified by the identifier in the
        URL.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnection` object
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpn_gateway_connection')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}'.format(
            *self.encode_path_vars(vpn_gateway_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_vpn_gateway_connection(
            self,
            vpn_gateway_id: str,
            id: str,
            *,
            admin_state_up: bool = None,
            dead_peer_detection: 'VPNGatewayConnectionDPDPrototype' = None,
            ike_policy: 'IKEPolicyIdentity' = None,
            ipsec_policy: 'IPsecPolicyIdentity' = None,
            name: str = None,
            peer_address: str = None,
            psk: str = None,
            **kwargs) -> DetailedResponse:
        """
        Update a VPN connection.

        This request updates the properties of an existing VPN connection.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param bool admin_state_up: (optional) If set to false, the VPN connection
               is shut down.
        :param VPNGatewayConnectionDPDPrototype dead_peer_detection: (optional) The
               Dead Peer Detection settings.
        :param IKEPolicyIdentity ike_policy: (optional) Optional IKE policy
               configuration. The absence of a policy indicates autonegotiation.
        :param IPsecPolicyIdentity ipsec_policy: (optional) Optional IPsec policy
               configuration. The absence of a policy indicates
               autonegotiation.
        :param str name: (optional) The user-defined name for this VPN connection.
        :param str peer_address: (optional) The IP address of the peer VPN gateway.
        :param str psk: (optional) The preshared key.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnection` object
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if dead_peer_detection is not None:
            dead_peer_detection = convert_model(dead_peer_detection)
        if ike_policy is not None:
            ike_policy = convert_model(ike_policy)
        if ipsec_policy is not None:
            ipsec_policy = convert_model(ipsec_policy)
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='update_vpn_gateway_connection')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {
            'admin_state_up': admin_state_up,
            'dead_peer_detection': dead_peer_detection,
            'ike_policy': ike_policy,
            'ipsec_policy': ipsec_policy,
            'name': name,
            'peer_address': peer_address,
            'psk': psk
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}'.format(
            *self.encode_path_vars(vpn_gateway_id, id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def list_vpn_gateway_connection_local_cidrs(self, vpn_gateway_id: str,
                                                id: str,
                                                **kwargs) -> DetailedResponse:
        """
        List all local CIDRs for a VPN gateway connection.

        This request lists all local CIDRs for a VPN gateway connection specified by the
        identifier in the URL.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnectionLocalCIDRs` object
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_vpn_gateway_connection_local_cidrs')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}/local_cidrs'.format(
            *self.encode_path_vars(vpn_gateway_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def remove_vpn_gateway_connection_local_cidr(self, vpn_gateway_id: str,
                                                 id: str, cidr_prefix: str,
                                                 prefix_length: str,
                                                 **kwargs) -> DetailedResponse:
        """
        Remove a local CIDR from a VPN gateway connection.

        This request removes a CIDR from a VPN gateway connection.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if cidr_prefix is None:
            raise ValueError('cidr_prefix must be provided')
        if prefix_length is None:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='remove_vpn_gateway_connection_local_cidr')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}/local_cidrs/{2}/{3}'.format(
            *self.encode_path_vars(vpn_gateway_id, id, cidr_prefix,
                                   prefix_length))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def check_vpn_gateway_connection_local_cidr(self, vpn_gateway_id: str,
                                                id: str, cidr_prefix: str,
                                                prefix_length: str,
                                                **kwargs) -> DetailedResponse:
        """
        Check if the specified local CIDR exists on a VPN gateway connection.

        This request succeeds if a CIDR exists on the specified VPN gateway connection and
        fails otherwise.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if cidr_prefix is None:
            raise ValueError('cidr_prefix must be provided')
        if prefix_length is None:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='check_vpn_gateway_connection_local_cidr')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}/local_cidrs/{2}/{3}'.format(
            *self.encode_path_vars(vpn_gateway_id, id, cidr_prefix,
                                   prefix_length))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def add_vpn_gateway_connection_local_cidr(self, vpn_gateway_id: str,
                                              id: str, cidr_prefix: str,
                                              prefix_length: str,
                                              **kwargs) -> DetailedResponse:
        """
        Set a local CIDR on a VPN gateway connection.

        This request adds the specified CIDR to the specified VPN gateway connection. A
        request body is not required, and if supplied, is ignored. This request succeeds
        if the CIDR already exists on the specified VPN gateway connection.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if cidr_prefix is None:
            raise ValueError('cidr_prefix must be provided')
        if prefix_length is None:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='add_vpn_gateway_connection_local_cidr')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}/local_cidrs/{2}/{3}'.format(
            *self.encode_path_vars(vpn_gateway_id, id, cidr_prefix,
                                   prefix_length))
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_vpn_gateway_connection_peer_cidrs(self, vpn_gateway_id: str,
                                               id: str,
                                               **kwargs) -> DetailedResponse:
        """
        List all peer CIDRs for a VPN gateway connection.

        This request lists all peer CIDRs for a VPN gateway connection specified by the
        identifier in the URL.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnectionPeerCIDRs` object
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='list_vpn_gateway_connection_peer_cidrs')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}/peer_cidrs'.format(
            *self.encode_path_vars(vpn_gateway_id, id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def remove_vpn_gateway_connection_peer_cidr(self, vpn_gateway_id: str,
                                                id: str, cidr_prefix: str,
                                                prefix_length: str,
                                                **kwargs) -> DetailedResponse:
        """
        Remove a peer CIDR from a VPN gateway connection.

        This request removes a CIDR from a VPN gateway connection.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if cidr_prefix is None:
            raise ValueError('cidr_prefix must be provided')
        if prefix_length is None:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='remove_vpn_gateway_connection_peer_cidr')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}/peer_cidrs/{2}/{3}'.format(
            *self.encode_path_vars(vpn_gateway_id, id, cidr_prefix,
                                   prefix_length))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def check_vpn_gateway_connection_peer_cidr(self, vpn_gateway_id: str,
                                               id: str, cidr_prefix: str,
                                               prefix_length: str,
                                               **kwargs) -> DetailedResponse:
        """
        Check if the specified peer CIDR exists on a VPN gateway connection.

        This request succeeds if a CIDR exists on the specified VPN gateway connection and
        fails otherwise.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if cidr_prefix is None:
            raise ValueError('cidr_prefix must be provided')
        if prefix_length is None:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='check_vpn_gateway_connection_peer_cidr')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}/peer_cidrs/{2}/{3}'.format(
            *self.encode_path_vars(vpn_gateway_id, id, cidr_prefix,
                                   prefix_length))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def add_vpn_gateway_connection_peer_cidr(self, vpn_gateway_id: str, id: str,
                                             cidr_prefix: str,
                                             prefix_length: str,
                                             **kwargs) -> DetailedResponse:
        """
        Set a peer CIDR on a VPN gateway connection.

        This request adds the specified CIDR to the specified VPN gateway connection. A
        request body is not required, and if supplied, is ignored. This request succeeds
        if the CIDR already exists on the specified VPN gateway connection.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if vpn_gateway_id is None:
            raise ValueError('vpn_gateway_id must be provided')
        if id is None:
            raise ValueError('id must be provided')
        if cidr_prefix is None:
            raise ValueError('cidr_prefix must be provided')
        if prefix_length is None:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(
            service_name=self.DEFAULT_SERVICE_NAME,
            service_version='V1',
            operation_id='add_vpn_gateway_connection_peer_cidr')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/vpn_gateways/{0}/connections/{1}/peer_cidrs/{2}/{3}'.format(
            *self.encode_path_vars(vpn_gateway_id, id, cidr_prefix,
                                   prefix_length))
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    #########################
    # volumes
    #########################

    def list_volume_profiles(self,
                             *,
                             start: str = None,
                             limit: int = None,
                             **kwargs) -> DetailedResponse:
        """
        List all volume profiles.

        This request lists all volume profiles available in the region. A volume profile
        specifies the performance characteristics and pricing model for a volume.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeProfileCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_volume_profiles')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/volume/profiles'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_volume_profile(self, name: str, **kwargs) -> DetailedResponse:
        """
        Retrieve specified volume profile.

        This request retrieves a single volume profile specified by the name in the URL.

        :param str name: The volume profile name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeProfile` object
        """

        if name is None:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_volume_profile')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/volume/profiles/{0}'.format(*self.encode_path_vars(name))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def list_volumes(self,
                     *,
                     start: str = None,
                     limit: int = None,
                     name: str = None,
                     zone_name: str = None,
                     **kwargs) -> DetailedResponse:
        """
        List all volumes.

        This request lists all volumes in the region. Volumes are network-connected block
        storage devices that may be attached to one or more instances in the same region.

        :param str start: (optional) A server-supplied token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str zone_name: (optional) Filters the collection to resources in the
               zone with the exact specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_volumes')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'name': name,
            'zone.name': zone_name
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/volumes'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def create_volume(self, volume_prototype: 'VolumePrototype',
                      **kwargs) -> DetailedResponse:
        """
        Create a volume.

        This request creates a new volume from a volume prototype object. The prototype
        object is structured in the same way as a retrieved volume, and contains the
        information necessary to create the new volume.

        :param VolumePrototype volume_prototype: The volume prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Volume` object
        """

        if volume_prototype is None:
            raise ValueError('volume_prototype must be provided')
        if isinstance(volume_prototype, VolumePrototype):
            volume_prototype = convert_model(volume_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_volume')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = json.dumps(volume_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/volumes'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response

    def delete_volume(self, id: str, **kwargs) -> DetailedResponse:
        """
        Delete specified volume.

        This request deletes a volume. This operation cannot be reversed. For this request
        to succeed, the volume must not be attached to any instances.

        :param str id: The volume identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_volume')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/volumes/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def get_volume(self, id: str, **kwargs) -> DetailedResponse:
        """
        Retrieve specified volume.

        This request retrieves a single volume specified by the identifier in the URL.

        :param str id: The volume identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Volume` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_volume')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/volumes/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response

    def update_volume(self,
                      id: str,
                      *,
                      name: str = None,
                      **kwargs) -> DetailedResponse:
        """
        Update specified volume.

        This request updates a volume with the information in a provided volume patch. The
        volume patch object is structured in the same way as a retrieved volume and
        contains only the information to be updated.

        :param str id: The volume identifier.
        :param str name: (optional) The unique user-defined name for this volume.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Volume` object
        """

        if id is None:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_volume')
        headers.update(sdk_headers)

        params = {'version': self.version, 'generation': self.generation}

        data = {'name': name}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        url = '/volumes/{0}'.format(*self.encode_path_vars(id))
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request)
        return response


class ListImagesEnums:
    """
    Enums for list_images parameters.
    """

    class Visibility(str, Enum):
        """
        Filters the collection to images with the specified visibility.
        """
        PRIVATE = 'private'
        PUBLIC = 'public'


class ListNetworkAclRulesEnums:
    """
    Enums for list_network_acl_rules parameters.
    """

    class Direction(str, Enum):
        """
        Filters the collection to rules with the specified direction.
        """
        INBOUND = 'inbound'
        OUTBOUND = 'outbound'


##############################################################################
# Models
##############################################################################


class AddressPrefix():
    """
    AddressPrefix.

    :attr str cidr: The CIDR block for this prefix.
    :attr datetime created_at: The date and time that the prefix was created.
    :attr bool has_subnets: Indicates whether subnets exist with addresses from this
          prefix.
    :attr str href: The URL for this address prefix.
    :attr str id: The unique identifier for this address prefix.
    :attr bool is_default: Indicates whether this is the default prefix for this
          zone in this VPC. If a default prefix was automatically created when the VPC was
          created, the prefix is automatically named using a hyphenated list of
          randomly-selected words, but may be updated with a user-specified name.
    :attr str name: The user-defined name for this address prefix. Names must be
          unique within the VPC the address prefix resides in.
    :attr ZoneReference zone: The zone this address prefix resides in.
    """

    def __init__(self, cidr: str, created_at: datetime, has_subnets: bool,
                 href: str, id: str, is_default: bool, name: str,
                 zone: 'ZoneReference') -> None:
        """
        Initialize a AddressPrefix object.

        :param str cidr: The CIDR block for this prefix.
        :param datetime created_at: The date and time that the prefix was created.
        :param bool has_subnets: Indicates whether subnets exist with addresses
               from this prefix.
        :param str href: The URL for this address prefix.
        :param str id: The unique identifier for this address prefix.
        :param bool is_default: Indicates whether this is the default prefix for
               this zone in this VPC. If a default prefix was automatically created when
               the VPC was created, the prefix is automatically named using a hyphenated
               list of randomly-selected words, but may be updated with a user-specified
               name.
        :param str name: The user-defined name for this address prefix. Names must
               be unique within the VPC the address prefix resides in.
        :param ZoneReference zone: The zone this address prefix resides in.
        """
        self.cidr = cidr
        self.created_at = created_at
        self.has_subnets = has_subnets
        self.href = href
        self.id = id
        self.is_default = is_default
        self.name = name
        self.zone = zone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'AddressPrefix':
        """Initialize a AddressPrefix object from a json dictionary."""
        args = {}
        if 'cidr' in _dict:
            args['cidr'] = _dict.get('cidr')
        else:
            raise ValueError(
                'Required property \'cidr\' not present in AddressPrefix JSON')
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in AddressPrefix JSON'
            )
        if 'has_subnets' in _dict:
            args['has_subnets'] = _dict.get('has_subnets')
        else:
            raise ValueError(
                'Required property \'has_subnets\' not present in AddressPrefix JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in AddressPrefix JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in AddressPrefix JSON')
        if 'is_default' in _dict:
            args['is_default'] = _dict.get('is_default')
        else:
            raise ValueError(
                'Required property \'is_default\' not present in AddressPrefix JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in AddressPrefix JSON')
        if 'zone' in _dict:
            args['zone'] = ZoneReference.from_dict(_dict.get('zone'))
        else:
            raise ValueError(
                'Required property \'zone\' not present in AddressPrefix JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a AddressPrefix object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'cidr') and self.cidr is not None:
            _dict['cidr'] = self.cidr
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'has_subnets') and self.has_subnets is not None:
            _dict['has_subnets'] = self.has_subnets
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'is_default') and self.is_default is not None:
            _dict['is_default'] = self.is_default
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'zone') and self.zone is not None:
            _dict['zone'] = self.zone.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this AddressPrefix object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'AddressPrefix') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'AddressPrefix') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class AddressPrefixCollection():
    """
    AddressPrefixCollection.

    :attr List[AddressPrefix] address_prefixes: Collection of address prefixes.
    :attr AddressPrefixCollectionFirst first: A reference to the first page of
          resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr AddressPrefixCollectionNext next: (optional) A reference to the next page
          of resources; this reference is included for all pages
          except the last page.
    """

    def __init__(self,
                 address_prefixes: List['AddressPrefix'],
                 first: 'AddressPrefixCollectionFirst',
                 limit: int,
                 *,
                 next: 'AddressPrefixCollectionNext' = None) -> None:
        """
        Initialize a AddressPrefixCollection object.

        :param List[AddressPrefix] address_prefixes: Collection of address
               prefixes.
        :param AddressPrefixCollectionFirst first: A reference to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param AddressPrefixCollectionNext next: (optional) A reference to the next
               page of resources; this reference is included for all pages
               except the last page.
        """
        self.address_prefixes = address_prefixes
        self.first = first
        self.limit = limit
        self.next = next

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'AddressPrefixCollection':
        """Initialize a AddressPrefixCollection object from a json dictionary."""
        args = {}
        if 'address_prefixes' in _dict:
            args['address_prefixes'] = [
                AddressPrefix.from_dict(x)
                for x in _dict.get('address_prefixes')
            ]
        else:
            raise ValueError(
                'Required property \'address_prefixes\' not present in AddressPrefixCollection JSON'
            )
        if 'first' in _dict:
            args['first'] = AddressPrefixCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in AddressPrefixCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in AddressPrefixCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = AddressPrefixCollectionNext.from_dict(
                _dict.get('next'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a AddressPrefixCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self,
                   'address_prefixes') and self.address_prefixes is not None:
            _dict['address_prefixes'] = [
                x.to_dict() for x in self.address_prefixes
            ]
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this AddressPrefixCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'AddressPrefixCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'AddressPrefixCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class AddressPrefixCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a AddressPrefixCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'AddressPrefixCollectionFirst':
        """Initialize a AddressPrefixCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in AddressPrefixCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a AddressPrefixCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this AddressPrefixCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'AddressPrefixCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'AddressPrefixCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class AddressPrefixCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a AddressPrefixCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'AddressPrefixCollectionNext':
        """Initialize a AddressPrefixCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in AddressPrefixCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a AddressPrefixCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this AddressPrefixCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'AddressPrefixCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'AddressPrefixCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class CertificateInstanceIdentity():
    """
    Identifies a certificate instance by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a CertificateInstanceIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['CertificateInstanceIdentityByCRN']))
        raise Exception(msg)


class CertificateInstanceReference():
    """
    CertificateInstanceReference.

    :attr str crn: The CRN for this certificate instance.
    """

    def __init__(self, crn: str) -> None:
        """
        Initialize a CertificateInstanceReference object.

        :param str crn: The CRN for this certificate instance.
        """
        self.crn = crn

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'CertificateInstanceReference':
        """Initialize a CertificateInstanceReference object from a json dictionary."""
        args = {}
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in CertificateInstanceReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a CertificateInstanceReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this CertificateInstanceReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'CertificateInstanceReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'CertificateInstanceReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class DefaultSecurityGroup():
    """
    Collection of rules in a default security group.

    :attr datetime created_at: The date and time that this security group was
          created.
    :attr str crn: The security group's CRN.
    :attr str href: The security group's canonical URL.
    :attr str id: The unique identifier for this security group.
    :attr str name: The name of the default security group created for a VPC. The
          name will be a hyphenated list of randomly-selected words at creation, but may
          be user-specified with a subsequent request.
    :attr List[SecurityGroupRule] rules: Array of rules for the default security
          group for a VPC. Defaults to allowing all outbound traffic, and allowing all
          inbound traffic from other interfaces in the VPC’s default security group. Rules
          in the default security group may be changed, added or removed.
    :attr VPCReference vpc: The VPC this security group is a part of.
    """

    def __init__(self, created_at: datetime, crn: str, href: str, id: str,
                 name: str, rules: List['SecurityGroupRule'],
                 vpc: 'VPCReference') -> None:
        """
        Initialize a DefaultSecurityGroup object.

        :param datetime created_at: The date and time that this security group was
               created.
        :param str crn: The security group's CRN.
        :param str href: The security group's canonical URL.
        :param str id: The unique identifier for this security group.
        :param str name: The name of the default security group created for a VPC.
               The name will be a hyphenated list of randomly-selected words at creation,
               but may be user-specified with a subsequent request.
        :param List[SecurityGroupRule] rules: Array of rules for the default
               security group for a VPC. Defaults to allowing all outbound traffic, and
               allowing all inbound traffic from other interfaces in the VPC’s default
               security group. Rules in the default security group may be changed, added
               or removed.
        :param VPCReference vpc: The VPC this security group is a part of.
        """
        self.created_at = created_at
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name
        self.rules = rules
        self.vpc = vpc

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'DefaultSecurityGroup':
        """Initialize a DefaultSecurityGroup object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in DefaultSecurityGroup JSON'
            )
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in DefaultSecurityGroup JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in DefaultSecurityGroup JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in DefaultSecurityGroup JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in DefaultSecurityGroup JSON'
            )
        if 'rules' in _dict:
            args['rules'] = [
                SecurityGroupRule.from_dict(x) for x in _dict.get('rules')
            ]
        else:
            raise ValueError(
                'Required property \'rules\' not present in DefaultSecurityGroup JSON'
            )
        if 'vpc' in _dict:
            args['vpc'] = VPCReference.from_dict(_dict.get('vpc'))
        else:
            raise ValueError(
                'Required property \'vpc\' not present in DefaultSecurityGroup JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a DefaultSecurityGroup object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'rules') and self.rules is not None:
            _dict['rules'] = [x.to_dict() for x in self.rules]
        if hasattr(self, 'vpc') and self.vpc is not None:
            _dict['vpc'] = self.vpc.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this DefaultSecurityGroup object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'DefaultSecurityGroup') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'DefaultSecurityGroup') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class EncryptionKeyIdentity():
    """
    Identifies an encryption key by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a EncryptionKeyIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['EncryptionKeyIdentityByCRN']))
        raise Exception(msg)


class EncryptionKeyReference():
    """
    EncryptionKeyReference.

    :attr str crn: The CRN of the [Key Protect Root
          Key](https://cloud.ibm.com/docs/key-protect?topic=key-protect-getting-started-tutorial)
          or [Hyper Protect Crypto Service Root
          Key](https://cloud.ibm.com/docs/hs-crypto?topic=hs-crypto-get-started) for this
          resource.
    """

    def __init__(self, crn: str) -> None:
        """
        Initialize a EncryptionKeyReference object.

        :param str crn: The CRN of the [Key Protect Root
               Key](https://cloud.ibm.com/docs/key-protect?topic=key-protect-getting-started-tutorial)
               or [Hyper Protect Crypto Service Root
               Key](https://cloud.ibm.com/docs/hs-crypto?topic=hs-crypto-get-started) for
               this resource.
        """
        self.crn = crn

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'EncryptionKeyReference':
        """Initialize a EncryptionKeyReference object from a json dictionary."""
        args = {}
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in EncryptionKeyReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a EncryptionKeyReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this EncryptionKeyReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'EncryptionKeyReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'EncryptionKeyReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class FloatingIP():
    """
    FloatingIP.

    :attr str address: The globally unique IP address.
    :attr datetime created_at: The date and time that the floating IP was created.
    :attr str crn: The CRN for this floating IP.
    :attr str href: The URL for this floating IP.
    :attr str id: The unique identifier for this floating IP.
    :attr str name: The user-defined name for this floating IP.
    :attr str status: The status of the floating IP.
    :attr FloatingIPTarget target: (optional) The target of this floating IP.
    :attr ZoneReference zone: The zone the floating IP resides in.
    """

    def __init__(self,
                 address: str,
                 created_at: datetime,
                 crn: str,
                 href: str,
                 id: str,
                 name: str,
                 status: str,
                 zone: 'ZoneReference',
                 *,
                 target: 'FloatingIPTarget' = None) -> None:
        """
        Initialize a FloatingIP object.

        :param str address: The globally unique IP address.
        :param datetime created_at: The date and time that the floating IP was
               created.
        :param str crn: The CRN for this floating IP.
        :param str href: The URL for this floating IP.
        :param str id: The unique identifier for this floating IP.
        :param str name: The user-defined name for this floating IP.
        :param str status: The status of the floating IP.
        :param ZoneReference zone: The zone the floating IP resides in.
        :param FloatingIPTarget target: (optional) The target of this floating IP.
        """
        self.address = address
        self.created_at = created_at
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name
        self.status = status
        self.target = target
        self.zone = zone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'FloatingIP':
        """Initialize a FloatingIP object from a json dictionary."""
        args = {}
        if 'address' in _dict:
            args['address'] = _dict.get('address')
        else:
            raise ValueError(
                'Required property \'address\' not present in FloatingIP JSON')
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in FloatingIP JSON'
            )
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in FloatingIP JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in FloatingIP JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in FloatingIP JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in FloatingIP JSON')
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in FloatingIP JSON')
        if 'target' in _dict:
            args['target'] = _dict.get('target')
        if 'zone' in _dict:
            args['zone'] = ZoneReference.from_dict(_dict.get('zone'))
        else:
            raise ValueError(
                'Required property \'zone\' not present in FloatingIP JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a FloatingIP object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'address') and self.address is not None:
            _dict['address'] = self.address
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'target') and self.target is not None:
            _dict['target'] = self.target
        if hasattr(self, 'zone') and self.zone is not None:
            _dict['zone'] = self.zone.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this FloatingIP object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'FloatingIP') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'FloatingIP') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class StatusEnum(str, Enum):
        """
        The status of the floating IP.
        """
        AVAILABLE = 'available'
        DELETING = 'deleting'
        FAILED = 'failed'
        PENDING = 'pending'


class FloatingIPCollection():
    """
    FloatingIPCollection.

    :attr FloatingIPCollectionFirst first: A reference to the first page of
          resources.
    :attr List[FloatingIP] floating_ips: Collection of floating IPs.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr FloatingIPCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    """

    def __init__(self,
                 first: 'FloatingIPCollectionFirst',
                 floating_ips: List['FloatingIP'],
                 limit: int,
                 *,
                 next: 'FloatingIPCollectionNext' = None) -> None:
        """
        Initialize a FloatingIPCollection object.

        :param FloatingIPCollectionFirst first: A reference to the first page of
               resources.
        :param List[FloatingIP] floating_ips: Collection of floating IPs.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param FloatingIPCollectionNext next: (optional) A reference to the next
               page of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.floating_ips = floating_ips
        self.limit = limit
        self.next = next

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'FloatingIPCollection':
        """Initialize a FloatingIPCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = FloatingIPCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in FloatingIPCollection JSON'
            )
        if 'floating_ips' in _dict:
            args['floating_ips'] = [
                FloatingIP.from_dict(x) for x in _dict.get('floating_ips')
            ]
        else:
            raise ValueError(
                'Required property \'floating_ips\' not present in FloatingIPCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in FloatingIPCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = FloatingIPCollectionNext.from_dict(_dict.get('next'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a FloatingIPCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'floating_ips') and self.floating_ips is not None:
            _dict['floating_ips'] = [x.to_dict() for x in self.floating_ips]
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this FloatingIPCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'FloatingIPCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'FloatingIPCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class FloatingIPCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a FloatingIPCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'FloatingIPCollectionFirst':
        """Initialize a FloatingIPCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in FloatingIPCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a FloatingIPCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this FloatingIPCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'FloatingIPCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'FloatingIPCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class FloatingIPCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a FloatingIPCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'FloatingIPCollectionNext':
        """Initialize a FloatingIPCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in FloatingIPCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a FloatingIPCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this FloatingIPCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'FloatingIPCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'FloatingIPCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class FloatingIPPrototype():
    """
    FloatingIPPrototype.

    :attr str name: (optional) The unique user-defined name for this floating IP. If
          unspecified, the name will be a hyphenated list of randomly-selected words.
    """

    def __init__(self, *, name: str = None) -> None:
        """
        Initialize a FloatingIPPrototype object.

        :param str name: (optional) The unique user-defined name for this floating
               IP. If unspecified, the name will be a hyphenated list of randomly-selected
               words.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'FloatingIPPrototypeFloatingIPByZone',
                'FloatingIPPrototypeFloatingIPByTarget'
            ]))
        raise Exception(msg)


class FloatingIPReference():
    """
    FloatingIPReference.

    :attr str address: The globally unique IP address.
    :attr str crn: The CRN for this floating IP.
    :attr str href: The URL for this floating IP.
    :attr str id: The unique identifier for this floating IP.
    :attr str name: The user-defined name for this floating IP.
    """

    def __init__(self, address: str, crn: str, href: str, id: str,
                 name: str) -> None:
        """
        Initialize a FloatingIPReference object.

        :param str address: The globally unique IP address.
        :param str crn: The CRN for this floating IP.
        :param str href: The URL for this floating IP.
        :param str id: The unique identifier for this floating IP.
        :param str name: The user-defined name for this floating IP.
        """
        self.address = address
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'FloatingIPReference':
        """Initialize a FloatingIPReference object from a json dictionary."""
        args = {}
        if 'address' in _dict:
            args['address'] = _dict.get('address')
        else:
            raise ValueError(
                'Required property \'address\' not present in FloatingIPReference JSON'
            )
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in FloatingIPReference JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in FloatingIPReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in FloatingIPReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in FloatingIPReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a FloatingIPReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'address') and self.address is not None:
            _dict['address'] = self.address
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this FloatingIPReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'FloatingIPReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'FloatingIPReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class FloatingIPTarget():
    """
    The target of this floating IP.

    """

    def __init__(self) -> None:
        """
        Initialize a FloatingIPTarget object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'FloatingIPTargetNetworkInterfaceReference',
                'FloatingIPTargetPublicGatewayReference'
            ]))
        raise Exception(msg)


class FloatingIPUnpaginatedCollection():
    """
    FloatingIPUnpaginatedCollection.

    :attr List[FloatingIP] floating_ips: Collection of floating IPs.
    """

    def __init__(self, floating_ips: List['FloatingIP']) -> None:
        """
        Initialize a FloatingIPUnpaginatedCollection object.

        :param List[FloatingIP] floating_ips: Collection of floating IPs.
        """
        self.floating_ips = floating_ips

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'FloatingIPUnpaginatedCollection':
        """Initialize a FloatingIPUnpaginatedCollection object from a json dictionary."""
        args = {}
        if 'floating_ips' in _dict:
            args['floating_ips'] = [
                FloatingIP.from_dict(x) for x in _dict.get('floating_ips')
            ]
        else:
            raise ValueError(
                'Required property \'floating_ips\' not present in FloatingIPUnpaginatedCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a FloatingIPUnpaginatedCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'floating_ips') and self.floating_ips is not None:
            _dict['floating_ips'] = [x.to_dict() for x in self.floating_ips]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this FloatingIPUnpaginatedCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'FloatingIPUnpaginatedCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'FloatingIPUnpaginatedCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class IKEPolicy():
    """
    IKEPolicy.

    :attr str authentication_algorithm: The authentication algorithm.
    :attr List[VPNGatewayConnectionReference] connections: Collection of references
          to VPN connections that use this IKE policy.
    :attr datetime created_at: The date and time that this IKE policy was created.
    :attr int dh_group: The Diffie-Hellman group.
    :attr str encryption_algorithm: The encryption algorithm.
    :attr str href: The IKE policy's canonical URL.
    :attr str id: The unique identifier for this IKE policy.
    :attr int ike_version: The IKE protocol version.
    :attr int key_lifetime: The key lifetime in seconds.
    :attr str name: The user-defined name for this IKE policy.
    :attr str negotiation_mode: The IKE negotiation mode. Only `main` is supported.
    :attr ResourceGroupReference resource_group: The resource group for this IKE
          policy.
    """

    def __init__(self, authentication_algorithm: str,
                 connections: List['VPNGatewayConnectionReference'],
                 created_at: datetime, dh_group: int, encryption_algorithm: str,
                 href: str, id: str, ike_version: int, key_lifetime: int,
                 name: str, negotiation_mode: str,
                 resource_group: 'ResourceGroupReference') -> None:
        """
        Initialize a IKEPolicy object.

        :param str authentication_algorithm: The authentication algorithm.
        :param List[VPNGatewayConnectionReference] connections: Collection of
               references to VPN connections that use this IKE policy.
        :param datetime created_at: The date and time that this IKE policy was
               created.
        :param int dh_group: The Diffie-Hellman group.
        :param str encryption_algorithm: The encryption algorithm.
        :param str href: The IKE policy's canonical URL.
        :param str id: The unique identifier for this IKE policy.
        :param int ike_version: The IKE protocol version.
        :param int key_lifetime: The key lifetime in seconds.
        :param str name: The user-defined name for this IKE policy.
        :param str negotiation_mode: The IKE negotiation mode. Only `main` is
               supported.
        :param ResourceGroupReference resource_group: The resource group for this
               IKE policy.
        """
        self.authentication_algorithm = authentication_algorithm
        self.connections = connections
        self.created_at = created_at
        self.dh_group = dh_group
        self.encryption_algorithm = encryption_algorithm
        self.href = href
        self.id = id
        self.ike_version = ike_version
        self.key_lifetime = key_lifetime
        self.name = name
        self.negotiation_mode = negotiation_mode
        self.resource_group = resource_group

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'IKEPolicy':
        """Initialize a IKEPolicy object from a json dictionary."""
        args = {}
        if 'authentication_algorithm' in _dict:
            args['authentication_algorithm'] = _dict.get(
                'authentication_algorithm')
        else:
            raise ValueError(
                'Required property \'authentication_algorithm\' not present in IKEPolicy JSON'
            )
        if 'connections' in _dict:
            args['connections'] = [
                VPNGatewayConnectionReference.from_dict(x)
                for x in _dict.get('connections')
            ]
        else:
            raise ValueError(
                'Required property \'connections\' not present in IKEPolicy JSON'
            )
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in IKEPolicy JSON'
            )
        if 'dh_group' in _dict:
            args['dh_group'] = _dict.get('dh_group')
        else:
            raise ValueError(
                'Required property \'dh_group\' not present in IKEPolicy JSON')
        if 'encryption_algorithm' in _dict:
            args['encryption_algorithm'] = _dict.get('encryption_algorithm')
        else:
            raise ValueError(
                'Required property \'encryption_algorithm\' not present in IKEPolicy JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in IKEPolicy JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in IKEPolicy JSON')
        if 'ike_version' in _dict:
            args['ike_version'] = _dict.get('ike_version')
        else:
            raise ValueError(
                'Required property \'ike_version\' not present in IKEPolicy JSON'
            )
        if 'key_lifetime' in _dict:
            args['key_lifetime'] = _dict.get('key_lifetime')
        else:
            raise ValueError(
                'Required property \'key_lifetime\' not present in IKEPolicy JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in IKEPolicy JSON')
        if 'negotiation_mode' in _dict:
            args['negotiation_mode'] = _dict.get('negotiation_mode')
        else:
            raise ValueError(
                'Required property \'negotiation_mode\' not present in IKEPolicy JSON'
            )
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(
                _dict.get('resource_group'))
        else:
            raise ValueError(
                'Required property \'resource_group\' not present in IKEPolicy JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a IKEPolicy object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'authentication_algorithm'
                  ) and self.authentication_algorithm is not None:
            _dict['authentication_algorithm'] = self.authentication_algorithm
        if hasattr(self, 'connections') and self.connections is not None:
            _dict['connections'] = [x.to_dict() for x in self.connections]
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'dh_group') and self.dh_group is not None:
            _dict['dh_group'] = self.dh_group
        if hasattr(self, 'encryption_algorithm'
                  ) and self.encryption_algorithm is not None:
            _dict['encryption_algorithm'] = self.encryption_algorithm
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'ike_version') and self.ike_version is not None:
            _dict['ike_version'] = self.ike_version
        if hasattr(self, 'key_lifetime') and self.key_lifetime is not None:
            _dict['key_lifetime'] = self.key_lifetime
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self,
                   'negotiation_mode') and self.negotiation_mode is not None:
            _dict['negotiation_mode'] = self.negotiation_mode
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            _dict['resource_group'] = self.resource_group.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this IKEPolicy object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'IKEPolicy') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'IKEPolicy') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class AuthenticationAlgorithmEnum(str, Enum):
        """
        The authentication algorithm.
        """
        MD5 = 'md5'
        SHA1 = 'sha1'
        SHA256 = 'sha256'

    class EncryptionAlgorithmEnum(str, Enum):
        """
        The encryption algorithm.
        """
        TRIPLE_DES = 'triple_des'
        AES128 = 'aes128'
        AES256 = 'aes256'

    class NegotiationModeEnum(str, Enum):
        """
        The IKE negotiation mode. Only `main` is supported.
        """
        MAIN = 'main'


class IKEPolicyCollection():
    """
    IKEPolicyCollection.

    :attr IKEPolicyCollectionFirst first: A reference to the first page of
          resources.
    :attr List[IKEPolicy] ike_policies: Collection of IKE policies.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr IKEPolicyCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 first: 'IKEPolicyCollectionFirst',
                 ike_policies: List['IKEPolicy'],
                 limit: int,
                 total_count: int,
                 *,
                 next: 'IKEPolicyCollectionNext' = None) -> None:
        """
        Initialize a IKEPolicyCollection object.

        :param IKEPolicyCollectionFirst first: A reference to the first page of
               resources.
        :param List[IKEPolicy] ike_policies: Collection of IKE policies.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param int total_count: The total number of resources across all pages.
        :param IKEPolicyCollectionNext next: (optional) A reference to the next
               page of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.ike_policies = ike_policies
        self.limit = limit
        self.next = next
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'IKEPolicyCollection':
        """Initialize a IKEPolicyCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = IKEPolicyCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in IKEPolicyCollection JSON'
            )
        if 'ike_policies' in _dict:
            args['ike_policies'] = [
                IKEPolicy.from_dict(x) for x in _dict.get('ike_policies')
            ]
        else:
            raise ValueError(
                'Required property \'ike_policies\' not present in IKEPolicyCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in IKEPolicyCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = IKEPolicyCollectionNext.from_dict(_dict.get('next'))
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError(
                'Required property \'total_count\' not present in IKEPolicyCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a IKEPolicyCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'ike_policies') and self.ike_policies is not None:
            _dict['ike_policies'] = [x.to_dict() for x in self.ike_policies]
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this IKEPolicyCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'IKEPolicyCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'IKEPolicyCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class IKEPolicyCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a IKEPolicyCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'IKEPolicyCollectionFirst':
        """Initialize a IKEPolicyCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in IKEPolicyCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a IKEPolicyCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this IKEPolicyCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'IKEPolicyCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'IKEPolicyCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class IKEPolicyCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a IKEPolicyCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'IKEPolicyCollectionNext':
        """Initialize a IKEPolicyCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in IKEPolicyCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a IKEPolicyCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this IKEPolicyCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'IKEPolicyCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'IKEPolicyCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class IKEPolicyIdentity():
    """
    Identifies an IKE policy by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a IKEPolicyIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['IKEPolicyIdentityById', 'IKEPolicyIdentityByHref']))
        raise Exception(msg)


class IP():
    """
    IP.

    :attr str address: The IP address. This property may add support for IPv6
          addresses in the future. When processing a value in this property, verify that
          the address is in an expected format. If it is not, log an error. Optionally
          halt processing and surface the error, or bypass the resource on which the
          unexpected IP address format was encountered.
    """

    def __init__(self, address: str) -> None:
        """
        Initialize a IP object.

        :param str address: The IP address. This property may add support for IPv6
               addresses in the future. When processing a value in this property, verify
               that the address is in an expected format. If it is not, log an error.
               Optionally halt processing and surface the error, or bypass the resource on
               which the unexpected IP address format was encountered.
        """
        self.address = address

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'IP':
        """Initialize a IP object from a json dictionary."""
        args = {}
        if 'address' in _dict:
            args['address'] = _dict.get('address')
        else:
            raise ValueError(
                'Required property \'address\' not present in IP JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a IP object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'address') and self.address is not None:
            _dict['address'] = self.address
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this IP object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'IP') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'IP') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class IPsecPolicy():
    """
    IPsecPolicy.

    :attr str authentication_algorithm: The authentication algorithm.
    :attr List[VPNGatewayConnectionReference] connections: Collection of references
          to VPN connections that use this IPsec policy.
    :attr datetime created_at: The date and time that this IPsec policy was created.
    :attr str encapsulation_mode: The encapsulation mode used. Only `tunnel` is
          supported.
    :attr str encryption_algorithm: The encryption algorithm.
    :attr str href: The IPsec policy's canonical URL.
    :attr str id: The unique identifier for this IPsec policy.
    :attr int key_lifetime: The key lifetime in seconds.
    :attr str name: The user-defined name for this IPsec policy.
    :attr str pfs: Perfect Forward Secrecy.
    :attr ResourceGroupReference resource_group: The resource group for this IPsec
          policy.
    :attr str transform_protocol: The transform protocol used. Only `esp` is
          supported.
    """

    def __init__(self, authentication_algorithm: str,
                 connections: List['VPNGatewayConnectionReference'],
                 created_at: datetime, encapsulation_mode: str,
                 encryption_algorithm: str, href: str, id: str,
                 key_lifetime: int, name: str, pfs: str,
                 resource_group: 'ResourceGroupReference',
                 transform_protocol: str) -> None:
        """
        Initialize a IPsecPolicy object.

        :param str authentication_algorithm: The authentication algorithm.
        :param List[VPNGatewayConnectionReference] connections: Collection of
               references to VPN connections that use this IPsec policy.
        :param datetime created_at: The date and time that this IPsec policy was
               created.
        :param str encapsulation_mode: The encapsulation mode used. Only `tunnel`
               is supported.
        :param str encryption_algorithm: The encryption algorithm.
        :param str href: The IPsec policy's canonical URL.
        :param str id: The unique identifier for this IPsec policy.
        :param int key_lifetime: The key lifetime in seconds.
        :param str name: The user-defined name for this IPsec policy.
        :param str pfs: Perfect Forward Secrecy.
        :param ResourceGroupReference resource_group: The resource group for this
               IPsec policy.
        :param str transform_protocol: The transform protocol used. Only `esp` is
               supported.
        """
        self.authentication_algorithm = authentication_algorithm
        self.connections = connections
        self.created_at = created_at
        self.encapsulation_mode = encapsulation_mode
        self.encryption_algorithm = encryption_algorithm
        self.href = href
        self.id = id
        self.key_lifetime = key_lifetime
        self.name = name
        self.pfs = pfs
        self.resource_group = resource_group
        self.transform_protocol = transform_protocol

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'IPsecPolicy':
        """Initialize a IPsecPolicy object from a json dictionary."""
        args = {}
        if 'authentication_algorithm' in _dict:
            args['authentication_algorithm'] = _dict.get(
                'authentication_algorithm')
        else:
            raise ValueError(
                'Required property \'authentication_algorithm\' not present in IPsecPolicy JSON'
            )
        if 'connections' in _dict:
            args['connections'] = [
                VPNGatewayConnectionReference.from_dict(x)
                for x in _dict.get('connections')
            ]
        else:
            raise ValueError(
                'Required property \'connections\' not present in IPsecPolicy JSON'
            )
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in IPsecPolicy JSON'
            )
        if 'encapsulation_mode' in _dict:
            args['encapsulation_mode'] = _dict.get('encapsulation_mode')
        else:
            raise ValueError(
                'Required property \'encapsulation_mode\' not present in IPsecPolicy JSON'
            )
        if 'encryption_algorithm' in _dict:
            args['encryption_algorithm'] = _dict.get('encryption_algorithm')
        else:
            raise ValueError(
                'Required property \'encryption_algorithm\' not present in IPsecPolicy JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in IPsecPolicy JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in IPsecPolicy JSON')
        if 'key_lifetime' in _dict:
            args['key_lifetime'] = _dict.get('key_lifetime')
        else:
            raise ValueError(
                'Required property \'key_lifetime\' not present in IPsecPolicy JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in IPsecPolicy JSON')
        if 'pfs' in _dict:
            args['pfs'] = _dict.get('pfs')
        else:
            raise ValueError(
                'Required property \'pfs\' not present in IPsecPolicy JSON')
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(
                _dict.get('resource_group'))
        else:
            raise ValueError(
                'Required property \'resource_group\' not present in IPsecPolicy JSON'
            )
        if 'transform_protocol' in _dict:
            args['transform_protocol'] = _dict.get('transform_protocol')
        else:
            raise ValueError(
                'Required property \'transform_protocol\' not present in IPsecPolicy JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a IPsecPolicy object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'authentication_algorithm'
                  ) and self.authentication_algorithm is not None:
            _dict['authentication_algorithm'] = self.authentication_algorithm
        if hasattr(self, 'connections') and self.connections is not None:
            _dict['connections'] = [x.to_dict() for x in self.connections]
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(
                self,
                'encapsulation_mode') and self.encapsulation_mode is not None:
            _dict['encapsulation_mode'] = self.encapsulation_mode
        if hasattr(self, 'encryption_algorithm'
                  ) and self.encryption_algorithm is not None:
            _dict['encryption_algorithm'] = self.encryption_algorithm
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'key_lifetime') and self.key_lifetime is not None:
            _dict['key_lifetime'] = self.key_lifetime
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'pfs') and self.pfs is not None:
            _dict['pfs'] = self.pfs
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(
                self,
                'transform_protocol') and self.transform_protocol is not None:
            _dict['transform_protocol'] = self.transform_protocol
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this IPsecPolicy object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'IPsecPolicy') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'IPsecPolicy') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class AuthenticationAlgorithmEnum(str, Enum):
        """
        The authentication algorithm.
        """
        MD5 = 'md5'
        SHA1 = 'sha1'
        SHA256 = 'sha256'

    class EncapsulationModeEnum(str, Enum):
        """
        The encapsulation mode used. Only `tunnel` is supported.
        """
        TUNNEL = 'tunnel'

    class EncryptionAlgorithmEnum(str, Enum):
        """
        The encryption algorithm.
        """
        TRIPLE_DES = 'triple_des'
        AES128 = 'aes128'
        AES256 = 'aes256'

    class PfsEnum(str, Enum):
        """
        Perfect Forward Secrecy.
        """
        DISABLED = 'disabled'
        GROUP_14 = 'group_14'
        GROUP_2 = 'group_2'
        GROUP_5 = 'group_5'

    class TransformProtocolEnum(str, Enum):
        """
        The transform protocol used. Only `esp` is supported.
        """
        ESP = 'esp'


class IPsecPolicyCollection():
    """
    IPsecPolicyCollection.

    :attr IPsecPolicyCollectionFirst first: A reference to the first page of
          resources.
    :attr List[IPsecPolicy] ipsec_policies: Collection of IPsec policies.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr IPsecPolicyCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 first: 'IPsecPolicyCollectionFirst',
                 ipsec_policies: List['IPsecPolicy'],
                 limit: int,
                 total_count: int,
                 *,
                 next: 'IPsecPolicyCollectionNext' = None) -> None:
        """
        Initialize a IPsecPolicyCollection object.

        :param IPsecPolicyCollectionFirst first: A reference to the first page of
               resources.
        :param List[IPsecPolicy] ipsec_policies: Collection of IPsec policies.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param int total_count: The total number of resources across all pages.
        :param IPsecPolicyCollectionNext next: (optional) A reference to the next
               page of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.ipsec_policies = ipsec_policies
        self.limit = limit
        self.next = next
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'IPsecPolicyCollection':
        """Initialize a IPsecPolicyCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = IPsecPolicyCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in IPsecPolicyCollection JSON'
            )
        if 'ipsec_policies' in _dict:
            args['ipsec_policies'] = [
                IPsecPolicy.from_dict(x) for x in _dict.get('ipsec_policies')
            ]
        else:
            raise ValueError(
                'Required property \'ipsec_policies\' not present in IPsecPolicyCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in IPsecPolicyCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = IPsecPolicyCollectionNext.from_dict(
                _dict.get('next'))
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError(
                'Required property \'total_count\' not present in IPsecPolicyCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a IPsecPolicyCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'ipsec_policies') and self.ipsec_policies is not None:
            _dict['ipsec_policies'] = [x.to_dict() for x in self.ipsec_policies]
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this IPsecPolicyCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'IPsecPolicyCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'IPsecPolicyCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class IPsecPolicyCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a IPsecPolicyCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'IPsecPolicyCollectionFirst':
        """Initialize a IPsecPolicyCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in IPsecPolicyCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a IPsecPolicyCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this IPsecPolicyCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'IPsecPolicyCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'IPsecPolicyCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class IPsecPolicyCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a IPsecPolicyCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'IPsecPolicyCollectionNext':
        """Initialize a IPsecPolicyCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in IPsecPolicyCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a IPsecPolicyCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this IPsecPolicyCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'IPsecPolicyCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'IPsecPolicyCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class IPsecPolicyIdentity():
    """
    Identifies an IPsec policy by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a IPsecPolicyIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['IPsecPolicyIdentityById', 'IPsecPolicyIdentityByHref']))
        raise Exception(msg)


class Image():
    """
    Image.

    :attr datetime created_at: The date and time that the image was created.
    :attr str crn: The CRN for this image.
    :attr ImageFile file: Details for the stored image file.
    :attr str href: The URL for this image.
    :attr str id: The unique identifier for this image.
    :attr int minimum_provisioned_size: (optional) The minimum size (in gigabytes)
          of a volume onto which this image may be provisioned.
          This property may be absent if the image has a `status` of `pending`,
          `tentative`, or
          `failed`.
    :attr str name: The user-defined or system-provided name for this image.
    :attr OperatingSystem operating_system: (optional) The operating system included
          in this image.
    :attr ResourceGroupReference resource_group: The resource group for this image.
    :attr str status: The status of this image.
    :attr str visibility: Whether the image is publicly visible or private to the
          account.
    """

    def __init__(self,
                 created_at: datetime,
                 crn: str,
                 file: 'ImageFile',
                 href: str,
                 id: str,
                 name: str,
                 resource_group: 'ResourceGroupReference',
                 status: str,
                 visibility: str,
                 *,
                 minimum_provisioned_size: int = None,
                 operating_system: 'OperatingSystem' = None) -> None:
        """
        Initialize a Image object.

        :param datetime created_at: The date and time that the image was created.
        :param str crn: The CRN for this image.
        :param ImageFile file: Details for the stored image file.
        :param str href: The URL for this image.
        :param str id: The unique identifier for this image.
        :param str name: The user-defined or system-provided name for this image.
        :param ResourceGroupReference resource_group: The resource group for this
               image.
        :param str status: The status of this image.
        :param str visibility: Whether the image is publicly visible or private to
               the account.
        :param int minimum_provisioned_size: (optional) The minimum size (in
               gigabytes) of a volume onto which this image may be provisioned.
               This property may be absent if the image has a `status` of `pending`,
               `tentative`, or
               `failed`.
        :param OperatingSystem operating_system: (optional) The operating system
               included in this image.
        """
        self.created_at = created_at
        self.crn = crn
        self.file = file
        self.href = href
        self.id = id
        self.minimum_provisioned_size = minimum_provisioned_size
        self.name = name
        self.operating_system = operating_system
        self.resource_group = resource_group
        self.status = status
        self.visibility = visibility

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'Image':
        """Initialize a Image object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in Image JSON')
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in Image JSON')
        if 'file' in _dict:
            args['file'] = ImageFile.from_dict(_dict.get('file'))
        else:
            raise ValueError(
                'Required property \'file\' not present in Image JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in Image JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in Image JSON')
        if 'minimum_provisioned_size' in _dict:
            args['minimum_provisioned_size'] = _dict.get(
                'minimum_provisioned_size')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in Image JSON')
        if 'operating_system' in _dict:
            args['operating_system'] = OperatingSystem.from_dict(
                _dict.get('operating_system'))
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(
                _dict.get('resource_group'))
        else:
            raise ValueError(
                'Required property \'resource_group\' not present in Image JSON'
            )
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in Image JSON')
        if 'visibility' in _dict:
            args['visibility'] = _dict.get('visibility')
        else:
            raise ValueError(
                'Required property \'visibility\' not present in Image JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a Image object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'file') and self.file is not None:
            _dict['file'] = self.file.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'minimum_provisioned_size'
                  ) and self.minimum_provisioned_size is not None:
            _dict['minimum_provisioned_size'] = self.minimum_provisioned_size
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self,
                   'operating_system') and self.operating_system is not None:
            _dict['operating_system'] = self.operating_system.to_dict()
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'visibility') and self.visibility is not None:
            _dict['visibility'] = self.visibility
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this Image object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'Image') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'Image') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class StatusEnum(str, Enum):
        """
        The status of this image.
        """
        AVAILABLE = 'available'
        DELETING = 'deleting'
        DEPRECATED = 'deprecated'
        FAILED = 'failed'
        PENDING = 'pending'
        TENTATIVE = 'tentative'

    class VisibilityEnum(str, Enum):
        """
        Whether the image is publicly visible or private to the account.
        """
        PRIVATE = 'private'
        PUBLIC = 'public'


class ImageCollection():
    """
    ImageCollection.

    :attr ImageCollectionFirst first: A reference to the first page of resources.
    :attr List[Image] images: Collection of images.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr ImageCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    """

    def __init__(self,
                 first: 'ImageCollectionFirst',
                 images: List['Image'],
                 limit: int,
                 *,
                 next: 'ImageCollectionNext' = None) -> None:
        """
        Initialize a ImageCollection object.

        :param ImageCollectionFirst first: A reference to the first page of
               resources.
        :param List[Image] images: Collection of images.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param ImageCollectionNext next: (optional) A reference to the next page of
               resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.images = images
        self.limit = limit
        self.next = next

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ImageCollection':
        """Initialize a ImageCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = ImageCollectionFirst.from_dict(_dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in ImageCollection JSON'
            )
        if 'images' in _dict:
            args['images'] = [Image.from_dict(x) for x in _dict.get('images')]
        else:
            raise ValueError(
                'Required property \'images\' not present in ImageCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in ImageCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = ImageCollectionNext.from_dict(_dict.get('next'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ImageCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'images') and self.images is not None:
            _dict['images'] = [x.to_dict() for x in self.images]
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ImageCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ImageCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ImageCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ImageCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a ImageCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ImageCollectionFirst':
        """Initialize a ImageCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in ImageCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ImageCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ImageCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ImageCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ImageCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ImageCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a ImageCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ImageCollectionNext':
        """Initialize a ImageCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in ImageCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ImageCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ImageCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ImageCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ImageCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ImageFile():
    """
    ImageFile.

    :attr int size: (optional) The size of the stored image file rounded up to the
          next gigabyte.
          This property may be absent if the associated image has a `status` of `pending`
          or
          `failed`.
    """

    def __init__(self, *, size: int = None) -> None:
        """
        Initialize a ImageFile object.

        :param int size: (optional) The size of the stored image file rounded up to
               the next gigabyte.
               This property may be absent if the associated image has a `status` of
               `pending` or
               `failed`.
        """
        self.size = size

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ImageFile':
        """Initialize a ImageFile object from a json dictionary."""
        args = {}
        if 'size' in _dict:
            args['size'] = _dict.get('size')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ImageFile object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'size') and self.size is not None:
            _dict['size'] = self.size
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ImageFile object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ImageFile') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ImageFile') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ImageFilePrototype():
    """
    ImageFilePrototype.

    :attr str href: The Cloud Object Store (COS) location of the image file.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a ImageFilePrototype object.

        :param str href: The Cloud Object Store (COS) location of the image file.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ImageFilePrototype':
        """Initialize a ImageFilePrototype object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in ImageFilePrototype JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ImageFilePrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ImageFilePrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ImageFilePrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ImageFilePrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ImageIdentity():
    """
    Identifies an image by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a ImageIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'ImageIdentityById', 'ImageIdentityByCRN', 'ImageIdentityByHref'
            ]))
        raise Exception(msg)


class ImagePrototype():
    """
    ImagePrototype.

    :attr str name: (optional) The unique user-defined name for this image. Names
          starting with "ibm-" are not allowed. If unspecified, the name will be a
          hyphenated list of randomly-selected words.
    :attr ResourceGroupIdentity resource_group: (optional) The resource group to
          use. If unspecified, the account's [default resource
          group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
    """

    def __init__(self,
                 *,
                 name: str = None,
                 resource_group: 'ResourceGroupIdentity' = None) -> None:
        """
        Initialize a ImagePrototype object.

        :param str name: (optional) The unique user-defined name for this image.
               Names starting with "ibm-" are not allowed. If unspecified, the name will
               be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['ImagePrototypeImageByFile']))
        raise Exception(msg)


class ImageReference():
    """
    ImageReference.

    :attr str crn: The CRN for this image.
    :attr str href: The URL for this image.
    :attr str id: The unique identifier for this image.
    :attr str name: The user-defined or system-provided name for this image.
    """

    def __init__(self, crn: str, href: str, id: str, name: str) -> None:
        """
        Initialize a ImageReference object.

        :param str crn: The CRN for this image.
        :param str href: The URL for this image.
        :param str id: The unique identifier for this image.
        :param str name: The user-defined or system-provided name for this image.
        """
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ImageReference':
        """Initialize a ImageReference object from a json dictionary."""
        args = {}
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in ImageReference JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in ImageReference JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in ImageReference JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in ImageReference JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ImageReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ImageReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ImageReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ImageReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class Instance():
    """
    Instance.

    :attr int bandwidth: The total bandwidth (in megabits per second) shared across
          the virtual server instance's network interfaces.
    :attr VolumeAttachmentReferenceInstanceContext boot_volume_attachment: Boot
          volume attachment.
    :attr datetime created_at: The date and time that the virtual server instance
          was created.
    :attr str crn: The CRN for this virtual server instance.
    :attr str href: The URL for this virtual server instance.
    :attr str id: The unique identifier for this virtual server instance.
    :attr ImageReference image: (optional) The image the virtual server instance was
          provisioned from.
    :attr int memory: The amount of memory in gigabytes.
    :attr str name: The user-defined name for this virtual server instance (and
          default system hostname).
    :attr List[NetworkInterfaceInstanceContextReference] network_interfaces:
          Collection of the virtual server instance's network interfaces, including the
          primary network interface.
    :attr NetworkInterfaceInstanceContextReference primary_network_interface:
          Primary network interface.
    :attr InstanceProfileReference profile: The profile this virtual server instance
          uses.
    :attr ResourceGroupReference resource_group: The resource group for this
          instance.
    :attr str status: The status of the virtual server instance.
    :attr InstanceVCPU vcpu: The virtual server instance VCPU configuration.
    :attr List[VolumeAttachmentReferenceInstanceContext] volume_attachments:
          Collection of the virtual server instance's volume attachments, including the
          boot volume attachment.
    :attr VPCReference vpc: The VPC the virtual server instance resides in.
    :attr ZoneReference zone: The zone the virtual server instance resides in.
    """

    def __init__(
            self,
            bandwidth: int,
            boot_volume_attachment: 'VolumeAttachmentReferenceInstanceContext',
            created_at: datetime,
            crn: str,
            href: str,
            id: str,
            memory: int,
            name: str,
            network_interfaces: List[
                'NetworkInterfaceInstanceContextReference'],
            primary_network_interface:
        'NetworkInterfaceInstanceContextReference',
            profile: 'InstanceProfileReference',
            resource_group: 'ResourceGroupReference',
            status: str,
            vcpu: 'InstanceVCPU',
            volume_attachments: List[
                'VolumeAttachmentReferenceInstanceContext'],
            vpc: 'VPCReference',
            zone: 'ZoneReference',
            *,
            image: 'ImageReference' = None) -> None:
        """
        Initialize a Instance object.

        :param int bandwidth: The total bandwidth (in megabits per second) shared
               across the virtual server instance's network interfaces.
        :param VolumeAttachmentReferenceInstanceContext boot_volume_attachment:
               Boot volume attachment.
        :param datetime created_at: The date and time that the virtual server
               instance was created.
        :param str crn: The CRN for this virtual server instance.
        :param str href: The URL for this virtual server instance.
        :param str id: The unique identifier for this virtual server instance.
        :param int memory: The amount of memory in gigabytes.
        :param str name: The user-defined name for this virtual server instance
               (and default system hostname).
        :param List[NetworkInterfaceInstanceContextReference] network_interfaces:
               Collection of the virtual server instance's network interfaces, including
               the primary network interface.
        :param NetworkInterfaceInstanceContextReference primary_network_interface:
               Primary network interface.
        :param InstanceProfileReference profile: The profile this virtual server
               instance uses.
        :param ResourceGroupReference resource_group: The resource group for this
               instance.
        :param str status: The status of the virtual server instance.
        :param InstanceVCPU vcpu: The virtual server instance VCPU configuration.
        :param List[VolumeAttachmentReferenceInstanceContext] volume_attachments:
               Collection of the virtual server instance's volume attachments, including
               the boot volume attachment.
        :param VPCReference vpc: The VPC the virtual server instance resides in.
        :param ZoneReference zone: The zone the virtual server instance resides in.
        :param ImageReference image: (optional) The image the virtual server
               instance was provisioned from.
        """
        self.bandwidth = bandwidth
        self.boot_volume_attachment = boot_volume_attachment
        self.created_at = created_at
        self.crn = crn
        self.href = href
        self.id = id
        self.image = image
        self.memory = memory
        self.name = name
        self.network_interfaces = network_interfaces
        self.primary_network_interface = primary_network_interface
        self.profile = profile
        self.resource_group = resource_group
        self.status = status
        self.vcpu = vcpu
        self.volume_attachments = volume_attachments
        self.vpc = vpc
        self.zone = zone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'Instance':
        """Initialize a Instance object from a json dictionary."""
        args = {}
        if 'bandwidth' in _dict:
            args['bandwidth'] = _dict.get('bandwidth')
        else:
            raise ValueError(
                'Required property \'bandwidth\' not present in Instance JSON')
        if 'boot_volume_attachment' in _dict:
            args[
                'boot_volume_attachment'] = VolumeAttachmentReferenceInstanceContext.from_dict(
                    _dict.get('boot_volume_attachment'))
        else:
            raise ValueError(
                'Required property \'boot_volume_attachment\' not present in Instance JSON'
            )
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in Instance JSON')
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in Instance JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in Instance JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in Instance JSON')
        if 'image' in _dict:
            args['image'] = ImageReference.from_dict(_dict.get('image'))
        if 'memory' in _dict:
            args['memory'] = _dict.get('memory')
        else:
            raise ValueError(
                'Required property \'memory\' not present in Instance JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in Instance JSON')
        if 'network_interfaces' in _dict:
            args['network_interfaces'] = [
                NetworkInterfaceInstanceContextReference.from_dict(x)
                for x in _dict.get('network_interfaces')
            ]
        else:
            raise ValueError(
                'Required property \'network_interfaces\' not present in Instance JSON'
            )
        if 'primary_network_interface' in _dict:
            args[
                'primary_network_interface'] = NetworkInterfaceInstanceContextReference.from_dict(
                    _dict.get('primary_network_interface'))
        else:
            raise ValueError(
                'Required property \'primary_network_interface\' not present in Instance JSON'
            )
        if 'profile' in _dict:
            args['profile'] = InstanceProfileReference.from_dict(
                _dict.get('profile'))
        else:
            raise ValueError(
                'Required property \'profile\' not present in Instance JSON')
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(
                _dict.get('resource_group'))
        else:
            raise ValueError(
                'Required property \'resource_group\' not present in Instance JSON'
            )
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in Instance JSON')
        if 'vcpu' in _dict:
            args['vcpu'] = InstanceVCPU.from_dict(_dict.get('vcpu'))
        else:
            raise ValueError(
                'Required property \'vcpu\' not present in Instance JSON')
        if 'volume_attachments' in _dict:
            args['volume_attachments'] = [
                VolumeAttachmentReferenceInstanceContext.from_dict(x)
                for x in _dict.get('volume_attachments')
            ]
        else:
            raise ValueError(
                'Required property \'volume_attachments\' not present in Instance JSON'
            )
        if 'vpc' in _dict:
            args['vpc'] = VPCReference.from_dict(_dict.get('vpc'))
        else:
            raise ValueError(
                'Required property \'vpc\' not present in Instance JSON')
        if 'zone' in _dict:
            args['zone'] = ZoneReference.from_dict(_dict.get('zone'))
        else:
            raise ValueError(
                'Required property \'zone\' not present in Instance JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a Instance object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'bandwidth') and self.bandwidth is not None:
            _dict['bandwidth'] = self.bandwidth
        if hasattr(self, 'boot_volume_attachment'
                  ) and self.boot_volume_attachment is not None:
            _dict[
                'boot_volume_attachment'] = self.boot_volume_attachment.to_dict(
                )
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'image') and self.image is not None:
            _dict['image'] = self.image.to_dict()
        if hasattr(self, 'memory') and self.memory is not None:
            _dict['memory'] = self.memory
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(
                self,
                'network_interfaces') and self.network_interfaces is not None:
            _dict['network_interfaces'] = [
                x.to_dict() for x in self.network_interfaces
            ]
        if hasattr(self, 'primary_network_interface'
                  ) and self.primary_network_interface is not None:
            _dict[
                'primary_network_interface'] = self.primary_network_interface.to_dict(
                )
        if hasattr(self, 'profile') and self.profile is not None:
            _dict['profile'] = self.profile.to_dict()
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'vcpu') and self.vcpu is not None:
            _dict['vcpu'] = self.vcpu.to_dict()
        if hasattr(
                self,
                'volume_attachments') and self.volume_attachments is not None:
            _dict['volume_attachments'] = [
                x.to_dict() for x in self.volume_attachments
            ]
        if hasattr(self, 'vpc') and self.vpc is not None:
            _dict['vpc'] = self.vpc.to_dict()
        if hasattr(self, 'zone') and self.zone is not None:
            _dict['zone'] = self.zone.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this Instance object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'Instance') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'Instance') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class StatusEnum(str, Enum):
        """
        The status of the virtual server instance.
        """
        FAILED = 'failed'
        PAUSED = 'paused'
        PAUSING = 'pausing'
        PENDING = 'pending'
        RESTARTING = 'restarting'
        RESUMING = 'resuming'
        RUNNING = 'running'
        STARTING = 'starting'
        STOPPED = 'stopped'
        STOPPING = 'stopping'


class InstanceAction():
    """
    InstanceAction.

    :attr datetime completed_at: (optional) The date and time that the action was
          completed.
    :attr datetime created_at: The date and time that the action was created.
    :attr str href: The URL for this instance action.
    :attr str id: The identifier for this instance action.
    :attr datetime started_at: (optional) The date and time that the action was
          started.
    :attr str status: The current status of this action.
    :attr str type: The type of action.
    """

    def __init__(self,
                 created_at: datetime,
                 href: str,
                 id: str,
                 status: str,
                 type: str,
                 *,
                 completed_at: datetime = None,
                 started_at: datetime = None) -> None:
        """
        Initialize a InstanceAction object.

        :param datetime created_at: The date and time that the action was created.
        :param str href: The URL for this instance action.
        :param str id: The identifier for this instance action.
        :param str status: The current status of this action.
        :param str type: The type of action.
        :param datetime completed_at: (optional) The date and time that the action
               was completed.
        :param datetime started_at: (optional) The date and time that the action
               was started.
        """
        self.completed_at = completed_at
        self.created_at = created_at
        self.href = href
        self.id = id
        self.started_at = started_at
        self.status = status
        self.type = type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceAction':
        """Initialize a InstanceAction object from a json dictionary."""
        args = {}
        if 'completed_at' in _dict:
            args['completed_at'] = string_to_datetime(_dict.get('completed_at'))
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in InstanceAction JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in InstanceAction JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in InstanceAction JSON')
        if 'started_at' in _dict:
            args['started_at'] = string_to_datetime(_dict.get('started_at'))
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in InstanceAction JSON'
            )
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in InstanceAction JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceAction object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'completed_at') and self.completed_at is not None:
            _dict['completed_at'] = datetime_to_string(self.completed_at)
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'started_at') and self.started_at is not None:
            _dict['started_at'] = datetime_to_string(self.started_at)
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceAction object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceAction') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceAction') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class StatusEnum(str, Enum):
        """
        The current status of this action.
        """
        COMPLETED = 'completed'
        FAILED = 'failed'
        PENDING = 'pending'
        RUNNING = 'running'

    class TypeEnum(str, Enum):
        """
        The type of action.
        """
        REBOOT = 'reboot'
        START = 'start'
        STOP = 'stop'
        RESET = 'reset'


class InstanceCollection():
    """
    InstanceCollection.

    :attr InstanceCollectionFirst first: A reference to the first page of resources.
    :attr List[Instance] instances: Collection of virtual server instances.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr InstanceCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 first: 'InstanceCollectionFirst',
                 instances: List['Instance'],
                 limit: int,
                 total_count: int,
                 *,
                 next: 'InstanceCollectionNext' = None) -> None:
        """
        Initialize a InstanceCollection object.

        :param InstanceCollectionFirst first: A reference to the first page of
               resources.
        :param List[Instance] instances: Collection of virtual server instances.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param int total_count: The total number of resources across all pages.
        :param InstanceCollectionNext next: (optional) A reference to the next page
               of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.instances = instances
        self.limit = limit
        self.next = next
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceCollection':
        """Initialize a InstanceCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = InstanceCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in InstanceCollection JSON'
            )
        if 'instances' in _dict:
            args['instances'] = [
                Instance.from_dict(x) for x in _dict.get('instances')
            ]
        else:
            raise ValueError(
                'Required property \'instances\' not present in InstanceCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in InstanceCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = InstanceCollectionNext.from_dict(_dict.get('next'))
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError(
                'Required property \'total_count\' not present in InstanceCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'instances') and self.instances is not None:
            _dict['instances'] = [x.to_dict() for x in self.instances]
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstanceCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a InstanceCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceCollectionFirst':
        """Initialize a InstanceCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in InstanceCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstanceCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a InstanceCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceCollectionNext':
        """Initialize a InstanceCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in InstanceCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstanceInitialization():
    """
    InstanceInitialization.

    :attr List[KeyReferenceInstanceInitializationContext] keys: Collection of
          references to public SSH keys used at instance initialization.
    :attr InstanceInitializationPassword password: (optional)
    """

    def __init__(self,
                 keys: List['KeyReferenceInstanceInitializationContext'],
                 *,
                 password: 'InstanceInitializationPassword' = None) -> None:
        """
        Initialize a InstanceInitialization object.

        :param List[KeyReferenceInstanceInitializationContext] keys: Collection of
               references to public SSH keys used at instance initialization.
        :param InstanceInitializationPassword password: (optional)
        """
        self.keys = keys
        self.password = password

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceInitialization':
        """Initialize a InstanceInitialization object from a json dictionary."""
        args = {}
        if 'keys' in _dict:
            args['keys'] = [
                KeyReferenceInstanceInitializationContext.from_dict(x)
                for x in _dict.get('keys')
            ]
        else:
            raise ValueError(
                'Required property \'keys\' not present in InstanceInitialization JSON'
            )
        if 'password' in _dict:
            args['password'] = InstanceInitializationPassword.from_dict(
                _dict.get('password'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceInitialization object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'keys') and self.keys is not None:
            _dict['keys'] = [x.to_dict() for x in self.keys]
        if hasattr(self, 'password') and self.password is not None:
            _dict['password'] = self.password.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceInitialization object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceInitialization') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceInitialization') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstanceInitializationPassword():
    """
    InstanceInitializationPassword.

    :attr bytes encrypted_password: The administrator password at initialization,
          encrypted using `encryption_key`, and returned base64-encoded.
    :attr KeyReferenceInstanceInitializationContext encryption_key: The reference to
          the public SSH key used to encrypt the administrator password.
    """

    def __init__(
            self, encrypted_password: bytes,
            encryption_key: 'KeyReferenceInstanceInitializationContext'
    ) -> None:
        """
        Initialize a InstanceInitializationPassword object.

        :param bytes encrypted_password: The administrator password at
               initialization, encrypted using `encryption_key`, and returned
               base64-encoded.
        :param KeyReferenceInstanceInitializationContext encryption_key: The
               reference to the public SSH key used to encrypt the administrator password.
        """
        self.encrypted_password = encrypted_password
        self.encryption_key = encryption_key

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceInitializationPassword':
        """Initialize a InstanceInitializationPassword object from a json dictionary."""
        args = {}
        if 'encrypted_password' in _dict:
            args['encrypted_password'] = base64.b64decode(
                _dict.get('encrypted_password'))
        else:
            raise ValueError(
                'Required property \'encrypted_password\' not present in InstanceInitializationPassword JSON'
            )
        if 'encryption_key' in _dict:
            args['encryption_key'] = _dict.get('encryption_key')
        else:
            raise ValueError(
                'Required property \'encryption_key\' not present in InstanceInitializationPassword JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceInitializationPassword object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(
                self,
                'encrypted_password') and self.encrypted_password is not None:
            _dict['encrypted_password'] = str(
                base64.b64encode(self.encrypted_password), 'utf-8')
        if hasattr(self, 'encryption_key') and self.encryption_key is not None:
            _dict['encryption_key'] = self.encryption_key
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceInitializationPassword object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceInitializationPassword') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceInitializationPassword') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstanceProfile():
    """
    InstanceProfile.

    :attr InstanceProfileBandwidth bandwidth:
    :attr str crn: The CRN for this virtual server instance profile.
    :attr str family: (optional) The product family this virtual server instance
          profile belongs to.
    :attr str href: The URL for this virtual server instance profile.
    :attr str name: The name for this virtual server instance profile.
    :attr InstanceProfilePortSpeed port_speed:
    """

    def __init__(self,
                 bandwidth: 'InstanceProfileBandwidth',
                 crn: str,
                 href: str,
                 name: str,
                 port_speed: 'InstanceProfilePortSpeed',
                 *,
                 family: str = None) -> None:
        """
        Initialize a InstanceProfile object.

        :param InstanceProfileBandwidth bandwidth:
        :param str crn: The CRN for this virtual server instance profile.
        :param str href: The URL for this virtual server instance profile.
        :param str name: The name for this virtual server instance profile.
        :param InstanceProfilePortSpeed port_speed:
        :param str family: (optional) The product family this virtual server
               instance profile belongs to.
        """
        self.bandwidth = bandwidth
        self.crn = crn
        self.family = family
        self.href = href
        self.name = name
        self.port_speed = port_speed

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceProfile':
        """Initialize a InstanceProfile object from a json dictionary."""
        args = {}
        if 'bandwidth' in _dict:
            args['bandwidth'] = _dict.get('bandwidth')
        else:
            raise ValueError(
                'Required property \'bandwidth\' not present in InstanceProfile JSON'
            )
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in InstanceProfile JSON')
        if 'family' in _dict:
            args['family'] = _dict.get('family')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in InstanceProfile JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in InstanceProfile JSON'
            )
        if 'port_speed' in _dict:
            args['port_speed'] = _dict.get('port_speed')
        else:
            raise ValueError(
                'Required property \'port_speed\' not present in InstanceProfile JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceProfile object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'bandwidth') and self.bandwidth is not None:
            _dict['bandwidth'] = self.bandwidth
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'family') and self.family is not None:
            _dict['family'] = self.family
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'port_speed') and self.port_speed is not None:
            _dict['port_speed'] = self.port_speed
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceProfile object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceProfile') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceProfile') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstanceProfileBandwidth():
    """
    InstanceProfileBandwidth.

    """

    def __init__(self) -> None:
        """
        Initialize a InstanceProfileBandwidth object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'InstanceProfileBandwidthFixed',
                'InstanceProfileBandwidthRange', 'InstanceProfileBandwidthEnum',
                'InstanceProfileBandwidthDependent'
            ]))
        raise Exception(msg)


class InstanceProfileCollection():
    """
    InstanceProfileCollection.

    :attr InstanceProfileCollectionFirst first: A reference to the first page of
          resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr InstanceProfileCollectionNext next: (optional) A reference to the next
          page of resources; this reference is included for all pages
          except the last page.
    :attr List[InstanceProfile] profiles: Collection of virtual server instance
          profiles.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 first: 'InstanceProfileCollectionFirst',
                 limit: int,
                 profiles: List['InstanceProfile'],
                 total_count: int,
                 *,
                 next: 'InstanceProfileCollectionNext' = None) -> None:
        """
        Initialize a InstanceProfileCollection object.

        :param InstanceProfileCollectionFirst first: A reference to the first page
               of resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param List[InstanceProfile] profiles: Collection of virtual server
               instance profiles.
        :param int total_count: The total number of resources across all pages.
        :param InstanceProfileCollectionNext next: (optional) A reference to the
               next page of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.next = next
        self.profiles = profiles
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceProfileCollection':
        """Initialize a InstanceProfileCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = InstanceProfileCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in InstanceProfileCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in InstanceProfileCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = InstanceProfileCollectionNext.from_dict(
                _dict.get('next'))
        if 'profiles' in _dict:
            args['profiles'] = [
                InstanceProfile.from_dict(x) for x in _dict.get('profiles')
            ]
        else:
            raise ValueError(
                'Required property \'profiles\' not present in InstanceProfileCollection JSON'
            )
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError(
                'Required property \'total_count\' not present in InstanceProfileCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceProfileCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'profiles') and self.profiles is not None:
            _dict['profiles'] = [x.to_dict() for x in self.profiles]
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceProfileCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceProfileCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceProfileCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstanceProfileCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a InstanceProfileCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceProfileCollectionFirst':
        """Initialize a InstanceProfileCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in InstanceProfileCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceProfileCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceProfileCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceProfileCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceProfileCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstanceProfileCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a InstanceProfileCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceProfileCollectionNext':
        """Initialize a InstanceProfileCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in InstanceProfileCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceProfileCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceProfileCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceProfileCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceProfileCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstanceProfileIdentity():
    """
    Identifies an instance profile by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a InstanceProfileIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'InstanceProfileIdentityByName', 'InstanceProfileIdentityByCRN',
                'InstanceProfileIdentityByHref'
            ]))
        raise Exception(msg)


class InstanceProfilePortSpeed():
    """
    InstanceProfilePortSpeed.

    """

    def __init__(self) -> None:
        """
        Initialize a InstanceProfilePortSpeed object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'InstanceProfilePortSpeedFixed',
                'InstanceProfilePortSpeedDependent'
            ]))
        raise Exception(msg)


class InstanceProfileReference():
    """
    InstanceProfileReference.

    :attr str crn: The CRN for this virtual server instance profile.
    :attr str href: The URL for this virtual server instance profile.
    :attr str name: The name for this virtual server instance profile.
    """

    def __init__(self, crn: str, href: str, name: str) -> None:
        """
        Initialize a InstanceProfileReference object.

        :param str crn: The CRN for this virtual server instance profile.
        :param str href: The URL for this virtual server instance profile.
        :param str name: The name for this virtual server instance profile.
        """
        self.crn = crn
        self.href = href
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceProfileReference':
        """Initialize a InstanceProfileReference object from a json dictionary."""
        args = {}
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in InstanceProfileReference JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in InstanceProfileReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in InstanceProfileReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceProfileReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceProfileReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceProfileReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceProfileReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstancePrototype():
    """
    InstancePrototype.

    :attr List[KeyIdentity] keys: (optional) The public SSH keys to install on the
          virtual server instance. Up to 10 keys may be provided; if no keys are provided
          the instance will be inaccessible unless the image used provides a means of
          access. For Windows instances, one of the keys will be used to encrypt the
          administrator password.
    :attr str name: (optional) The user-defined name for this virtual server
          instance (and default system hostname). If unspecified, the name will be a
          hyphenated list of randomly-selected words.
    :attr List[NetworkInterfacePrototype] network_interfaces: (optional) Collection
          of additional network interfaces to create for the virtual server instance.
    :attr InstanceProfileIdentity profile: (optional) The profile to use for this
          virtual server instance.
    :attr ResourceGroupIdentity resource_group: (optional) The resource group to
          use. If unspecified, the account's [default resource
          group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
    :attr str user_data: (optional) User data to be made available when setting up
          the virtual server instance.
    :attr List[VolumeAttachmentPrototypeInstanceContext] volume_attachments:
          (optional) Collection of volume attachments.
    :attr VPCIdentity vpc: (optional) The VPC the virtual server instance is to be a
          part of. If provided, must match the
          VPC tied to the subnets of the instance's network interfaces.
    """

    def __init__(self,
                 *,
                 keys: List['KeyIdentity'] = None,
                 name: str = None,
                 network_interfaces: List['NetworkInterfacePrototype'] = None,
                 profile: 'InstanceProfileIdentity' = None,
                 resource_group: 'ResourceGroupIdentity' = None,
                 user_data: str = None,
                 volume_attachments: List[
                     'VolumeAttachmentPrototypeInstanceContext'] = None,
                 vpc: 'VPCIdentity' = None) -> None:
        """
        Initialize a InstancePrototype object.

        :param List[KeyIdentity] keys: (optional) The public SSH keys to install on
               the virtual server instance. Up to 10 keys may be provided; if no keys are
               provided the instance will be inaccessible unless the image used provides a
               means of access. For Windows instances, one of the keys will be used to
               encrypt the administrator password.
        :param str name: (optional) The user-defined name for this virtual server
               instance (and default system hostname). If unspecified, the name will be a
               hyphenated list of randomly-selected words.
        :param List[NetworkInterfacePrototype] network_interfaces: (optional)
               Collection of additional network interfaces to create for the virtual
               server instance.
        :param InstanceProfileIdentity profile: (optional) The profile to use for
               this virtual server instance.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param str user_data: (optional) User data to be made available when
               setting up the virtual server instance.
        :param List[VolumeAttachmentPrototypeInstanceContext] volume_attachments:
               (optional) Collection of volume attachments.
        :param VPCIdentity vpc: (optional) The VPC the virtual server instance is
               to be a part of. If provided, must match the
               VPC tied to the subnets of the instance's network interfaces.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['InstancePrototypeInstanceByImage']))
        raise Exception(msg)


class InstanceReference():
    """
    InstanceReference.

    :attr str crn: The CRN for this virtual server instance.
    :attr str href: The URL for this virtual server instance.
    :attr str id: The unique identifier for this virtual server instance.
    :attr str name: The user-defined name for this virtual server instance (and
          default system hostname).
    """

    def __init__(self, crn: str, href: str, id: str, name: str) -> None:
        """
        Initialize a InstanceReference object.

        :param str crn: The CRN for this virtual server instance.
        :param str href: The URL for this virtual server instance.
        :param str id: The unique identifier for this virtual server instance.
        :param str name: The user-defined name for this virtual server instance
               (and default system hostname).
        """
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceReference':
        """Initialize a InstanceReference object from a json dictionary."""
        args = {}
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in InstanceReference JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in InstanceReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in InstanceReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in InstanceReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class InstanceVCPU():
    """
    The virtual server instance VCPU configuration.

    :attr str architecture: The VCPU architecture.
    :attr int count: The number of VCPUs assigned.
    """

    def __init__(self, architecture: str, count: int) -> None:
        """
        Initialize a InstanceVCPU object.

        :param str architecture: The VCPU architecture.
        :param int count: The number of VCPUs assigned.
        """
        self.architecture = architecture
        self.count = count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'InstanceVCPU':
        """Initialize a InstanceVCPU object from a json dictionary."""
        args = {}
        if 'architecture' in _dict:
            args['architecture'] = _dict.get('architecture')
        else:
            raise ValueError(
                'Required property \'architecture\' not present in InstanceVCPU JSON'
            )
        if 'count' in _dict:
            args['count'] = _dict.get('count')
        else:
            raise ValueError(
                'Required property \'count\' not present in InstanceVCPU JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a InstanceVCPU object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'architecture') and self.architecture is not None:
            _dict['architecture'] = self.architecture
        if hasattr(self, 'count') and self.count is not None:
            _dict['count'] = self.count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this InstanceVCPU object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'InstanceVCPU') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'InstanceVCPU') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class Key():
    """
    Key.

    :attr datetime created_at: The date and time that the key was created.
    :attr str crn: The CRN for this key.
    :attr str fingerprint: The fingerprint for this key.  The value is returned
          base64-encoded and prefixed with the hash algorithm (always `SHA256`).
    :attr str href: The URL for this key.
    :attr str id: The unique identifier for this key.
    :attr int length: The length of this key (in bits).
    :attr str name: The user-defined name for this key.
    :attr str public_key: The public SSH key.
    :attr ResourceGroupReference resource_group: The resource group for this key.
    :attr str type: The cryptosystem used by this key.
    """

    def __init__(self, created_at: datetime, crn: str, fingerprint: str,
                 href: str, id: str, length: int, name: str, public_key: str,
                 resource_group: 'ResourceGroupReference', type: str) -> None:
        """
        Initialize a Key object.

        :param datetime created_at: The date and time that the key was created.
        :param str crn: The CRN for this key.
        :param str fingerprint: The fingerprint for this key.  The value is
               returned base64-encoded and prefixed with the hash algorithm (always
               `SHA256`).
        :param str href: The URL for this key.
        :param str id: The unique identifier for this key.
        :param int length: The length of this key (in bits).
        :param str name: The user-defined name for this key.
        :param str public_key: The public SSH key.
        :param ResourceGroupReference resource_group: The resource group for this
               key.
        :param str type: The cryptosystem used by this key.
        """
        self.created_at = created_at
        self.crn = crn
        self.fingerprint = fingerprint
        self.href = href
        self.id = id
        self.length = length
        self.name = name
        self.public_key = public_key
        self.resource_group = resource_group
        self.type = type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'Key':
        """Initialize a Key object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in Key JSON')
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in Key JSON')
        if 'fingerprint' in _dict:
            args['fingerprint'] = _dict.get('fingerprint')
        else:
            raise ValueError(
                'Required property \'fingerprint\' not present in Key JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in Key JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError('Required property \'id\' not present in Key JSON')
        if 'length' in _dict:
            args['length'] = _dict.get('length')
        else:
            raise ValueError(
                'Required property \'length\' not present in Key JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in Key JSON')
        if 'public_key' in _dict:
            args['public_key'] = _dict.get('public_key')
        else:
            raise ValueError(
                'Required property \'public_key\' not present in Key JSON')
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(
                _dict.get('resource_group'))
        else:
            raise ValueError(
                'Required property \'resource_group\' not present in Key JSON')
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in Key JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a Key object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'fingerprint') and self.fingerprint is not None:
            _dict['fingerprint'] = self.fingerprint
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'length') and self.length is not None:
            _dict['length'] = self.length
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'public_key') and self.public_key is not None:
            _dict['public_key'] = self.public_key
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this Key object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'Key') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'Key') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class TypeEnum(str, Enum):
        """
        The cryptosystem used by this key.
        """
        RSA = 'rsa'


class KeyCollection():
    """
    KeyCollection.

    :attr KeyCollectionFirst first: A reference to the first page of resources.
    :attr List[Key] keys: Collection of keys.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr KeyCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 first: 'KeyCollectionFirst',
                 keys: List['Key'],
                 limit: int,
                 total_count: int,
                 *,
                 next: 'KeyCollectionNext' = None) -> None:
        """
        Initialize a KeyCollection object.

        :param KeyCollectionFirst first: A reference to the first page of
               resources.
        :param List[Key] keys: Collection of keys.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param int total_count: The total number of resources across all pages.
        :param KeyCollectionNext next: (optional) A reference to the next page of
               resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.keys = keys
        self.limit = limit
        self.next = next
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'KeyCollection':
        """Initialize a KeyCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = KeyCollectionFirst.from_dict(_dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in KeyCollection JSON')
        if 'keys' in _dict:
            args['keys'] = [Key.from_dict(x) for x in _dict.get('keys')]
        else:
            raise ValueError(
                'Required property \'keys\' not present in KeyCollection JSON')
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in KeyCollection JSON')
        if 'next' in _dict:
            args['next'] = KeyCollectionNext.from_dict(_dict.get('next'))
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError(
                'Required property \'total_count\' not present in KeyCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a KeyCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'keys') and self.keys is not None:
            _dict['keys'] = [x.to_dict() for x in self.keys]
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this KeyCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'KeyCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'KeyCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class KeyCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a KeyCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'KeyCollectionFirst':
        """Initialize a KeyCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in KeyCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a KeyCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this KeyCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'KeyCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'KeyCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class KeyCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a KeyCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'KeyCollectionNext':
        """Initialize a KeyCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in KeyCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a KeyCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this KeyCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'KeyCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'KeyCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class KeyIdentity():
    """
    Identifies a key by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a KeyIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'KeyIdentityById', 'KeyIdentityByCRN', 'KeyIdentityByHref',
                'KeyIdentityByFingerprint'
            ]))
        raise Exception(msg)


class KeyReferenceInstanceInitializationContext():
    """
    KeyReferenceInstanceInitializationContext.

    """

    def __init__(self) -> None:
        """
        Initialize a KeyReferenceInstanceInitializationContext object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'KeyReferenceInstanceInitializationContextKeyReference',
                'KeyReferenceInstanceInitializationContextKeyIdentityByFingerprint'
            ]))
        raise Exception(msg)


class LoadBalancer():
    """
    LoadBalancer.

    :attr datetime created_at: The date and time that this load balancer was
          created.
    :attr str crn: The load balancer's CRN.
    :attr str hostname: Fully qualified domain name assigned to this load balancer.
    :attr str href: The load balancer's canonical URL.
    :attr str id: The unique identifier for this load balancer.
    :attr bool is_public: The type of this load balancer, public or private.
    :attr List[LoadBalancerListenerReference] listeners: The listeners of this load
          balancer.
    :attr str name: The unique user-defined name for this load balancer.
    :attr str operating_status: The operating status of this load balancer.
    :attr List[LoadBalancerPoolReference] pools: The pools of this load balancer.
    :attr List[IP] private_ips: The private IP addresses assigned to this load
          balancer.
    :attr str provisioning_status: The provisioning status of this load balancer.
    :attr List[IP] public_ips: The public IP addresses assigned to this load
          balancer. These are applicable only for public load balancers.
    :attr ResourceGroupReference resource_group: The resource group for this load
          balancer.
    :attr List[SubnetReference] subnets: The subnets this load balancer is part of.
    """

    def __init__(self, created_at: datetime, crn: str, hostname: str, href: str,
                 id: str, is_public: bool,
                 listeners: List['LoadBalancerListenerReference'], name: str,
                 operating_status: str,
                 pools: List['LoadBalancerPoolReference'],
                 private_ips: List['IP'], provisioning_status: str,
                 public_ips: List['IP'],
                 resource_group: 'ResourceGroupReference',
                 subnets: List['SubnetReference']) -> None:
        """
        Initialize a LoadBalancer object.

        :param datetime created_at: The date and time that this load balancer was
               created.
        :param str crn: The load balancer's CRN.
        :param str hostname: Fully qualified domain name assigned to this load
               balancer.
        :param str href: The load balancer's canonical URL.
        :param str id: The unique identifier for this load balancer.
        :param bool is_public: The type of this load balancer, public or private.
        :param List[LoadBalancerListenerReference] listeners: The listeners of this
               load balancer.
        :param str name: The unique user-defined name for this load balancer.
        :param str operating_status: The operating status of this load balancer.
        :param List[LoadBalancerPoolReference] pools: The pools of this load
               balancer.
        :param List[IP] private_ips: The private IP addresses assigned to this load
               balancer.
        :param str provisioning_status: The provisioning status of this load
               balancer.
        :param List[IP] public_ips: The public IP addresses assigned to this load
               balancer. These are applicable only for public load balancers.
        :param ResourceGroupReference resource_group: The resource group for this
               load balancer.
        :param List[SubnetReference] subnets: The subnets this load balancer is
               part of.
        """
        self.created_at = created_at
        self.crn = crn
        self.hostname = hostname
        self.href = href
        self.id = id
        self.is_public = is_public
        self.listeners = listeners
        self.name = name
        self.operating_status = operating_status
        self.pools = pools
        self.private_ips = private_ips
        self.provisioning_status = provisioning_status
        self.public_ips = public_ips
        self.resource_group = resource_group
        self.subnets = subnets

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancer':
        """Initialize a LoadBalancer object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in LoadBalancer JSON'
            )
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in LoadBalancer JSON')
        if 'hostname' in _dict:
            args['hostname'] = _dict.get('hostname')
        else:
            raise ValueError(
                'Required property \'hostname\' not present in LoadBalancer JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancer JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancer JSON')
        if 'is_public' in _dict:
            args['is_public'] = _dict.get('is_public')
        else:
            raise ValueError(
                'Required property \'is_public\' not present in LoadBalancer JSON'
            )
        if 'listeners' in _dict:
            args['listeners'] = [
                LoadBalancerListenerReference.from_dict(x)
                for x in _dict.get('listeners')
            ]
        else:
            raise ValueError(
                'Required property \'listeners\' not present in LoadBalancer JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in LoadBalancer JSON')
        if 'operating_status' in _dict:
            args['operating_status'] = _dict.get('operating_status')
        else:
            raise ValueError(
                'Required property \'operating_status\' not present in LoadBalancer JSON'
            )
        if 'pools' in _dict:
            args['pools'] = [
                LoadBalancerPoolReference.from_dict(x)
                for x in _dict.get('pools')
            ]
        else:
            raise ValueError(
                'Required property \'pools\' not present in LoadBalancer JSON')
        if 'private_ips' in _dict:
            args['private_ips'] = [
                IP.from_dict(x) for x in _dict.get('private_ips')
            ]
        else:
            raise ValueError(
                'Required property \'private_ips\' not present in LoadBalancer JSON'
            )
        if 'provisioning_status' in _dict:
            args['provisioning_status'] = _dict.get('provisioning_status')
        else:
            raise ValueError(
                'Required property \'provisioning_status\' not present in LoadBalancer JSON'
            )
        if 'public_ips' in _dict:
            args['public_ips'] = [
                IP.from_dict(x) for x in _dict.get('public_ips')
            ]
        else:
            raise ValueError(
                'Required property \'public_ips\' not present in LoadBalancer JSON'
            )
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(
                _dict.get('resource_group'))
        else:
            raise ValueError(
                'Required property \'resource_group\' not present in LoadBalancer JSON'
            )
        if 'subnets' in _dict:
            args['subnets'] = [
                SubnetReference.from_dict(x) for x in _dict.get('subnets')
            ]
        else:
            raise ValueError(
                'Required property \'subnets\' not present in LoadBalancer JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancer object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'hostname') and self.hostname is not None:
            _dict['hostname'] = self.hostname
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'is_public') and self.is_public is not None:
            _dict['is_public'] = self.is_public
        if hasattr(self, 'listeners') and self.listeners is not None:
            _dict['listeners'] = [x.to_dict() for x in self.listeners]
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self,
                   'operating_status') and self.operating_status is not None:
            _dict['operating_status'] = self.operating_status
        if hasattr(self, 'pools') and self.pools is not None:
            _dict['pools'] = [x.to_dict() for x in self.pools]
        if hasattr(self, 'private_ips') and self.private_ips is not None:
            _dict['private_ips'] = [x.to_dict() for x in self.private_ips]
        if hasattr(
                self,
                'provisioning_status') and self.provisioning_status is not None:
            _dict['provisioning_status'] = self.provisioning_status
        if hasattr(self, 'public_ips') and self.public_ips is not None:
            _dict['public_ips'] = [x.to_dict() for x in self.public_ips]
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(self, 'subnets') and self.subnets is not None:
            _dict['subnets'] = [x.to_dict() for x in self.subnets]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancer object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancer') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancer') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class OperatingStatusEnum(str, Enum):
        """
        The operating status of this load balancer.
        """
        OFFLINE = 'offline'
        ONLINE = 'online'

    class ProvisioningStatusEnum(str, Enum):
        """
        The provisioning status of this load balancer.
        """
        ACTIVE = 'active'
        CREATE_PENDING = 'create_pending'
        DELETE_PENDING = 'delete_pending'
        FAILED = 'failed'
        MAINTENANCE_PENDING = 'maintenance_pending'
        UPDATE_PENDING = 'update_pending'


class LoadBalancerCollection():
    """
    LoadBalancerCollection.

    :attr List[LoadBalancer] load_balancers: Collection of load balancers.
    """

    def __init__(self, load_balancers: List['LoadBalancer']) -> None:
        """
        Initialize a LoadBalancerCollection object.

        :param List[LoadBalancer] load_balancers: Collection of load balancers.
        """
        self.load_balancers = load_balancers

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerCollection':
        """Initialize a LoadBalancerCollection object from a json dictionary."""
        args = {}
        if 'load_balancers' in _dict:
            args['load_balancers'] = [
                LoadBalancer.from_dict(x) for x in _dict.get('load_balancers')
            ]
        else:
            raise ValueError(
                'Required property \'load_balancers\' not present in LoadBalancerCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'load_balancers') and self.load_balancers is not None:
            _dict['load_balancers'] = [x.to_dict() for x in self.load_balancers]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerListener():
    """
    LoadBalancerListener.

    :attr CertificateInstanceReference certificate_instance: (optional) The
          certificate instance used for SSL termination. It is applicable only to `https`
          protocol.
    :attr int connection_limit: (optional) The connection limit of the listener.
    :attr datetime created_at: The date and time that this listener was created.
    :attr LoadBalancerPoolReference default_pool: (optional) The default pool
          associated with the listener.
    :attr str href: The listener's canonical URL.
    :attr str id: The unique identifier for this load balancer listener.
    :attr List[LoadBalancerListenerPolicyReference] policies: (optional) The list of
          policies of this listener.
    :attr int port: The listener port number.
    :attr str protocol: The listener protocol.
    :attr str provisioning_status: The provisioning status of this listener.
    """

    def __init__(
            self,
            created_at: datetime,
            href: str,
            id: str,
            port: int,
            protocol: str,
            provisioning_status: str,
            *,
            certificate_instance: 'CertificateInstanceReference' = None,
            connection_limit: int = None,
            default_pool: 'LoadBalancerPoolReference' = None,
            policies: List['LoadBalancerListenerPolicyReference'] = None
    ) -> None:
        """
        Initialize a LoadBalancerListener object.

        :param datetime created_at: The date and time that this listener was
               created.
        :param str href: The listener's canonical URL.
        :param str id: The unique identifier for this load balancer listener.
        :param int port: The listener port number.
        :param str protocol: The listener protocol.
        :param str provisioning_status: The provisioning status of this listener.
        :param CertificateInstanceReference certificate_instance: (optional) The
               certificate instance used for SSL termination. It is applicable only to
               `https`
               protocol.
        :param int connection_limit: (optional) The connection limit of the
               listener.
        :param LoadBalancerPoolReference default_pool: (optional) The default pool
               associated with the listener.
        :param List[LoadBalancerListenerPolicyReference] policies: (optional) The
               list of policies of this listener.
        """
        self.certificate_instance = certificate_instance
        self.connection_limit = connection_limit
        self.created_at = created_at
        self.default_pool = default_pool
        self.href = href
        self.id = id
        self.policies = policies
        self.port = port
        self.protocol = protocol
        self.provisioning_status = provisioning_status

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerListener':
        """Initialize a LoadBalancerListener object from a json dictionary."""
        args = {}
        if 'certificate_instance' in _dict:
            args[
                'certificate_instance'] = CertificateInstanceReference.from_dict(
                    _dict.get('certificate_instance'))
        if 'connection_limit' in _dict:
            args['connection_limit'] = _dict.get('connection_limit')
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in LoadBalancerListener JSON'
            )
        if 'default_pool' in _dict:
            args['default_pool'] = LoadBalancerPoolReference.from_dict(
                _dict.get('default_pool'))
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancerListener JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancerListener JSON'
            )
        if 'policies' in _dict:
            args['policies'] = [
                LoadBalancerListenerPolicyReference.from_dict(x)
                for x in _dict.get('policies')
            ]
        if 'port' in _dict:
            args['port'] = _dict.get('port')
        else:
            raise ValueError(
                'Required property \'port\' not present in LoadBalancerListener JSON'
            )
        if 'protocol' in _dict:
            args['protocol'] = _dict.get('protocol')
        else:
            raise ValueError(
                'Required property \'protocol\' not present in LoadBalancerListener JSON'
            )
        if 'provisioning_status' in _dict:
            args['provisioning_status'] = _dict.get('provisioning_status')
        else:
            raise ValueError(
                'Required property \'provisioning_status\' not present in LoadBalancerListener JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListener object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'certificate_instance'
                  ) and self.certificate_instance is not None:
            _dict['certificate_instance'] = self.certificate_instance.to_dict()
        if hasattr(self,
                   'connection_limit') and self.connection_limit is not None:
            _dict['connection_limit'] = self.connection_limit
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'default_pool') and self.default_pool is not None:
            _dict['default_pool'] = self.default_pool.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'policies') and self.policies is not None:
            _dict['policies'] = [x.to_dict() for x in self.policies]
        if hasattr(self, 'port') and self.port is not None:
            _dict['port'] = self.port
        if hasattr(self, 'protocol') and self.protocol is not None:
            _dict['protocol'] = self.protocol
        if hasattr(
                self,
                'provisioning_status') and self.provisioning_status is not None:
            _dict['provisioning_status'] = self.provisioning_status
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListener object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListener') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListener') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ProtocolEnum(str, Enum):
        """
        The listener protocol.
        """
        HTTP = 'http'
        HTTPS = 'https'
        TCP = 'tcp'

    class ProvisioningStatusEnum(str, Enum):
        """
        The provisioning status of this listener.
        """
        ACTIVE = 'active'
        CREATE_PENDING = 'create_pending'
        DELETE_PENDING = 'delete_pending'
        FAILED = 'failed'
        MAINTENANCE_PENDING = 'maintenance_pending'
        UPDATE_PENDING = 'update_pending'


class LoadBalancerListenerCollection():
    """
    LoadBalancerListenerCollection.

    :attr List[LoadBalancerListener] listeners: Collection of listeners.
    """

    def __init__(self, listeners: List['LoadBalancerListener']) -> None:
        """
        Initialize a LoadBalancerListenerCollection object.

        :param List[LoadBalancerListener] listeners: Collection of listeners.
        """
        self.listeners = listeners

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerListenerCollection':
        """Initialize a LoadBalancerListenerCollection object from a json dictionary."""
        args = {}
        if 'listeners' in _dict:
            args['listeners'] = [
                LoadBalancerListener.from_dict(x)
                for x in _dict.get('listeners')
            ]
        else:
            raise ValueError(
                'Required property \'listeners\' not present in LoadBalancerListenerCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'listeners') and self.listeners is not None:
            _dict['listeners'] = [x.to_dict() for x in self.listeners]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListenerCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListenerCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerListenerPolicy():
    """
    LoadBalancerListenerPolicy.

    :attr str action: The policy action.
    :attr datetime created_at: The date and time that this policy was created.
    :attr str href: The listener policy's canonical URL.
    :attr str id: The policy's unique identifier.
    :attr str name: The user-defined name for this policy.
    :attr int priority: Priority of the policy. Lower value indicates higher
          priority.
    :attr str provisioning_status: The provisioning status of this policy.
    :attr List[LoadBalancerListenerPolicyRuleReference] rules: The rules of this
          policy.
    :attr LoadBalancerListenerPolicyTargetReference target: (optional)
          `LoadBalancerPoolReference` is in the response if `action` is `forward`.
          `LoadBalancerListenerPolicyRedirectURL` is in the response if `action` is
          `redirect`.
    """

    def __init__(
            self,
            action: str,
            created_at: datetime,
            href: str,
            id: str,
            name: str,
            priority: int,
            provisioning_status: str,
            rules: List['LoadBalancerListenerPolicyRuleReference'],
            *,
            target: 'LoadBalancerListenerPolicyTargetReference' = None) -> None:
        """
        Initialize a LoadBalancerListenerPolicy object.

        :param str action: The policy action.
        :param datetime created_at: The date and time that this policy was created.
        :param str href: The listener policy's canonical URL.
        :param str id: The policy's unique identifier.
        :param str name: The user-defined name for this policy.
        :param int priority: Priority of the policy. Lower value indicates higher
               priority.
        :param str provisioning_status: The provisioning status of this policy.
        :param List[LoadBalancerListenerPolicyRuleReference] rules: The rules of
               this policy.
        :param LoadBalancerListenerPolicyTargetReference target: (optional)
               `LoadBalancerPoolReference` is in the response if `action` is `forward`.
               `LoadBalancerListenerPolicyRedirectURL` is in the response if `action` is
               `redirect`.
        """
        self.action = action
        self.created_at = created_at
        self.href = href
        self.id = id
        self.name = name
        self.priority = priority
        self.provisioning_status = provisioning_status
        self.rules = rules
        self.target = target

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerListenerPolicy':
        """Initialize a LoadBalancerListenerPolicy object from a json dictionary."""
        args = {}
        if 'action' in _dict:
            args['action'] = _dict.get('action')
        else:
            raise ValueError(
                'Required property \'action\' not present in LoadBalancerListenerPolicy JSON'
            )
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in LoadBalancerListenerPolicy JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancerListenerPolicy JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancerListenerPolicy JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in LoadBalancerListenerPolicy JSON'
            )
        if 'priority' in _dict:
            args['priority'] = _dict.get('priority')
        else:
            raise ValueError(
                'Required property \'priority\' not present in LoadBalancerListenerPolicy JSON'
            )
        if 'provisioning_status' in _dict:
            args['provisioning_status'] = _dict.get('provisioning_status')
        else:
            raise ValueError(
                'Required property \'provisioning_status\' not present in LoadBalancerListenerPolicy JSON'
            )
        if 'rules' in _dict:
            args['rules'] = [
                LoadBalancerListenerPolicyRuleReference.from_dict(x)
                for x in _dict.get('rules')
            ]
        else:
            raise ValueError(
                'Required property \'rules\' not present in LoadBalancerListenerPolicy JSON'
            )
        if 'target' in _dict:
            args['target'] = _dict.get('target')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerPolicy object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'action') and self.action is not None:
            _dict['action'] = self.action
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'priority') and self.priority is not None:
            _dict['priority'] = self.priority
        if hasattr(
                self,
                'provisioning_status') and self.provisioning_status is not None:
            _dict['provisioning_status'] = self.provisioning_status
        if hasattr(self, 'rules') and self.rules is not None:
            _dict['rules'] = [x.to_dict() for x in self.rules]
        if hasattr(self, 'target') and self.target is not None:
            _dict['target'] = self.target
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerPolicy object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListenerPolicy') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListenerPolicy') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ActionEnum(str, Enum):
        """
        The policy action.
        """
        FORWARD = 'forward'
        REDIRECT = 'redirect'
        REJECT = 'reject'

    class ProvisioningStatusEnum(str, Enum):
        """
        The provisioning status of this policy.
        """
        ACTIVE = 'active'
        CREATE_PENDING = 'create_pending'
        DELETE_PENDING = 'delete_pending'
        FAILED = 'failed'
        MAINTENANCE_PENDING = 'maintenance_pending'
        UPDATE_PENDING = 'update_pending'


class LoadBalancerListenerPolicyCollection():
    """
    LoadBalancerListenerPolicyCollection.

    :attr List[LoadBalancerListenerPolicy] policies: Collection of policies.
    """

    def __init__(self, policies: List['LoadBalancerListenerPolicy']) -> None:
        """
        Initialize a LoadBalancerListenerPolicyCollection object.

        :param List[LoadBalancerListenerPolicy] policies: Collection of policies.
        """
        self.policies = policies

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerListenerPolicyCollection':
        """Initialize a LoadBalancerListenerPolicyCollection object from a json dictionary."""
        args = {}
        if 'policies' in _dict:
            args['policies'] = [
                LoadBalancerListenerPolicy.from_dict(x)
                for x in _dict.get('policies')
            ]
        else:
            raise ValueError(
                'Required property \'policies\' not present in LoadBalancerListenerPolicyCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerPolicyCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'policies') and self.policies is not None:
            _dict['policies'] = [x.to_dict() for x in self.policies]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerPolicyCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListenerPolicyCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListenerPolicyCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerListenerPolicyPatchTarget():
    """
    When `action` is `forward`, `LoadBalancerPoolIdentity` specifies which pool the load
    balancer forwards the traffic to. When `action` is `redirect`,
    `LoadBalancerListenerPolicyRedirectURLPatch` specifies the url and http status code
    used in the redirect response.

    """

    def __init__(self) -> None:
        """
        Initialize a LoadBalancerListenerPolicyPatchTarget object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'LoadBalancerListenerPolicyPatchTargetLoadBalancerPoolIdentity',
                'LoadBalancerListenerPolicyPatchTargetLoadBalancerListenerPolicyRedirectURLPatch'
            ]))
        raise Exception(msg)


class LoadBalancerListenerPolicyPrototype():
    """
    LoadBalancerListenerPolicyPrototype.

    :attr str action: The policy action.
    :attr str name: (optional) The user-defined name for this policy. Names must be
          unique within the load balancer listener the policy resides in.
    :attr int priority: Priority of the policy. Lower value indicates higher
          priority.
    :attr List[LoadBalancerListenerPolicyRulePrototype] rules: (optional) The list
          of rules of this policy.
    :attr LoadBalancerListenerPolicyPrototypeTarget target: (optional) When `action`
          is `forward`, `LoadBalancerPoolIdentity` is required to specify which
          pool the load balancer forwards the traffic to. When `action` is `redirect`,
          `LoadBalancerListenerPolicyRedirectURLPrototype` is required to specify the url
          and
          http status code used in the redirect response.
    """

    def __init__(
            self,
            action: str,
            priority: int,
            *,
            name: str = None,
            rules: List['LoadBalancerListenerPolicyRulePrototype'] = None,
            target: 'LoadBalancerListenerPolicyPrototypeTarget' = None) -> None:
        """
        Initialize a LoadBalancerListenerPolicyPrototype object.

        :param str action: The policy action.
        :param int priority: Priority of the policy. Lower value indicates higher
               priority.
        :param str name: (optional) The user-defined name for this policy. Names
               must be unique within the load balancer listener the policy resides in.
        :param List[LoadBalancerListenerPolicyRulePrototype] rules: (optional) The
               list of rules of this policy.
        :param LoadBalancerListenerPolicyPrototypeTarget target: (optional) When
               `action` is `forward`, `LoadBalancerPoolIdentity` is required to specify
               which
               pool the load balancer forwards the traffic to. When `action` is
               `redirect`,
               `LoadBalancerListenerPolicyRedirectURLPrototype` is required to specify the
               url and
               http status code used in the redirect response.
        """
        self.action = action
        self.name = name
        self.priority = priority
        self.rules = rules
        self.target = target

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerListenerPolicyPrototype':
        """Initialize a LoadBalancerListenerPolicyPrototype object from a json dictionary."""
        args = {}
        if 'action' in _dict:
            args['action'] = _dict.get('action')
        else:
            raise ValueError(
                'Required property \'action\' not present in LoadBalancerListenerPolicyPrototype JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        if 'priority' in _dict:
            args['priority'] = _dict.get('priority')
        else:
            raise ValueError(
                'Required property \'priority\' not present in LoadBalancerListenerPolicyPrototype JSON'
            )
        if 'rules' in _dict:
            args['rules'] = [
                LoadBalancerListenerPolicyRulePrototype.from_dict(x)
                for x in _dict.get('rules')
            ]
        if 'target' in _dict:
            args['target'] = _dict.get('target')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerPolicyPrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'action') and self.action is not None:
            _dict['action'] = self.action
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'priority') and self.priority is not None:
            _dict['priority'] = self.priority
        if hasattr(self, 'rules') and self.rules is not None:
            _dict['rules'] = [x.to_dict() for x in self.rules]
        if hasattr(self, 'target') and self.target is not None:
            _dict['target'] = self.target
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerPolicyPrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListenerPolicyPrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListenerPolicyPrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ActionEnum(str, Enum):
        """
        The policy action.
        """
        FORWARD = 'forward'
        REDIRECT = 'redirect'
        REJECT = 'reject'


class LoadBalancerListenerPolicyPrototypeTarget():
    """
    When `action` is `forward`, `LoadBalancerPoolIdentity` is required to specify which
    pool the load balancer forwards the traffic to. When `action` is `redirect`,
    `LoadBalancerListenerPolicyRedirectURLPrototype` is required to specify the url and
    http status code used in the redirect response.

    """

    def __init__(self) -> None:
        """
        Initialize a LoadBalancerListenerPolicyPrototypeTarget object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'LoadBalancerListenerPolicyPrototypeTargetLoadBalancerPoolIdentity',
                'LoadBalancerListenerPolicyPrototypeTargetLoadBalancerListenerPolicyRedirectURLPrototype'
            ]))
        raise Exception(msg)


class LoadBalancerListenerPolicyReference():
    """
    LoadBalancerListenerPolicyReference.

    :attr str href: The listener policy's canonical URL.
    :attr str id: The policy's unique identifier.
    """

    def __init__(self, href: str, id: str) -> None:
        """
        Initialize a LoadBalancerListenerPolicyReference object.

        :param str href: The listener policy's canonical URL.
        :param str id: The policy's unique identifier.
        """
        self.href = href
        self.id = id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerListenerPolicyReference':
        """Initialize a LoadBalancerListenerPolicyReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancerListenerPolicyReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancerListenerPolicyReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerPolicyReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerPolicyReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListenerPolicyReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListenerPolicyReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerListenerPolicyRule():
    """
    LoadBalancerListenerPolicyRule.

    :attr str condition: The condition of the rule.
    :attr datetime created_at: The date and time that this rule was created.
    :attr str field: (optional) HTTP header field. This is only applicable to
          "header" rule type.
    :attr str href: The rule's canonical URL.
    :attr str id: The rule's unique identifier.
    :attr str provisioning_status: The provisioning status of this rule.
    :attr str type: The type of the rule.
    :attr str value: Value to be matched for rule condition.
    """

    def __init__(self,
                 condition: str,
                 created_at: datetime,
                 href: str,
                 id: str,
                 provisioning_status: str,
                 type: str,
                 value: str,
                 *,
                 field: str = None) -> None:
        """
        Initialize a LoadBalancerListenerPolicyRule object.

        :param str condition: The condition of the rule.
        :param datetime created_at: The date and time that this rule was created.
        :param str href: The rule's canonical URL.
        :param str id: The rule's unique identifier.
        :param str provisioning_status: The provisioning status of this rule.
        :param str type: The type of the rule.
        :param str value: Value to be matched for rule condition.
        :param str field: (optional) HTTP header field. This is only applicable to
               "header" rule type.
        """
        self.condition = condition
        self.created_at = created_at
        self.field = field
        self.href = href
        self.id = id
        self.provisioning_status = provisioning_status
        self.type = type
        self.value = value

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerListenerPolicyRule':
        """Initialize a LoadBalancerListenerPolicyRule object from a json dictionary."""
        args = {}
        if 'condition' in _dict:
            args['condition'] = _dict.get('condition')
        else:
            raise ValueError(
                'Required property \'condition\' not present in LoadBalancerListenerPolicyRule JSON'
            )
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in LoadBalancerListenerPolicyRule JSON'
            )
        if 'field' in _dict:
            args['field'] = _dict.get('field')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancerListenerPolicyRule JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancerListenerPolicyRule JSON'
            )
        if 'provisioning_status' in _dict:
            args['provisioning_status'] = _dict.get('provisioning_status')
        else:
            raise ValueError(
                'Required property \'provisioning_status\' not present in LoadBalancerListenerPolicyRule JSON'
            )
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LoadBalancerListenerPolicyRule JSON'
            )
        if 'value' in _dict:
            args['value'] = _dict.get('value')
        else:
            raise ValueError(
                'Required property \'value\' not present in LoadBalancerListenerPolicyRule JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerPolicyRule object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'condition') and self.condition is not None:
            _dict['condition'] = self.condition
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'field') and self.field is not None:
            _dict['field'] = self.field
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(
                self,
                'provisioning_status') and self.provisioning_status is not None:
            _dict['provisioning_status'] = self.provisioning_status
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'value') and self.value is not None:
            _dict['value'] = self.value
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerPolicyRule object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListenerPolicyRule') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListenerPolicyRule') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ConditionEnum(str, Enum):
        """
        The condition of the rule.
        """
        CONTAINS = 'contains'
        EQUALS = 'equals'
        MATCHES_REGEX = 'matches_regex'

    class ProvisioningStatusEnum(str, Enum):
        """
        The provisioning status of this rule.
        """
        ACTIVE = 'active'
        CREATE_PENDING = 'create_pending'
        DELETE_PENDING = 'delete_pending'
        FAILED = 'failed'
        MAINTENANCE_PENDING = 'maintenance_pending'
        UPDATE_PENDING = 'update_pending'

    class TypeEnum(str, Enum):
        """
        The type of the rule.
        """
        HEADER = 'header'
        HOSTNAME = 'hostname'
        PATH = 'path'


class LoadBalancerListenerPolicyRuleCollection():
    """
    LoadBalancerListenerPolicyRuleCollection.

    :attr List[LoadBalancerListenerPolicyRule] rules: Collection of rules.
    """

    def __init__(self, rules: List['LoadBalancerListenerPolicyRule']) -> None:
        """
        Initialize a LoadBalancerListenerPolicyRuleCollection object.

        :param List[LoadBalancerListenerPolicyRule] rules: Collection of rules.
        """
        self.rules = rules

    @classmethod
    def from_dict(cls,
                  _dict: Dict) -> 'LoadBalancerListenerPolicyRuleCollection':
        """Initialize a LoadBalancerListenerPolicyRuleCollection object from a json dictionary."""
        args = {}
        if 'rules' in _dict:
            args['rules'] = [
                LoadBalancerListenerPolicyRule.from_dict(x)
                for x in _dict.get('rules')
            ]
        else:
            raise ValueError(
                'Required property \'rules\' not present in LoadBalancerListenerPolicyRuleCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerPolicyRuleCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'rules') and self.rules is not None:
            _dict['rules'] = [x.to_dict() for x in self.rules]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerPolicyRuleCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListenerPolicyRuleCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListenerPolicyRuleCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerListenerPolicyRulePrototype():
    """
    LoadBalancerListenerPolicyRulePrototype.

    :attr str condition: The condition of the rule.
    :attr str field: (optional) HTTP header field. This is only applicable to
          "header" rule type.
    :attr str type: The type of the rule.
    :attr str value: Value to be matched for rule condition.
    """

    def __init__(self,
                 condition: str,
                 type: str,
                 value: str,
                 *,
                 field: str = None) -> None:
        """
        Initialize a LoadBalancerListenerPolicyRulePrototype object.

        :param str condition: The condition of the rule.
        :param str type: The type of the rule.
        :param str value: Value to be matched for rule condition.
        :param str field: (optional) HTTP header field. This is only applicable to
               "header" rule type.
        """
        self.condition = condition
        self.field = field
        self.type = type
        self.value = value

    @classmethod
    def from_dict(cls,
                  _dict: Dict) -> 'LoadBalancerListenerPolicyRulePrototype':
        """Initialize a LoadBalancerListenerPolicyRulePrototype object from a json dictionary."""
        args = {}
        if 'condition' in _dict:
            args['condition'] = _dict.get('condition')
        else:
            raise ValueError(
                'Required property \'condition\' not present in LoadBalancerListenerPolicyRulePrototype JSON'
            )
        if 'field' in _dict:
            args['field'] = _dict.get('field')
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LoadBalancerListenerPolicyRulePrototype JSON'
            )
        if 'value' in _dict:
            args['value'] = _dict.get('value')
        else:
            raise ValueError(
                'Required property \'value\' not present in LoadBalancerListenerPolicyRulePrototype JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerPolicyRulePrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'condition') and self.condition is not None:
            _dict['condition'] = self.condition
        if hasattr(self, 'field') and self.field is not None:
            _dict['field'] = self.field
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'value') and self.value is not None:
            _dict['value'] = self.value
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerPolicyRulePrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListenerPolicyRulePrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListenerPolicyRulePrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ConditionEnum(str, Enum):
        """
        The condition of the rule.
        """
        CONTAINS = 'contains'
        EQUALS = 'equals'
        MATCHES_REGEX = 'matches_regex'

    class TypeEnum(str, Enum):
        """
        The type of the rule.
        """
        HEADER = 'header'
        HOSTNAME = 'hostname'
        PATH = 'path'


class LoadBalancerListenerPolicyRuleReference():
    """
    LoadBalancerListenerPolicyRuleReference.

    :attr str href: The rule's canonical URL.
    :attr str id: The rule's unique identifier.
    """

    def __init__(self, href: str, id: str) -> None:
        """
        Initialize a LoadBalancerListenerPolicyRuleReference object.

        :param str href: The rule's canonical URL.
        :param str id: The rule's unique identifier.
        """
        self.href = href
        self.id = id

    @classmethod
    def from_dict(cls,
                  _dict: Dict) -> 'LoadBalancerListenerPolicyRuleReference':
        """Initialize a LoadBalancerListenerPolicyRuleReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancerListenerPolicyRuleReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancerListenerPolicyRuleReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerPolicyRuleReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerPolicyRuleReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListenerPolicyRuleReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListenerPolicyRuleReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerListenerPolicyTargetReference():
    """
    `LoadBalancerPoolReference` is in the response if `action` is `forward`.
    `LoadBalancerListenerPolicyRedirectURL` is in the response if `action` is `redirect`.

    """

    def __init__(self) -> None:
        """
        Initialize a LoadBalancerListenerPolicyTargetReference object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'LoadBalancerListenerPolicyTargetReferenceLoadBalancerPoolReference',
                'LoadBalancerListenerPolicyTargetReferenceLoadBalancerListenerPolicyRedirectURL'
            ]))
        raise Exception(msg)


class LoadBalancerListenerPrototypeLoadBalancerContext():
    """
    LoadBalancerListenerPrototypeLoadBalancerContext.

    :attr int connection_limit: (optional) The connection limit of the listener.
    :attr LoadBalancerPoolIdentityByName default_pool: (optional) The default pool
          associated with the listener.
    :attr int port: The listener port number.
    :attr str protocol: The listener protocol.
    """

    def __init__(self,
                 port: int,
                 protocol: str,
                 *,
                 connection_limit: int = None,
                 default_pool: 'LoadBalancerPoolIdentityByName' = None) -> None:
        """
        Initialize a LoadBalancerListenerPrototypeLoadBalancerContext object.

        :param int port: The listener port number.
        :param str protocol: The listener protocol.
        :param int connection_limit: (optional) The connection limit of the
               listener.
        :param LoadBalancerPoolIdentityByName default_pool: (optional) The default
               pool associated with the listener.
        """
        self.connection_limit = connection_limit
        self.default_pool = default_pool
        self.port = port
        self.protocol = protocol

    @classmethod
    def from_dict(
            cls,
            _dict: Dict) -> 'LoadBalancerListenerPrototypeLoadBalancerContext':
        """Initialize a LoadBalancerListenerPrototypeLoadBalancerContext object from a json dictionary."""
        args = {}
        if 'connection_limit' in _dict:
            args['connection_limit'] = _dict.get('connection_limit')
        if 'default_pool' in _dict:
            args['default_pool'] = LoadBalancerPoolIdentityByName.from_dict(
                _dict.get('default_pool'))
        if 'port' in _dict:
            args['port'] = _dict.get('port')
        else:
            raise ValueError(
                'Required property \'port\' not present in LoadBalancerListenerPrototypeLoadBalancerContext JSON'
            )
        if 'protocol' in _dict:
            args['protocol'] = _dict.get('protocol')
        else:
            raise ValueError(
                'Required property \'protocol\' not present in LoadBalancerListenerPrototypeLoadBalancerContext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerPrototypeLoadBalancerContext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self,
                   'connection_limit') and self.connection_limit is not None:
            _dict['connection_limit'] = self.connection_limit
        if hasattr(self, 'default_pool') and self.default_pool is not None:
            _dict['default_pool'] = self.default_pool.to_dict()
        if hasattr(self, 'port') and self.port is not None:
            _dict['port'] = self.port
        if hasattr(self, 'protocol') and self.protocol is not None:
            _dict['protocol'] = self.protocol
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerPrototypeLoadBalancerContext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(
            self,
            other: 'LoadBalancerListenerPrototypeLoadBalancerContext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(
            self,
            other: 'LoadBalancerListenerPrototypeLoadBalancerContext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ProtocolEnum(str, Enum):
        """
        The listener protocol.
        """
        HTTP = 'http'
        HTTPS = 'https'
        TCP = 'tcp'


class LoadBalancerListenerReference():
    """
    LoadBalancerListenerReference.

    :attr str href: The listener's canonical URL.
    :attr str id: The unique identifier for this load balancer listener.
    """

    def __init__(self, href: str, id: str) -> None:
        """
        Initialize a LoadBalancerListenerReference object.

        :param str href: The listener's canonical URL.
        :param str id: The unique identifier for this load balancer listener.
        """
        self.href = href
        self.id = id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerListenerReference':
        """Initialize a LoadBalancerListenerReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancerListenerReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancerListenerReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerListenerReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerListenerReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerListenerReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerListenerReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerPool():
    """
    LoadBalancerPool.

    :attr str algorithm: The load balancing algorithm.
    :attr datetime created_at: The date and time that this pool was created.
    :attr LoadBalancerPoolHealthMonitor health_monitor: The health monitor of this
          pool.
    :attr str href: The pool's canonical URL.
    :attr str id: The unique identifier for this load balancer pool.
    :attr List[LoadBalancerPoolMemberReference] members: (optional) The backend
          server members of the pool.
    :attr str name: The user-defined name for this load balancer pool.
    :attr str protocol: The protocol used for this load balancer pool.
          The enumerated values for this property are expected to expand in the future.
          When processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the pool on which the unexpected
          property value was encountered.
    :attr str provisioning_status: The provisioning status of this pool.
    :attr LoadBalancerPoolSessionPersistence session_persistence: (optional) The
          session persistence of this pool.
    """

    def __init__(
        self,
        algorithm: str,
        created_at: datetime,
        health_monitor: 'LoadBalancerPoolHealthMonitor',
        href: str,
        id: str,
        name: str,
        protocol: str,
        provisioning_status: str,
        *,
        members: List['LoadBalancerPoolMemberReference'] = None,
        session_persistence: 'LoadBalancerPoolSessionPersistence' = None
    ) -> None:
        """
        Initialize a LoadBalancerPool object.

        :param str algorithm: The load balancing algorithm.
        :param datetime created_at: The date and time that this pool was created.
        :param LoadBalancerPoolHealthMonitor health_monitor: The health monitor of
               this pool.
        :param str href: The pool's canonical URL.
        :param str id: The unique identifier for this load balancer pool.
        :param str name: The user-defined name for this load balancer pool.
        :param str protocol: The protocol used for this load balancer pool.
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the pool on
               which the unexpected property value was encountered.
        :param str provisioning_status: The provisioning status of this pool.
        :param List[LoadBalancerPoolMemberReference] members: (optional) The
               backend server members of the pool.
        :param LoadBalancerPoolSessionPersistence session_persistence: (optional)
               The session persistence of this pool.
        """
        self.algorithm = algorithm
        self.created_at = created_at
        self.health_monitor = health_monitor
        self.href = href
        self.id = id
        self.members = members
        self.name = name
        self.protocol = protocol
        self.provisioning_status = provisioning_status
        self.session_persistence = session_persistence

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPool':
        """Initialize a LoadBalancerPool object from a json dictionary."""
        args = {}
        if 'algorithm' in _dict:
            args['algorithm'] = _dict.get('algorithm')
        else:
            raise ValueError(
                'Required property \'algorithm\' not present in LoadBalancerPool JSON'
            )
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in LoadBalancerPool JSON'
            )
        if 'health_monitor' in _dict:
            args['health_monitor'] = LoadBalancerPoolHealthMonitor.from_dict(
                _dict.get('health_monitor'))
        else:
            raise ValueError(
                'Required property \'health_monitor\' not present in LoadBalancerPool JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancerPool JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancerPool JSON')
        if 'members' in _dict:
            args['members'] = [
                LoadBalancerPoolMemberReference.from_dict(x)
                for x in _dict.get('members')
            ]
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in LoadBalancerPool JSON'
            )
        if 'protocol' in _dict:
            args['protocol'] = _dict.get('protocol')
        else:
            raise ValueError(
                'Required property \'protocol\' not present in LoadBalancerPool JSON'
            )
        if 'provisioning_status' in _dict:
            args['provisioning_status'] = _dict.get('provisioning_status')
        else:
            raise ValueError(
                'Required property \'provisioning_status\' not present in LoadBalancerPool JSON'
            )
        if 'session_persistence' in _dict:
            args[
                'session_persistence'] = LoadBalancerPoolSessionPersistence.from_dict(
                    _dict.get('session_persistence'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPool object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'algorithm') and self.algorithm is not None:
            _dict['algorithm'] = self.algorithm
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'health_monitor') and self.health_monitor is not None:
            _dict['health_monitor'] = self.health_monitor.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'members') and self.members is not None:
            _dict['members'] = [x.to_dict() for x in self.members]
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'protocol') and self.protocol is not None:
            _dict['protocol'] = self.protocol
        if hasattr(
                self,
                'provisioning_status') and self.provisioning_status is not None:
            _dict['provisioning_status'] = self.provisioning_status
        if hasattr(
                self,
                'session_persistence') and self.session_persistence is not None:
            _dict['session_persistence'] = self.session_persistence.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPool object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPool') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPool') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class AlgorithmEnum(str, Enum):
        """
        The load balancing algorithm.
        """
        LEAST_CONNECTIONS = 'least_connections'
        ROUND_ROBIN = 'round_robin'
        WEIGHTED_ROUND_ROBIN = 'weighted_round_robin'

    class ProtocolEnum(str, Enum):
        """
        The protocol used for this load balancer pool.
        The enumerated values for this property are expected to expand in the future. When
        processing this property, check for and log unknown values. Optionally halt
        processing and surface the error, or bypass the pool on which the unexpected
        property value was encountered.
        """
        HTTP = 'http'
        TCP = 'tcp'
        HTTPS = 'https'

    class ProvisioningStatusEnum(str, Enum):
        """
        The provisioning status of this pool.
        """
        ACTIVE = 'active'
        CREATE_PENDING = 'create_pending'
        DELETE_PENDING = 'delete_pending'
        FAILED = 'failed'
        MAINTENANCE_PENDING = 'maintenance_pending'
        UPDATE_PENDING = 'update_pending'


class LoadBalancerPoolCollection():
    """
    LoadBalancerPoolCollection.

    :attr List[LoadBalancerPool] pools: Collection of pools.
    """

    def __init__(self, pools: List['LoadBalancerPool']) -> None:
        """
        Initialize a LoadBalancerPoolCollection object.

        :param List[LoadBalancerPool] pools: Collection of pools.
        """
        self.pools = pools

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolCollection':
        """Initialize a LoadBalancerPoolCollection object from a json dictionary."""
        args = {}
        if 'pools' in _dict:
            args['pools'] = [
                LoadBalancerPool.from_dict(x) for x in _dict.get('pools')
            ]
        else:
            raise ValueError(
                'Required property \'pools\' not present in LoadBalancerPoolCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'pools') and self.pools is not None:
            _dict['pools'] = [x.to_dict() for x in self.pools]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerPoolHealthMonitor():
    """
    LoadBalancerPoolHealthMonitor.

    :attr int delay: The health check interval in seconds. Interval must be greater
          than timeout value.
    :attr int max_retries: The health check max retries.
    :attr int port: (optional) The health check port number. If specified, this
          overrides the ports specified in the server member resources.
    :attr int timeout: The health check timeout in seconds.
    :attr str type: The protocol type of this load balancer pool health monitor.
          The enumerated values for this property are expected to expand in the future.
          When processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the health monitor on which the
          unexpected property value was encountered.
    :attr str url_path: (optional) The health check URL. This is applicable only to
          `http` type of health monitor.
    """

    def __init__(self,
                 delay: int,
                 max_retries: int,
                 timeout: int,
                 type: str,
                 *,
                 port: int = None,
                 url_path: str = None) -> None:
        """
        Initialize a LoadBalancerPoolHealthMonitor object.

        :param int delay: The health check interval in seconds. Interval must be
               greater than timeout value.
        :param int max_retries: The health check max retries.
        :param int timeout: The health check timeout in seconds.
        :param str type: The protocol type of this load balancer pool health
               monitor.
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the health
               monitor on which the unexpected property value was encountered.
        :param int port: (optional) The health check port number. If specified,
               this overrides the ports specified in the server member resources.
        :param str url_path: (optional) The health check URL. This is applicable
               only to `http` type of health monitor.
        """
        self.delay = delay
        self.max_retries = max_retries
        self.port = port
        self.timeout = timeout
        self.type = type
        self.url_path = url_path

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolHealthMonitor':
        """Initialize a LoadBalancerPoolHealthMonitor object from a json dictionary."""
        args = {}
        if 'delay' in _dict:
            args['delay'] = _dict.get('delay')
        else:
            raise ValueError(
                'Required property \'delay\' not present in LoadBalancerPoolHealthMonitor JSON'
            )
        if 'max_retries' in _dict:
            args['max_retries'] = _dict.get('max_retries')
        else:
            raise ValueError(
                'Required property \'max_retries\' not present in LoadBalancerPoolHealthMonitor JSON'
            )
        if 'port' in _dict:
            args['port'] = _dict.get('port')
        if 'timeout' in _dict:
            args['timeout'] = _dict.get('timeout')
        else:
            raise ValueError(
                'Required property \'timeout\' not present in LoadBalancerPoolHealthMonitor JSON'
            )
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LoadBalancerPoolHealthMonitor JSON'
            )
        if 'url_path' in _dict:
            args['url_path'] = _dict.get('url_path')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolHealthMonitor object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'delay') and self.delay is not None:
            _dict['delay'] = self.delay
        if hasattr(self, 'max_retries') and self.max_retries is not None:
            _dict['max_retries'] = self.max_retries
        if hasattr(self, 'port') and self.port is not None:
            _dict['port'] = self.port
        if hasattr(self, 'timeout') and self.timeout is not None:
            _dict['timeout'] = self.timeout
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'url_path') and self.url_path is not None:
            _dict['url_path'] = self.url_path
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolHealthMonitor object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolHealthMonitor') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolHealthMonitor') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class TypeEnum(str, Enum):
        """
        The protocol type of this load balancer pool health monitor.
        The enumerated values for this property are expected to expand in the future. When
        processing this property, check for and log unknown values. Optionally halt
        processing and surface the error, or bypass the health monitor on which the
        unexpected property value was encountered.
        """
        HTTP = 'http'
        TCP = 'tcp'
        HTTPS = 'https'


class LoadBalancerPoolHealthMonitorPatch():
    """
    LoadBalancerPoolHealthMonitorPatch.

    :attr int delay: The health check interval in seconds. Interval must be greater
          than timeout value.
    :attr int max_retries: The health check max retries.
    :attr int port: (optional) The health check port number. If specified, this
          overrides the ports specified in the server member resources. Specify `null` to
          remove an existing port value.
    :attr int timeout: The health check timeout in seconds.
    :attr str type: The protocol type of this load balancer pool health monitor.
          The enumerated values for this property are expected to expand in the future.
          When processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the health monitor on which the
          unexpected property value was encountered.
    :attr str url_path: (optional) The health check URL. This is applicable only to
          `http` type of health monitor.
    """

    def __init__(self,
                 delay: int,
                 max_retries: int,
                 timeout: int,
                 type: str,
                 *,
                 port: int = None,
                 url_path: str = None) -> None:
        """
        Initialize a LoadBalancerPoolHealthMonitorPatch object.

        :param int delay: The health check interval in seconds. Interval must be
               greater than timeout value.
        :param int max_retries: The health check max retries.
        :param int timeout: The health check timeout in seconds.
        :param str type: The protocol type of this load balancer pool health
               monitor.
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the health
               monitor on which the unexpected property value was encountered.
        :param int port: (optional) The health check port number. If specified,
               this overrides the ports specified in the server member resources. Specify
               `null` to remove an existing port value.
        :param str url_path: (optional) The health check URL. This is applicable
               only to `http` type of health monitor.
        """
        self.delay = delay
        self.max_retries = max_retries
        self.port = port
        self.timeout = timeout
        self.type = type
        self.url_path = url_path

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolHealthMonitorPatch':
        """Initialize a LoadBalancerPoolHealthMonitorPatch object from a json dictionary."""
        args = {}
        if 'delay' in _dict:
            args['delay'] = _dict.get('delay')
        else:
            raise ValueError(
                'Required property \'delay\' not present in LoadBalancerPoolHealthMonitorPatch JSON'
            )
        if 'max_retries' in _dict:
            args['max_retries'] = _dict.get('max_retries')
        else:
            raise ValueError(
                'Required property \'max_retries\' not present in LoadBalancerPoolHealthMonitorPatch JSON'
            )
        if 'port' in _dict:
            args['port'] = _dict.get('port')
        if 'timeout' in _dict:
            args['timeout'] = _dict.get('timeout')
        else:
            raise ValueError(
                'Required property \'timeout\' not present in LoadBalancerPoolHealthMonitorPatch JSON'
            )
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LoadBalancerPoolHealthMonitorPatch JSON'
            )
        if 'url_path' in _dict:
            args['url_path'] = _dict.get('url_path')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolHealthMonitorPatch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'delay') and self.delay is not None:
            _dict['delay'] = self.delay
        if hasattr(self, 'max_retries') and self.max_retries is not None:
            _dict['max_retries'] = self.max_retries
        if hasattr(self, 'port') and self.port is not None:
            _dict['port'] = self.port
        if hasattr(self, 'timeout') and self.timeout is not None:
            _dict['timeout'] = self.timeout
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'url_path') and self.url_path is not None:
            _dict['url_path'] = self.url_path
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolHealthMonitorPatch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolHealthMonitorPatch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolHealthMonitorPatch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class TypeEnum(str, Enum):
        """
        The protocol type of this load balancer pool health monitor.
        The enumerated values for this property are expected to expand in the future. When
        processing this property, check for and log unknown values. Optionally halt
        processing and surface the error, or bypass the health monitor on which the
        unexpected property value was encountered.
        """
        HTTP = 'http'
        TCP = 'tcp'
        HTTPS = 'https'


class LoadBalancerPoolHealthMonitorPrototype():
    """
    LoadBalancerPoolHealthMonitorPrototype.

    :attr int delay: The health check interval in seconds. Interval must be greater
          than timeout value.
    :attr int max_retries: The health check max retries.
    :attr int port: (optional) The health check port number. If specified, this
          overrides the ports specified in the server member resources.
    :attr int timeout: The health check timeout in seconds.
    :attr str type: The protocol type of this load balancer pool health monitor.
          The enumerated values for this property are expected to expand in the future.
          When processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the health monitor on which the
          unexpected property value was encountered.
    :attr str url_path: (optional) The health check URL. This is applicable only to
          `http` type of health monitor.
    """

    def __init__(self,
                 delay: int,
                 max_retries: int,
                 timeout: int,
                 type: str,
                 *,
                 port: int = None,
                 url_path: str = None) -> None:
        """
        Initialize a LoadBalancerPoolHealthMonitorPrototype object.

        :param int delay: The health check interval in seconds. Interval must be
               greater than timeout value.
        :param int max_retries: The health check max retries.
        :param int timeout: The health check timeout in seconds.
        :param str type: The protocol type of this load balancer pool health
               monitor.
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the health
               monitor on which the unexpected property value was encountered.
        :param int port: (optional) The health check port number. If specified,
               this overrides the ports specified in the server member resources.
        :param str url_path: (optional) The health check URL. This is applicable
               only to `http` type of health monitor.
        """
        self.delay = delay
        self.max_retries = max_retries
        self.port = port
        self.timeout = timeout
        self.type = type
        self.url_path = url_path

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolHealthMonitorPrototype':
        """Initialize a LoadBalancerPoolHealthMonitorPrototype object from a json dictionary."""
        args = {}
        if 'delay' in _dict:
            args['delay'] = _dict.get('delay')
        else:
            raise ValueError(
                'Required property \'delay\' not present in LoadBalancerPoolHealthMonitorPrototype JSON'
            )
        if 'max_retries' in _dict:
            args['max_retries'] = _dict.get('max_retries')
        else:
            raise ValueError(
                'Required property \'max_retries\' not present in LoadBalancerPoolHealthMonitorPrototype JSON'
            )
        if 'port' in _dict:
            args['port'] = _dict.get('port')
        if 'timeout' in _dict:
            args['timeout'] = _dict.get('timeout')
        else:
            raise ValueError(
                'Required property \'timeout\' not present in LoadBalancerPoolHealthMonitorPrototype JSON'
            )
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LoadBalancerPoolHealthMonitorPrototype JSON'
            )
        if 'url_path' in _dict:
            args['url_path'] = _dict.get('url_path')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolHealthMonitorPrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'delay') and self.delay is not None:
            _dict['delay'] = self.delay
        if hasattr(self, 'max_retries') and self.max_retries is not None:
            _dict['max_retries'] = self.max_retries
        if hasattr(self, 'port') and self.port is not None:
            _dict['port'] = self.port
        if hasattr(self, 'timeout') and self.timeout is not None:
            _dict['timeout'] = self.timeout
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'url_path') and self.url_path is not None:
            _dict['url_path'] = self.url_path
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolHealthMonitorPrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolHealthMonitorPrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolHealthMonitorPrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class TypeEnum(str, Enum):
        """
        The protocol type of this load balancer pool health monitor.
        The enumerated values for this property are expected to expand in the future. When
        processing this property, check for and log unknown values. Optionally halt
        processing and surface the error, or bypass the health monitor on which the
        unexpected property value was encountered.
        """
        HTTP = 'http'
        TCP = 'tcp'
        HTTPS = 'https'


class LoadBalancerPoolIdentity():
    """
    Identifies a load balancer pool by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a LoadBalancerPoolIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'LoadBalancerPoolIdentityById', 'LoadBalancerPoolIdentityByHref'
            ]))
        raise Exception(msg)


class LoadBalancerPoolIdentityByName():
    """
    LoadBalancerPoolIdentityByName.

    :attr str name: The user-defined name for this load balancer pool.
    """

    def __init__(self, name: str) -> None:
        """
        Initialize a LoadBalancerPoolIdentityByName object.

        :param str name: The user-defined name for this load balancer pool.
        """
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolIdentityByName':
        """Initialize a LoadBalancerPoolIdentityByName object from a json dictionary."""
        args = {}
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in LoadBalancerPoolIdentityByName JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolIdentityByName object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolIdentityByName object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolIdentityByName') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolIdentityByName') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerPoolMember():
    """
    LoadBalancerPoolMember.

    :attr datetime created_at: The date and time that this member was created.
    :attr str health: Health of the server member in the pool.
    :attr str href: The member's canonical URL.
    :attr str id: The unique identifier for this load balancer pool member.
    :attr int port: The port number of the application running in the server member.
    :attr str provisioning_status: The provisioning status of this member.
    :attr LoadBalancerPoolMemberTarget target: The pool member target.
    :attr int weight: (optional) Weight of the server member. This takes effect only
          when the load balancing algorithm of its belonging pool is
          `weighted_round_robin`.
    """

    def __init__(self,
                 created_at: datetime,
                 health: str,
                 href: str,
                 id: str,
                 port: int,
                 provisioning_status: str,
                 target: 'LoadBalancerPoolMemberTarget',
                 *,
                 weight: int = None) -> None:
        """
        Initialize a LoadBalancerPoolMember object.

        :param datetime created_at: The date and time that this member was created.
        :param str health: Health of the server member in the pool.
        :param str href: The member's canonical URL.
        :param str id: The unique identifier for this load balancer pool member.
        :param int port: The port number of the application running in the server
               member.
        :param str provisioning_status: The provisioning status of this member.
        :param LoadBalancerPoolMemberTarget target: The pool member target.
        :param int weight: (optional) Weight of the server member. This takes
               effect only when the load balancing algorithm of its belonging pool is
               `weighted_round_robin`.
        """
        self.created_at = created_at
        self.health = health
        self.href = href
        self.id = id
        self.port = port
        self.provisioning_status = provisioning_status
        self.target = target
        self.weight = weight

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolMember':
        """Initialize a LoadBalancerPoolMember object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in LoadBalancerPoolMember JSON'
            )
        if 'health' in _dict:
            args['health'] = _dict.get('health')
        else:
            raise ValueError(
                'Required property \'health\' not present in LoadBalancerPoolMember JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancerPoolMember JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancerPoolMember JSON'
            )
        if 'port' in _dict:
            args['port'] = _dict.get('port')
        else:
            raise ValueError(
                'Required property \'port\' not present in LoadBalancerPoolMember JSON'
            )
        if 'provisioning_status' in _dict:
            args['provisioning_status'] = _dict.get('provisioning_status')
        else:
            raise ValueError(
                'Required property \'provisioning_status\' not present in LoadBalancerPoolMember JSON'
            )
        if 'target' in _dict:
            args['target'] = _dict.get('target')
        else:
            raise ValueError(
                'Required property \'target\' not present in LoadBalancerPoolMember JSON'
            )
        if 'weight' in _dict:
            args['weight'] = _dict.get('weight')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolMember object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'health') and self.health is not None:
            _dict['health'] = self.health
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'port') and self.port is not None:
            _dict['port'] = self.port
        if hasattr(
                self,
                'provisioning_status') and self.provisioning_status is not None:
            _dict['provisioning_status'] = self.provisioning_status
        if hasattr(self, 'target') and self.target is not None:
            _dict['target'] = self.target
        if hasattr(self, 'weight') and self.weight is not None:
            _dict['weight'] = self.weight
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolMember object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolMember') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolMember') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class HealthEnum(str, Enum):
        """
        Health of the server member in the pool.
        """
        FAULTED = 'faulted'
        OK = 'ok'
        UNKNOWN = 'unknown'

    class ProvisioningStatusEnum(str, Enum):
        """
        The provisioning status of this member.
        """
        ACTIVE = 'active'
        CREATE_PENDING = 'create_pending'
        DELETE_PENDING = 'delete_pending'
        FAILED = 'failed'
        MAINTENANCE_PENDING = 'maintenance_pending'
        UPDATE_PENDING = 'update_pending'


class LoadBalancerPoolMemberCollection():
    """
    LoadBalancerPoolMemberCollection.

    :attr List[LoadBalancerPoolMember] members: Collection of members.
    """

    def __init__(self, members: List['LoadBalancerPoolMember']) -> None:
        """
        Initialize a LoadBalancerPoolMemberCollection object.

        :param List[LoadBalancerPoolMember] members: Collection of members.
        """
        self.members = members

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolMemberCollection':
        """Initialize a LoadBalancerPoolMemberCollection object from a json dictionary."""
        args = {}
        if 'members' in _dict:
            args['members'] = [
                LoadBalancerPoolMember.from_dict(x)
                for x in _dict.get('members')
            ]
        else:
            raise ValueError(
                'Required property \'members\' not present in LoadBalancerPoolMemberCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolMemberCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'members') and self.members is not None:
            _dict['members'] = [x.to_dict() for x in self.members]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolMemberCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolMemberCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolMemberCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerPoolMemberPrototype():
    """
    LoadBalancerPoolMemberPrototype.

    :attr int port: The port number of the application running in the server member.
    :attr LoadBalancerPoolMemberTargetPrototype target: The pool member target.
    :attr int weight: (optional) Weight of the server member. This takes effect only
          when the load balancing algorithm of its belonging pool is
          `weighted_round_robin`.
    """

    def __init__(self,
                 port: int,
                 target: 'LoadBalancerPoolMemberTargetPrototype',
                 *,
                 weight: int = None) -> None:
        """
        Initialize a LoadBalancerPoolMemberPrototype object.

        :param int port: The port number of the application running in the server
               member.
        :param LoadBalancerPoolMemberTargetPrototype target: The pool member
               target.
        :param int weight: (optional) Weight of the server member. This takes
               effect only when the load balancing algorithm of its belonging pool is
               `weighted_round_robin`.
        """
        self.port = port
        self.target = target
        self.weight = weight

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolMemberPrototype':
        """Initialize a LoadBalancerPoolMemberPrototype object from a json dictionary."""
        args = {}
        if 'port' in _dict:
            args['port'] = _dict.get('port')
        else:
            raise ValueError(
                'Required property \'port\' not present in LoadBalancerPoolMemberPrototype JSON'
            )
        if 'target' in _dict:
            args['target'] = _dict.get('target')
        else:
            raise ValueError(
                'Required property \'target\' not present in LoadBalancerPoolMemberPrototype JSON'
            )
        if 'weight' in _dict:
            args['weight'] = _dict.get('weight')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolMemberPrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'port') and self.port is not None:
            _dict['port'] = self.port
        if hasattr(self, 'target') and self.target is not None:
            _dict['target'] = self.target
        if hasattr(self, 'weight') and self.weight is not None:
            _dict['weight'] = self.weight
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolMemberPrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolMemberPrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolMemberPrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerPoolMemberReference():
    """
    LoadBalancerPoolMemberReference.

    :attr str href: The member's canonical URL.
    :attr str id: The unique identifier for this load balancer pool member.
    """

    def __init__(self, href: str, id: str) -> None:
        """
        Initialize a LoadBalancerPoolMemberReference object.

        :param str href: The member's canonical URL.
        :param str id: The unique identifier for this load balancer pool member.
        """
        self.href = href
        self.id = id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolMemberReference':
        """Initialize a LoadBalancerPoolMemberReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancerPoolMemberReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancerPoolMemberReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolMemberReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolMemberReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolMemberReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolMemberReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerPoolMemberTarget():
    """
    The pool member target.

    """

    def __init__(self) -> None:
        """
        Initialize a LoadBalancerPoolMemberTarget object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['LoadBalancerPoolMemberTargetIP']))
        raise Exception(msg)


class LoadBalancerPoolMemberTargetPrototype():
    """
    The pool member target.

    """

    def __init__(self) -> None:
        """
        Initialize a LoadBalancerPoolMemberTargetPrototype object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['LoadBalancerPoolMemberTargetPrototypeIP']))
        raise Exception(msg)


class LoadBalancerPoolPrototype():
    """
    LoadBalancerPoolPrototype.

    :attr str algorithm: The load balancing algorithm.
    :attr LoadBalancerPoolHealthMonitorPrototype health_monitor: The health monitor
          of this pool.
    :attr List[LoadBalancerPoolMemberPrototype] members: (optional) The members for
          this load balancer pool. For load balancers in the `network` family, the same
          `port` and `target` tuple cannot be shared by a member of any other load
          balancer.
    :attr str name: (optional) The user-defined name for this load balancer pool. If
          unspecified, the name will be a hyphenated list of randomly-selected words.
    :attr str protocol: The protocol used for this load balancer pool.
          The enumerated values for this property are expected to expand in the future.
          When processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the pool on which the unexpected
          property value was encountered.
    :attr LoadBalancerPoolSessionPersistencePrototype session_persistence:
          (optional) The session persistence of this pool.
    """

    def __init__(
        self,
        algorithm: str,
        health_monitor: 'LoadBalancerPoolHealthMonitorPrototype',
        protocol: str,
        *,
        members: List['LoadBalancerPoolMemberPrototype'] = None,
        name: str = None,
        session_persistence: 'LoadBalancerPoolSessionPersistencePrototype' = None
    ) -> None:
        """
        Initialize a LoadBalancerPoolPrototype object.

        :param str algorithm: The load balancing algorithm.
        :param LoadBalancerPoolHealthMonitorPrototype health_monitor: The health
               monitor of this pool.
        :param str protocol: The protocol used for this load balancer pool.
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the pool on
               which the unexpected property value was encountered.
        :param List[LoadBalancerPoolMemberPrototype] members: (optional) The
               members for this load balancer pool. For load balancers in the `network`
               family, the same `port` and `target` tuple cannot be shared by a member of
               any other load balancer.
        :param str name: (optional) The user-defined name for this load balancer
               pool. If unspecified, the name will be a hyphenated list of
               randomly-selected words.
        :param LoadBalancerPoolSessionPersistencePrototype session_persistence:
               (optional) The session persistence of this pool.
        """
        self.algorithm = algorithm
        self.health_monitor = health_monitor
        self.members = members
        self.name = name
        self.protocol = protocol
        self.session_persistence = session_persistence

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolPrototype':
        """Initialize a LoadBalancerPoolPrototype object from a json dictionary."""
        args = {}
        if 'algorithm' in _dict:
            args['algorithm'] = _dict.get('algorithm')
        else:
            raise ValueError(
                'Required property \'algorithm\' not present in LoadBalancerPoolPrototype JSON'
            )
        if 'health_monitor' in _dict:
            args[
                'health_monitor'] = LoadBalancerPoolHealthMonitorPrototype.from_dict(
                    _dict.get('health_monitor'))
        else:
            raise ValueError(
                'Required property \'health_monitor\' not present in LoadBalancerPoolPrototype JSON'
            )
        if 'members' in _dict:
            args['members'] = [
                LoadBalancerPoolMemberPrototype.from_dict(x)
                for x in _dict.get('members')
            ]
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        if 'protocol' in _dict:
            args['protocol'] = _dict.get('protocol')
        else:
            raise ValueError(
                'Required property \'protocol\' not present in LoadBalancerPoolPrototype JSON'
            )
        if 'session_persistence' in _dict:
            args[
                'session_persistence'] = LoadBalancerPoolSessionPersistencePrototype.from_dict(
                    _dict.get('session_persistence'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolPrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'algorithm') and self.algorithm is not None:
            _dict['algorithm'] = self.algorithm
        if hasattr(self, 'health_monitor') and self.health_monitor is not None:
            _dict['health_monitor'] = self.health_monitor.to_dict()
        if hasattr(self, 'members') and self.members is not None:
            _dict['members'] = [x.to_dict() for x in self.members]
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'protocol') and self.protocol is not None:
            _dict['protocol'] = self.protocol
        if hasattr(
                self,
                'session_persistence') and self.session_persistence is not None:
            _dict['session_persistence'] = self.session_persistence.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolPrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolPrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolPrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class AlgorithmEnum(str, Enum):
        """
        The load balancing algorithm.
        """
        LEAST_CONNECTIONS = 'least_connections'
        ROUND_ROBIN = 'round_robin'
        WEIGHTED_ROUND_ROBIN = 'weighted_round_robin'

    class ProtocolEnum(str, Enum):
        """
        The protocol used for this load balancer pool.
        The enumerated values for this property are expected to expand in the future. When
        processing this property, check for and log unknown values. Optionally halt
        processing and surface the error, or bypass the pool on which the unexpected
        property value was encountered.
        """
        HTTP = 'http'
        TCP = 'tcp'
        HTTPS = 'https'


class LoadBalancerPoolReference():
    """
    LoadBalancerPoolReference.

    :attr str href: The pool's canonical URL.
    :attr str id: The unique identifier for this load balancer pool.
    :attr str name: The user-defined name for this load balancer pool.
    """

    def __init__(self, href: str, id: str, name: str) -> None:
        """
        Initialize a LoadBalancerPoolReference object.

        :param str href: The pool's canonical URL.
        :param str id: The unique identifier for this load balancer pool.
        :param str name: The user-defined name for this load balancer pool.
        """
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolReference':
        """Initialize a LoadBalancerPoolReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in LoadBalancerPoolReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in LoadBalancerPoolReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in LoadBalancerPoolReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LoadBalancerPoolSessionPersistence():
    """
    LoadBalancerPoolSessionPersistence.

    :attr str type: The session persistence type.
    """

    def __init__(self, type: str) -> None:
        """
        Initialize a LoadBalancerPoolSessionPersistence object.

        :param str type: The session persistence type.
        """
        self.type = type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerPoolSessionPersistence':
        """Initialize a LoadBalancerPoolSessionPersistence object from a json dictionary."""
        args = {}
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LoadBalancerPoolSessionPersistence JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolSessionPersistence object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolSessionPersistence object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolSessionPersistence') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolSessionPersistence') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class TypeEnum(str, Enum):
        """
        The session persistence type.
        """
        SOURCE_IP = 'source_ip'


class LoadBalancerPoolSessionPersistencePatch():
    """
    LoadBalancerPoolSessionPersistencePatch.

    :attr str type: The session persistence type.
    """

    def __init__(self, type: str) -> None:
        """
        Initialize a LoadBalancerPoolSessionPersistencePatch object.

        :param str type: The session persistence type.
        """
        self.type = type

    @classmethod
    def from_dict(cls,
                  _dict: Dict) -> 'LoadBalancerPoolSessionPersistencePatch':
        """Initialize a LoadBalancerPoolSessionPersistencePatch object from a json dictionary."""
        args = {}
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LoadBalancerPoolSessionPersistencePatch JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolSessionPersistencePatch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolSessionPersistencePatch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerPoolSessionPersistencePatch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerPoolSessionPersistencePatch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class TypeEnum(str, Enum):
        """
        The session persistence type.
        """
        SOURCE_IP = 'source_ip'


class LoadBalancerPoolSessionPersistencePrototype():
    """
    LoadBalancerPoolSessionPersistencePrototype.

    :attr str type: The session persistence type.
    """

    def __init__(self, type: str) -> None:
        """
        Initialize a LoadBalancerPoolSessionPersistencePrototype object.

        :param str type: The session persistence type.
        """
        self.type = type

    @classmethod
    def from_dict(cls,
                  _dict: Dict) -> 'LoadBalancerPoolSessionPersistencePrototype':
        """Initialize a LoadBalancerPoolSessionPersistencePrototype object from a json dictionary."""
        args = {}
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LoadBalancerPoolSessionPersistencePrototype JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerPoolSessionPersistencePrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerPoolSessionPersistencePrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self,
               other: 'LoadBalancerPoolSessionPersistencePrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self,
               other: 'LoadBalancerPoolSessionPersistencePrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class TypeEnum(str, Enum):
        """
        The session persistence type.
        """
        SOURCE_IP = 'source_ip'


class LoadBalancerStatistics():
    """
    LoadBalancerStatistics.

    :attr int active_connections: Number of active connections of this load
          balancer.
    :attr float connection_rate: Current connection rate (connections per second) of
          this load balancer.
    :attr int data_processed_this_month: Total number of data processed (bytes) of
          this load balancer within current calendar month.
    :attr float throughput: Current throughput (Mbps) of this load balancer.
    """

    def __init__(self, active_connections: int, connection_rate: float,
                 data_processed_this_month: int, throughput: float) -> None:
        """
        Initialize a LoadBalancerStatistics object.

        :param int active_connections: Number of active connections of this load
               balancer.
        :param float connection_rate: Current connection rate (connections per
               second) of this load balancer.
        :param int data_processed_this_month: Total number of data processed
               (bytes) of this load balancer within current calendar month.
        :param float throughput: Current throughput (Mbps) of this load balancer.
        """
        self.active_connections = active_connections
        self.connection_rate = connection_rate
        self.data_processed_this_month = data_processed_this_month
        self.throughput = throughput

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LoadBalancerStatistics':
        """Initialize a LoadBalancerStatistics object from a json dictionary."""
        args = {}
        if 'active_connections' in _dict:
            args['active_connections'] = _dict.get('active_connections')
        else:
            raise ValueError(
                'Required property \'active_connections\' not present in LoadBalancerStatistics JSON'
            )
        if 'connection_rate' in _dict:
            args['connection_rate'] = _dict.get('connection_rate')
        else:
            raise ValueError(
                'Required property \'connection_rate\' not present in LoadBalancerStatistics JSON'
            )
        if 'data_processed_this_month' in _dict:
            args['data_processed_this_month'] = _dict.get(
                'data_processed_this_month')
        else:
            raise ValueError(
                'Required property \'data_processed_this_month\' not present in LoadBalancerStatistics JSON'
            )
        if 'throughput' in _dict:
            args['throughput'] = _dict.get('throughput')
        else:
            raise ValueError(
                'Required property \'throughput\' not present in LoadBalancerStatistics JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LoadBalancerStatistics object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(
                self,
                'active_connections') and self.active_connections is not None:
            _dict['active_connections'] = self.active_connections
        if hasattr(self,
                   'connection_rate') and self.connection_rate is not None:
            _dict['connection_rate'] = self.connection_rate
        if hasattr(self, 'data_processed_this_month'
                  ) and self.data_processed_this_month is not None:
            _dict['data_processed_this_month'] = self.data_processed_this_month
        if hasattr(self, 'throughput') and self.throughput is not None:
            _dict['throughput'] = self.throughput
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LoadBalancerStatistics object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LoadBalancerStatistics') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LoadBalancerStatistics') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkACL():
    """
    NetworkACL.

    :attr datetime created_at: The date and time that the network ACL was created.
    :attr str href: The URL for this network ACL.
    :attr str id: The unique identifier for this network ACL.
    :attr str name: The user-defined name for this network ACL.
    :attr List[NetworkACLRuleItem] rules: The ordered rules for this network ACL. If
          no rules exist, all traffic will be allowed.
    :attr List[SubnetReference] subnets: The subnets to which this network ACL is
          attached.
    """

    def __init__(self, created_at: datetime, href: str, id: str, name: str,
                 rules: List['NetworkACLRuleItem'],
                 subnets: List['SubnetReference']) -> None:
        """
        Initialize a NetworkACL object.

        :param datetime created_at: The date and time that the network ACL was
               created.
        :param str href: The URL for this network ACL.
        :param str id: The unique identifier for this network ACL.
        :param str name: The user-defined name for this network ACL.
        :param List[NetworkACLRuleItem] rules: The ordered rules for this network
               ACL. If no rules exist, all traffic will be allowed.
        :param List[SubnetReference] subnets: The subnets to which this network ACL
               is attached.
        """
        self.created_at = created_at
        self.href = href
        self.id = id
        self.name = name
        self.rules = rules
        self.subnets = subnets

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACL':
        """Initialize a NetworkACL object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in NetworkACL JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in NetworkACL JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in NetworkACL JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in NetworkACL JSON')
        if 'rules' in _dict:
            args['rules'] = [
                NetworkACLRuleItem.from_dict(x) for x in _dict.get('rules')
            ]
        else:
            raise ValueError(
                'Required property \'rules\' not present in NetworkACL JSON')
        if 'subnets' in _dict:
            args['subnets'] = [
                SubnetReference.from_dict(x) for x in _dict.get('subnets')
            ]
        else:
            raise ValueError(
                'Required property \'subnets\' not present in NetworkACL JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkACL object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'rules') and self.rules is not None:
            _dict['rules'] = [x.to_dict() for x in self.rules]
        if hasattr(self, 'subnets') and self.subnets is not None:
            _dict['subnets'] = [x.to_dict() for x in self.subnets]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkACL object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkACL') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkACL') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkACLCollection():
    """
    NetworkACLCollection.

    :attr NetworkACLCollectionFirst first: A reference to the first page of
          resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr List[NetworkACL] network_acls: Collection of network ACLs.
    :attr NetworkACLCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    """

    def __init__(self,
                 first: 'NetworkACLCollectionFirst',
                 limit: int,
                 network_acls: List['NetworkACL'],
                 *,
                 next: 'NetworkACLCollectionNext' = None) -> None:
        """
        Initialize a NetworkACLCollection object.

        :param NetworkACLCollectionFirst first: A reference to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param List[NetworkACL] network_acls: Collection of network ACLs.
        :param NetworkACLCollectionNext next: (optional) A reference to the next
               page of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.network_acls = network_acls
        self.next = next

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLCollection':
        """Initialize a NetworkACLCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = NetworkACLCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in NetworkACLCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in NetworkACLCollection JSON'
            )
        if 'network_acls' in _dict:
            args['network_acls'] = [
                NetworkACL.from_dict(x) for x in _dict.get('network_acls')
            ]
        else:
            raise ValueError(
                'Required property \'network_acls\' not present in NetworkACLCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = NetworkACLCollectionNext.from_dict(_dict.get('next'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkACLCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'network_acls') and self.network_acls is not None:
            _dict['network_acls'] = [x.to_dict() for x in self.network_acls]
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkACLCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkACLCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkACLCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkACLCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a NetworkACLCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLCollectionFirst':
        """Initialize a NetworkACLCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in NetworkACLCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkACLCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkACLCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkACLCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkACLCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkACLCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a NetworkACLCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLCollectionNext':
        """Initialize a NetworkACLCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in NetworkACLCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkACLCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkACLCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkACLCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkACLCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkACLIdentity():
    """
    Identifies a network ACL by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a NetworkACLIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['NetworkACLIdentityById', 'NetworkACLIdentityByHref']))
        raise Exception(msg)


class NetworkACLPrototype():
    """
    NetworkACLPrototype.

    :attr str name: (optional) The unique user-defined name for this network ACL. If
          unspecified, the name will be a hyphenated list of randomly-selected words.
    """

    def __init__(self, *, name: str = None) -> None:
        """
        Initialize a NetworkACLPrototype object.

        :param str name: (optional) The unique user-defined name for this network
               ACL. If unspecified, the name will be a hyphenated list of
               randomly-selected words.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'NetworkACLPrototypeNetworkACLByRules',
                'NetworkACLPrototypeNetworkACLBySourceNetworkACL'
            ]))
        raise Exception(msg)


class NetworkACLReference():
    """
    NetworkACLReference.

    :attr str href: The URL for this network ACL.
    :attr str id: The unique identifier for this network ACL.
    :attr str name: The user-defined name for this network ACL.
    """

    def __init__(self, href: str, id: str, name: str) -> None:
        """
        Initialize a NetworkACLReference object.

        :param str href: The URL for this network ACL.
        :param str id: The unique identifier for this network ACL.
        :param str name: The user-defined name for this network ACL.
        """
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLReference':
        """Initialize a NetworkACLReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in NetworkACLReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in NetworkACLReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in NetworkACLReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkACLReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkACLReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkACLReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkACLReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkACLRule():
    """
    NetworkACLRule.

    :attr str action: Whether to allow or deny matching traffic.
    :attr NetworkACLRuleReference before: (optional) The rule that this rule is
          immediately before. If absent, this is the last rule.
    :attr datetime created_at: The date and time that the rule was created.
    :attr str destination: The destination CIDR block. The CIDR block `0.0.0.0/0`
          applies to all addresses.
    :attr str direction: Whether the traffic to be matched is `inbound` or
          `outbound`.
    :attr str href: The URL for this Network ACL rule.
    :attr str id: The unique identifier for this Network ACL rule.
    :attr str ip_version: The IP version for this rule.
    :attr str name: The user-defined name for this rule. Names must be unique within
          the network ACL the rule resides in. If unspecified, the name will be a
          hyphenated list of randomly-selected words.
    :attr str protocol: (optional) The protocol to enforce.
    :attr str source: The source CIDR block. The CIDR block `0.0.0.0/0` applies to
          all addresses.
    """

    def __init__(self,
                 action: str,
                 created_at: datetime,
                 destination: str,
                 direction: str,
                 href: str,
                 id: str,
                 ip_version: str,
                 name: str,
                 source: str,
                 *,
                 before: 'NetworkACLRuleReference' = None,
                 protocol: str = None) -> None:
        """
        Initialize a NetworkACLRule object.

        :param str action: Whether to allow or deny matching traffic.
        :param datetime created_at: The date and time that the rule was created.
        :param str destination: The destination CIDR block. The CIDR block
               `0.0.0.0/0` applies to all addresses.
        :param str direction: Whether the traffic to be matched is `inbound` or
               `outbound`.
        :param str href: The URL for this Network ACL rule.
        :param str id: The unique identifier for this Network ACL rule.
        :param str ip_version: The IP version for this rule.
        :param str name: The user-defined name for this rule. Names must be unique
               within the network ACL the rule resides in. If unspecified, the name will
               be a hyphenated list of randomly-selected words.
        :param str source: The source CIDR block. The CIDR block `0.0.0.0/0`
               applies to all addresses.
        :param NetworkACLRuleReference before: (optional) The rule that this rule
               is immediately before. If absent, this is the last rule.
        :param str protocol: (optional) The protocol to enforce.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'NetworkACLRuleProtocolTCPUDP', 'NetworkACLRuleProtocolICMP',
                'NetworkACLRuleProtocolAll'
            ]))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLRule':
        """Initialize a NetworkACLRule object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = (
            "Cannot convert dictionary into an instance of base class 'NetworkACLRule'. "
            + "The discriminator value should map to a valid subclass: {1}"
        ).format(", ".join([
            'NetworkACLRuleProtocolTCPUDP', 'NetworkACLRuleProtocolICMP',
            'NetworkACLRuleProtocolAll'
        ]))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a NetworkACLRule object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping['all'] = 'NetworkACLRuleProtocolAll'
        mapping['icmp'] = 'NetworkACLRuleProtocolICMP'
        mapping['tcp'] = 'NetworkACLRuleProtocolTCPUDP'
        mapping['udp'] = 'NetworkACLRuleProtocolTCPUDP'
        disc_value = _dict.get('protocol')
        if disc_value is None:
            raise ValueError(
                'Discriminator property \'protocol\' not found in NetworkACLRule JSON'
            )
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)

    class ActionEnum(str, Enum):
        """
        Whether to allow or deny matching traffic.
        """
        ALLOW = 'allow'
        DENY = 'deny'

    class DirectionEnum(str, Enum):
        """
        Whether the traffic to be matched is `inbound` or `outbound`.
        """
        INBOUND = 'inbound'
        OUTBOUND = 'outbound'

    class IpVersionEnum(str, Enum):
        """
        The IP version for this rule.
        """
        IPV4 = 'ipv4'
        IPV6 = 'ipv6'

    class ProtocolEnum(str, Enum):
        """
        The protocol to enforce.
        """
        ALL = 'all'
        ICMP = 'icmp'
        TCP = 'tcp'
        UDP = 'udp'


class NetworkACLRuleCollection():
    """
    NetworkACLRuleCollection.

    :attr NetworkACLRuleCollectionFirst first: A reference to the first page of
          resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr NetworkACLRuleCollectionNext next: (optional) A reference to the next page
          of resources; this reference is included for all pages
          except the last page.
    :attr List[NetworkACLRuleItem] rules: Ordered collection of Network ACL rules.
    """

    def __init__(self,
                 first: 'NetworkACLRuleCollectionFirst',
                 limit: int,
                 rules: List['NetworkACLRuleItem'],
                 *,
                 next: 'NetworkACLRuleCollectionNext' = None) -> None:
        """
        Initialize a NetworkACLRuleCollection object.

        :param NetworkACLRuleCollectionFirst first: A reference to the first page
               of resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param List[NetworkACLRuleItem] rules: Ordered collection of Network ACL
               rules.
        :param NetworkACLRuleCollectionNext next: (optional) A reference to the
               next page of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.next = next
        self.rules = rules

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLRuleCollection':
        """Initialize a NetworkACLRuleCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = NetworkACLRuleCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in NetworkACLRuleCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in NetworkACLRuleCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = NetworkACLRuleCollectionNext.from_dict(
                _dict.get('next'))
        if 'rules' in _dict:
            args['rules'] = [
                NetworkACLRuleItem.from_dict(x) for x in _dict.get('rules')
            ]
        else:
            raise ValueError(
                'Required property \'rules\' not present in NetworkACLRuleCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkACLRuleCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'rules') and self.rules is not None:
            _dict['rules'] = [x.to_dict() for x in self.rules]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkACLRuleCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkACLRuleCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkACLRuleCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkACLRuleCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a NetworkACLRuleCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLRuleCollectionFirst':
        """Initialize a NetworkACLRuleCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in NetworkACLRuleCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkACLRuleCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkACLRuleCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkACLRuleCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkACLRuleCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkACLRuleCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a NetworkACLRuleCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLRuleCollectionNext':
        """Initialize a NetworkACLRuleCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in NetworkACLRuleCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkACLRuleCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkACLRuleCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkACLRuleCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkACLRuleCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkACLRuleItem():
    """
    NetworkACLRuleItem.

    :attr str action: Whether to allow or deny matching traffic.
    :attr NetworkACLRuleReference before: (optional) The rule that this rule is
          immediately before. In a rule collection, this always
          refers to the next item in the collection. If absent, this is the last rule.
    :attr datetime created_at: The date and time that the rule was created.
    :attr str destination: The destination CIDR block. The CIDR block `0.0.0.0/0`
          applies to all addresses.
    :attr str direction: Whether the traffic to be matched is `inbound` or
          `outbound`.
    :attr str href: The URL for this Network ACL rule.
    :attr str id: The unique identifier for this Network ACL rule.
    :attr str ip_version: The IP version for this rule.
    :attr str name: The user-defined name for this rule. Names must be unique within
          the network ACL the rule resides in. If unspecified, the name will be a
          hyphenated list of randomly-selected words.
    :attr str protocol: (optional) The protocol to enforce.
    :attr str source: The source CIDR block. The CIDR block `0.0.0.0/0` applies to
          all addresses.
    """

    def __init__(self,
                 action: str,
                 created_at: datetime,
                 destination: str,
                 direction: str,
                 href: str,
                 id: str,
                 ip_version: str,
                 name: str,
                 source: str,
                 *,
                 before: 'NetworkACLRuleReference' = None,
                 protocol: str = None) -> None:
        """
        Initialize a NetworkACLRuleItem object.

        :param str action: Whether to allow or deny matching traffic.
        :param datetime created_at: The date and time that the rule was created.
        :param str destination: The destination CIDR block. The CIDR block
               `0.0.0.0/0` applies to all addresses.
        :param str direction: Whether the traffic to be matched is `inbound` or
               `outbound`.
        :param str href: The URL for this Network ACL rule.
        :param str id: The unique identifier for this Network ACL rule.
        :param str ip_version: The IP version for this rule.
        :param str name: The user-defined name for this rule. Names must be unique
               within the network ACL the rule resides in. If unspecified, the name will
               be a hyphenated list of randomly-selected words.
        :param str source: The source CIDR block. The CIDR block `0.0.0.0/0`
               applies to all addresses.
        :param NetworkACLRuleReference before: (optional) The rule that this rule
               is immediately before. In a rule collection, this always
               refers to the next item in the collection. If absent, this is the last
               rule.
        :param str protocol: (optional) The protocol to enforce.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'NetworkACLRuleItemNetworkACLRuleProtocolTCPUDP',
                'NetworkACLRuleItemNetworkACLRuleProtocolICMP',
                'NetworkACLRuleItemNetworkACLRuleProtocolAll'
            ]))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLRuleItem':
        """Initialize a NetworkACLRuleItem object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = (
            "Cannot convert dictionary into an instance of base class 'NetworkACLRuleItem'. "
            + "The discriminator value should map to a valid subclass: {1}"
        ).format(", ".join([
            'NetworkACLRuleItemNetworkACLRuleProtocolTCPUDP',
            'NetworkACLRuleItemNetworkACLRuleProtocolICMP',
            'NetworkACLRuleItemNetworkACLRuleProtocolAll'
        ]))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a NetworkACLRuleItem object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping['all'] = 'NetworkACLRuleItemNetworkACLRuleProtocolAll'
        mapping['icmp'] = 'NetworkACLRuleItemNetworkACLRuleProtocolICMP'
        mapping['tcp'] = 'NetworkACLRuleItemNetworkACLRuleProtocolTCPUDP'
        mapping['udp'] = 'NetworkACLRuleItemNetworkACLRuleProtocolTCPUDP'
        disc_value = _dict.get('protocol')
        if disc_value is None:
            raise ValueError(
                'Discriminator property \'protocol\' not found in NetworkACLRuleItem JSON'
            )
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)

    class ActionEnum(str, Enum):
        """
        Whether to allow or deny matching traffic.
        """
        ALLOW = 'allow'
        DENY = 'deny'

    class DirectionEnum(str, Enum):
        """
        Whether the traffic to be matched is `inbound` or `outbound`.
        """
        INBOUND = 'inbound'
        OUTBOUND = 'outbound'

    class IpVersionEnum(str, Enum):
        """
        The IP version for this rule.
        """
        IPV4 = 'ipv4'
        IPV6 = 'ipv6'

    class ProtocolEnum(str, Enum):
        """
        The protocol to enforce.
        """
        ALL = 'all'
        ICMP = 'icmp'
        TCP = 'tcp'
        UDP = 'udp'


class NetworkACLRulePatch():
    """
    NetworkACLRulePatch.

    :attr str action: (optional) Whether to allow or deny matching traffic.
    :attr NetworkACLRulePatchBefore before: (optional) The rule to move this rule
          immediately before. Specify `null` to move this rule after
          all existing rules.
    :attr str destination: (optional) The destination IP address or CIDR block. The
          CIDR block `0.0.0.0/0` applies to all addresses.
    :attr str direction: (optional) Whether the traffic to be matched is `inbound`
          or `outbound`.
    :attr str name: (optional) The user-defined name for this rule. Names must be
          unique within the network ACL the rule resides in.
    :attr str protocol: (optional) The protocol to enforce.
    :attr str source: (optional) The source IP address or CIDR block.  The CIDR
          block `0.0.0.0/0` applies to all addresses.
    """

    def __init__(self,
                 *,
                 action: str = None,
                 before: 'NetworkACLRulePatchBefore' = None,
                 destination: str = None,
                 direction: str = None,
                 name: str = None,
                 protocol: str = None,
                 source: str = None) -> None:
        """
        Initialize a NetworkACLRulePatch object.

        :param str action: (optional) Whether to allow or deny matching traffic.
        :param NetworkACLRulePatchBefore before: (optional) The rule to move this
               rule immediately before. Specify `null` to move this rule after
               all existing rules.
        :param str destination: (optional) The destination IP address or CIDR
               block. The CIDR block `0.0.0.0/0` applies to all addresses.
        :param str direction: (optional) Whether the traffic to be matched is
               `inbound` or `outbound`.
        :param str name: (optional) The user-defined name for this rule. Names must
               be unique within the network ACL the rule resides in.
        :param str protocol: (optional) The protocol to enforce.
        :param str source: (optional) The source IP address or CIDR block.  The
               CIDR block `0.0.0.0/0` applies to all addresses.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'NetworkACLRulePatchNetworkACLRuleProtocolTCPUDP',
                'NetworkACLRulePatchNetworkACLRuleProtocolICMP',
                'NetworkACLRulePatchNetworkACLRuleProtocolAll'
            ]))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLRulePatch':
        """Initialize a NetworkACLRulePatch object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = (
            "Cannot convert dictionary into an instance of base class 'NetworkACLRulePatch'. "
            + "The discriminator value should map to a valid subclass: {1}"
        ).format(", ".join([
            'NetworkACLRulePatchNetworkACLRuleProtocolTCPUDP',
            'NetworkACLRulePatchNetworkACLRuleProtocolICMP',
            'NetworkACLRulePatchNetworkACLRuleProtocolAll'
        ]))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a NetworkACLRulePatch object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping['all'] = 'NetworkACLRulePatchNetworkACLRuleProtocolAll'
        mapping['icmp'] = 'NetworkACLRulePatchNetworkACLRuleProtocolICMP'
        mapping['tcp'] = 'NetworkACLRulePatchNetworkACLRuleProtocolTCPUDP'
        mapping['udp'] = 'NetworkACLRulePatchNetworkACLRuleProtocolTCPUDP'
        disc_value = _dict.get('protocol')
        if disc_value is None:
            raise ValueError(
                'Discriminator property \'protocol\' not found in NetworkACLRulePatch JSON'
            )
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)

    class ActionEnum(str, Enum):
        """
        Whether to allow or deny matching traffic.
        """
        ALLOW = 'allow'
        DENY = 'deny'

    class DirectionEnum(str, Enum):
        """
        Whether the traffic to be matched is `inbound` or `outbound`.
        """
        INBOUND = 'inbound'
        OUTBOUND = 'outbound'

    class ProtocolEnum(str, Enum):
        """
        The protocol to enforce.
        """
        ALL = 'all'
        ICMP = 'icmp'
        TCP = 'tcp'
        UDP = 'udp'


class NetworkACLRulePatchBefore():
    """
    The rule to move this rule immediately before. Specify `null` to move this rule after
    all existing rules.

    """

    def __init__(self) -> None:
        """
        Initialize a NetworkACLRulePatchBefore object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'NetworkACLRulePatchBeforeNetworkACLRuleIdentityById',
                'NetworkACLRulePatchBeforeNetworkACLRuleIdentityByHref'
            ]))
        raise Exception(msg)


class NetworkACLRulePrototype():
    """
    NetworkACLRulePrototype.

    :attr str action: Whether to allow or deny matching traffic.
    :attr NetworkACLRulePrototypeBefore before: (optional) The rule to insert this
          rule immediately before. If omitted, this rule will be
          inserted after all existing rules.
    :attr str destination: The destination IP address or CIDR block. The CIDR block
          `0.0.0.0/0` applies to all addresses.
    :attr str direction: Whether the traffic to be matched is `inbound` or
          `outbound`.
    :attr str name: (optional) The user-defined name for this rule. Names must be
          unique within the network ACL the rule resides in. If unspecified, the name will
          be a hyphenated list of randomly-selected words.
    :attr str protocol: (optional) The protocol to enforce.
    :attr str source: The source IP address or CIDR block.  The CIDR block
          `0.0.0.0/0` applies to all addresses.
    """

    def __init__(self,
                 action: str,
                 destination: str,
                 direction: str,
                 source: str,
                 *,
                 before: 'NetworkACLRulePrototypeBefore' = None,
                 name: str = None,
                 protocol: str = None) -> None:
        """
        Initialize a NetworkACLRulePrototype object.

        :param str action: Whether to allow or deny matching traffic.
        :param str destination: The destination IP address or CIDR block. The CIDR
               block `0.0.0.0/0` applies to all addresses.
        :param str direction: Whether the traffic to be matched is `inbound` or
               `outbound`.
        :param str source: The source IP address or CIDR block.  The CIDR block
               `0.0.0.0/0` applies to all addresses.
        :param NetworkACLRulePrototypeBefore before: (optional) The rule to insert
               this rule immediately before. If omitted, this rule will be
               inserted after all existing rules.
        :param str name: (optional) The user-defined name for this rule. Names must
               be unique within the network ACL the rule resides in. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        :param str protocol: (optional) The protocol to enforce.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'NetworkACLRulePrototypeNetworkACLRuleProtocolTCPUDP',
                'NetworkACLRulePrototypeNetworkACLRuleProtocolICMP',
                'NetworkACLRulePrototypeNetworkACLRuleProtocolAll'
            ]))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLRulePrototype':
        """Initialize a NetworkACLRulePrototype object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = (
            "Cannot convert dictionary into an instance of base class 'NetworkACLRulePrototype'. "
            + "The discriminator value should map to a valid subclass: {1}"
        ).format(", ".join([
            'NetworkACLRulePrototypeNetworkACLRuleProtocolTCPUDP',
            'NetworkACLRulePrototypeNetworkACLRuleProtocolICMP',
            'NetworkACLRulePrototypeNetworkACLRuleProtocolAll'
        ]))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a NetworkACLRulePrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping['all'] = 'NetworkACLRulePrototypeNetworkACLRuleProtocolAll'
        mapping['icmp'] = 'NetworkACLRulePrototypeNetworkACLRuleProtocolICMP'
        mapping['tcp'] = 'NetworkACLRulePrototypeNetworkACLRuleProtocolTCPUDP'
        mapping['udp'] = 'NetworkACLRulePrototypeNetworkACLRuleProtocolTCPUDP'
        disc_value = _dict.get('protocol')
        if disc_value is None:
            raise ValueError(
                'Discriminator property \'protocol\' not found in NetworkACLRulePrototype JSON'
            )
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)

    class ActionEnum(str, Enum):
        """
        Whether to allow or deny matching traffic.
        """
        ALLOW = 'allow'
        DENY = 'deny'

    class DirectionEnum(str, Enum):
        """
        Whether the traffic to be matched is `inbound` or `outbound`.
        """
        INBOUND = 'inbound'
        OUTBOUND = 'outbound'

    class ProtocolEnum(str, Enum):
        """
        The protocol to enforce.
        """
        ALL = 'all'
        ICMP = 'icmp'
        TCP = 'tcp'
        UDP = 'udp'


class NetworkACLRulePrototypeBefore():
    """
    The rule to insert this rule immediately before. If omitted, this rule will be
    inserted after all existing rules.

    """

    def __init__(self) -> None:
        """
        Initialize a NetworkACLRulePrototypeBefore object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'NetworkACLRulePrototypeBeforeNetworkACLRuleIdentityById',
                'NetworkACLRulePrototypeBeforeNetworkACLRuleIdentityByHref'
            ]))
        raise Exception(msg)


class NetworkACLRulePrototypeNetworkACLContext():
    """
    NetworkACLRulePrototypeNetworkACLContext.

    :attr str action: Whether to allow or deny matching traffic.
    :attr str destination: The destination IP address or CIDR block. The CIDR block
          `0.0.0.0/0` applies to all addresses.
    :attr str direction: Whether the traffic to be matched is `inbound` or
          `outbound`.
    :attr str name: (optional) The user-defined name for this rule. Names must be
          unique within the network ACL the rule resides in. If unspecified, the name will
          be a hyphenated list of randomly-selected words.
    :attr str protocol: (optional) The protocol to enforce.
    :attr str source: The source IP address or CIDR block.  The CIDR block
          `0.0.0.0/0` applies to all addresses.
    """

    def __init__(self,
                 action: str,
                 destination: str,
                 direction: str,
                 source: str,
                 *,
                 name: str = None,
                 protocol: str = None) -> None:
        """
        Initialize a NetworkACLRulePrototypeNetworkACLContext object.

        :param str action: Whether to allow or deny matching traffic.
        :param str destination: The destination IP address or CIDR block. The CIDR
               block `0.0.0.0/0` applies to all addresses.
        :param str direction: Whether the traffic to be matched is `inbound` or
               `outbound`.
        :param str source: The source IP address or CIDR block.  The CIDR block
               `0.0.0.0/0` applies to all addresses.
        :param str name: (optional) The user-defined name for this rule. Names must
               be unique within the network ACL the rule resides in. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        :param str protocol: (optional) The protocol to enforce.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolTCPUDP',
                'NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolICMP',
                'NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll'
            ]))
        raise Exception(msg)

    @classmethod
    def from_dict(cls,
                  _dict: Dict) -> 'NetworkACLRulePrototypeNetworkACLContext':
        """Initialize a NetworkACLRulePrototypeNetworkACLContext object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = (
            "Cannot convert dictionary into an instance of base class 'NetworkACLRulePrototypeNetworkACLContext'. "
            + "The discriminator value should map to a valid subclass: {1}"
        ).format(", ".join([
            'NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolTCPUDP',
            'NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolICMP',
            'NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll'
        ]))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a NetworkACLRulePrototypeNetworkACLContext object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping[
            'all'] = 'NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolAll'
        mapping[
            'icmp'] = 'NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolICMP'
        mapping[
            'tcp'] = 'NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolTCPUDP'
        mapping[
            'udp'] = 'NetworkACLRulePrototypeNetworkACLContextNetworkACLRuleProtocolTCPUDP'
        disc_value = _dict.get('protocol')
        if disc_value is None:
            raise ValueError(
                'Discriminator property \'protocol\' not found in NetworkACLRulePrototypeNetworkACLContext JSON'
            )
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)

    class ActionEnum(str, Enum):
        """
        Whether to allow or deny matching traffic.
        """
        ALLOW = 'allow'
        DENY = 'deny'

    class DirectionEnum(str, Enum):
        """
        Whether the traffic to be matched is `inbound` or `outbound`.
        """
        INBOUND = 'inbound'
        OUTBOUND = 'outbound'

    class ProtocolEnum(str, Enum):
        """
        The protocol to enforce.
        """
        ALL = 'all'
        ICMP = 'icmp'
        TCP = 'tcp'
        UDP = 'udp'


class NetworkACLRuleReference():
    """
    NetworkACLRuleReference.

    :attr str href: The URL for this Network ACL rule.
    :attr str id: The unique identifier for this Network ACL rule.
    :attr str name: The user-defined name for this Network ACL rule.
    """

    def __init__(self, href: str, id: str, name: str) -> None:
        """
        Initialize a NetworkACLRuleReference object.

        :param str href: The URL for this Network ACL rule.
        :param str id: The unique identifier for this Network ACL rule.
        :param str name: The user-defined name for this Network ACL rule.
        """
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkACLRuleReference':
        """Initialize a NetworkACLRuleReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in NetworkACLRuleReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in NetworkACLRuleReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in NetworkACLRuleReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkACLRuleReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkACLRuleReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkACLRuleReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkACLRuleReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkInterface():
    """
    NetworkInterface.

    :attr datetime created_at: The date and time that the network interface was
          created.
    :attr List[FloatingIPReference] floating_ips: (optional) Array of references to
          floating IPs associated with this network interface.
    :attr str href: The URL for this network interface.
    :attr str id: The unique identifier for this network interface.
    :attr str name: The user-defined name for this network interface.
    :attr int port_speed: The network interface port speed in Mbps.
    :attr str primary_ipv4_address: The primary IPv4 address.
    :attr str resource_type: The type of resource referenced.
    :attr List[SecurityGroupReference] security_groups: Collection of security
          groups.
    :attr str status: The status of the network interface.
    :attr SubnetReference subnet: The associated subnet.
    :attr str type: The type of this network interface as it relates to an instance.
    """

    def __init__(self,
                 created_at: datetime,
                 href: str,
                 id: str,
                 name: str,
                 port_speed: int,
                 primary_ipv4_address: str,
                 resource_type: str,
                 security_groups: List['SecurityGroupReference'],
                 status: str,
                 subnet: 'SubnetReference',
                 type: str,
                 *,
                 floating_ips: List['FloatingIPReference'] = None) -> None:
        """
        Initialize a NetworkInterface object.

        :param datetime created_at: The date and time that the network interface
               was created.
        :param str href: The URL for this network interface.
        :param str id: The unique identifier for this network interface.
        :param str name: The user-defined name for this network interface.
        :param int port_speed: The network interface port speed in Mbps.
        :param str primary_ipv4_address: The primary IPv4 address.
        :param str resource_type: The type of resource referenced.
        :param List[SecurityGroupReference] security_groups: Collection of security
               groups.
        :param str status: The status of the network interface.
        :param SubnetReference subnet: The associated subnet.
        :param str type: The type of this network interface as it relates to an
               instance.
        :param List[FloatingIPReference] floating_ips: (optional) Array of
               references to floating IPs associated with this network interface.
        """
        self.created_at = created_at
        self.floating_ips = floating_ips
        self.href = href
        self.id = id
        self.name = name
        self.port_speed = port_speed
        self.primary_ipv4_address = primary_ipv4_address
        self.resource_type = resource_type
        self.security_groups = security_groups
        self.status = status
        self.subnet = subnet
        self.type = type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkInterface':
        """Initialize a NetworkInterface object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in NetworkInterface JSON'
            )
        if 'floating_ips' in _dict:
            args['floating_ips'] = [
                FloatingIPReference.from_dict(x)
                for x in _dict.get('floating_ips')
            ]
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in NetworkInterface JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in NetworkInterface JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in NetworkInterface JSON'
            )
        if 'port_speed' in _dict:
            args['port_speed'] = _dict.get('port_speed')
        else:
            raise ValueError(
                'Required property \'port_speed\' not present in NetworkInterface JSON'
            )
        if 'primary_ipv4_address' in _dict:
            args['primary_ipv4_address'] = _dict.get('primary_ipv4_address')
        else:
            raise ValueError(
                'Required property \'primary_ipv4_address\' not present in NetworkInterface JSON'
            )
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError(
                'Required property \'resource_type\' not present in NetworkInterface JSON'
            )
        if 'security_groups' in _dict:
            args['security_groups'] = [
                SecurityGroupReference.from_dict(x)
                for x in _dict.get('security_groups')
            ]
        else:
            raise ValueError(
                'Required property \'security_groups\' not present in NetworkInterface JSON'
            )
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in NetworkInterface JSON'
            )
        if 'subnet' in _dict:
            args['subnet'] = SubnetReference.from_dict(_dict.get('subnet'))
        else:
            raise ValueError(
                'Required property \'subnet\' not present in NetworkInterface JSON'
            )
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in NetworkInterface JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkInterface object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'floating_ips') and self.floating_ips is not None:
            _dict['floating_ips'] = [x.to_dict() for x in self.floating_ips]
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'port_speed') and self.port_speed is not None:
            _dict['port_speed'] = self.port_speed
        if hasattr(self, 'primary_ipv4_address'
                  ) and self.primary_ipv4_address is not None:
            _dict['primary_ipv4_address'] = self.primary_ipv4_address
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        if hasattr(self,
                   'security_groups') and self.security_groups is not None:
            _dict['security_groups'] = [
                x.to_dict() for x in self.security_groups
            ]
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'subnet') and self.subnet is not None:
            _dict['subnet'] = self.subnet.to_dict()
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkInterface object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkInterface') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkInterface') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ResourceTypeEnum(str, Enum):
        """
        The type of resource referenced.
        """
        NETWORK_INTERFACE = 'network_interface'

    class StatusEnum(str, Enum):
        """
        The status of the network interface.
        """
        AVAILABLE = 'available'
        FAILED = 'failed'
        PENDING = 'pending'

    class TypeEnum(str, Enum):
        """
        The type of this network interface as it relates to an instance.
        """
        PRIMARY = 'primary'
        SECONDARY = 'secondary'


class NetworkInterfaceCollection():
    """
    NetworkInterfaceCollection.

    :attr List[NetworkInterface] network_interfaces: Collection of network
          interfaces.
    """

    def __init__(self, network_interfaces: List['NetworkInterface']) -> None:
        """
        Initialize a NetworkInterfaceCollection object.

        :param List[NetworkInterface] network_interfaces: Collection of network
               interfaces.
        """
        self.network_interfaces = network_interfaces

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkInterfaceCollection':
        """Initialize a NetworkInterfaceCollection object from a json dictionary."""
        args = {}
        if 'network_interfaces' in _dict:
            args['network_interfaces'] = [
                NetworkInterface.from_dict(x)
                for x in _dict.get('network_interfaces')
            ]
        else:
            raise ValueError(
                'Required property \'network_interfaces\' not present in NetworkInterfaceCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkInterfaceCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(
                self,
                'network_interfaces') and self.network_interfaces is not None:
            _dict['network_interfaces'] = [
                x.to_dict() for x in self.network_interfaces
            ]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkInterfaceCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkInterfaceCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkInterfaceCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkInterfaceIdentity():
    """
    Identifies a network interface by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a NetworkInterfaceIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'NetworkInterfaceIdentityById', 'NetworkInterfaceIdentityByHref'
            ]))
        raise Exception(msg)


class NetworkInterfaceInstanceContextReference():
    """
    NetworkInterfaceInstanceContextReference.

    :attr str href: The URL for this network interface.
    :attr str id: The unique identifier for this network interface.
    :attr str name: The user-defined name for this network interface.
    :attr str primary_ipv4_address: The primary IPv4 address.
    :attr str resource_type: The type of resource referenced.
    :attr SubnetReference subnet: The associated subnet.
    """

    def __init__(self, href: str, id: str, name: str, primary_ipv4_address: str,
                 resource_type: str, subnet: 'SubnetReference') -> None:
        """
        Initialize a NetworkInterfaceInstanceContextReference object.

        :param str href: The URL for this network interface.
        :param str id: The unique identifier for this network interface.
        :param str name: The user-defined name for this network interface.
        :param str primary_ipv4_address: The primary IPv4 address.
        :param str resource_type: The type of resource referenced.
        :param SubnetReference subnet: The associated subnet.
        """
        self.href = href
        self.id = id
        self.name = name
        self.primary_ipv4_address = primary_ipv4_address
        self.resource_type = resource_type
        self.subnet = subnet

    @classmethod
    def from_dict(cls,
                  _dict: Dict) -> 'NetworkInterfaceInstanceContextReference':
        """Initialize a NetworkInterfaceInstanceContextReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in NetworkInterfaceInstanceContextReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in NetworkInterfaceInstanceContextReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in NetworkInterfaceInstanceContextReference JSON'
            )
        if 'primary_ipv4_address' in _dict:
            args['primary_ipv4_address'] = _dict.get('primary_ipv4_address')
        else:
            raise ValueError(
                'Required property \'primary_ipv4_address\' not present in NetworkInterfaceInstanceContextReference JSON'
            )
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError(
                'Required property \'resource_type\' not present in NetworkInterfaceInstanceContextReference JSON'
            )
        if 'subnet' in _dict:
            args['subnet'] = SubnetReference.from_dict(_dict.get('subnet'))
        else:
            raise ValueError(
                'Required property \'subnet\' not present in NetworkInterfaceInstanceContextReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkInterfaceInstanceContextReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'primary_ipv4_address'
                  ) and self.primary_ipv4_address is not None:
            _dict['primary_ipv4_address'] = self.primary_ipv4_address
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        if hasattr(self, 'subnet') and self.subnet is not None:
            _dict['subnet'] = self.subnet.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkInterfaceInstanceContextReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkInterfaceInstanceContextReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkInterfaceInstanceContextReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ResourceTypeEnum(str, Enum):
        """
        The type of resource referenced.
        """
        NETWORK_INTERFACE = 'network_interface'


class NetworkInterfacePrototype():
    """
    NetworkInterfacePrototype.

    :attr str name: (optional) The user-defined name for this network interface. If
          unspecified, the name will be a hyphenated list of randomly-selected words.
    :attr str primary_ipv4_address: (optional) The primary IPv4 address.
    :attr List[SecurityGroupIdentity] security_groups: (optional) Collection of
          security groups.
    :attr SubnetIdentity subnet: The associated subnet.
    """

    def __init__(self,
                 subnet: 'SubnetIdentity',
                 *,
                 name: str = None,
                 primary_ipv4_address: str = None,
                 security_groups: List['SecurityGroupIdentity'] = None) -> None:
        """
        Initialize a NetworkInterfacePrototype object.

        :param SubnetIdentity subnet: The associated subnet.
        :param str name: (optional) The user-defined name for this network
               interface. If unspecified, the name will be a hyphenated list of
               randomly-selected words.
        :param str primary_ipv4_address: (optional) The primary IPv4 address.
        :param List[SecurityGroupIdentity] security_groups: (optional) Collection
               of security groups.
        """
        self.name = name
        self.primary_ipv4_address = primary_ipv4_address
        self.security_groups = security_groups
        self.subnet = subnet

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkInterfacePrototype':
        """Initialize a NetworkInterfacePrototype object from a json dictionary."""
        args = {}
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        if 'primary_ipv4_address' in _dict:
            args['primary_ipv4_address'] = _dict.get('primary_ipv4_address')
        if 'security_groups' in _dict:
            args['security_groups'] = [
                SecurityGroupIdentity.from_dict(x)
                for x in _dict.get('security_groups')
            ]
        if 'subnet' in _dict:
            args['subnet'] = _dict.get('subnet')
        else:
            raise ValueError(
                'Required property \'subnet\' not present in NetworkInterfacePrototype JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkInterfacePrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'primary_ipv4_address'
                  ) and self.primary_ipv4_address is not None:
            _dict['primary_ipv4_address'] = self.primary_ipv4_address
        if hasattr(self,
                   'security_groups') and self.security_groups is not None:
            _dict['security_groups'] = [
                x.to_dict() for x in self.security_groups
            ]
        if hasattr(self, 'subnet') and self.subnet is not None:
            _dict['subnet'] = self.subnet
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkInterfacePrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkInterfacePrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkInterfacePrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class NetworkInterfaceReference():
    """
    NetworkInterfaceReference.

    :attr str href: The URL for this network interface.
    :attr str id: The unique identifier for this network interface.
    :attr str name: The user-defined name for this network interface.
    :attr str primary_ipv4_address: The primary IPv4 address.
    :attr str resource_type: The type of resource referenced.
    """

    def __init__(self, href: str, id: str, name: str, primary_ipv4_address: str,
                 resource_type: str) -> None:
        """
        Initialize a NetworkInterfaceReference object.

        :param str href: The URL for this network interface.
        :param str id: The unique identifier for this network interface.
        :param str name: The user-defined name for this network interface.
        :param str primary_ipv4_address: The primary IPv4 address.
        :param str resource_type: The type of resource referenced.
        """
        self.href = href
        self.id = id
        self.name = name
        self.primary_ipv4_address = primary_ipv4_address
        self.resource_type = resource_type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'NetworkInterfaceReference':
        """Initialize a NetworkInterfaceReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in NetworkInterfaceReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in NetworkInterfaceReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in NetworkInterfaceReference JSON'
            )
        if 'primary_ipv4_address' in _dict:
            args['primary_ipv4_address'] = _dict.get('primary_ipv4_address')
        else:
            raise ValueError(
                'Required property \'primary_ipv4_address\' not present in NetworkInterfaceReference JSON'
            )
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError(
                'Required property \'resource_type\' not present in NetworkInterfaceReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a NetworkInterfaceReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'primary_ipv4_address'
                  ) and self.primary_ipv4_address is not None:
            _dict['primary_ipv4_address'] = self.primary_ipv4_address
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this NetworkInterfaceReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'NetworkInterfaceReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'NetworkInterfaceReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ResourceTypeEnum(str, Enum):
        """
        The type of resource referenced.
        """
        NETWORK_INTERFACE = 'network_interface'


class OperatingSystem():
    """
    OperatingSystem.

    :attr str architecture: The operating system architecture.
    :attr str display_name: A unique, display-friendly name for the operating
          system.
    :attr str family: The name of the software family this operating system belongs
          to.
    :attr str href: The URL for this operating system.
    :attr str name: The unique name of the operating system.
    :attr str vendor: The vendor of the operating system.
    :attr str version: The major release version of this operating system.
    """

    def __init__(self, architecture: str, display_name: str, family: str,
                 href: str, name: str, vendor: str, version: str) -> None:
        """
        Initialize a OperatingSystem object.

        :param str architecture: The operating system architecture.
        :param str display_name: A unique, display-friendly name for the operating
               system.
        :param str family: The name of the software family this operating system
               belongs to.
        :param str href: The URL for this operating system.
        :param str name: The unique name of the operating system.
        :param str vendor: The vendor of the operating system.
        :param str version: The major release version of this operating system.
        """
        self.architecture = architecture
        self.display_name = display_name
        self.family = family
        self.href = href
        self.name = name
        self.vendor = vendor
        self.version = version

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'OperatingSystem':
        """Initialize a OperatingSystem object from a json dictionary."""
        args = {}
        if 'architecture' in _dict:
            args['architecture'] = _dict.get('architecture')
        else:
            raise ValueError(
                'Required property \'architecture\' not present in OperatingSystem JSON'
            )
        if 'display_name' in _dict:
            args['display_name'] = _dict.get('display_name')
        else:
            raise ValueError(
                'Required property \'display_name\' not present in OperatingSystem JSON'
            )
        if 'family' in _dict:
            args['family'] = _dict.get('family')
        else:
            raise ValueError(
                'Required property \'family\' not present in OperatingSystem JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in OperatingSystem JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in OperatingSystem JSON'
            )
        if 'vendor' in _dict:
            args['vendor'] = _dict.get('vendor')
        else:
            raise ValueError(
                'Required property \'vendor\' not present in OperatingSystem JSON'
            )
        if 'version' in _dict:
            args['version'] = _dict.get('version')
        else:
            raise ValueError(
                'Required property \'version\' not present in OperatingSystem JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a OperatingSystem object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'architecture') and self.architecture is not None:
            _dict['architecture'] = self.architecture
        if hasattr(self, 'display_name') and self.display_name is not None:
            _dict['display_name'] = self.display_name
        if hasattr(self, 'family') and self.family is not None:
            _dict['family'] = self.family
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'vendor') and self.vendor is not None:
            _dict['vendor'] = self.vendor
        if hasattr(self, 'version') and self.version is not None:
            _dict['version'] = self.version
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this OperatingSystem object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'OperatingSystem') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'OperatingSystem') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class OperatingSystemCollection():
    """
    OperatingSystemCollection.

    :attr OperatingSystemCollectionFirst first: A reference to the first page of
          resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr OperatingSystemCollectionNext next: (optional) A reference to the next
          page of resources; this reference is included for all pages
          except the last page.
    :attr List[OperatingSystem] operating_systems: Collection of operating systems.
    """

    def __init__(self,
                 first: 'OperatingSystemCollectionFirst',
                 limit: int,
                 operating_systems: List['OperatingSystem'],
                 *,
                 next: 'OperatingSystemCollectionNext' = None) -> None:
        """
        Initialize a OperatingSystemCollection object.

        :param OperatingSystemCollectionFirst first: A reference to the first page
               of resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param List[OperatingSystem] operating_systems: Collection of operating
               systems.
        :param OperatingSystemCollectionNext next: (optional) A reference to the
               next page of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.next = next
        self.operating_systems = operating_systems

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'OperatingSystemCollection':
        """Initialize a OperatingSystemCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = OperatingSystemCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in OperatingSystemCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in OperatingSystemCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = OperatingSystemCollectionNext.from_dict(
                _dict.get('next'))
        if 'operating_systems' in _dict:
            args['operating_systems'] = [
                OperatingSystem.from_dict(x)
                for x in _dict.get('operating_systems')
            ]
        else:
            raise ValueError(
                'Required property \'operating_systems\' not present in OperatingSystemCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a OperatingSystemCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self,
                   'operating_systems') and self.operating_systems is not None:
            _dict['operating_systems'] = [
                x.to_dict() for x in self.operating_systems
            ]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this OperatingSystemCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'OperatingSystemCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'OperatingSystemCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class OperatingSystemCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a OperatingSystemCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'OperatingSystemCollectionFirst':
        """Initialize a OperatingSystemCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in OperatingSystemCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a OperatingSystemCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this OperatingSystemCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'OperatingSystemCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'OperatingSystemCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class OperatingSystemCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a OperatingSystemCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'OperatingSystemCollectionNext':
        """Initialize a OperatingSystemCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in OperatingSystemCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a OperatingSystemCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this OperatingSystemCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'OperatingSystemCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'OperatingSystemCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class OperatingSystemIdentity():
    """
    Identifies an operating system by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a OperatingSystemIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'OperatingSystemIdentityByName', 'OperatingSystemIdentityByHref'
            ]))
        raise Exception(msg)


class PublicGateway():
    """
    PublicGateway.

    :attr datetime created_at: The date and time that the public gateway was
          created.
    :attr str crn: The CRN for this public gateway.
    :attr PublicGatewayFloatingIp floating_ip: Reference to the floating IP which is
          bound to this public gateway.
    :attr str href: The URL for this public gateway.
    :attr str id: The unique identifier for this public gateway.
    :attr str name: The user-defined name for this public gateway.
    :attr str resource_type: The type of resource referenced.
    :attr str status: The status of the volume.
    :attr VPCReference vpc: The VPC this public gateway serves.
    :attr ZoneReference zone: The zone where this public gateway lives.
    """

    def __init__(self, created_at: datetime, crn: str,
                 floating_ip: 'PublicGatewayFloatingIp', href: str, id: str,
                 name: str, resource_type: str, status: str,
                 vpc: 'VPCReference', zone: 'ZoneReference') -> None:
        """
        Initialize a PublicGateway object.

        :param datetime created_at: The date and time that the public gateway was
               created.
        :param str crn: The CRN for this public gateway.
        :param PublicGatewayFloatingIp floating_ip: Reference to the floating IP
               which is bound to this public gateway.
        :param str href: The URL for this public gateway.
        :param str id: The unique identifier for this public gateway.
        :param str name: The user-defined name for this public gateway.
        :param str resource_type: The type of resource referenced.
        :param str status: The status of the volume.
        :param VPCReference vpc: The VPC this public gateway serves.
        :param ZoneReference zone: The zone where this public gateway lives.
        """
        self.created_at = created_at
        self.crn = crn
        self.floating_ip = floating_ip
        self.href = href
        self.id = id
        self.name = name
        self.resource_type = resource_type
        self.status = status
        self.vpc = vpc
        self.zone = zone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'PublicGateway':
        """Initialize a PublicGateway object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in PublicGateway JSON'
            )
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in PublicGateway JSON')
        if 'floating_ip' in _dict:
            args['floating_ip'] = PublicGatewayFloatingIp.from_dict(
                _dict.get('floating_ip'))
        else:
            raise ValueError(
                'Required property \'floating_ip\' not present in PublicGateway JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in PublicGateway JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in PublicGateway JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in PublicGateway JSON')
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError(
                'Required property \'resource_type\' not present in PublicGateway JSON'
            )
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in PublicGateway JSON'
            )
        if 'vpc' in _dict:
            args['vpc'] = VPCReference.from_dict(_dict.get('vpc'))
        else:
            raise ValueError(
                'Required property \'vpc\' not present in PublicGateway JSON')
        if 'zone' in _dict:
            args['zone'] = ZoneReference.from_dict(_dict.get('zone'))
        else:
            raise ValueError(
                'Required property \'zone\' not present in PublicGateway JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a PublicGateway object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'floating_ip') and self.floating_ip is not None:
            _dict['floating_ip'] = self.floating_ip.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'vpc') and self.vpc is not None:
            _dict['vpc'] = self.vpc.to_dict()
        if hasattr(self, 'zone') and self.zone is not None:
            _dict['zone'] = self.zone.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this PublicGateway object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'PublicGateway') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'PublicGateway') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ResourceTypeEnum(str, Enum):
        """
        The type of resource referenced.
        """
        PUBLIC_GATEWAY = 'public_gateway'

    class StatusEnum(str, Enum):
        """
        The status of the volume.
        """
        AVAILABLE = 'available'
        DELETING = 'deleting'
        FAILED = 'failed'
        PENDING = 'pending'


class PublicGatewayCollection():
    """
    PublicGatewayCollection.

    :attr PublicGatewayCollectionFirst first: A reference to the first page of
          resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr PublicGatewayCollectionNext next: (optional) A reference to the next page
          of resources; this reference is included for all pages
          except the last page.
    :attr List[PublicGateway] public_gateways: Collection of public gateways.
    """

    def __init__(self,
                 first: 'PublicGatewayCollectionFirst',
                 limit: int,
                 public_gateways: List['PublicGateway'],
                 *,
                 next: 'PublicGatewayCollectionNext' = None) -> None:
        """
        Initialize a PublicGatewayCollection object.

        :param PublicGatewayCollectionFirst first: A reference to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param List[PublicGateway] public_gateways: Collection of public gateways.
        :param PublicGatewayCollectionNext next: (optional) A reference to the next
               page of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.next = next
        self.public_gateways = public_gateways

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'PublicGatewayCollection':
        """Initialize a PublicGatewayCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = PublicGatewayCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in PublicGatewayCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in PublicGatewayCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = PublicGatewayCollectionNext.from_dict(
                _dict.get('next'))
        if 'public_gateways' in _dict:
            args['public_gateways'] = [
                PublicGateway.from_dict(x) for x in _dict.get('public_gateways')
            ]
        else:
            raise ValueError(
                'Required property \'public_gateways\' not present in PublicGatewayCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a PublicGatewayCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self,
                   'public_gateways') and self.public_gateways is not None:
            _dict['public_gateways'] = [
                x.to_dict() for x in self.public_gateways
            ]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this PublicGatewayCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'PublicGatewayCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'PublicGatewayCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class PublicGatewayCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a PublicGatewayCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'PublicGatewayCollectionFirst':
        """Initialize a PublicGatewayCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in PublicGatewayCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a PublicGatewayCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this PublicGatewayCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'PublicGatewayCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'PublicGatewayCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class PublicGatewayCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a PublicGatewayCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'PublicGatewayCollectionNext':
        """Initialize a PublicGatewayCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in PublicGatewayCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a PublicGatewayCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this PublicGatewayCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'PublicGatewayCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'PublicGatewayCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class PublicGatewayFloatingIp():
    """
    Reference to the floating IP which is bound to this public gateway.

    :attr str address: The globally unique IP address.
    :attr str crn: The CRN for this floating IP.
    :attr str href: The URL for this floating IP.
    :attr str id: The unique identifier for this floating IP.
    :attr str name: The user-defined name for this floating IP.
    """

    def __init__(self, address: str, crn: str, href: str, id: str,
                 name: str) -> None:
        """
        Initialize a PublicGatewayFloatingIp object.

        :param str address: The globally unique IP address.
        :param str crn: The CRN for this floating IP.
        :param str href: The URL for this floating IP.
        :param str id: The unique identifier for this floating IP.
        :param str name: The user-defined name for this floating IP.
        """
        self.address = address
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'PublicGatewayFloatingIp':
        """Initialize a PublicGatewayFloatingIp object from a json dictionary."""
        args = {}
        if 'address' in _dict:
            args['address'] = _dict.get('address')
        else:
            raise ValueError(
                'Required property \'address\' not present in PublicGatewayFloatingIp JSON'
            )
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in PublicGatewayFloatingIp JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in PublicGatewayFloatingIp JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in PublicGatewayFloatingIp JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in PublicGatewayFloatingIp JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a PublicGatewayFloatingIp object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'address') and self.address is not None:
            _dict['address'] = self.address
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this PublicGatewayFloatingIp object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'PublicGatewayFloatingIp') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'PublicGatewayFloatingIp') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class PublicGatewayIdentity():
    """
    Identifies a public gateway by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a PublicGatewayIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'PublicGatewayIdentityById', 'PublicGatewayIdentityByCRN',
                'PublicGatewayIdentityByHref'
            ]))
        raise Exception(msg)


class PublicGatewayPrototypeFloatingIp():
    """
    PublicGatewayPrototypeFloatingIp.

    """

    def __init__(self) -> None:
        """
        Initialize a PublicGatewayPrototypeFloatingIp object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'PublicGatewayPrototypeFloatingIpFloatingIPIdentity',
                'PublicGatewayPrototypeFloatingIpFloatingIPPrototypeTargetContext'
            ]))
        raise Exception(msg)


class PublicGatewayReference():
    """
    PublicGatewayReference.

    :attr str crn: The CRN for this public gateway.
    :attr str href: The URL for this public gateway.
    :attr str id: The unique identifier for this public gateway.
    :attr str name: The user-defined name for this public gateway.
    :attr str resource_type: The type of resource referenced.
    """

    def __init__(self, crn: str, href: str, id: str, name: str,
                 resource_type: str) -> None:
        """
        Initialize a PublicGatewayReference object.

        :param str crn: The CRN for this public gateway.
        :param str href: The URL for this public gateway.
        :param str id: The unique identifier for this public gateway.
        :param str name: The user-defined name for this public gateway.
        :param str resource_type: The type of resource referenced.
        """
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name
        self.resource_type = resource_type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'PublicGatewayReference':
        """Initialize a PublicGatewayReference object from a json dictionary."""
        args = {}
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in PublicGatewayReference JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in PublicGatewayReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in PublicGatewayReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in PublicGatewayReference JSON'
            )
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError(
                'Required property \'resource_type\' not present in PublicGatewayReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a PublicGatewayReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this PublicGatewayReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'PublicGatewayReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'PublicGatewayReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ResourceTypeEnum(str, Enum):
        """
        The type of resource referenced.
        """
        PUBLIC_GATEWAY = 'public_gateway'


class Region():
    """
    Region.

    :attr str endpoint: The API endpoint for this region.
    :attr str href: The URL for this region.
    :attr str name: The name for this region.
    :attr str status: The availability status of this region.
    """

    def __init__(self, endpoint: str, href: str, name: str,
                 status: str) -> None:
        """
        Initialize a Region object.

        :param str endpoint: The API endpoint for this region.
        :param str href: The URL for this region.
        :param str name: The name for this region.
        :param str status: The availability status of this region.
        """
        self.endpoint = endpoint
        self.href = href
        self.name = name
        self.status = status

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'Region':
        """Initialize a Region object from a json dictionary."""
        args = {}
        if 'endpoint' in _dict:
            args['endpoint'] = _dict.get('endpoint')
        else:
            raise ValueError(
                'Required property \'endpoint\' not present in Region JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in Region JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in Region JSON')
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in Region JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a Region object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'endpoint') and self.endpoint is not None:
            _dict['endpoint'] = self.endpoint
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this Region object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'Region') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'Region') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class StatusEnum(str, Enum):
        """
        The availability status of this region.
        """
        AVAILABLE = 'available'
        UNAVAILABLE = 'unavailable'


class RegionCollection():
    """
    RegionCollection.

    :attr List[Region] regions: Array of Region objects.
    """

    def __init__(self, regions: List['Region']) -> None:
        """
        Initialize a RegionCollection object.

        :param List[Region] regions: Array of Region objects.
        """
        self.regions = regions

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'RegionCollection':
        """Initialize a RegionCollection object from a json dictionary."""
        args = {}
        if 'regions' in _dict:
            args['regions'] = [
                Region.from_dict(x) for x in _dict.get('regions')
            ]
        else:
            raise ValueError(
                'Required property \'regions\' not present in RegionCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RegionCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'regions') and self.regions is not None:
            _dict['regions'] = [x.to_dict() for x in self.regions]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RegionCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'RegionCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'RegionCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RegionReference():
    """
    RegionReference.

    :attr str href: The URL for this region.
    :attr str name: The name for this region.
    """

    def __init__(self, href: str, name: str) -> None:
        """
        Initialize a RegionReference object.

        :param str href: The URL for this region.
        :param str name: The name for this region.
        """
        self.href = href
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'RegionReference':
        """Initialize a RegionReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in RegionReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in RegionReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RegionReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RegionReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'RegionReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'RegionReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ResourceGroupIdentity():
    """
    The resource group to use. If unspecified, the account's [default resource
    group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.

    """

    def __init__(self) -> None:
        """
        Initialize a ResourceGroupIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['ResourceGroupIdentityById']))
        raise Exception(msg)


class ResourceGroupReference():
    """
    ResourceGroupReference.

    :attr str href: The URL for this resource group.
    :attr str id: The unique identifier for this resource group.
    """

    def __init__(self, href: str, id: str) -> None:
        """
        Initialize a ResourceGroupReference object.

        :param str href: The URL for this resource group.
        :param str id: The unique identifier for this resource group.
        """
        self.href = href
        self.id = id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ResourceGroupReference':
        """Initialize a ResourceGroupReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in ResourceGroupReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in ResourceGroupReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ResourceGroupReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ResourceGroupReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ResourceGroupReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ResourceGroupReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class Route():
    """
    Route.

    :attr datetime created_at: The date and time that the route was created.
    :attr str destination: The destination of the route.
    :attr str href: The URL for this route.
    :attr str id: The unique identifier for this route.
    :attr str lifecycle_state: The lifecycle state of the route.
    :attr str name: The user-defined name for this route.
    :attr RouteNextHop next_hop: The next hop that packets will be delivered to.
    :attr ZoneReference zone: The zone the route applies to. (Traffic from subnets
          in this zone will be
          subject to this route.).
    """

    def __init__(self, created_at: datetime, destination: str, href: str,
                 id: str, lifecycle_state: str, name: str,
                 next_hop: 'RouteNextHop', zone: 'ZoneReference') -> None:
        """
        Initialize a Route object.

        :param datetime created_at: The date and time that the route was created.
        :param str destination: The destination of the route.
        :param str href: The URL for this route.
        :param str id: The unique identifier for this route.
        :param str lifecycle_state: The lifecycle state of the route.
        :param str name: The user-defined name for this route.
        :param RouteNextHop next_hop: The next hop that packets will be delivered
               to.
        :param ZoneReference zone: The zone the route applies to. (Traffic from
               subnets in this zone will be
               subject to this route.).
        """
        self.created_at = created_at
        self.destination = destination
        self.href = href
        self.id = id
        self.lifecycle_state = lifecycle_state
        self.name = name
        self.next_hop = next_hop
        self.zone = zone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'Route':
        """Initialize a Route object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in Route JSON')
        if 'destination' in _dict:
            args['destination'] = _dict.get('destination')
        else:
            raise ValueError(
                'Required property \'destination\' not present in Route JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in Route JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in Route JSON')
        if 'lifecycle_state' in _dict:
            args['lifecycle_state'] = _dict.get('lifecycle_state')
        else:
            raise ValueError(
                'Required property \'lifecycle_state\' not present in Route JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in Route JSON')
        if 'next_hop' in _dict:
            args['next_hop'] = _dict.get('next_hop')
        else:
            raise ValueError(
                'Required property \'next_hop\' not present in Route JSON')
        if 'zone' in _dict:
            args['zone'] = ZoneReference.from_dict(_dict.get('zone'))
        else:
            raise ValueError(
                'Required property \'zone\' not present in Route JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a Route object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'destination') and self.destination is not None:
            _dict['destination'] = self.destination
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self,
                   'lifecycle_state') and self.lifecycle_state is not None:
            _dict['lifecycle_state'] = self.lifecycle_state
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'next_hop') and self.next_hop is not None:
            _dict['next_hop'] = self.next_hop
        if hasattr(self, 'zone') and self.zone is not None:
            _dict['zone'] = self.zone.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this Route object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'Route') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'Route') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class LifecycleStateEnum(str, Enum):
        """
        The lifecycle state of the route.
        """
        DELETED = 'deleted'
        DELETING = 'deleting'
        FAILED = 'failed'
        PENDING = 'pending'
        STABLE = 'stable'
        UPDATING = 'updating'
        WAITING = 'waiting'


class RouteCollection():
    """
    RouteCollection.

    :attr List[Route] routes: Collection of routes.
    """

    def __init__(self, routes: List['Route']) -> None:
        """
        Initialize a RouteCollection object.

        :param List[Route] routes: Collection of routes.
        """
        self.routes = routes

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'RouteCollection':
        """Initialize a RouteCollection object from a json dictionary."""
        args = {}
        if 'routes' in _dict:
            args['routes'] = [Route.from_dict(x) for x in _dict.get('routes')]
        else:
            raise ValueError(
                'Required property \'routes\' not present in RouteCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RouteCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'routes') and self.routes is not None:
            _dict['routes'] = [x.to_dict() for x in self.routes]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RouteCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'RouteCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'RouteCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RouteNextHop():
    """
    RouteNextHop.

    """

    def __init__(self) -> None:
        """
        Initialize a RouteNextHop object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['RouteNextHopIP']))
        raise Exception(msg)


class RouteNextHopPrototype():
    """
    RouteNextHopPrototype.

    """

    def __init__(self) -> None:
        """
        Initialize a RouteNextHopPrototype object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['RouteNextHopPrototypeRouteNextHopIP']))
        raise Exception(msg)


class SecurityGroup():
    """
    SecurityGroup.

    :attr datetime created_at: The date and time that this security group was
          created.
    :attr str crn: The security group's CRN.
    :attr str href: The security group's canonical URL.
    :attr str id: The unique identifier for this security group.
    :attr str name: The user-defined name for this security group. Security group
          names must be unique, within the scope of an account.
    :attr List[NetworkInterfaceReference] network_interfaces: Array of references to
          network interfaces.
    :attr ResourceGroupReference resource_group: The resource group for this
          security group.
    :attr List[SecurityGroupRule] rules: Array of rules for this security group. If
          no rules exist, all traffic will be denied.
    :attr VPCReference vpc: The VPC this security group is a part of.
    """

    def __init__(self, created_at: datetime, crn: str, href: str, id: str,
                 name: str,
                 network_interfaces: List['NetworkInterfaceReference'],
                 resource_group: 'ResourceGroupReference',
                 rules: List['SecurityGroupRule'], vpc: 'VPCReference') -> None:
        """
        Initialize a SecurityGroup object.

        :param datetime created_at: The date and time that this security group was
               created.
        :param str crn: The security group's CRN.
        :param str href: The security group's canonical URL.
        :param str id: The unique identifier for this security group.
        :param str name: The user-defined name for this security group. Security
               group names must be unique, within the scope of an account.
        :param List[NetworkInterfaceReference] network_interfaces: Array of
               references to network interfaces.
        :param ResourceGroupReference resource_group: The resource group for this
               security group.
        :param List[SecurityGroupRule] rules: Array of rules for this security
               group. If no rules exist, all traffic will be denied.
        :param VPCReference vpc: The VPC this security group is a part of.
        """
        self.created_at = created_at
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name
        self.network_interfaces = network_interfaces
        self.resource_group = resource_group
        self.rules = rules
        self.vpc = vpc

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SecurityGroup':
        """Initialize a SecurityGroup object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in SecurityGroup JSON'
            )
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in SecurityGroup JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in SecurityGroup JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in SecurityGroup JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in SecurityGroup JSON')
        if 'network_interfaces' in _dict:
            args['network_interfaces'] = [
                NetworkInterfaceReference.from_dict(x)
                for x in _dict.get('network_interfaces')
            ]
        else:
            raise ValueError(
                'Required property \'network_interfaces\' not present in SecurityGroup JSON'
            )
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(
                _dict.get('resource_group'))
        else:
            raise ValueError(
                'Required property \'resource_group\' not present in SecurityGroup JSON'
            )
        if 'rules' in _dict:
            args['rules'] = [
                SecurityGroupRule.from_dict(x) for x in _dict.get('rules')
            ]
        else:
            raise ValueError(
                'Required property \'rules\' not present in SecurityGroup JSON')
        if 'vpc' in _dict:
            args['vpc'] = VPCReference.from_dict(_dict.get('vpc'))
        else:
            raise ValueError(
                'Required property \'vpc\' not present in SecurityGroup JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SecurityGroup object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(
                self,
                'network_interfaces') and self.network_interfaces is not None:
            _dict['network_interfaces'] = [
                x.to_dict() for x in self.network_interfaces
            ]
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(self, 'rules') and self.rules is not None:
            _dict['rules'] = [x.to_dict() for x in self.rules]
        if hasattr(self, 'vpc') and self.vpc is not None:
            _dict['vpc'] = self.vpc.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SecurityGroup object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SecurityGroup') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SecurityGroup') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SecurityGroupCollection():
    """
    SecurityGroupCollection.

    :attr List[SecurityGroup] security_groups: Collection of security groups.
    """

    def __init__(self, security_groups: List['SecurityGroup']) -> None:
        """
        Initialize a SecurityGroupCollection object.

        :param List[SecurityGroup] security_groups: Collection of security groups.
        """
        self.security_groups = security_groups

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SecurityGroupCollection':
        """Initialize a SecurityGroupCollection object from a json dictionary."""
        args = {}
        if 'security_groups' in _dict:
            args['security_groups'] = [
                SecurityGroup.from_dict(x) for x in _dict.get('security_groups')
            ]
        else:
            raise ValueError(
                'Required property \'security_groups\' not present in SecurityGroupCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SecurityGroupCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self,
                   'security_groups') and self.security_groups is not None:
            _dict['security_groups'] = [
                x.to_dict() for x in self.security_groups
            ]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SecurityGroupCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SecurityGroupCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SecurityGroupCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SecurityGroupIdentity():
    """
    Identifies a security group by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a SecurityGroupIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupIdentityById', 'SecurityGroupIdentityByCRN',
                'SecurityGroupIdentityByHref'
            ]))
        raise Exception(msg)


class SecurityGroupReference():
    """
    SecurityGroupReference.

    :attr str crn: The security group's CRN.
    :attr str href: The security group's canonical URL.
    :attr str id: The unique identifier for this security group.
    :attr str name: The user-defined name for this security group. Security group
          names must be unique, within the scope of an account.
    """

    def __init__(self, crn: str, href: str, id: str, name: str) -> None:
        """
        Initialize a SecurityGroupReference object.

        :param str crn: The security group's CRN.
        :param str href: The security group's canonical URL.
        :param str id: The unique identifier for this security group.
        :param str name: The user-defined name for this security group. Security
               group names must be unique, within the scope of an account.
        """
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SecurityGroupReference':
        """Initialize a SecurityGroupReference object from a json dictionary."""
        args = {}
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in SecurityGroupReference JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in SecurityGroupReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in SecurityGroupReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in SecurityGroupReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SecurityGroupReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SecurityGroupReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SecurityGroupReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SecurityGroupReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SecurityGroupRule():
    """
    SecurityGroupRule.

    :attr str direction: The direction of traffic to enforce, either `inbound` or
          `outbound`.
    :attr str id: The unique identifier for this security group rule.
    :attr str ip_version: (optional) The IP version to enforce. The format of
          `remote.address` or `remote.cidr_block` must match this field, if they are used.
          Alternatively, if `remote` references a security group, then this rule only
          applies to IP addresses (network interfaces) in that group matching this IP
          version.
    :attr str protocol: (optional) The protocol to enforce.
    :attr SecurityGroupRuleRemote remote: (optional) The IP addresses or security
          groups from which this rule allows traffic (or to
          which, for outbound rules). Can be specified as an IP address, a CIDR block, or
          a
          security group. If omitted, then traffic is allowed from any source (or to any
          source, for outbound rules).
    """

    def __init__(self,
                 direction: str,
                 id: str,
                 *,
                 ip_version: str = None,
                 protocol: str = None,
                 remote: 'SecurityGroupRuleRemote' = None) -> None:
        """
        Initialize a SecurityGroupRule object.

        :param str direction: The direction of traffic to enforce, either `inbound`
               or `outbound`.
        :param str id: The unique identifier for this security group rule.
        :param str ip_version: (optional) The IP version to enforce. The format of
               `remote.address` or `remote.cidr_block` must match this field, if they are
               used. Alternatively, if `remote` references a security group, then this
               rule only applies to IP addresses (network interfaces) in that group
               matching this IP version.
        :param str protocol: (optional) The protocol to enforce.
        :param SecurityGroupRuleRemote remote: (optional) The IP addresses or
               security groups from which this rule allows traffic (or to
               which, for outbound rules). Can be specified as an IP address, a CIDR
               block, or a
               security group. If omitted, then traffic is allowed from any source (or to
               any
               source, for outbound rules).
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRuleProtocolAll', 'SecurityGroupRuleProtocolICMP',
                'SecurityGroupRuleProtocolTCPUDP'
            ]))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SecurityGroupRule':
        """Initialize a SecurityGroupRule object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = (
            "Cannot convert dictionary into an instance of base class 'SecurityGroupRule'. "
            + "The discriminator value should map to a valid subclass: {1}"
        ).format(", ".join([
            'SecurityGroupRuleProtocolAll', 'SecurityGroupRuleProtocolICMP',
            'SecurityGroupRuleProtocolTCPUDP'
        ]))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a SecurityGroupRule object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping['all'] = 'SecurityGroupRuleProtocolAll'
        mapping['icmp'] = 'SecurityGroupRuleProtocolICMP'
        mapping['tcp'] = 'SecurityGroupRuleProtocolTCPUDP'
        mapping['udp'] = 'SecurityGroupRuleProtocolTCPUDP'
        disc_value = _dict.get('protocol')
        if disc_value is None:
            raise ValueError(
                'Discriminator property \'protocol\' not found in SecurityGroupRule JSON'
            )
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)

    class DirectionEnum(str, Enum):
        """
        The direction of traffic to enforce, either `inbound` or `outbound`.
        """
        INBOUND = 'inbound'
        OUTBOUND = 'outbound'

    class IpVersionEnum(str, Enum):
        """
        The IP version to enforce. The format of `remote.address` or `remote.cidr_block`
        must match this field, if they are used. Alternatively, if `remote` references a
        security group, then this rule only applies to IP addresses (network interfaces)
        in that group matching this IP version.
        """
        IPV4 = 'ipv4'

    class ProtocolEnum(str, Enum):
        """
        The protocol to enforce.
        """
        ALL = 'all'
        ICMP = 'icmp'
        TCP = 'tcp'
        UDP = 'udp'


class SecurityGroupRuleCollection():
    """
    Collection of rules in a security group.

    :attr List[SecurityGroupRule] rules: Array of rules.
    """

    def __init__(self, rules: List['SecurityGroupRule']) -> None:
        """
        Initialize a SecurityGroupRuleCollection object.

        :param List[SecurityGroupRule] rules: Array of rules.
        """
        self.rules = rules

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SecurityGroupRuleCollection':
        """Initialize a SecurityGroupRuleCollection object from a json dictionary."""
        args = {}
        if 'rules' in _dict:
            args['rules'] = [
                SecurityGroupRule.from_dict(x) for x in _dict.get('rules')
            ]
        else:
            raise ValueError(
                'Required property \'rules\' not present in SecurityGroupRuleCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SecurityGroupRuleCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'rules') and self.rules is not None:
            _dict['rules'] = [x.to_dict() for x in self.rules]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SecurityGroupRuleCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SecurityGroupRuleCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SecurityGroupRuleCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SecurityGroupRulePatch():
    """
    SecurityGroupRulePatch.

    :attr str direction: (optional) The direction of traffic to enforce, either
          `inbound` or `outbound`.
    :attr str ip_version: (optional) The IP version to enforce. The format of
          `remote.address` or `remote.cidr_block` must match this field, if they are used.
          Alternatively, if `remote` references a security group, then this rule only
          applies to IP addresses (network interfaces) in that group matching this IP
          version.
    :attr str protocol: (optional) The protocol to enforce.
    :attr SecurityGroupRulePatchRemote remote: (optional) The IP addresses or
          security groups from which this rule will allow traffic (or to
          which, for outbound rules). Can be specified as an IP address, a CIDR block, or
          a
          security group. If omitted, then traffic will be allowed from any source (or to
          any
          source, for outbound rules).
    """

    def __init__(self,
                 *,
                 direction: str = None,
                 ip_version: str = None,
                 protocol: str = None,
                 remote: 'SecurityGroupRulePatchRemote' = None) -> None:
        """
        Initialize a SecurityGroupRulePatch object.

        :param str direction: (optional) The direction of traffic to enforce,
               either `inbound` or `outbound`.
        :param str ip_version: (optional) The IP version to enforce. The format of
               `remote.address` or `remote.cidr_block` must match this field, if they are
               used. Alternatively, if `remote` references a security group, then this
               rule only applies to IP addresses (network interfaces) in that group
               matching this IP version.
        :param str protocol: (optional) The protocol to enforce.
        :param SecurityGroupRulePatchRemote remote: (optional) The IP addresses or
               security groups from which this rule will allow traffic (or to
               which, for outbound rules). Can be specified as an IP address, a CIDR
               block, or a
               security group. If omitted, then traffic will be allowed from any source
               (or to any
               source, for outbound rules).
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRulePatchSecurityGroupRuleProtocolAll',
                'SecurityGroupRulePatchSecurityGroupRuleProtocolICMP',
                'SecurityGroupRulePatchSecurityGroupRuleProtocolTCPUDP'
            ]))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SecurityGroupRulePatch':
        """Initialize a SecurityGroupRulePatch object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = (
            "Cannot convert dictionary into an instance of base class 'SecurityGroupRulePatch'. "
            + "The discriminator value should map to a valid subclass: {1}"
        ).format(", ".join([
            'SecurityGroupRulePatchSecurityGroupRuleProtocolAll',
            'SecurityGroupRulePatchSecurityGroupRuleProtocolICMP',
            'SecurityGroupRulePatchSecurityGroupRuleProtocolTCPUDP'
        ]))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a SecurityGroupRulePatch object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping['all'] = 'SecurityGroupRulePatchSecurityGroupRuleProtocolAll'
        mapping['icmp'] = 'SecurityGroupRulePatchSecurityGroupRuleProtocolICMP'
        mapping['tcp'] = 'SecurityGroupRulePatchSecurityGroupRuleProtocolTCPUDP'
        mapping['udp'] = 'SecurityGroupRulePatchSecurityGroupRuleProtocolTCPUDP'
        disc_value = _dict.get('protocol')
        if disc_value is None:
            raise ValueError(
                'Discriminator property \'protocol\' not found in SecurityGroupRulePatch JSON'
            )
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)

    class DirectionEnum(str, Enum):
        """
        The direction of traffic to enforce, either `inbound` or `outbound`.
        """
        INBOUND = 'inbound'
        OUTBOUND = 'outbound'

    class IpVersionEnum(str, Enum):
        """
        The IP version to enforce. The format of `remote.address` or `remote.cidr_block`
        must match this field, if they are used. Alternatively, if `remote` references a
        security group, then this rule only applies to IP addresses (network interfaces)
        in that group matching this IP version.
        """
        IPV4 = 'ipv4'

    class ProtocolEnum(str, Enum):
        """
        The protocol to enforce.
        """
        ALL = 'all'
        ICMP = 'icmp'
        TCP = 'tcp'
        UDP = 'udp'


class SecurityGroupRulePatchRemote():
    """
    The IP addresses or security groups from which this rule will allow traffic (or to
    which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
    security group. If omitted, then traffic will be allowed from any source (or to any
    source, for outbound rules).

    """

    def __init__(self) -> None:
        """
        Initialize a SecurityGroupRulePatchRemote object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRulePatchRemoteIP',
                'SecurityGroupRulePatchRemoteCIDR',
                'SecurityGroupRulePatchRemoteSecurityGroupIdentity'
            ]))
        raise Exception(msg)


class SecurityGroupRulePatchSecurityGroupRuleProtocolAllRemote():
    """
    The IP addresses or security groups from which this rule will allow traffic (or to
    which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
    security group. If omitted, then traffic will be allowed from any source (or to any
    source, for outbound rules).

    """

    def __init__(self) -> None:
        """
        Initialize a SecurityGroupRulePatchSecurityGroupRuleProtocolAllRemote object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRulePatchSecurityGroupRuleProtocolAllRemoteIP',
                'SecurityGroupRulePatchSecurityGroupRuleProtocolAllRemoteCIDR',
                'SecurityGroupRulePatchSecurityGroupRuleProtocolAllRemoteSecurityGroupIdentity'
            ]))
        raise Exception(msg)


class SecurityGroupRulePatchSecurityGroupRuleProtocolICMPRemote():
    """
    The IP addresses or security groups from which this rule will allow traffic (or to
    which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
    security group. If omitted, then traffic will be allowed from any source (or to any
    source, for outbound rules).

    """

    def __init__(self) -> None:
        """
        Initialize a SecurityGroupRulePatchSecurityGroupRuleProtocolICMPRemote object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRulePatchSecurityGroupRuleProtocolICMPRemoteIP',
                'SecurityGroupRulePatchSecurityGroupRuleProtocolICMPRemoteCIDR',
                'SecurityGroupRulePatchSecurityGroupRuleProtocolICMPRemoteSecurityGroupIdentity'
            ]))
        raise Exception(msg)


class SecurityGroupRulePatchSecurityGroupRuleProtocolTCPUDPRemote():
    """
    The IP addresses or security groups from which this rule will allow traffic (or to
    which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
    security group. If omitted, then traffic will be allowed from any source (or to any
    source, for outbound rules).

    """

    def __init__(self) -> None:
        """
        Initialize a SecurityGroupRulePatchSecurityGroupRuleProtocolTCPUDPRemote object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRulePatchSecurityGroupRuleProtocolTCPUDPRemoteIP',
                'SecurityGroupRulePatchSecurityGroupRuleProtocolTCPUDPRemoteCIDR',
                'SecurityGroupRulePatchSecurityGroupRuleProtocolTCPUDPRemoteSecurityGroupIdentity'
            ]))
        raise Exception(msg)


class SecurityGroupRulePrototype():
    """
    SecurityGroupRulePrototype.

    :attr str direction: The direction of traffic to enforce, either `inbound` or
          `outbound`.
    :attr str ip_version: (optional) The IP version to enforce. The format of
          `remote.address` or `remote.cidr_block` must match this field, if they are used.
          Alternatively, if `remote` references a security group, then this rule only
          applies to IP addresses (network interfaces) in that group matching this IP
          version.
    :attr str protocol: (optional) The protocol to enforce.
    :attr SecurityGroupRulePrototypeRemote remote: (optional) The IP addresses or
          security groups from which this rule will allow traffic (or to
          which, for outbound rules). Can be specified as an IP address, a CIDR block, or
          a
          security group. If omitted, then traffic will be allowed from any source (or to
          any
          source, for outbound rules).
    """

    def __init__(self,
                 direction: str,
                 *,
                 ip_version: str = None,
                 protocol: str = None,
                 remote: 'SecurityGroupRulePrototypeRemote' = None) -> None:
        """
        Initialize a SecurityGroupRulePrototype object.

        :param str direction: The direction of traffic to enforce, either `inbound`
               or `outbound`.
        :param str ip_version: (optional) The IP version to enforce. The format of
               `remote.address` or `remote.cidr_block` must match this field, if they are
               used. Alternatively, if `remote` references a security group, then this
               rule only applies to IP addresses (network interfaces) in that group
               matching this IP version.
        :param str protocol: (optional) The protocol to enforce.
        :param SecurityGroupRulePrototypeRemote remote: (optional) The IP addresses
               or security groups from which this rule will allow traffic (or to
               which, for outbound rules). Can be specified as an IP address, a CIDR
               block, or a
               security group. If omitted, then traffic will be allowed from any source
               (or to any
               source, for outbound rules).
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolAll',
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolICMP',
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolTCPUDP'
            ]))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SecurityGroupRulePrototype':
        """Initialize a SecurityGroupRulePrototype object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = (
            "Cannot convert dictionary into an instance of base class 'SecurityGroupRulePrototype'. "
            + "The discriminator value should map to a valid subclass: {1}"
        ).format(", ".join([
            'SecurityGroupRulePrototypeSecurityGroupRuleProtocolAll',
            'SecurityGroupRulePrototypeSecurityGroupRuleProtocolICMP',
            'SecurityGroupRulePrototypeSecurityGroupRuleProtocolTCPUDP'
        ]))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a SecurityGroupRulePrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping[
            'all'] = 'SecurityGroupRulePrototypeSecurityGroupRuleProtocolAll'
        mapping[
            'icmp'] = 'SecurityGroupRulePrototypeSecurityGroupRuleProtocolICMP'
        mapping[
            'tcp'] = 'SecurityGroupRulePrototypeSecurityGroupRuleProtocolTCPUDP'
        mapping[
            'udp'] = 'SecurityGroupRulePrototypeSecurityGroupRuleProtocolTCPUDP'
        disc_value = _dict.get('protocol')
        if disc_value is None:
            raise ValueError(
                'Discriminator property \'protocol\' not found in SecurityGroupRulePrototype JSON'
            )
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)

    class DirectionEnum(str, Enum):
        """
        The direction of traffic to enforce, either `inbound` or `outbound`.
        """
        INBOUND = 'inbound'
        OUTBOUND = 'outbound'

    class IpVersionEnum(str, Enum):
        """
        The IP version to enforce. The format of `remote.address` or `remote.cidr_block`
        must match this field, if they are used. Alternatively, if `remote` references a
        security group, then this rule only applies to IP addresses (network interfaces)
        in that group matching this IP version.
        """
        IPV4 = 'ipv4'

    class ProtocolEnum(str, Enum):
        """
        The protocol to enforce.
        """
        ALL = 'all'
        ICMP = 'icmp'
        TCP = 'tcp'
        UDP = 'udp'


class SecurityGroupRulePrototypeRemote():
    """
    The IP addresses or security groups from which this rule will allow traffic (or to
    which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
    security group. If omitted, then traffic will be allowed from any source (or to any
    source, for outbound rules).

    """

    def __init__(self) -> None:
        """
        Initialize a SecurityGroupRulePrototypeRemote object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRulePrototypeRemoteIP',
                'SecurityGroupRulePrototypeRemoteCIDR',
                'SecurityGroupRulePrototypeRemoteSecurityGroupIdentity'
            ]))
        raise Exception(msg)


class SecurityGroupRulePrototypeSecurityGroupRuleProtocolAllRemote():
    """
    The IP addresses or security groups from which this rule will allow traffic (or to
    which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
    security group. If omitted, then traffic will be allowed from any source (or to any
    source, for outbound rules).

    """

    def __init__(self) -> None:
        """
        Initialize a SecurityGroupRulePrototypeSecurityGroupRuleProtocolAllRemote object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolAllRemoteIP',
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolAllRemoteCIDR',
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolAllRemoteSecurityGroupIdentity'
            ]))
        raise Exception(msg)


class SecurityGroupRulePrototypeSecurityGroupRuleProtocolICMPRemote():
    """
    The IP addresses or security groups from which this rule will allow traffic (or to
    which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
    security group. If omitted, then traffic will be allowed from any source (or to any
    source, for outbound rules).

    """

    def __init__(self) -> None:
        """
        Initialize a SecurityGroupRulePrototypeSecurityGroupRuleProtocolICMPRemote object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolICMPRemoteIP',
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolICMPRemoteCIDR',
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolICMPRemoteSecurityGroupIdentity'
            ]))
        raise Exception(msg)


class SecurityGroupRulePrototypeSecurityGroupRuleProtocolTCPUDPRemote():
    """
    The IP addresses or security groups from which this rule will allow traffic (or to
    which, for outbound rules). Can be specified as an IP address, a CIDR block, or a
    security group. If omitted, then traffic will be allowed from any source (or to any
    source, for outbound rules).

    """

    def __init__(self) -> None:
        """
        Initialize a SecurityGroupRulePrototypeSecurityGroupRuleProtocolTCPUDPRemote object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolTCPUDPRemoteIP',
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolTCPUDPRemoteCIDR',
                'SecurityGroupRulePrototypeSecurityGroupRuleProtocolTCPUDPRemoteSecurityGroupIdentity'
            ]))
        raise Exception(msg)


class SecurityGroupRuleRemote():
    """
    The IP addresses or security groups from which this rule allows traffic (or to which,
    for outbound rules). Can be specified as an IP address, a CIDR block, or a security
    group. If omitted, then traffic is allowed from any source (or to any source, for
    outbound rules).

    """

    def __init__(self) -> None:
        """
        Initialize a SecurityGroupRuleRemote object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SecurityGroupRuleRemoteIP', 'SecurityGroupRuleRemoteCIDR',
                'SecurityGroupRuleRemoteSecurityGroupReference'
            ]))
        raise Exception(msg)


class Subnet():
    """
    Subnet.

    :attr int available_ipv4_address_count: The number of IPv4 addresses in this
          subnet that are not in-use, and have not been reserved by the user or the
          provider.
    :attr datetime created_at: The date and time that the subnet was created.
    :attr str crn: The CRN for this subnet.
    :attr str href: The URL for this subnet.
    :attr str id: The unique identifier for this subnet.
    :attr str ipv4_cidr_block: (optional) The IPv4 range of the subnet, expressed in
          CIDR format.
    :attr str name: The user-defined name for this subnet.
    :attr NetworkACLReference network_acl: The network ACL for this subnet.
    :attr PublicGatewayReference public_gateway: (optional) The public gateway to
          handle internet bound traffic for this subnet.
    :attr str status: The status of the subnet.
    :attr int total_ipv4_address_count: The total number of IPv4 addresses in this
          subnet.
          Note: This is calculated as 2<sup>(32 − prefix length)</sup>. For example, the
          prefix length `/24` gives:<br> 2<sup>(32 − 24)</sup> = 2<sup>8</sup> = 256
          addresses.
    :attr VPCReference vpc: The VPC this subnet is a part of.
    :attr ZoneReference zone: The zone this subnet resides in.
    """

    def __init__(self,
                 available_ipv4_address_count: int,
                 created_at: datetime,
                 crn: str,
                 href: str,
                 id: str,
                 name: str,
                 network_acl: 'NetworkACLReference',
                 status: str,
                 total_ipv4_address_count: int,
                 vpc: 'VPCReference',
                 zone: 'ZoneReference',
                 *,
                 ipv4_cidr_block: str = None,
                 public_gateway: 'PublicGatewayReference' = None) -> None:
        """
        Initialize a Subnet object.

        :param int available_ipv4_address_count: The number of IPv4 addresses in
               this subnet that are not in-use, and have not been reserved by the user or
               the provider.
        :param datetime created_at: The date and time that the subnet was created.
        :param str crn: The CRN for this subnet.
        :param str href: The URL for this subnet.
        :param str id: The unique identifier for this subnet.
        :param str name: The user-defined name for this subnet.
        :param NetworkACLReference network_acl: The network ACL for this subnet.
        :param str status: The status of the subnet.
        :param int total_ipv4_address_count: The total number of IPv4 addresses in
               this subnet.
               Note: This is calculated as 2<sup>(32 − prefix length)</sup>. For example,
               the prefix length `/24` gives:<br> 2<sup>(32 − 24)</sup> = 2<sup>8</sup> =
               256 addresses.
        :param VPCReference vpc: The VPC this subnet is a part of.
        :param ZoneReference zone: The zone this subnet resides in.
        :param str ipv4_cidr_block: (optional) The IPv4 range of the subnet,
               expressed in CIDR format.
        :param PublicGatewayReference public_gateway: (optional) The public gateway
               to handle internet bound traffic for this subnet.
        """
        self.available_ipv4_address_count = available_ipv4_address_count
        self.created_at = created_at
        self.crn = crn
        self.href = href
        self.id = id
        self.ipv4_cidr_block = ipv4_cidr_block
        self.name = name
        self.network_acl = network_acl
        self.public_gateway = public_gateway
        self.status = status
        self.total_ipv4_address_count = total_ipv4_address_count
        self.vpc = vpc
        self.zone = zone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'Subnet':
        """Initialize a Subnet object from a json dictionary."""
        args = {}
        if 'available_ipv4_address_count' in _dict:
            args['available_ipv4_address_count'] = _dict.get(
                'available_ipv4_address_count')
        else:
            raise ValueError(
                'Required property \'available_ipv4_address_count\' not present in Subnet JSON'
            )
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in Subnet JSON')
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in Subnet JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in Subnet JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in Subnet JSON')
        if 'ipv4_cidr_block' in _dict:
            args['ipv4_cidr_block'] = _dict.get('ipv4_cidr_block')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in Subnet JSON')
        if 'network_acl' in _dict:
            args['network_acl'] = NetworkACLReference.from_dict(
                _dict.get('network_acl'))
        else:
            raise ValueError(
                'Required property \'network_acl\' not present in Subnet JSON')
        if 'public_gateway' in _dict:
            args['public_gateway'] = PublicGatewayReference.from_dict(
                _dict.get('public_gateway'))
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in Subnet JSON')
        if 'total_ipv4_address_count' in _dict:
            args['total_ipv4_address_count'] = _dict.get(
                'total_ipv4_address_count')
        else:
            raise ValueError(
                'Required property \'total_ipv4_address_count\' not present in Subnet JSON'
            )
        if 'vpc' in _dict:
            args['vpc'] = VPCReference.from_dict(_dict.get('vpc'))
        else:
            raise ValueError(
                'Required property \'vpc\' not present in Subnet JSON')
        if 'zone' in _dict:
            args['zone'] = ZoneReference.from_dict(_dict.get('zone'))
        else:
            raise ValueError(
                'Required property \'zone\' not present in Subnet JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a Subnet object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'available_ipv4_address_count'
                  ) and self.available_ipv4_address_count is not None:
            _dict[
                'available_ipv4_address_count'] = self.available_ipv4_address_count
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self,
                   'ipv4_cidr_block') and self.ipv4_cidr_block is not None:
            _dict['ipv4_cidr_block'] = self.ipv4_cidr_block
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'network_acl') and self.network_acl is not None:
            _dict['network_acl'] = self.network_acl.to_dict()
        if hasattr(self, 'public_gateway') and self.public_gateway is not None:
            _dict['public_gateway'] = self.public_gateway.to_dict()
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'total_ipv4_address_count'
                  ) and self.total_ipv4_address_count is not None:
            _dict['total_ipv4_address_count'] = self.total_ipv4_address_count
        if hasattr(self, 'vpc') and self.vpc is not None:
            _dict['vpc'] = self.vpc.to_dict()
        if hasattr(self, 'zone') and self.zone is not None:
            _dict['zone'] = self.zone.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this Subnet object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'Subnet') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'Subnet') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class StatusEnum(str, Enum):
        """
        The status of the subnet.
        """
        AVAILABLE = 'available'
        DELETING = 'deleting'
        FAILED = 'failed'
        PENDING = 'pending'


class SubnetCollection():
    """
    SubnetCollection.

    :attr SubnetCollectionFirst first: A reference to the first page of resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr SubnetCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    :attr List[Subnet] subnets: Collection of subnets.
    """

    def __init__(self,
                 first: 'SubnetCollectionFirst',
                 limit: int,
                 subnets: List['Subnet'],
                 *,
                 next: 'SubnetCollectionNext' = None) -> None:
        """
        Initialize a SubnetCollection object.

        :param SubnetCollectionFirst first: A reference to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param List[Subnet] subnets: Collection of subnets.
        :param SubnetCollectionNext next: (optional) A reference to the next page
               of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.next = next
        self.subnets = subnets

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SubnetCollection':
        """Initialize a SubnetCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = SubnetCollectionFirst.from_dict(_dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in SubnetCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in SubnetCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = SubnetCollectionNext.from_dict(_dict.get('next'))
        if 'subnets' in _dict:
            args['subnets'] = [
                Subnet.from_dict(x) for x in _dict.get('subnets')
            ]
        else:
            raise ValueError(
                'Required property \'subnets\' not present in SubnetCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SubnetCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'subnets') and self.subnets is not None:
            _dict['subnets'] = [x.to_dict() for x in self.subnets]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SubnetCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SubnetCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SubnetCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SubnetCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a SubnetCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SubnetCollectionFirst':
        """Initialize a SubnetCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in SubnetCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SubnetCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SubnetCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SubnetCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SubnetCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SubnetCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a SubnetCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SubnetCollectionNext':
        """Initialize a SubnetCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in SubnetCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SubnetCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SubnetCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SubnetCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SubnetCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SubnetIdentity():
    """
    Identifies a subnet by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a SubnetIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SubnetIdentityById', 'SubnetIdentityByCRN',
                'SubnetIdentityByHref'
            ]))
        raise Exception(msg)


class SubnetPrototype():
    """
    SubnetPrototype.

    :attr str name: (optional) The user-defined name for this subnet. Names must be
          unique within the VPC the subnet resides in. If unspecified, the name will be a
          hyphenated list of randomly-selected words.
    :attr NetworkACLIdentity network_acl: (optional) The network ACL to use for this
          subnet.
    :attr PublicGatewayIdentity public_gateway: (optional) The public gateway to
          handle internet bound traffic for this subnet.
    :attr VPCIdentity vpc: The VPC the subnet is to be a part of.
    """

    def __init__(self,
                 vpc: 'VPCIdentity',
                 *,
                 name: str = None,
                 network_acl: 'NetworkACLIdentity' = None,
                 public_gateway: 'PublicGatewayIdentity' = None) -> None:
        """
        Initialize a SubnetPrototype object.

        :param VPCIdentity vpc: The VPC the subnet is to be a part of.
        :param str name: (optional) The user-defined name for this subnet. Names
               must be unique within the VPC the subnet resides in. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        :param NetworkACLIdentity network_acl: (optional) The network ACL to use
               for this subnet.
        :param PublicGatewayIdentity public_gateway: (optional) The public gateway
               to handle internet bound traffic for this subnet.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'SubnetPrototypeSubnetByTotalCount',
                'SubnetPrototypeSubnetByCIDR'
            ]))
        raise Exception(msg)


class SubnetReference():
    """
    SubnetReference.

    :attr str crn: The CRN for this subnet.
    :attr str href: The URL for this subnet.
    :attr str id: The unique identifier for this subnet.
    :attr str name: The user-defined name for this subnet.
    """

    def __init__(self, crn: str, href: str, id: str, name: str) -> None:
        """
        Initialize a SubnetReference object.

        :param str crn: The CRN for this subnet.
        :param str href: The URL for this subnet.
        :param str id: The unique identifier for this subnet.
        :param str name: The user-defined name for this subnet.
        """
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SubnetReference':
        """Initialize a SubnetReference object from a json dictionary."""
        args = {}
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in SubnetReference JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in SubnetReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in SubnetReference JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in SubnetReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SubnetReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SubnetReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SubnetReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SubnetReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPC():
    """
    VPC.

    :attr bool classic_access: Indicates whether this VPC is connected to Classic
          Infrastructure. If true, this VPC's resources have private network connectivity
          to the account's Classic Infrastructure resources. Only one VPC, per region, may
          be connected in this way. This value is set at creation and subsequently
          immutable.
    :attr datetime created_at: The date and time that the VPC was created.
    :attr str crn: The CRN for this VPC.
    :attr List[VPCCSESourceIP] cse_source_ips: (optional) Array of CSE ([Cloud
          Service
          Endpoint](https://cloud.ibm.com/docs/resources?topic=resources-service-endpoints))
          source IP addresses for the VPC. The VPC will have one CSE source IP address per
          zone.
    :attr NetworkACLReference default_network_acl: The default network ACL to use
          for subnets created in this VPC.
    :attr SecurityGroupReference default_security_group: The default security group
          to use for network interfaces created in this VPC.
    :attr str href: The URL for this VPC.
    :attr str id: The unique identifier for this VPC.
    :attr str name: The user-defined name for this VPC.
    :attr ResourceGroupReference resource_group: The resource group for this VPC.
    :attr str status: The status of this VPC.
    """

    def __init__(self,
                 classic_access: bool,
                 created_at: datetime,
                 crn: str,
                 default_network_acl: 'NetworkACLReference',
                 default_security_group: 'SecurityGroupReference',
                 href: str,
                 id: str,
                 name: str,
                 resource_group: 'ResourceGroupReference',
                 status: str,
                 *,
                 cse_source_ips: List['VPCCSESourceIP'] = None) -> None:
        """
        Initialize a VPC object.

        :param bool classic_access: Indicates whether this VPC is connected to
               Classic Infrastructure. If true, this VPC's resources have private network
               connectivity to the account's Classic Infrastructure resources. Only one
               VPC, per region, may be connected in this way. This value is set at
               creation and subsequently immutable.
        :param datetime created_at: The date and time that the VPC was created.
        :param str crn: The CRN for this VPC.
        :param NetworkACLReference default_network_acl: The default network ACL to
               use for subnets created in this VPC.
        :param SecurityGroupReference default_security_group: The default security
               group to use for network interfaces created in this VPC.
        :param str href: The URL for this VPC.
        :param str id: The unique identifier for this VPC.
        :param str name: The user-defined name for this VPC.
        :param ResourceGroupReference resource_group: The resource group for this
               VPC.
        :param str status: The status of this VPC.
        :param List[VPCCSESourceIP] cse_source_ips: (optional) Array of CSE ([Cloud
               Service
               Endpoint](https://cloud.ibm.com/docs/resources?topic=resources-service-endpoints))
               source IP addresses for the VPC. The VPC will have one CSE source IP
               address per zone.
        """
        self.classic_access = classic_access
        self.created_at = created_at
        self.crn = crn
        self.cse_source_ips = cse_source_ips
        self.default_network_acl = default_network_acl
        self.default_security_group = default_security_group
        self.href = href
        self.id = id
        self.name = name
        self.resource_group = resource_group
        self.status = status

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPC':
        """Initialize a VPC object from a json dictionary."""
        args = {}
        if 'classic_access' in _dict:
            args['classic_access'] = _dict.get('classic_access')
        else:
            raise ValueError(
                'Required property \'classic_access\' not present in VPC JSON')
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in VPC JSON')
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in VPC JSON')
        if 'cse_source_ips' in _dict:
            args['cse_source_ips'] = [
                VPCCSESourceIP.from_dict(x) for x in _dict.get('cse_source_ips')
            ]
        if 'default_network_acl' in _dict:
            args['default_network_acl'] = NetworkACLReference.from_dict(
                _dict.get('default_network_acl'))
        else:
            raise ValueError(
                'Required property \'default_network_acl\' not present in VPC JSON'
            )
        if 'default_security_group' in _dict:
            args['default_security_group'] = SecurityGroupReference.from_dict(
                _dict.get('default_security_group'))
        else:
            raise ValueError(
                'Required property \'default_security_group\' not present in VPC JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VPC JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError('Required property \'id\' not present in VPC JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VPC JSON')
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(
                _dict.get('resource_group'))
        else:
            raise ValueError(
                'Required property \'resource_group\' not present in VPC JSON')
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in VPC JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPC object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'classic_access') and self.classic_access is not None:
            _dict['classic_access'] = self.classic_access
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'cse_source_ips') and self.cse_source_ips is not None:
            _dict['cse_source_ips'] = [x.to_dict() for x in self.cse_source_ips]
        if hasattr(
                self,
                'default_network_acl') and self.default_network_acl is not None:
            _dict['default_network_acl'] = self.default_network_acl.to_dict()
        if hasattr(self, 'default_security_group'
                  ) and self.default_security_group is not None:
            _dict[
                'default_security_group'] = self.default_security_group.to_dict(
                )
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPC object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPC') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPC') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class StatusEnum(str, Enum):
        """
        The status of this VPC.
        """
        AVAILABLE = 'available'
        DELETING = 'deleting'
        FAILED = 'failed'
        PENDING = 'pending'


class VPCCSESourceIP():
    """
    VPCCSESourceIP.

    :attr IP ip: The Cloud Service Endpoint source IP address for this zone.
    :attr ZoneReference zone: The zone this Cloud Service Endpoint source IP belongs
          to.
    """

    def __init__(self, ip: 'IP', zone: 'ZoneReference') -> None:
        """
        Initialize a VPCCSESourceIP object.

        :param IP ip: The Cloud Service Endpoint source IP address for this zone.
        :param ZoneReference zone: The zone this Cloud Service Endpoint source IP
               belongs to.
        """
        self.ip = ip
        self.zone = zone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPCCSESourceIP':
        """Initialize a VPCCSESourceIP object from a json dictionary."""
        args = {}
        if 'ip' in _dict:
            args['ip'] = IP.from_dict(_dict.get('ip'))
        else:
            raise ValueError(
                'Required property \'ip\' not present in VPCCSESourceIP JSON')
        if 'zone' in _dict:
            args['zone'] = ZoneReference.from_dict(_dict.get('zone'))
        else:
            raise ValueError(
                'Required property \'zone\' not present in VPCCSESourceIP JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPCCSESourceIP object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'ip') and self.ip is not None:
            _dict['ip'] = self.ip.to_dict()
        if hasattr(self, 'zone') and self.zone is not None:
            _dict['zone'] = self.zone.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPCCSESourceIP object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPCCSESourceIP') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPCCSESourceIP') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPCCollection():
    """
    VPCCollection.

    :attr VPCCollectionFirst first: A reference to the first page of resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr VPCCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    :attr List[VPC] vpcs: Collection of VPCs.
    """

    def __init__(self,
                 first: 'VPCCollectionFirst',
                 limit: int,
                 vpcs: List['VPC'],
                 *,
                 next: 'VPCCollectionNext' = None) -> None:
        """
        Initialize a VPCCollection object.

        :param VPCCollectionFirst first: A reference to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param List[VPC] vpcs: Collection of VPCs.
        :param VPCCollectionNext next: (optional) A reference to the next page of
               resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.next = next
        self.vpcs = vpcs

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPCCollection':
        """Initialize a VPCCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = VPCCollectionFirst.from_dict(_dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in VPCCollection JSON')
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in VPCCollection JSON')
        if 'next' in _dict:
            args['next'] = VPCCollectionNext.from_dict(_dict.get('next'))
        if 'vpcs' in _dict:
            args['vpcs'] = [VPC.from_dict(x) for x in _dict.get('vpcs')]
        else:
            raise ValueError(
                'Required property \'vpcs\' not present in VPCCollection JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPCCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'vpcs') and self.vpcs is not None:
            _dict['vpcs'] = [x.to_dict() for x in self.vpcs]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPCCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPCCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPCCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPCCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a VPCCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPCCollectionFirst':
        """Initialize a VPCCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VPCCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPCCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPCCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPCCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPCCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPCCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a VPCCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPCCollectionNext':
        """Initialize a VPCCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VPCCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPCCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPCCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPCCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPCCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPCIdentity():
    """
    Identifies a VPC by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a VPCIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(
                ['VPCIdentityById', 'VPCIdentityByCRN', 'VPCIdentityByHref']))
        raise Exception(msg)


class VPCReference():
    """
    VPCReference.

    :attr str crn: The CRN for this VPC.
    :attr str href: The URL for this VPC.
    :attr str id: The unique identifier for this VPC.
    :attr str name: The user-defined name for this VPC.
    """

    def __init__(self, crn: str, href: str, id: str, name: str) -> None:
        """
        Initialize a VPCReference object.

        :param str crn: The CRN for this VPC.
        :param str href: The URL for this VPC.
        :param str id: The unique identifier for this VPC.
        :param str name: The user-defined name for this VPC.
        """
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPCReference':
        """Initialize a VPCReference object from a json dictionary."""
        args = {}
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in VPCReference JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VPCReference JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in VPCReference JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VPCReference JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPCReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPCReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPCReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPCReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPNGateway():
    """
    VPNGateway.

    :attr List[VPNGatewayConnectionReference] connections: Collection of references
          to VPN connections.
    :attr datetime created_at: The date and time that this VPN gateway was created.
    :attr str crn: The VPN gateway's CRN.
    :attr str href: The VPN gateway's canonical URL.
    :attr str id: The unique identifier for this VPN gateway.
    :attr str name: The user-defined name for this VPN gateway.
    :attr VPNGatewayPublicIp public_ip: The public IP address assigned to this VPN
          gateway.
    :attr ResourceGroupReference resource_group: The resource group for this VPN
          gateway.
    :attr str status: The status of the VPN gateway.
    :attr SubnetReference subnet:
    """

    def __init__(self, connections: List['VPNGatewayConnectionReference'],
                 created_at: datetime, crn: str, href: str, id: str, name: str,
                 public_ip: 'VPNGatewayPublicIp',
                 resource_group: 'ResourceGroupReference', status: str,
                 subnet: 'SubnetReference') -> None:
        """
        Initialize a VPNGateway object.

        :param List[VPNGatewayConnectionReference] connections: Collection of
               references to VPN connections.
        :param datetime created_at: The date and time that this VPN gateway was
               created.
        :param str crn: The VPN gateway's CRN.
        :param str href: The VPN gateway's canonical URL.
        :param str id: The unique identifier for this VPN gateway.
        :param str name: The user-defined name for this VPN gateway.
        :param VPNGatewayPublicIp public_ip: The public IP address assigned to this
               VPN gateway.
        :param ResourceGroupReference resource_group: The resource group for this
               VPN gateway.
        :param str status: The status of the VPN gateway.
        :param SubnetReference subnet:
        """
        self.connections = connections
        self.created_at = created_at
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name
        self.public_ip = public_ip
        self.resource_group = resource_group
        self.status = status
        self.subnet = subnet

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGateway':
        """Initialize a VPNGateway object from a json dictionary."""
        args = {}
        if 'connections' in _dict:
            args['connections'] = [
                VPNGatewayConnectionReference.from_dict(x)
                for x in _dict.get('connections')
            ]
        else:
            raise ValueError(
                'Required property \'connections\' not present in VPNGateway JSON'
            )
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in VPNGateway JSON'
            )
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in VPNGateway JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VPNGateway JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in VPNGateway JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VPNGateway JSON')
        if 'public_ip' in _dict:
            args['public_ip'] = VPNGatewayPublicIp.from_dict(
                _dict.get('public_ip'))
        else:
            raise ValueError(
                'Required property \'public_ip\' not present in VPNGateway JSON'
            )
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(
                _dict.get('resource_group'))
        else:
            raise ValueError(
                'Required property \'resource_group\' not present in VPNGateway JSON'
            )
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in VPNGateway JSON')
        if 'subnet' in _dict:
            args['subnet'] = SubnetReference.from_dict(_dict.get('subnet'))
        else:
            raise ValueError(
                'Required property \'subnet\' not present in VPNGateway JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGateway object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'connections') and self.connections is not None:
            _dict['connections'] = [x.to_dict() for x in self.connections]
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'public_ip') and self.public_ip is not None:
            _dict['public_ip'] = self.public_ip.to_dict()
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'subnet') and self.subnet is not None:
            _dict['subnet'] = self.subnet.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGateway object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGateway') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGateway') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class StatusEnum(str, Enum):
        """
        The status of the VPN gateway.
        """
        AVAILABLE = 'available'
        DELETING = 'deleting'
        FAILED = 'failed'
        PENDING = 'pending'


class VPNGatewayCollection():
    """
    VPNGatewayCollection.

    :attr VPNGatewayCollectionFirst first: A reference to the first page of
          resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr VPNGatewayCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    :attr int total_count: The total number of resources across all pages.
    :attr List[VPNGateway] vpn_gateways: Collection of VPN gateways.
    """

    def __init__(self,
                 first: 'VPNGatewayCollectionFirst',
                 limit: int,
                 total_count: int,
                 vpn_gateways: List['VPNGateway'],
                 *,
                 next: 'VPNGatewayCollectionNext' = None) -> None:
        """
        Initialize a VPNGatewayCollection object.

        :param VPNGatewayCollectionFirst first: A reference to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param int total_count: The total number of resources across all pages.
        :param List[VPNGateway] vpn_gateways: Collection of VPN gateways.
        :param VPNGatewayCollectionNext next: (optional) A reference to the next
               page of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.next = next
        self.total_count = total_count
        self.vpn_gateways = vpn_gateways

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayCollection':
        """Initialize a VPNGatewayCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = VPNGatewayCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in VPNGatewayCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in VPNGatewayCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = VPNGatewayCollectionNext.from_dict(_dict.get('next'))
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError(
                'Required property \'total_count\' not present in VPNGatewayCollection JSON'
            )
        if 'vpn_gateways' in _dict:
            args['vpn_gateways'] = [
                VPNGateway.from_dict(x) for x in _dict.get('vpn_gateways')
            ]
        else:
            raise ValueError(
                'Required property \'vpn_gateways\' not present in VPNGatewayCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        if hasattr(self, 'vpn_gateways') and self.vpn_gateways is not None:
            _dict['vpn_gateways'] = [x.to_dict() for x in self.vpn_gateways]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPNGatewayCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a VPNGatewayCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayCollectionFirst':
        """Initialize a VPNGatewayCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VPNGatewayCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPNGatewayCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a VPNGatewayCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayCollectionNext':
        """Initialize a VPNGatewayCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VPNGatewayCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPNGatewayConnection():
    """
    VPNGatewayConnection.

    :attr bool admin_state_up: If set to false, the VPN connection is shut down.
    :attr str authentication_mode: The authentication mode. Only `psk` is currently
          supported.
    :attr datetime created_at: The date and time that this VPN connection was
          created.
    :attr VPNGatewayConnectionDPD dead_peer_detection: The Dead Peer Detection
          settings.
    :attr str href: The VPN connection's canonical URL.
    :attr str id: The unique identifier for this VPN connection.
    :attr IKEPolicyIdentity ike_policy: (optional) Optional IKE policy
          configuration. The absence of a policy indicates autonegotiation.
    :attr IPsecPolicyIdentity ipsec_policy: (optional) Optional IPsec policy
          configuration. The absence of a policy indicates
          autonegotiation.
    :attr List[str] local_cidrs: A collection of local CIDRs for this resource.
    :attr str name: The user-defined name for this VPN connection.
    :attr str peer_address: The IP address of the peer VPN gateway.
    :attr List[str] peer_cidrs: A collection of peer CIDRs for this resource.
    :attr str psk: The preshared key.
    :attr str route_mode: The routing mode. Only `policy` is currently supported.
    :attr str status: The status of a VPN connection.
    """

    def __init__(self,
                 admin_state_up: bool,
                 authentication_mode: str,
                 created_at: datetime,
                 dead_peer_detection: 'VPNGatewayConnectionDPD',
                 href: str,
                 id: str,
                 local_cidrs: List[str],
                 name: str,
                 peer_address: str,
                 peer_cidrs: List[str],
                 psk: str,
                 route_mode: str,
                 status: str,
                 *,
                 ike_policy: 'IKEPolicyIdentity' = None,
                 ipsec_policy: 'IPsecPolicyIdentity' = None) -> None:
        """
        Initialize a VPNGatewayConnection object.

        :param bool admin_state_up: If set to false, the VPN connection is shut
               down.
        :param str authentication_mode: The authentication mode. Only `psk` is
               currently supported.
        :param datetime created_at: The date and time that this VPN connection was
               created.
        :param VPNGatewayConnectionDPD dead_peer_detection: The Dead Peer Detection
               settings.
        :param str href: The VPN connection's canonical URL.
        :param str id: The unique identifier for this VPN connection.
        :param List[str] local_cidrs: A collection of local CIDRs for this
               resource.
        :param str name: The user-defined name for this VPN connection.
        :param str peer_address: The IP address of the peer VPN gateway.
        :param List[str] peer_cidrs: A collection of peer CIDRs for this resource.
        :param str psk: The preshared key.
        :param str route_mode: The routing mode. Only `policy` is currently
               supported.
        :param str status: The status of a VPN connection.
        :param IKEPolicyIdentity ike_policy: (optional) Optional IKE policy
               configuration. The absence of a policy indicates autonegotiation.
        :param IPsecPolicyIdentity ipsec_policy: (optional) Optional IPsec policy
               configuration. The absence of a policy indicates
               autonegotiation.
        """
        self.admin_state_up = admin_state_up
        self.authentication_mode = authentication_mode
        self.created_at = created_at
        self.dead_peer_detection = dead_peer_detection
        self.href = href
        self.id = id
        self.ike_policy = ike_policy
        self.ipsec_policy = ipsec_policy
        self.local_cidrs = local_cidrs
        self.name = name
        self.peer_address = peer_address
        self.peer_cidrs = peer_cidrs
        self.psk = psk
        self.route_mode = route_mode
        self.status = status

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayConnection':
        """Initialize a VPNGatewayConnection object from a json dictionary."""
        args = {}
        if 'admin_state_up' in _dict:
            args['admin_state_up'] = _dict.get('admin_state_up')
        else:
            raise ValueError(
                'Required property \'admin_state_up\' not present in VPNGatewayConnection JSON'
            )
        if 'authentication_mode' in _dict:
            args['authentication_mode'] = _dict.get('authentication_mode')
        else:
            raise ValueError(
                'Required property \'authentication_mode\' not present in VPNGatewayConnection JSON'
            )
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in VPNGatewayConnection JSON'
            )
        if 'dead_peer_detection' in _dict:
            args['dead_peer_detection'] = VPNGatewayConnectionDPD.from_dict(
                _dict.get('dead_peer_detection'))
        else:
            raise ValueError(
                'Required property \'dead_peer_detection\' not present in VPNGatewayConnection JSON'
            )
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VPNGatewayConnection JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in VPNGatewayConnection JSON'
            )
        if 'ike_policy' in _dict:
            args['ike_policy'] = _dict.get('ike_policy')
        if 'ipsec_policy' in _dict:
            args['ipsec_policy'] = _dict.get('ipsec_policy')
        if 'local_cidrs' in _dict:
            args['local_cidrs'] = _dict.get('local_cidrs')
        else:
            raise ValueError(
                'Required property \'local_cidrs\' not present in VPNGatewayConnection JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VPNGatewayConnection JSON'
            )
        if 'peer_address' in _dict:
            args['peer_address'] = _dict.get('peer_address')
        else:
            raise ValueError(
                'Required property \'peer_address\' not present in VPNGatewayConnection JSON'
            )
        if 'peer_cidrs' in _dict:
            args['peer_cidrs'] = _dict.get('peer_cidrs')
        else:
            raise ValueError(
                'Required property \'peer_cidrs\' not present in VPNGatewayConnection JSON'
            )
        if 'psk' in _dict:
            args['psk'] = _dict.get('psk')
        else:
            raise ValueError(
                'Required property \'psk\' not present in VPNGatewayConnection JSON'
            )
        if 'route_mode' in _dict:
            args['route_mode'] = _dict.get('route_mode')
        else:
            raise ValueError(
                'Required property \'route_mode\' not present in VPNGatewayConnection JSON'
            )
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in VPNGatewayConnection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayConnection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'admin_state_up') and self.admin_state_up is not None:
            _dict['admin_state_up'] = self.admin_state_up
        if hasattr(
                self,
                'authentication_mode') and self.authentication_mode is not None:
            _dict['authentication_mode'] = self.authentication_mode
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(
                self,
                'dead_peer_detection') and self.dead_peer_detection is not None:
            _dict['dead_peer_detection'] = self.dead_peer_detection.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'ike_policy') and self.ike_policy is not None:
            _dict['ike_policy'] = self.ike_policy
        if hasattr(self, 'ipsec_policy') and self.ipsec_policy is not None:
            _dict['ipsec_policy'] = self.ipsec_policy
        if hasattr(self, 'local_cidrs') and self.local_cidrs is not None:
            _dict['local_cidrs'] = self.local_cidrs
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'peer_address') and self.peer_address is not None:
            _dict['peer_address'] = self.peer_address
        if hasattr(self, 'peer_cidrs') and self.peer_cidrs is not None:
            _dict['peer_cidrs'] = self.peer_cidrs
        if hasattr(self, 'psk') and self.psk is not None:
            _dict['psk'] = self.psk
        if hasattr(self, 'route_mode') and self.route_mode is not None:
            _dict['route_mode'] = self.route_mode
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayConnection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayConnection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayConnection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class AuthenticationModeEnum(str, Enum):
        """
        The authentication mode. Only `psk` is currently supported.
        """
        PSK = 'psk'

    class RouteModeEnum(str, Enum):
        """
        The routing mode. Only `policy` is currently supported.
        """
        POLICY = 'policy'

    class StatusEnum(str, Enum):
        """
        The status of a VPN connection.
        """
        DOWN = 'down'
        UP = 'up'


class VPNGatewayConnectionCollection():
    """
    Collection of VPN connections in a VPN gateway.

    :attr List[VPNGatewayConnection] connections: Array of VPN connections.
    """

    def __init__(self, connections: List['VPNGatewayConnection']) -> None:
        """
        Initialize a VPNGatewayConnectionCollection object.

        :param List[VPNGatewayConnection] connections: Array of VPN connections.
        """
        self.connections = connections

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayConnectionCollection':
        """Initialize a VPNGatewayConnectionCollection object from a json dictionary."""
        args = {}
        if 'connections' in _dict:
            args['connections'] = [
                VPNGatewayConnection.from_dict(x)
                for x in _dict.get('connections')
            ]
        else:
            raise ValueError(
                'Required property \'connections\' not present in VPNGatewayConnectionCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayConnectionCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'connections') and self.connections is not None:
            _dict['connections'] = [x.to_dict() for x in self.connections]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayConnectionCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayConnectionCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayConnectionCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPNGatewayConnectionDPD():
    """
    The Dead Peer Detection settings.

    :attr str action: Dead Peer Detection actions.
    :attr int interval: Dead Peer Detection interval in seconds.
    :attr int timeout: Dead Peer Detection timeout in seconds. Must be at least the
          interval.
    """

    def __init__(self, action: str, interval: int, timeout: int) -> None:
        """
        Initialize a VPNGatewayConnectionDPD object.

        :param str action: Dead Peer Detection actions.
        :param int interval: Dead Peer Detection interval in seconds.
        :param int timeout: Dead Peer Detection timeout in seconds. Must be at
               least the interval.
        """
        self.action = action
        self.interval = interval
        self.timeout = timeout

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayConnectionDPD':
        """Initialize a VPNGatewayConnectionDPD object from a json dictionary."""
        args = {}
        if 'action' in _dict:
            args['action'] = _dict.get('action')
        else:
            raise ValueError(
                'Required property \'action\' not present in VPNGatewayConnectionDPD JSON'
            )
        if 'interval' in _dict:
            args['interval'] = _dict.get('interval')
        else:
            raise ValueError(
                'Required property \'interval\' not present in VPNGatewayConnectionDPD JSON'
            )
        if 'timeout' in _dict:
            args['timeout'] = _dict.get('timeout')
        else:
            raise ValueError(
                'Required property \'timeout\' not present in VPNGatewayConnectionDPD JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayConnectionDPD object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'action') and self.action is not None:
            _dict['action'] = self.action
        if hasattr(self, 'interval') and self.interval is not None:
            _dict['interval'] = self.interval
        if hasattr(self, 'timeout') and self.timeout is not None:
            _dict['timeout'] = self.timeout
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayConnectionDPD object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayConnectionDPD') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayConnectionDPD') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ActionEnum(str, Enum):
        """
        Dead Peer Detection actions.
        """
        CLEAR = 'clear'
        HOLD = 'hold'
        NONE = 'none'
        RESTART = 'restart'


class VPNGatewayConnectionDPDPrototype():
    """
    The Dead Peer Detection settings.

    :attr str action: (optional) Dead Peer Detection actions.
    :attr int interval: (optional) Dead Peer Detection interval in seconds.
    :attr int timeout: (optional) Dead Peer Detection timeout in seconds. Must be at
          least the interval.
    """

    def __init__(self,
                 *,
                 action: str = None,
                 interval: int = None,
                 timeout: int = None) -> None:
        """
        Initialize a VPNGatewayConnectionDPDPrototype object.

        :param str action: (optional) Dead Peer Detection actions.
        :param int interval: (optional) Dead Peer Detection interval in seconds.
        :param int timeout: (optional) Dead Peer Detection timeout in seconds. Must
               be at least the interval.
        """
        self.action = action
        self.interval = interval
        self.timeout = timeout

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayConnectionDPDPrototype':
        """Initialize a VPNGatewayConnectionDPDPrototype object from a json dictionary."""
        args = {}
        if 'action' in _dict:
            args['action'] = _dict.get('action')
        if 'interval' in _dict:
            args['interval'] = _dict.get('interval')
        if 'timeout' in _dict:
            args['timeout'] = _dict.get('timeout')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayConnectionDPDPrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'action') and self.action is not None:
            _dict['action'] = self.action
        if hasattr(self, 'interval') and self.interval is not None:
            _dict['interval'] = self.interval
        if hasattr(self, 'timeout') and self.timeout is not None:
            _dict['timeout'] = self.timeout
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayConnectionDPDPrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayConnectionDPDPrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayConnectionDPDPrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ActionEnum(str, Enum):
        """
        Dead Peer Detection actions.
        """
        CLEAR = 'clear'
        HOLD = 'hold'
        NONE = 'none'
        RESTART = 'restart'


class VPNGatewayConnectionLocalCIDRs():
    """
    VPNGatewayConnectionLocalCIDRs.

    :attr List[str] local_cidrs: (optional) A collection of local CIDRs for this
          resource.
    """

    def __init__(self, *, local_cidrs: List[str] = None) -> None:
        """
        Initialize a VPNGatewayConnectionLocalCIDRs object.

        :param List[str] local_cidrs: (optional) A collection of local CIDRs for
               this resource.
        """
        self.local_cidrs = local_cidrs

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayConnectionLocalCIDRs':
        """Initialize a VPNGatewayConnectionLocalCIDRs object from a json dictionary."""
        args = {}
        if 'local_cidrs' in _dict:
            args['local_cidrs'] = _dict.get('local_cidrs')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayConnectionLocalCIDRs object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'local_cidrs') and self.local_cidrs is not None:
            _dict['local_cidrs'] = self.local_cidrs
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayConnectionLocalCIDRs object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayConnectionLocalCIDRs') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayConnectionLocalCIDRs') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPNGatewayConnectionPeerCIDRs():
    """
    VPNGatewayConnectionPeerCIDRs.

    :attr List[str] peer_cidrs: (optional) A collection of peer CIDRs for this
          resource.
    """

    def __init__(self, *, peer_cidrs: List[str] = None) -> None:
        """
        Initialize a VPNGatewayConnectionPeerCIDRs object.

        :param List[str] peer_cidrs: (optional) A collection of peer CIDRs for this
               resource.
        """
        self.peer_cidrs = peer_cidrs

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayConnectionPeerCIDRs':
        """Initialize a VPNGatewayConnectionPeerCIDRs object from a json dictionary."""
        args = {}
        if 'peer_cidrs' in _dict:
            args['peer_cidrs'] = _dict.get('peer_cidrs')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayConnectionPeerCIDRs object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'peer_cidrs') and self.peer_cidrs is not None:
            _dict['peer_cidrs'] = self.peer_cidrs
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayConnectionPeerCIDRs object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayConnectionPeerCIDRs') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayConnectionPeerCIDRs') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPNGatewayConnectionReference():
    """
    VPNGatewayConnectionReference.

    :attr str href: The VPN connection's canonical URL.
    :attr str id: The unique identifier for this VPN connection.
    :attr str name: The user-defined name for this VPN connection.
    """

    def __init__(self, href: str, id: str, name: str) -> None:
        """
        Initialize a VPNGatewayConnectionReference object.

        :param str href: The VPN connection's canonical URL.
        :param str id: The unique identifier for this VPN connection.
        :param str name: The user-defined name for this VPN connection.
        """
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayConnectionReference':
        """Initialize a VPNGatewayConnectionReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VPNGatewayConnectionReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in VPNGatewayConnectionReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VPNGatewayConnectionReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayConnectionReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayConnectionReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayConnectionReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayConnectionReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VPNGatewayPublicIp():
    """
    The public IP address assigned to this VPN gateway.

    :attr str address: The IP address. This property may add support for IPv6
          addresses in the future. When processing a value in this property, verify that
          the address is in an expected format. If it is not, log an error. Optionally
          halt processing and surface the error, or bypass the resource on which the
          unexpected IP address format was encountered.
    """

    def __init__(self, address: str) -> None:
        """
        Initialize a VPNGatewayPublicIp object.

        :param str address: The IP address. This property may add support for IPv6
               addresses in the future. When processing a value in this property, verify
               that the address is in an expected format. If it is not, log an error.
               Optionally halt processing and surface the error, or bypass the resource on
               which the unexpected IP address format was encountered.
        """
        self.address = address

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VPNGatewayPublicIp':
        """Initialize a VPNGatewayPublicIp object from a json dictionary."""
        args = {}
        if 'address' in _dict:
            args['address'] = _dict.get('address')
        else:
            raise ValueError(
                'Required property \'address\' not present in VPNGatewayPublicIp JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VPNGatewayPublicIp object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'address') and self.address is not None:
            _dict['address'] = self.address
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VPNGatewayPublicIp object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VPNGatewayPublicIp') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VPNGatewayPublicIp') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class Volume():
    """
    Volume.

    :attr int capacity: The capacity of the volume in gigabytes. Note that the
          specified minimum and maximum capacity values for creating or updating volumes
          may expand in the future.
    :attr datetime created_at: The date and time that the volume was created.
    :attr str crn: The CRN for this volume.
    :attr str encryption: The type of encryption used on the volume.
    :attr EncryptionKeyReference encryption_key: (optional) A reference to the root
          key used to wrap the data encryption key for the volume.
          This property will be present for volumes with an `encryption` type of
          `user_managed`.
    :attr str href: The URL for this volume.
    :attr str id: The unique identifier for this volume.
    :attr int iops: The bandwidth for the volume.
    :attr str name: The unique user-defined name for this volume.
    :attr VolumeProfileReference profile: The profile this volume uses.
    :attr ResourceGroupReference resource_group: The resource group for this volume.
    :attr str status: The status of the volume.
    :attr List[VolumeAttachmentReferenceVolumeContext] volume_attachments: The
          collection of volume attachments attaching instances to the volume.
    :attr ZoneReference zone: The zone this volume resides in.
    """

    def __init__(
            self,
            capacity: int,
            created_at: datetime,
            crn: str,
            encryption: str,
            href: str,
            id: str,
            iops: int,
            name: str,
            profile: 'VolumeProfileReference',
            resource_group: 'ResourceGroupReference',
            status: str,
            volume_attachments: List['VolumeAttachmentReferenceVolumeContext'],
            zone: 'ZoneReference',
            *,
            encryption_key: 'EncryptionKeyReference' = None) -> None:
        """
        Initialize a Volume object.

        :param int capacity: The capacity of the volume in gigabytes. Note that the
               specified minimum and maximum capacity values for creating or updating
               volumes may expand in the future.
        :param datetime created_at: The date and time that the volume was created.
        :param str crn: The CRN for this volume.
        :param str encryption: The type of encryption used on the volume.
        :param str href: The URL for this volume.
        :param str id: The unique identifier for this volume.
        :param int iops: The bandwidth for the volume.
        :param str name: The unique user-defined name for this volume.
        :param VolumeProfileReference profile: The profile this volume uses.
        :param ResourceGroupReference resource_group: The resource group for this
               volume.
        :param str status: The status of the volume.
        :param List[VolumeAttachmentReferenceVolumeContext] volume_attachments: The
               collection of volume attachments attaching instances to the volume.
        :param ZoneReference zone: The zone this volume resides in.
        :param EncryptionKeyReference encryption_key: (optional) A reference to the
               root key used to wrap the data encryption key for the volume.
               This property will be present for volumes with an `encryption` type of
               `user_managed`.
        """
        self.capacity = capacity
        self.created_at = created_at
        self.crn = crn
        self.encryption = encryption
        self.encryption_key = encryption_key
        self.href = href
        self.id = id
        self.iops = iops
        self.name = name
        self.profile = profile
        self.resource_group = resource_group
        self.status = status
        self.volume_attachments = volume_attachments
        self.zone = zone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'Volume':
        """Initialize a Volume object from a json dictionary."""
        args = {}
        if 'capacity' in _dict:
            args['capacity'] = _dict.get('capacity')
        else:
            raise ValueError(
                'Required property \'capacity\' not present in Volume JSON')
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in Volume JSON')
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in Volume JSON')
        if 'encryption' in _dict:
            args['encryption'] = _dict.get('encryption')
        else:
            raise ValueError(
                'Required property \'encryption\' not present in Volume JSON')
        if 'encryption_key' in _dict:
            args['encryption_key'] = EncryptionKeyReference.from_dict(
                _dict.get('encryption_key'))
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in Volume JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in Volume JSON')
        if 'iops' in _dict:
            args['iops'] = _dict.get('iops')
        else:
            raise ValueError(
                'Required property \'iops\' not present in Volume JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in Volume JSON')
        if 'profile' in _dict:
            args['profile'] = VolumeProfileReference.from_dict(
                _dict.get('profile'))
        else:
            raise ValueError(
                'Required property \'profile\' not present in Volume JSON')
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(
                _dict.get('resource_group'))
        else:
            raise ValueError(
                'Required property \'resource_group\' not present in Volume JSON'
            )
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in Volume JSON')
        if 'volume_attachments' in _dict:
            args['volume_attachments'] = [
                VolumeAttachmentReferenceVolumeContext.from_dict(x)
                for x in _dict.get('volume_attachments')
            ]
        else:
            raise ValueError(
                'Required property \'volume_attachments\' not present in Volume JSON'
            )
        if 'zone' in _dict:
            args['zone'] = ZoneReference.from_dict(_dict.get('zone'))
        else:
            raise ValueError(
                'Required property \'zone\' not present in Volume JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a Volume object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'capacity') and self.capacity is not None:
            _dict['capacity'] = self.capacity
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'encryption') and self.encryption is not None:
            _dict['encryption'] = self.encryption
        if hasattr(self, 'encryption_key') and self.encryption_key is not None:
            _dict['encryption_key'] = self.encryption_key.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'iops') and self.iops is not None:
            _dict['iops'] = self.iops
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'profile') and self.profile is not None:
            _dict['profile'] = self.profile.to_dict()
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(
                self,
                'volume_attachments') and self.volume_attachments is not None:
            _dict['volume_attachments'] = [
                x.to_dict() for x in self.volume_attachments
            ]
        if hasattr(self, 'zone') and self.zone is not None:
            _dict['zone'] = self.zone.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this Volume object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'Volume') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'Volume') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class EncryptionEnum(str, Enum):
        """
        The type of encryption used on the volume.
        """
        PROVIDER_MANAGED = 'provider_managed'
        USER_MANAGED = 'user_managed'

    class StatusEnum(str, Enum):
        """
        The status of the volume.
        """
        AVAILABLE = 'available'
        FAILED = 'failed'
        PENDING = 'pending'
        PENDING_DELETION = 'pending_deletion'


class VolumeAttachment():
    """
    VolumeAttachment.

    :attr datetime created_at: The date and time that the volume was attached.
    :attr bool delete_volume_on_instance_delete: (optional) If set to true, when
          deleting the instance the volume will also be deleted.
    :attr VolumeAttachmentDevice device: (optional) Information about how the volume
          is exposed to the instance operating system.
          This property may be absent if the volume attachment's `status` is not
          `attached`.
    :attr str href: The URL for this volume attachment.
    :attr str id: The unique identifier for this volume attachment.
    :attr str name: The user-defined name for this volume attachment.
    :attr str status: The status of this volume attachment.
    :attr str type: The type of volume attachment.
    :attr VolumeReference volume: The attached volume.
    """

    def __init__(self,
                 created_at: datetime,
                 href: str,
                 id: str,
                 name: str,
                 status: str,
                 type: str,
                 volume: 'VolumeReference',
                 *,
                 delete_volume_on_instance_delete: bool = None,
                 device: 'VolumeAttachmentDevice' = None) -> None:
        """
        Initialize a VolumeAttachment object.

        :param datetime created_at: The date and time that the volume was attached.
        :param str href: The URL for this volume attachment.
        :param str id: The unique identifier for this volume attachment.
        :param str name: The user-defined name for this volume attachment.
        :param str status: The status of this volume attachment.
        :param str type: The type of volume attachment.
        :param VolumeReference volume: The attached volume.
        :param bool delete_volume_on_instance_delete: (optional) If set to true,
               when deleting the instance the volume will also be deleted.
        :param VolumeAttachmentDevice device: (optional) Information about how the
               volume is exposed to the instance operating system.
               This property may be absent if the volume attachment's `status` is not
               `attached`.
        """
        self.created_at = created_at
        self.delete_volume_on_instance_delete = delete_volume_on_instance_delete
        self.device = device
        self.href = href
        self.id = id
        self.name = name
        self.status = status
        self.type = type
        self.volume = volume

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeAttachment':
        """Initialize a VolumeAttachment object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError(
                'Required property \'created_at\' not present in VolumeAttachment JSON'
            )
        if 'delete_volume_on_instance_delete' in _dict:
            args['delete_volume_on_instance_delete'] = _dict.get(
                'delete_volume_on_instance_delete')
        if 'device' in _dict:
            args['device'] = VolumeAttachmentDevice.from_dict(
                _dict.get('device'))
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VolumeAttachment JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in VolumeAttachment JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VolumeAttachment JSON'
            )
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in VolumeAttachment JSON'
            )
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in VolumeAttachment JSON'
            )
        if 'volume' in _dict:
            args['volume'] = VolumeReference.from_dict(_dict.get('volume'))
        else:
            raise ValueError(
                'Required property \'volume\' not present in VolumeAttachment JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeAttachment object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'delete_volume_on_instance_delete'
                  ) and self.delete_volume_on_instance_delete is not None:
            _dict[
                'delete_volume_on_instance_delete'] = self.delete_volume_on_instance_delete
        if hasattr(self, 'device') and self.device is not None:
            _dict['device'] = self.device.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'volume') and self.volume is not None:
            _dict['volume'] = self.volume.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeAttachment object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeAttachment') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeAttachment') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class StatusEnum(str, Enum):
        """
        The status of this volume attachment.
        """
        ATTACHED = 'attached'
        ATTACHING = 'attaching'
        DETACHING = 'detaching'

    class TypeEnum(str, Enum):
        """
        The type of volume attachment.
        """
        BOOT = 'boot'
        DATA = 'data'


class VolumeAttachmentCollection():
    """
    VolumeAttachmentCollection.

    :attr List[VolumeAttachment] volume_attachments: Collection of volume
          attachments.
    """

    def __init__(self, volume_attachments: List['VolumeAttachment']) -> None:
        """
        Initialize a VolumeAttachmentCollection object.

        :param List[VolumeAttachment] volume_attachments: Collection of volume
               attachments.
        """
        self.volume_attachments = volume_attachments

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeAttachmentCollection':
        """Initialize a VolumeAttachmentCollection object from a json dictionary."""
        args = {}
        if 'volume_attachments' in _dict:
            args['volume_attachments'] = [
                VolumeAttachment.from_dict(x)
                for x in _dict.get('volume_attachments')
            ]
        else:
            raise ValueError(
                'Required property \'volume_attachments\' not present in VolumeAttachmentCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeAttachmentCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(
                self,
                'volume_attachments') and self.volume_attachments is not None:
            _dict['volume_attachments'] = [
                x.to_dict() for x in self.volume_attachments
            ]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeAttachmentCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeAttachmentCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeAttachmentCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeAttachmentDevice():
    """
    VolumeAttachmentDevice.

    :attr str id: (optional) A unique identifier for the device which is exposed to
          the instance operating system.
    """

    def __init__(self, *, id: str = None) -> None:
        """
        Initialize a VolumeAttachmentDevice object.

        :param str id: (optional) A unique identifier for the device which is
               exposed to the instance operating system.
        """
        self.id = id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeAttachmentDevice':
        """Initialize a VolumeAttachmentDevice object from a json dictionary."""
        args = {}
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeAttachmentDevice object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeAttachmentDevice object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeAttachmentDevice') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeAttachmentDevice') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeAttachmentPrototypeInstanceByImageContext():
    """
    VolumeAttachmentPrototypeInstanceByImageContext.

    :attr bool delete_volume_on_instance_delete: (optional) If set to true, when
          deleting the instance the volume will also be deleted.
    :attr str name: (optional) The user-defined name for this volume attachment.
    :attr VolumePrototypeInstanceByImageContext volume: The identity of the volume
          to attach to the instance, or a prototype object for a
          new volume.
    """

    def __init__(self,
                 volume: 'VolumePrototypeInstanceByImageContext',
                 *,
                 delete_volume_on_instance_delete: bool = None,
                 name: str = None) -> None:
        """
        Initialize a VolumeAttachmentPrototypeInstanceByImageContext object.

        :param VolumePrototypeInstanceByImageContext volume: The identity of the
               volume to attach to the instance, or a prototype object for a
               new volume.
        :param bool delete_volume_on_instance_delete: (optional) If set to true,
               when deleting the instance the volume will also be deleted.
        :param str name: (optional) The user-defined name for this volume
               attachment.
        """
        self.delete_volume_on_instance_delete = delete_volume_on_instance_delete
        self.name = name
        self.volume = volume

    @classmethod
    def from_dict(
            cls,
            _dict: Dict) -> 'VolumeAttachmentPrototypeInstanceByImageContext':
        """Initialize a VolumeAttachmentPrototypeInstanceByImageContext object from a json dictionary."""
        args = {}
        if 'delete_volume_on_instance_delete' in _dict:
            args['delete_volume_on_instance_delete'] = _dict.get(
                'delete_volume_on_instance_delete')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        if 'volume' in _dict:
            args['volume'] = VolumePrototypeInstanceByImageContext.from_dict(
                _dict.get('volume'))
        else:
            raise ValueError(
                'Required property \'volume\' not present in VolumeAttachmentPrototypeInstanceByImageContext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeAttachmentPrototypeInstanceByImageContext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'delete_volume_on_instance_delete'
                  ) and self.delete_volume_on_instance_delete is not None:
            _dict[
                'delete_volume_on_instance_delete'] = self.delete_volume_on_instance_delete
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'volume') and self.volume is not None:
            _dict['volume'] = self.volume.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeAttachmentPrototypeInstanceByImageContext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(
            self,
            other: 'VolumeAttachmentPrototypeInstanceByImageContext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(
            self,
            other: 'VolumeAttachmentPrototypeInstanceByImageContext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeAttachmentPrototypeInstanceContext():
    """
    VolumeAttachmentPrototypeInstanceContext.

    :attr bool delete_volume_on_instance_delete: (optional) If set to true, when
          deleting the instance the volume will also be deleted.
    :attr str name: (optional) The user-defined name for this volume attachment.
    :attr VolumeAttachmentPrototypeInstanceContextVolume volume: The identity of the
          volume to attach to the instance, or a prototype object for a new
          volume.
    """

    def __init__(self,
                 volume: 'VolumeAttachmentPrototypeInstanceContextVolume',
                 *,
                 delete_volume_on_instance_delete: bool = None,
                 name: str = None) -> None:
        """
        Initialize a VolumeAttachmentPrototypeInstanceContext object.

        :param VolumeAttachmentPrototypeInstanceContextVolume volume: The identity
               of the volume to attach to the instance, or a prototype object for a new
               volume.
        :param bool delete_volume_on_instance_delete: (optional) If set to true,
               when deleting the instance the volume will also be deleted.
        :param str name: (optional) The user-defined name for this volume
               attachment.
        """
        self.delete_volume_on_instance_delete = delete_volume_on_instance_delete
        self.name = name
        self.volume = volume

    @classmethod
    def from_dict(cls,
                  _dict: Dict) -> 'VolumeAttachmentPrototypeInstanceContext':
        """Initialize a VolumeAttachmentPrototypeInstanceContext object from a json dictionary."""
        args = {}
        if 'delete_volume_on_instance_delete' in _dict:
            args['delete_volume_on_instance_delete'] = _dict.get(
                'delete_volume_on_instance_delete')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        if 'volume' in _dict:
            args['volume'] = _dict.get('volume')
        else:
            raise ValueError(
                'Required property \'volume\' not present in VolumeAttachmentPrototypeInstanceContext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeAttachmentPrototypeInstanceContext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'delete_volume_on_instance_delete'
                  ) and self.delete_volume_on_instance_delete is not None:
            _dict[
                'delete_volume_on_instance_delete'] = self.delete_volume_on_instance_delete
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'volume') and self.volume is not None:
            _dict['volume'] = self.volume
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeAttachmentPrototypeInstanceContext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeAttachmentPrototypeInstanceContext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeAttachmentPrototypeInstanceContext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeAttachmentPrototypeInstanceContextVolume():
    """
    The identity of the volume to attach to the instance, or a prototype object for a new
    volume.

    """

    def __init__(self) -> None:
        """
        Initialize a VolumeAttachmentPrototypeInstanceContextVolume object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'VolumeAttachmentPrototypeInstanceContextVolumeVolumePrototypeInstanceContext'
            ]))
        raise Exception(msg)


class VolumeAttachmentReferenceInstanceContext():
    """
    VolumeAttachmentReferenceInstanceContext.

    :attr VolumeAttachmentDevice device: (optional) Information about how the volume
          is exposed to the instance operating system.
          This property may be absent if the volume attachment's `status` is not
          `attached`.
    :attr str href: The URL for this volume attachment.
    :attr str id: The unique identifier for this volume attachment.
    :attr str name: The user-defined name for this volume attachment.
    :attr VolumeReference volume: The attached volume.
    """

    def __init__(self,
                 href: str,
                 id: str,
                 name: str,
                 volume: 'VolumeReference',
                 *,
                 device: 'VolumeAttachmentDevice' = None) -> None:
        """
        Initialize a VolumeAttachmentReferenceInstanceContext object.

        :param str href: The URL for this volume attachment.
        :param str id: The unique identifier for this volume attachment.
        :param str name: The user-defined name for this volume attachment.
        :param VolumeReference volume: The attached volume.
        :param VolumeAttachmentDevice device: (optional) Information about how the
               volume is exposed to the instance operating system.
               This property may be absent if the volume attachment's `status` is not
               `attached`.
        """
        self.device = device
        self.href = href
        self.id = id
        self.name = name
        self.volume = volume

    @classmethod
    def from_dict(cls,
                  _dict: Dict) -> 'VolumeAttachmentReferenceInstanceContext':
        """Initialize a VolumeAttachmentReferenceInstanceContext object from a json dictionary."""
        args = {}
        if 'device' in _dict:
            args['device'] = VolumeAttachmentDevice.from_dict(
                _dict.get('device'))
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VolumeAttachmentReferenceInstanceContext JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in VolumeAttachmentReferenceInstanceContext JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VolumeAttachmentReferenceInstanceContext JSON'
            )
        if 'volume' in _dict:
            args['volume'] = VolumeReference.from_dict(_dict.get('volume'))
        else:
            raise ValueError(
                'Required property \'volume\' not present in VolumeAttachmentReferenceInstanceContext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeAttachmentReferenceInstanceContext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'device') and self.device is not None:
            _dict['device'] = self.device.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'volume') and self.volume is not None:
            _dict['volume'] = self.volume.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeAttachmentReferenceInstanceContext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeAttachmentReferenceInstanceContext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeAttachmentReferenceInstanceContext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeAttachmentReferenceVolumeContext():
    """
    VolumeAttachmentReferenceVolumeContext.

    :attr VolumeAttachmentDevice device: (optional) Information about how the volume
          is exposed to the instance operating system.
          This property may be absent if the volume attachment's `status` is not
          `attached`.
    :attr str href: The URL for this volume attachment.
    :attr str id: The unique identifier for this volume attachment.
    :attr InstanceReference instance: The attached instance.
    :attr str name: The user-defined name for this volume attachment.
    :attr str type: The type of volume attachment.
    """

    def __init__(self,
                 href: str,
                 id: str,
                 instance: 'InstanceReference',
                 name: str,
                 type: str,
                 *,
                 device: 'VolumeAttachmentDevice' = None) -> None:
        """
        Initialize a VolumeAttachmentReferenceVolumeContext object.

        :param str href: The URL for this volume attachment.
        :param str id: The unique identifier for this volume attachment.
        :param InstanceReference instance: The attached instance.
        :param str name: The user-defined name for this volume attachment.
        :param str type: The type of volume attachment.
        :param VolumeAttachmentDevice device: (optional) Information about how the
               volume is exposed to the instance operating system.
               This property may be absent if the volume attachment's `status` is not
               `attached`.
        """
        self.device = device
        self.href = href
        self.id = id
        self.instance = instance
        self.name = name
        self.type = type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeAttachmentReferenceVolumeContext':
        """Initialize a VolumeAttachmentReferenceVolumeContext object from a json dictionary."""
        args = {}
        if 'device' in _dict:
            args['device'] = VolumeAttachmentDevice.from_dict(
                _dict.get('device'))
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VolumeAttachmentReferenceVolumeContext JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in VolumeAttachmentReferenceVolumeContext JSON'
            )
        if 'instance' in _dict:
            args['instance'] = InstanceReference.from_dict(
                _dict.get('instance'))
        else:
            raise ValueError(
                'Required property \'instance\' not present in VolumeAttachmentReferenceVolumeContext JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VolumeAttachmentReferenceVolumeContext JSON'
            )
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in VolumeAttachmentReferenceVolumeContext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeAttachmentReferenceVolumeContext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'device') and self.device is not None:
            _dict['device'] = self.device.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'instance') and self.instance is not None:
            _dict['instance'] = self.instance.to_dict()
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeAttachmentReferenceVolumeContext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeAttachmentReferenceVolumeContext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeAttachmentReferenceVolumeContext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class TypeEnum(str, Enum):
        """
        The type of volume attachment.
        """
        BOOT = 'boot'
        DATA = 'data'


class VolumeCollection():
    """
    VolumeCollection.

    :attr VolumeCollectionFirst first: A reference to the first page of resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr VolumeCollectionNext next: (optional) A reference to the next page of
          resources; this reference is included for all pages
          except the last page.
    :attr List[Volume] volumes: Collection of volumes.
    """

    def __init__(self,
                 first: 'VolumeCollectionFirst',
                 limit: int,
                 volumes: List['Volume'],
                 *,
                 next: 'VolumeCollectionNext' = None) -> None:
        """
        Initialize a VolumeCollection object.

        :param VolumeCollectionFirst first: A reference to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param List[Volume] volumes: Collection of volumes.
        :param VolumeCollectionNext next: (optional) A reference to the next page
               of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.next = next
        self.volumes = volumes

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeCollection':
        """Initialize a VolumeCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = VolumeCollectionFirst.from_dict(_dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in VolumeCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in VolumeCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = VolumeCollectionNext.from_dict(_dict.get('next'))
        if 'volumes' in _dict:
            args['volumes'] = [
                Volume.from_dict(x) for x in _dict.get('volumes')
            ]
        else:
            raise ValueError(
                'Required property \'volumes\' not present in VolumeCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'volumes') and self.volumes is not None:
            _dict['volumes'] = [x.to_dict() for x in self.volumes]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a VolumeCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeCollectionFirst':
        """Initialize a VolumeCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VolumeCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a VolumeCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeCollectionNext':
        """Initialize a VolumeCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VolumeCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeIdentity():
    """
    Identifies a volume by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a VolumeIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'VolumeIdentityById', 'VolumeIdentityByCRN',
                'VolumeIdentityByHref'
            ]))
        raise Exception(msg)


class VolumeProfile():
    """
    VolumeProfile.

    :attr str family: (optional) The product family this volume profile belongs to.
    :attr str href: The URL for this volume profile.
    :attr str name: The name for this volume profile.
    """

    def __init__(self, href: str, name: str, *, family: str = None) -> None:
        """
        Initialize a VolumeProfile object.

        :param str href: The URL for this volume profile.
        :param str name: The name for this volume profile.
        :param str family: (optional) The product family this volume profile
               belongs to.
        """
        self.family = family
        self.href = href
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeProfile':
        """Initialize a VolumeProfile object from a json dictionary."""
        args = {}
        if 'family' in _dict:
            args['family'] = _dict.get('family')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VolumeProfile JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VolumeProfile JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeProfile object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'family') and self.family is not None:
            _dict['family'] = self.family
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeProfile object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeProfile') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeProfile') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeProfileCollection():
    """
    VolumeProfileCollection.

    :attr VolumeProfileCollectionFirst first: A reference to the first page of
          resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr VolumeProfileCollectionNext next: (optional) A reference to the next page
          of resources; this reference is included for all pages
          except the last page.
    :attr List[VolumeProfile] profiles: Collection of volume profiles.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 first: 'VolumeProfileCollectionFirst',
                 limit: int,
                 profiles: List['VolumeProfile'],
                 total_count: int,
                 *,
                 next: 'VolumeProfileCollectionNext' = None) -> None:
        """
        Initialize a VolumeProfileCollection object.

        :param VolumeProfileCollectionFirst first: A reference to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param List[VolumeProfile] profiles: Collection of volume profiles.
        :param int total_count: The total number of resources across all pages.
        :param VolumeProfileCollectionNext next: (optional) A reference to the next
               page of resources; this reference is included for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.next = next
        self.profiles = profiles
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeProfileCollection':
        """Initialize a VolumeProfileCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = VolumeProfileCollectionFirst.from_dict(
                _dict.get('first'))
        else:
            raise ValueError(
                'Required property \'first\' not present in VolumeProfileCollection JSON'
            )
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError(
                'Required property \'limit\' not present in VolumeProfileCollection JSON'
            )
        if 'next' in _dict:
            args['next'] = VolumeProfileCollectionNext.from_dict(
                _dict.get('next'))
        if 'profiles' in _dict:
            args['profiles'] = [
                VolumeProfile.from_dict(x) for x in _dict.get('profiles')
            ]
        else:
            raise ValueError(
                'Required property \'profiles\' not present in VolumeProfileCollection JSON'
            )
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError(
                'Required property \'total_count\' not present in VolumeProfileCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeProfileCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            _dict['next'] = self.next.to_dict()
        if hasattr(self, 'profiles') and self.profiles is not None:
            _dict['profiles'] = [x.to_dict() for x in self.profiles]
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeProfileCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeProfileCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeProfileCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeProfileCollectionFirst():
    """
    A reference to the first page of resources.

    :attr str href: The URL for the first page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a VolumeProfileCollectionFirst object.

        :param str href: The URL for the first page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeProfileCollectionFirst':
        """Initialize a VolumeProfileCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VolumeProfileCollectionFirst JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeProfileCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeProfileCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeProfileCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeProfileCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeProfileCollectionNext():
    """
    A reference to the next page of resources; this reference is included for all pages
    except the last page.

    :attr str href: The URL for the next page of resources.
    """

    def __init__(self, href: str) -> None:
        """
        Initialize a VolumeProfileCollectionNext object.

        :param str href: The URL for the next page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeProfileCollectionNext':
        """Initialize a VolumeProfileCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VolumeProfileCollectionNext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeProfileCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeProfileCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeProfileCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeProfileCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeProfileIdentity():
    """
    Identifies a volume profile by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a VolumeProfileIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(
                ['VolumeProfileIdentityByName', 'VolumeProfileIdentityByHref']))
        raise Exception(msg)


class VolumeProfileReference():
    """
    VolumeProfileReference.

    :attr str href: The URL for this volume profile.
    :attr str name: The name for this volume profile.
    """

    def __init__(self, href: str, name: str) -> None:
        """
        Initialize a VolumeProfileReference object.

        :param str href: The URL for this volume profile.
        :param str name: The name for this volume profile.
        """
        self.href = href
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeProfileReference':
        """Initialize a VolumeProfileReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VolumeProfileReference JSON'
            )
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VolumeProfileReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeProfileReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeProfileReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeProfileReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeProfileReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumePrototype():
    """
    VolumePrototype.

    :attr EncryptionKeyIdentity encryption_key: (optional) A reference to the root
          key to use to wrap the data encryption key for the volume.
          If this property is not provided, the `encryption` type for the volume will be
          `provider_managed`.
    :attr int iops: (optional) The bandwidth for the volume.
    :attr str name: (optional) The unique user-defined name for this volume.
    :attr VolumeProfileIdentity profile: The profile to use for this volume.
    :attr ResourceGroupIdentity resource_group: (optional) The resource group to
          use. If unspecified, the account's [default resource
          group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is used.
    :attr ZoneIdentity zone: The location of the volume.
    """

    def __init__(self,
                 profile: 'VolumeProfileIdentity',
                 zone: 'ZoneIdentity',
                 *,
                 encryption_key: 'EncryptionKeyIdentity' = None,
                 iops: int = None,
                 name: str = None,
                 resource_group: 'ResourceGroupIdentity' = None) -> None:
        """
        Initialize a VolumePrototype object.

        :param VolumeProfileIdentity profile: The profile to use for this volume.
        :param ZoneIdentity zone: The location of the volume.
        :param EncryptionKeyIdentity encryption_key: (optional) A reference to the
               root key to use to wrap the data encryption key for the volume.
               If this property is not provided, the `encryption` type for the volume will
               be
               `provider_managed`.
        :param int iops: (optional) The bandwidth for the volume.
        :param str name: (optional) The unique user-defined name for this volume.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['VolumePrototypeVolumeByCapacity']))
        raise Exception(msg)


class VolumePrototypeInstanceByImageContext():
    """
    VolumePrototypeInstanceByImageContext.

    :attr int capacity: (optional) The capacity of the volume in gigabytes. Note
          that the specified minimum and maximum capacity values for creating or updating
          volumes may expand in the future.
    :attr EncryptionKeyIdentity encryption_key: (optional) A reference to the root
          key to use to wrap the data encryption key for the volume.
          If this property is not provided, the `encryption` type for the volume will be
          `provider_managed`.
    :attr int iops: (optional) The bandwidth for the volume.
    :attr str name: (optional) The unique user-defined name for this volume.
    :attr VolumeProfileIdentity profile: The profile to use for this volume.
    """

    def __init__(self,
                 profile: 'VolumeProfileIdentity',
                 *,
                 capacity: int = None,
                 encryption_key: 'EncryptionKeyIdentity' = None,
                 iops: int = None,
                 name: str = None) -> None:
        """
        Initialize a VolumePrototypeInstanceByImageContext object.

        :param VolumeProfileIdentity profile: The profile to use for this volume.
        :param int capacity: (optional) The capacity of the volume in gigabytes.
               Note that the specified minimum and maximum capacity values for creating or
               updating volumes may expand in the future.
        :param EncryptionKeyIdentity encryption_key: (optional) A reference to the
               root key to use to wrap the data encryption key for the volume.
               If this property is not provided, the `encryption` type for the volume will
               be
               `provider_managed`.
        :param int iops: (optional) The bandwidth for the volume.
        :param str name: (optional) The unique user-defined name for this volume.
        """
        self.capacity = capacity
        self.encryption_key = encryption_key
        self.iops = iops
        self.name = name
        self.profile = profile

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumePrototypeInstanceByImageContext':
        """Initialize a VolumePrototypeInstanceByImageContext object from a json dictionary."""
        args = {}
        if 'capacity' in _dict:
            args['capacity'] = _dict.get('capacity')
        if 'encryption_key' in _dict:
            args['encryption_key'] = _dict.get('encryption_key')
        if 'iops' in _dict:
            args['iops'] = _dict.get('iops')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        if 'profile' in _dict:
            args['profile'] = _dict.get('profile')
        else:
            raise ValueError(
                'Required property \'profile\' not present in VolumePrototypeInstanceByImageContext JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumePrototypeInstanceByImageContext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'capacity') and self.capacity is not None:
            _dict['capacity'] = self.capacity
        if hasattr(self, 'encryption_key') and self.encryption_key is not None:
            _dict['encryption_key'] = self.encryption_key
        if hasattr(self, 'iops') and self.iops is not None:
            _dict['iops'] = self.iops
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'profile') and self.profile is not None:
            _dict['profile'] = self.profile
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumePrototypeInstanceByImageContext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumePrototypeInstanceByImageContext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumePrototypeInstanceByImageContext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class VolumeReference():
    """
    VolumeReference.

    :attr str crn: The CRN for this volume.
    :attr str href: The URL for this volume.
    :attr str id: The unique identifier for this volume.
    :attr str name: The unique user-defined name for this volume.
    """

    def __init__(self, crn: str, href: str, id: str, name: str) -> None:
        """
        Initialize a VolumeReference object.

        :param str crn: The CRN for this volume.
        :param str href: The URL for this volume.
        :param str id: The unique identifier for this volume.
        :param str name: The unique user-defined name for this volume.
        """
        self.crn = crn
        self.href = href
        self.id = id
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'VolumeReference':
        """Initialize a VolumeReference object from a json dictionary."""
        args = {}
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError(
                'Required property \'crn\' not present in VolumeReference JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in VolumeReference JSON'
            )
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in VolumeReference JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in VolumeReference JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a VolumeReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this VolumeReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'VolumeReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'VolumeReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class Zone():
    """
    Zone.

    :attr str href: The URL for this zone.
    :attr str name: The name for this zone.
    :attr RegionReference region: The region this zone belongs to.
    :attr str status: The availability status of this zone.
    """

    def __init__(self, href: str, name: str, region: 'RegionReference',
                 status: str) -> None:
        """
        Initialize a Zone object.

        :param str href: The URL for this zone.
        :param str name: The name for this zone.
        :param RegionReference region: The region this zone belongs to.
        :param str status: The availability status of this zone.
        """
        self.href = href
        self.name = name
        self.region = region
        self.status = status

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'Zone':
        """Initialize a Zone object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in Zone JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in Zone JSON')
        if 'region' in _dict:
            args['region'] = RegionReference.from_dict(_dict.get('region'))
        else:
            raise ValueError(
                'Required property \'region\' not present in Zone JSON')
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError(
                'Required property \'status\' not present in Zone JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a Zone object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'region') and self.region is not None:
            _dict['region'] = self.region.to_dict()
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this Zone object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'Zone') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'Zone') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class StatusEnum(str, Enum):
        """
        The availability status of this zone.
        """
        AVAILABLE = 'available'
        IMPAIRED = 'impaired'
        UNAVAILABLE = 'unavailable'


class ZoneCollection():
    """
    ZoneCollection.

    :attr List[Zone] zones: Collection of zones.
    """

    def __init__(self, zones: List['Zone']) -> None:
        """
        Initialize a ZoneCollection object.

        :param List[Zone] zones: Collection of zones.
        """
        self.zones = zones

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ZoneCollection':
        """Initialize a ZoneCollection object from a json dictionary."""
        args = {}
        if 'zones' in _dict:
            args['zones'] = [Zone.from_dict(x) for x in _dict.get('zones')]
        else:
            raise ValueError(
                'Required property \'zones\' not present in ZoneCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ZoneCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'zones') and self.zones is not None:
            _dict['zones'] = [x.to_dict() for x in self.zones]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ZoneCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ZoneCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ZoneCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ZoneIdentity():
    """
    Identifies a zone by a unique property.

    """

    def __init__(self) -> None:
        """
        Initialize a ZoneIdentity object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join(['ZoneIdentityByName', 'ZoneIdentityByHref']))
        raise Exception(msg)


class ZoneReference():
    """
    ZoneReference.

    :attr str href: The URL for this zone.
    :attr str name: The name for this zone.
    """

    def __init__(self, href: str, name: str) -> None:
        """
        Initialize a ZoneReference object.

        :param str href: The URL for this zone.
        :param str name: The name for this zone.
        """
        self.href = href
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ZoneReference':
        """Initialize a ZoneReference object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError(
                'Required property \'href\' not present in ZoneReference JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in ZoneReference JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ZoneReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ZoneReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ZoneReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ZoneReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class CertificateI