from __future__ import annotations

from typing import Dict, Tuple

SSL_TEST_PRIVATE_CA_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDKTCCAhGgAwIBAgIUaAgNa9r45P7lGSz9yjEAMmSYuA4wDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMB4XDTIyMDczMTAyMDgwOFoXDTMyMDcy
ODAyMDgwOFowRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8G
A1UECwwYT3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMIIBIjANBgkqhkiG9w0BAQEF
AAOCAQ8AMIIBCgKCAQEAnTwel56+tLKkGLedJB0CRNyiCYiWS0sNuoCYpRN6MTnj
o5pYCN6rFQvdRvkaBRTodlU4stoEAy7b/v2ny7h00M8iWkVXqiDIB8u1pefEE53G
p7ezsPNSf5sX4+1kvm/+TJd4UEeAnSYaN6LWnTbEov18Kb7Xgy9yoGPpnYHmoYme
lmnV9rSXmI+nFzLIMWKTFfaGW4icTI3d8m53Teqhj0XiPI9VU7PHjUakAEkybyPM
tGJ0f7yq4OG29haDfWicyAKjYm3KxAQwdRUR4AuUWjXhk3flSyEnFsu5aevHd5xJ
88W7sp7UJWw3NDPHIxrehVeo+ZWNiMJWC/Ff1bVMBwIDAQABoxMwETAPBgNVHRMB
Af8EBTADAQH/MA0GCSqGSIb3DQEBCwUAA4IBAQCLXqq05Wp9/J3atNhSo+qoFETF
aO2VokcUYonsiXxNqTKruNtTHMSIHZwerwhIPqEA0KGUVR1oyJvvdBu2VAxv8IpC
vR4aJAtI8X2UgDYElsDkmahEiCja+1f3+w7AK78HGCzm+5d2kz3p4vQHGkE4N57c
g6uZVoouma3LJMpKS9R++Y2JBjg1EfXoleYdh+CPac0AihHMvV64/7tkTt7Wd++O
4pe/yUy7qcZL1b4JKoSb2bKAwDTYMljJ1buLLvBlhjIwqxOiYjnEZJK1MHfEp+sF
HPPfgGE01n8iANfXwjob4OY1SB0n7vxz2tCMEBMqAdOt0T7ThGhCVRGILcKH
-----END CERTIFICATE-----
"""

SSL_TEST_PRIVATE_CA_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEogIBAAKCAQEAnTwel56+tLKkGLedJB0CRNyiCYiWS0sNuoCYpRN6MTnjo5pY
CN6rFQvdRvkaBRTodlU4stoEAy7b/v2ny7h00M8iWkVXqiDIB8u1pefEE53Gp7ez
sPNSf5sX4+1kvm/+TJd4UEeAnSYaN6LWnTbEov18Kb7Xgy9yoGPpnYHmoYmelmnV
9rSXmI+nFzLIMWKTFfaGW4icTI3d8m53Teqhj0XiPI9VU7PHjUakAEkybyPMtGJ0
f7yq4OG29haDfWicyAKjYm3KxAQwdRUR4AuUWjXhk3flSyEnFsu5aevHd5xJ88W7
sp7UJWw3NDPHIxrehVeo+ZWNiMJWC/Ff1bVMBwIDAQABAoIBAGI8l5h/VAMtwZy3
y3Qus6Y2ev6d2dj88Xzer4AjpSpwUqt62RnhlN43ZW7cX8dhAsQ8Boqn7+3VLxon
fNipTELRx1LDFDx08sT8H2eiYD0pTF6pRZl+zMJxMgHz5gedjTJpSIBPzCljDupH
0PK0rzJuUU77r0aFNIbFJcaEzIWLrVWW2DiRjHrOc6gCW2T8refphTDzjJPP3Hg3
97N7aGW7wbVV0ObRbCKbnQNeWdky1VuFPh5JKa7tKZld4eXHWMTPkm/ooJrJj97F
3OP7d79uUwdKaUR/nzvzH0OOdilz//dS26XnURY+6OoUEZScEMD2Blipnxfn45ar
E6ZnvlECgYEAynSTcsc0cy+PYNyEfw2ue0vm1z4f4bqZFv20ISL+VRP69Zv8Mu+0
kHrSOg3nNftLmdc036lAk5Es6edEXcF1J6wd//m2OlLVze8EKWrok4rXXL3Ez6R8
OADnM1Ht/f/+lFtKm4uvAls/kXboW60OcQCO77EXbZAUtfsipuzSh5sCgYEAxtHY
s7EVcv9PIsoKzv1Dz7xpAfcal6/tGQiU+W1J7nej9U+WaLK7/HbgzaMieCXJ4Lif
0pyk+vBH5eNYxeQbYVOCjzr7dY/D/o/7zYyhSRd0tgqfAwBbXEaINayrPEtmYa8F
YRgzYurF57F8A3V2iwW5KQeSGIJhs9TYaT+LUgUCgYAy6z0Geimr6sNR5kSAvA0t
P3rQmfRkzoygZrC+QNHczgipa1Dz8qvkRq/HICYNwqVSDPlkY2o1cybrsVnVhsmT
b8ufzMtTQfA6Oio+zH+wVBQ0ZBoy6zi+0SNVGf8UiiUaZRePpYMfwZlsWY65N/5Q
14YbrNuql7lNx9sARsHwjwKBgDcdxW1LEh8nF9EVIZnwK/RSzul0jzKt2pSuBF6B
JZWuJQ3EWnX71pFOg1N+fffAwg5CIExKkJHgOo5IUDvL2KtKc08xkrQsfrD7SVra
OXwBvKEaoKo8Vxdlli0LCQ1V+bPqyRr5koVBgEuSsEZlMc+oboQpTR7uRheMUdIb
g4vhAoGAGVEUwhOaeEWRUwyzBmxRcBJTSijppz5AnX4QqzYfVms3biTsDve9bje7
deI9IEP1RRYMXIX5aSExF9eWQqEr2fvXVzpijQWLzzw6+vDSmUygM1p4UYQdcyPo
12dGJg8QCIW6kcId8cZCMOvyo/U/RH4NtCAFoJqCe63OwHKBTB0=
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_FULLNODE_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUHx4JBOGyU12gOWGuB5dBegczdfcwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQCqyyHVlhZpFR0rHSfSunLrREk0Q9MJSU/pcsbnmYIBpUZS
HHnEStu5B984swApxNmBaPWiW7UaJ2jq8h4NS8Oo2tCasIJF2zWGWgJDEieTEU+d
vpJ/Sfu0MZUpu4obbWERXkmVnCyrDx7v+kSsxukSAWF5iAHT3Oqnouc0AwPJCuU7
5MRD9/0xC6GFurSJQndquKcZJ8wGWkC1WNU9C3VwMy4pYLT84Pmu8bh/6uzupVet
TDDDNgh0SGqbabk0TCG9kgS41ww3GTRf0sOoJdT6Zb2UYA6SQ7iGmu5ZPZ7bQykC
4zGkna+MtM5WldUilkfGN9rPt6xvCycBpif7nUOLAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQBlSm8+WQWNqQ/6XZ6RFxCS
Lq3aFKAGGZpX97Y/Q88ZcbJFd+67yORmhCa8mvLYfhIpXXG/IoX4E1PEBohskeah
/nPyeaPAklwSjFYKMKbukYUb3j8hQgs0jFh6+mMvLUcf7bQbmD2ZgUgt7xKVPTHG
59KlnZv+cIMlS50JQU6n3oHv0a9+STMU7GZV8RN14INGGX+RD8shO3xWVZB/BzNj
sjoud8pSpnBJYLH4uucnTXNjrzM8a+S+uaU0kxCdRY3TE8y6PsqC2XaPQctf9O3J
hnFoH9yydMD2leovDedRyENxs05wNlVK+GzUB8/g6++3dqC0Sq4U84yemYh73Pa0
-----END CERTIFICATE-----
"""

SSL_TEST_FULLNODE_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpQIBAAKCAQEAqssh1ZYWaRUdKx0n0rpy60RJNEPTCUlP6XLG55mCAaVGUhx5
xErbuQffOLMAKcTZgWj1olu1Gido6vIeDUvDqNrQmrCCRds1hloCQxInkxFPnb6S
f0n7tDGVKbuKG21hEV5JlZwsqw8e7/pErMbpEgFheYgB09zqp6LnNAMDyQrlO+TE
Q/f9MQuhhbq0iUJ3arinGSfMBlpAtVjVPQt1cDMuKWC0/OD5rvG4f+rs7qVXrUww
wzYIdEhqm2m5NEwhvZIEuNcMNxk0X9LDqCXU+mW9lGAOkkO4hpruWT2e20MpAuMx
pJ2vjLTOVpXVIpZHxjfaz7esbwsnAaYn+51DiwIDAQABAoIBAQCRC6vzQOBNY7H+
0GbmjpNT/A0TRtptk4Y7Is+LFpITJbI/v5Elw9EwJWBMPbBQyK1TOrkIpkt6J6fx
h6c/8Wz79MCd7nLkMpMVQ1GSdlOHkXcE5zhww3rwf/9Gjs9uFYkO//0MArLonxHO
CcPS26ZKNUP7uyaaL45UyYMCZ+EKFUOVvYA6EDsLSLcn7O4R4B7FgfLmbymx6FKe
P4EU0RA/w8Lv5GfYwmHSrdgfouAJoI/vG5HDUDIb7fWR0kX6gIst4PEKLn5eWGGt
xevgiuVUaL1nUHZQKeoLfXcqDTCdRpkmPSF2YTAxcF2ZC58JiPmmA+cXqWrCKmN6
xkOSzpLpAoGBAOD5p/kxe8ckFfV6cwspFt3Q97V5C4vpX9JOE5fmDAaMQdTkLciI
33Z/C/+vHrzcFb/99SbVTzcTT9mJiT3hXvuy0M2nVk2znjAI46y6EgBr9sS5cLUj
qcTjeoJVkLjo2DBipp1kp2Z5JCKFb4GGdsRjtM26IZwanINTbLERjkadAoGBAMJY
sA81RkQ6lGEXMnUdqxyc3UL/87MjwzhG+w9xi8fOW1a0kBlV1gjKm0Fbbyc07lPy
t5zTyjYFp57fzmyz7MllGE8WQHiwEyc7QDNW74RcZhFwmO8KVpXB6ytZzSERbWlg
by0UFV3ux5AIlBh/6eduR56B30v01Cwv0JeNgAZHAoGBANy2MQO5dyyki2J8Efdm
BUfyZcJEkX5eoyIodPJ1v2CBI2ifvwbYuRqAg54BCM/RvwjPNj410PRAXvsVAm6R
9uRJxq2ruHCcgdmwLuTa2yRMxxiysQ8UgM5nOV46YbP2w3mcnscU42IWAMxZSk6x
Qz1icBPQwn5qYRg5tQ12IrXFAoGANc4X8B3vtjQBoF+ruIDePq2vFD2IWrAr/Aiq
uTvWVxrqbrbOrjT1bLmrAg0uWMxurw8jhmecNBicr3+GxtGJXLiAYmTmtIMKynG8
9Hm9Fgf+uhVSQdu63PszAG18PQt2HxZzsSc+QyM4XMgDOEUxnMELUvbhu0Ws+A7E
0TB7oUECgYEAhNcqwwzWtHK93YQPVzekpTeGoLgnFhTTx3Wh/sQddJuJeqzWWYqS
LDObEYARXNGlGOoJJ9eObYZQ7pMFt9IXDSAtLSDHnh092rC3OgP+brTcUjHYXYEl
8yZ3z5xGm0Jou25wBNoAE8OKFPju+yjOcVf8L0RIsqMIOGOPRTbLQlc=
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_FULLNODE_PUBLIC_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUFhaPwCH7jH8uqupV1SQQhWjqr7QwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQC0T15f3tKXrJr19pL+meyy+AxHrhlOfgID0n5BCn6sYGcg
7OzPjMf7JFBavWzj9WGvufAke4mOCOrQ8XbbeTW6lKDow1mSAQYWe29N0Xuz1SQH
xwbFTf83tPzsbDWwxWZdDRG8SmkJ2WigJkG/0PGSlkPm+ttzga7Ehw9E+gUNTYNF
wbVj0+2zNVf6VzbQ++vnh4atHRvTcuMIXXJ9kfJPg9HupgykSi1EMum7fTNQQSRK
VYAyD/qnb4t7xbmzqIciC6uz26ziwMXLOhiV4zibe2gqoeLJBVboMV8WCWJnhSmA
MEXaGgq4cgZ1MhfJdlWa1TB/tCFz7vCHB6o3n5/rAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQC229ik5XvaUZsMoNKWpzPe
7VA5XwAMPImgppwDsQrMlo8cLX7DEUOi0GPHstS8vx6cozRihVhr3YwfUh87MFIb
B2WkIy9dXyutI+KMK9dQWSUKaPcavvLvWBvMGN4bkx/93/CVq/lD7aUkoZ0XNulZ
Pf1SkL6qFjUYZb7l5xJhFeygYqz1cEdf/0u9FJpZPjyI8kRywTJv6cawDWCL3oeS
de1AN/0IWU9ePMiBlFjVjM6dhxP+qobq9WwZkJon3zWHuD5dbegEjS5tmN2g5VYT
1hZaamRbV7aKDLr9Koq42owHByfGYxh1o5pBwnnOdEHB2SMT7djNC1gkDj+PxajW
-----END CERTIFICATE-----
"""

SSL_TEST_FULLNODE_PUBLIC_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAtE9eX97Sl6ya9faS/pnssvgMR64ZTn4CA9J+QQp+rGBnIOzs
z4zH+yRQWr1s4/Vhr7nwJHuJjgjq0PF223k1upSg6MNZkgEGFntvTdF7s9UkB8cG
xU3/N7T87Gw1sMVmXQ0RvEppCdlooCZBv9DxkpZD5vrbc4GuxIcPRPoFDU2DRcG1
Y9PtszVX+lc20Pvr54eGrR0b03LjCF1yfZHyT4PR7qYMpEotRDLpu30zUEEkSlWA
Mg/6p2+Le8W5s6iHIgurs9us4sDFyzoYleM4m3toKqHiyQVW6DFfFgliZ4UpgDBF
2hoKuHIGdTIXyXZVmtUwf7Qhc+7whweqN5+f6wIDAQABAoIBAE4TWBA3UgUvTmJX
BMuOJzpPwjF/p6GfWowBJ35OZw+JJt7Q+LHeifGrhcfk4jPLWyNWZi/F9og0QEl9
ZTQfxi4mkY54mF5qZxJxHH8byG7USie7/KiO34ZlrmdJa+5Ba9O7BwT7S4RiSbWf
GzVN6mzI+FoaIdBgRScAPJJnmKxrO4aISBQ+r6eM/JTdIKvBKH4Juq+rxAMAuVAA
PQ0En3MJajiZsU5n2PsqCFIiLvAOQfff8UdBsGcdeOwn2zQg1e2HcEAnVb9p+hIk
oJVKx0iySmBaxeP/cuVyuQwHo59u1SjxkJgHH+RRpVGn63BFrZ19Wuejt+VieqXA
eatCR2ECgYEA5X90Z5/1enWi+iVNddVArOxY8hMf2QwzjunWSmDbcsEEro2rBuWR
qoCWs5HeikptvJtOt96vIa7xcT7gaU/rbeOVUHN964aGlEd/Y+GWHRI8wfJhLzcj
Gae4r1eOXs1ZkUFMNxMDR3Zb0eGeilUsB7sRwm/j5PvQ5um4ooN+szkCgYEAySHL
oR11lxHwp1G2Qk0s2u07vqUnP27/Ib0x023VU/zoIdO6uJiNG1+cin11AmIi/RuI
YwtYaCn3HF7FmgmsAx78CF6XLzGvuX9EQK0N1Byrwi3i3K5S+gAsM42dv5Lb7uwS
JDvFH2wlq7cN2bLZr6bnYmXHU9wjvRPfHrEUeEMCgYAOarxQSi8smzk6x58fGN9w
PYS06Fim1OpkPvJUR9NhWRjHMmUdEMzezicpbq9xNo3APG1shZOw5QGL4KxOUVPz
2kkl/ataYWAQ7pDlIGXCu8OPV3+dbLRKx0uF0FVERVgsFNMJ3qrlEV1hSPAhtooG
jkzmTwsc9mjAKMyNgP8zmQKBgQDEXzyi91JBxwBBPGRIY3Ux4oF8Z17kbgim9kNZ
fFjWAYzayg/5/ud2HGmIRWkihJJIsML+gKs2OBLOHjGrbVoGShAkU8YkC8zXvUv0
IYHo4H1JvudKMuFMuTJuuwuTaV1V7iLgQYL6ZOp6tuTM8fPbrYcn6NsuIza5eldT
bezgZQKBgDvsa1sqjJlQCSNYSk7P3vDVBIPaWa7RM9B1gwg/v/pmF3BWYxZNbV25
jOqJbmHeqWiyWX148jVLAXouL7Gxw4MglisrrddXaWSUnIJOrySmN42ZODVttvRj
u8Kj4nm2cfosy42ar8FAqj0DfTo7GNo5+R8B8lbfO8/gcMi4j5hb
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_WALLET_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUPgSiFTu/c9pvcsiavcITg6l+XbUwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQC0AXXqjde3+iG4nSDVvntiMpMM+vm7Gg6R8yShZEp1zT/h
uov3smxNzKpSAbNA2nPifDIsLI0oVOWgDrZXOi1kXipC98GMjXE0mXItsDojSyih
Y78HKIAeJBtBfLkt2JvyoGxZ6TBBgfgemRQiz/b24QKLLIYjFIlH7VFI6XGMb2PT
rzB8OqwxnBe9orFnCJkGtYVrSQvcrIUX2Vt6SP+7QVXxgJXmm1SpetNjcmAYQ1Dp
QR9leIYRPy91TItvrUUnmP3pC2aJ7RmC7Ts2Fomqed3qnuMEgk1BhTILnZAC4oqb
lKMo082HapNoxbGcjBNIsm/Ns85la/DP3V2FdZINAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQBjkhqh1RyS9YGL3OOQumPy
fYzt81tI9dfBcVDISmDYApgBQ9x2z5/JDZjReQuXacNbVYydJjHL2JEbhQvq5ebd
uv56pvNAUwMiM8Ihkos1cgN2QvALBAFYxusFEP02rI5sb9GwW9eSCYDbuJtFsvdN
D974kPNL4B2hLkWyO+f2/epxVJ3i4TSf5GwXIjJERIjNa+LKBCLEDq/IJ3sjQrbs
KZOiHm/NlgaEoqEIZGDE8J0FfhTUrnXHd31/fWe2iFDcbQkOuUCbpphtr/3iChXV
tgkL/OjUF3EID5NlttW4AbIXCwvW+buScK80DLtLEbaxueF7CUooOObsbECh6ZOU
-----END CERTIFICATE-----
"""

SSL_TEST_WALLET_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAtAF16o3Xt/ohuJ0g1b57YjKTDPr5uxoOkfMkoWRKdc0/4bqL
97JsTcyqUgGzQNpz4nwyLCyNKFTloA62VzotZF4qQvfBjI1xNJlyLbA6I0sooWO/
ByiAHiQbQXy5Ldib8qBsWekwQYH4HpkUIs/29uECiyyGIxSJR+1RSOlxjG9j068w
fDqsMZwXvaKxZwiZBrWFa0kL3KyFF9lbekj/u0FV8YCV5ptUqXrTY3JgGENQ6UEf
ZXiGET8vdUyLb61FJ5j96Qtmie0Zgu07NhaJqnnd6p7jBIJNQYUyC52QAuKKm5Sj
KNPNh2qTaMWxnIwTSLJvzbPOZWvwz91dhXWSDQIDAQABAoIBAQCaTuo7oU68q1N/
BEvij8ELpBlFNl0g1KyK1ymepO/j3oiVxCo6s5x+cSXM2SJGRT60eA02qxZy7ZxP
Qm1zhy7kbIjEEcm86H+UPLqRihG8sbBp8a0PwNtbe05WLEo+kALX0kgwJ2OztGv0
JvwI/8hRSQC6N/fhWLh6jWlyHJS3clGyqI+SBT50EJGNhL3evIxfQT+dviff3ooe
KmO7fkFQbAz3BjIqAyRk8z16qLYgAL9wWy5q17Bvg4oGL38BS38gHA9iAsPYx5LO
hf4eSHTvlj5vQQYs9I6piRb9G6YYj8YsWQvvwjB+bBIpwiV5hxuwLNeh6JzJr8Pr
L8d/GQmtAoGBANwjGRK1xWJBczvaK3PvYYZ6xwSzHgoGCYQUnQephc3VSy/TEWmQ
E+ai6yxvojlV+JzI3Ly+7M3MwIEmx406Kx6vpDUXPSdKleiO1OPeMtOLvVwiJk20
pLpEeIiQWINuVhkbzBdLk9VH09UTKXe7t9q5qloRtDCeNM9hpLnFZmK7AoGBANFU
qt3RvKGEOZVyM/2UjJ/29zmUikofCnNCc/hA8Hweilg/LrtCCmteX/8IrDyJZuhX
XtYZWfNDIewXn2AcVbvz88jHOW0FhMmWv1diEQkxbVfi8qtB09oXPz+GDp0mVhZy
VLyLXGACw3AAPl25BNzbvus4s1tUvUlFHwX4cQXXAoGAO8a8FDyPsLKdG++ORRkH
noDQchJBYe9pvhfq3B3IW7BDjmT+IXAiN0qeGZ7mqjMSI2DXr9kYbHng/DEPluXo
t5hyOoho7rZuJoxCeu3nV1JmxRHw3MDkyj+1iz/HCUl4z7CgpuK67ftnJkDhzcWH
kzk/bN3J1DnhlT6qgFd+CekCgYBvv40/wXm7l8TN3RuYIbc4pjSwjnWMwClqAm26
d6BqAjvEhGZEHStGyijhExYvcwJ2wj2kegAk9w+i47F3M2kpX0AjTQeBBdMvcsRO
t9F0YOiIZGU7u3aHFKX1GXGvvwlyDjLiPFth/dercNXaXq0T4lN3atuExCYQeU/o
dgqQHQKBgClExHG7vW5XUuRNMYnvsm58V5dIqwNe48JVs9KN19q8R8MdH0Cg4sQ2
yh94hRTezAMgJTm6if6t3nI63xwPshwPnJ3sesh1zc03Onwio+GXijMfLFzr9LeK
rZTS1woX8S0JWJ3TelbC7ocstOGbVwGEQFiWjFO+ig028I5wawxH
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_WALLET_PUBLIC_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUa6usaYtQvU3SpCXwC1JZ+v6+V+owDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQCfSnAPjNYAaVYFm8sz9GPz3PytW3pZhdlEp41qyD7JKgdQ
znBMLwGHx8k2eT//mKovi7/4ASCrtgoti7k06GIRCM34ltcHBtBhwksJ3nlzvW9b
e2r/MzXN9nxNqVFf28Wpp577hngtnhm1Y6Jm7eVZP2C5nrEfOE4wwXkaG1wQn6ED
TvIz5G9pF0BkuqCKFajygtyHl5s3GeuysABMOLnj02K8IClStUd2sD3tuXoOhxZ7
dqtmN0HHKgA6GEeq1YNMpPmHAorJ/qgTWXTvAsHvxHIYAOvmOAFDBwj4dKBR3vCz
YpO7z/AfkHNJQjVMCZS+TilS8KvEOqrXg0jk8wf7AgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQCYJQ+LL15Qfnbw/Fi45U8Y
oBqdURidfgfqYD2z61hIE7MalIKy8zqsIadQc3+3xnIJRF5QrQ8ctpZuWKBjZ7Lx
3c5NOENmNK4eiD/M5Ip9nA7XyyxLWkNIgpGLvJGmjTsH5XJBQjnUhxWP+hvapcox
t0uhkXafa7iQ2PCZfhAkdS+4rKl5aC2+gNPNN9unbTyrI3pUKv+kfWiV1rW66gdh
QMY98qQwlDA5FP+3Sldln9XbVUwZipEoBB6U4TwMye6ZuCvHUhi7eE037kN/RRWr
vvQxYJQy+J/dMVfCZ+DGdBMp6Hdf/uH+C7Dh9hFK7txq6LvU8sRCJT7CM+WSNNmB
-----END CERTIFICATE-----
"""

SSL_TEST_WALLET_PUBLIC_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAn0pwD4zWAGlWBZvLM/Rj89z8rVt6WYXZRKeNasg+ySoHUM5w
TC8Bh8fJNnk//5iqL4u/+AEgq7YKLYu5NOhiEQjN+JbXBwbQYcJLCd55c71vW3tq
/zM1zfZ8TalRX9vFqaee+4Z4LZ4ZtWOiZu3lWT9guZ6xHzhOMMF5GhtcEJ+hA07y
M+RvaRdAZLqgihWo8oLch5ebNxnrsrAATDi549NivCApUrVHdrA97bl6DocWe3ar
ZjdBxyoAOhhHqtWDTKT5hwKKyf6oE1l07wLB78RyGADr5jgBQwcI+HSgUd7ws2KT
u8/wH5BzSUI1TAmUvk4pUvCrxDqq14NI5PMH+wIDAQABAoIBACgVp1WCb2fzGf6X
Ue9lFLINbliBqKxWjnJEZ8XzVjlxJh4jlQbW3V+52tymboB3fnwwgsJ6JiqPyMfl
XVn/EI6Q1KQ18S6HfHDdX9y/63hAt4Eh8a7+yA9oc6dncF7bA0bKM6vADwYLvxRy
Wi/ktc5bsI+Fa7UbijGivkLfl69m1o9jKScb/Kiba1JjoqSRpaNnqpmDseAg7Pjc
xF0CSUi6+7vlJ0+h+S7QIRHNLfIfHL+Zf7wZVOWxXK3kRBolOUSoyEwOw4qeb08t
NKpTKF/fG0ZiN9IZb0EmsCm3lwTDlZ3ldutBJtydvAyHLKKyNDjMRit547vYTQxR
n58ZacECgYEAytCnPTFDldiOCFg6oW/vHKBRpU/KjQGIdfvQnfhH8FoFaUHyq7Pt
g9SCYAYRNoOaPEgxApVWJ/9a24MO1s938gnstREc2snUbNUJE7BSbN69/XEUTA1Y
1/P7begrDug1gSu/N5lSmlq0y7KAFz6wJ4H1jvYK9DM2rr2zYiVw0IMCgYEAyQ/r
cI0Sqm2zsKm616vfe2qC9UJiBUVbnC7OCrC5nXgaxgV+14Pv7JnmlqgqaO6gX8cD
W10wEItMOCk0/2E2ZbSvvwO6GyXSxxvYfDMUoQS9Q5PG9pU8RsMAKR7x0opX36SG
Se5VEUFnGZ1KM8E+fwJYfqye8ic2Y3jkw8egYSkCgYAfA8VvYCc7tQATWpqssVdZ
g1K4+vi4k3FwyW9fmOY3ZSzKqeEYTM2XiVneoU7kjT961HdaM8gXGb+lLeeJBfUL
uGTQU6ub8Z1gfzuCHUadi3p2VWKLNsdNTuw2JHjhMLVt6Wi2ceXl4NUhCUz5QQbj
I8aPdCS8r1A/bJcP/TXBTQKBgQDAthiE/UI5xgHnvxcNpngCk7lHmEPZJ3xB+36T
MoYGD1siF5xWtufckDo6hp53ith+VIK8LFpoUQtR/PUGHcgEvaWBYPIdGSionODn
UhJgruk0/0xBhhKLBFaAE1/W27CT896et+gjLNvD6kV3y626EKZ/a7bBBKKVvwcN
O6ZV2QKBgGbB2TsEWuO+FRS0eQzxx1+NsHY1jCkeSEadYZGMKMTAvro5TAkiegZ0
ta2ECvu/eTPrJ/vZFVXxL9bTLn/rGleKChvarYY8W/jLKkMimAhdVqjOo7pu3aQ+
O3LoYNPP7Wgx0+sisxU609LCSwd7aDlf+O+1tt0+iNEgb65K7ooB
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_FARMER_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUNoDaXQ+0Go9hQxTxGLs0B7vxTDQwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQCwHxpAJMp1U0WJHLmyRi9bii71RoGv1NjkH+PuRZxigaFB
VMsjZndD+Is3DdYDWvPD/yQbe+pA0Ne/h9gLA+JAyRClPAhvx0hl20OQ/5AQDHkO
fFcgSyZsnIqqkzprHNHwEu+wJahTHNuRJb9iesNxDqXpV3Ve11wJf2Sh9o807a24
2zTMb40CxdhF47Y2ZypDO6c7DbS/WJmEdTihCh4T+kgqSscXHlGAGlq9M8Vl9Mtd
uDxNzybWV37wHY0XNaHyJSmwlwznVE5Xz7BCjtyL2/Tw2Cp62mil4RvyHv3An3N+
lgC45Mo35mWX84VcBxqXfyNTGtw6oQ2SWllbGjUNAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQA9A7G1S48zTgsU7elNwxTm
ctADVimn5RWpVSHWJnRXCM7XQjzqvZ6LDyQGZAks8bE6cX1yHjJU+bJYJ2foYk3k
cVE0Xqozc9c2a54REY301NnHZB9ks8ptXjf4JZrifqStALW+wotqWIo1OSzcgMYJ
vaP4cUo4Tx29eYCq51jsqW331gY2ZqClzMn6DbhRJVQBH7avnTTbWWShnZ4sZxM1
un7BkM/d3t2ehx3FFBdoMSTCcXa/TolLO4DlCH4b9dojvW4HN42RrTcruwMx6m3/
KgcTOMtluX0W/Ti8H+SqCghru8FK/6HkE3EViCvYnHsr4AyEVXGzk+owTXfJq6vB
-----END CERTIFICATE-----
"""

SSL_TEST_FARMER_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAsB8aQCTKdVNFiRy5skYvW4ou9UaBr9TY5B/j7kWcYoGhQVTL
I2Z3Q/iLNw3WA1rzw/8kG3vqQNDXv4fYCwPiQMkQpTwIb8dIZdtDkP+QEAx5DnxX
IEsmbJyKqpM6axzR8BLvsCWoUxzbkSW/YnrDcQ6l6Vd1XtdcCX9kofaPNO2tuNs0
zG+NAsXYReO2NmcqQzunOw20v1iZhHU4oQoeE/pIKkrHFx5RgBpavTPFZfTLXbg8
Tc8m1ld+8B2NFzWh8iUpsJcM51ROV8+wQo7ci9v08NgqetpopeEb8h79wJ9zfpYA
uOTKN+Zll/OFXAcal38jUxrcOqENklpZWxo1DQIDAQABAoIBAFQliBU9fr2vm3l3
EAHtJpNF4kmt9ihcOYRE+FHPUHiAFttv995iBnmjNmCHa8kOm4j5TfcTrUhqf6oy
7TXWFTN+3UUU+bLuM9j2ecG9WMdSLkSNg9MaFLGv2VDGJxMmIefiJEK0u0d6ci1X
K6mRne9WYE6sSpeVrZOcGiiH9MQEQCEvLcG9RnGoEPdkOfMDcM+I14GwR3vkI7JG
mq4BsWP8Min6Exg3dS2gHnHThdkFG4Gdl/0fIM84dXsOzN0BRwNWVJc55V+pqsnB
5BA3ZGgOWQxzB2Y1JWYoFJekRBjAbkFDfRVipkMb4eJa5MsT24z5d2c2Vk9ktXR6
AJtgJAECgYEA1vkJCHs+rOnL0/qFxbp9HCW1na7W3XSrFNYJd7ML6Cm0/Oj3aWeL
/PJDa3fE2OkxmM2gFEFPuVZS0hIlEfPQ4pP+8X3CfcQCmZOo5G3UNz0fo5+Cp5mW
cVGNa+rt1WPfHE8UlBK7xHqUqWAw41S6CdZL6zXmsJ/mHpZKbuNX6Z0CgYEA0bvm
+x/I0AQwi0wb/7rbPffmx+ywrubnVzypISuBKKgZa+hnZ3pgHp4l2MFtbA7KWu3Q
ABA7rwz1VO6t++RbbU+FH+MtihbE3dA2pHBu3hWtydap/BFNSiwOqVvi1fmO94w8
wumpYpG9vtk1jwMLV+jtEWNWdqTX7yvQKy0PFjECgYBFHdagqndxQs4s/nwqIZZ+
CUXGR+N/WjZzQYnTuXZXk6lEM2bwtcEVd5Ljp6NblkZC+cHBIWqPjUidkAXpKrn7
T1gFrshshSvM0uPXZdR4WielBPEOD9yF3JGao/MJy819HZPhM9C5ECxh9LJjNOeH
qCXKnw7ntNFY/js6QdAPJQKBgQC87RFdZz1CPHPqFs4l55E02gmWUBy8AJ2l1KvZ
iFHZaMs1i+0RpofNLNKU1XCW5R/dFtnGJNlOj1KzhilNfV/qMLaPidnsySPfV37R
U8Ja5ZZd+bO3r3I6/ULbxjyoWRwy262kOk1t0DDnKMx5Hdnq/5QFlME/MLCe7o2f
RwCPEQKBgFQBIaa7ws2is2B5Depi3t9h/ho7WshHheSXVyzwOZpq2B2pX0+cO7Ar
mrydNDqbYLkRXjiBRaut3NKzmYfI1YPExS2ldkdwVWC9i69gnNoJrTfgGAWzPSCs
5PyRUk46eUdzd9TDNoi/fdEL5Y+fjeFIcMhRwrNyjrGga1sunhTd
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_FARMER_PUBLIC_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUIMjA9usezqPG812Y++0Upkri/xMwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDLxomVu498bb2IIN/b6aEe/TAO7KlTEHNMSuSA9OtKK9Rn
idxm41gWsyAb5JWqXkIOPg6mhuFCD5Xp1NOO3n541pEbE0P1Bc1XmDyDUq0+VqJp
hoevlCEmDbnfTPpqUWcybpGLeWfteQBWaMyiesi+PD0ABQxdvgDfmoeBzL4QDUhM
WkwHe2IthuUvAwW589PKiKFucxkZnHaZKYAcPzlWsG1+i2NSzL2TbCwFA9+pPOVJ
Aw3QbPmTviIgrfE3/oJbgFY+g1yoh3YALpe4vUjo9g1qW/joqhzLq7ZRMgxkZwyR
kZp/CXFUR4S2iLmU/4ST521avzBi/Mt+YEATvjbPAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQBGQ0xssTKZRLCHx6yhG6Wz
eA7v0UIrUUtLBIwN8/EIVuYScv2o29kWQ3ANYhO10Jbj7uh5Pblk1CGkJO+kT54V
Xcl0okSDZ+IEV2D3ogUfOK7KLliHEy5WLfzOs829B38Sm5gQiAG8JdTHj9gPkb65
Cai4YYB2CESOdB16Xa0h0YKwtp+ELgaOOnGrjzUDEp+H+1tL6FOO4+rHxzE5wUAp
uljmdZ4LMqHh9vigEf+G2qpjuQzDgB0yPaQlDNnfus2MkTkyGgRBwPr27bdnNmLF
nJhA0ZnehIuVWm7CVRf1NJD5RbHBqVY3Lr/JElU9DAudrlHnxd6vh1qCquzBoRyr
-----END CERTIFICATE-----
"""

SSL_TEST_FARMER_PUBLIC_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEAy8aJlbuPfG29iCDf2+mhHv0wDuypUxBzTErkgPTrSivUZ4nc
ZuNYFrMgG+SVql5CDj4OpobhQg+V6dTTjt5+eNaRGxND9QXNV5g8g1KtPlaiaYaH
r5QhJg2530z6alFnMm6Ri3ln7XkAVmjMonrIvjw9AAUMXb4A35qHgcy+EA1ITFpM
B3tiLYblLwMFufPTyoihbnMZGZx2mSmAHD85VrBtfotjUsy9k2wsBQPfqTzlSQMN
0Gz5k74iIK3xN/6CW4BWPoNcqId2AC6XuL1I6PYNalv46Kocy6u2UTIMZGcMkZGa
fwlxVEeEtoi5lP+Ek+dtWr8wYvzLfmBAE742zwIDAQABAoIBAAJxpSDImRXy8vz4
9/ZZPPd0Lelf6T/3yIo3s7ngNhozoF6FhsEtJNdQ2qe6/BiI/N3S3LYelAEI0tF1
HfxuqrGkUSUz+/KqWSYdy0v/59gBjf0aAfKXIjakn4AQGkjAJS6zqMc1c4ujkd93
bkzckB/H9StU5CsTpksIPWvPGtByctkIHvbzjhjm4bhGCclmN7WASjrhf4i27xPA
dmOGHuIBp9QFPdZy1VDWkn9N9gO0B3IZgRUxhBEd6fb4L2Yn7uQy3J1CAPL24gRj
fKH4lLPZKPMGBIWlaLbfgeU5NViy661CFMGISS4Gy/5YOc3RpZNVOfNrfTMiij/o
rJebx8ECgYEA8Nx6EYoKrTiKM5pn/ZcWpL7Crz4h/OiN71HPx/KugbtpqyN8sUe5
bLmHtCoIGKAJ/A0N5L2VrM28URpdZAGyGalS8y9PnvN1V+Ceo/0DCstrNyR3GEgE
Ge551NTB1Y/FXV+Uh8MxnI8XQDfY7O39bRLEi03ygB1Vim3Glv59rJECgYEA2JVX
b441FoSgpNPewffexunmmoUWWVqYCqNZBa7vsGZWclbM80i9aGiV3YOL9WUIMiYm
7t8vetEWTxvDE/DQ9A10H+0gyHc+F9i4qFj4jr/JHJB9c/QM0J1JFLVm6LFed8Sh
32YroTU8ozo35L1Vqw2TzvLtDoWTQt9xwwwC3V8CgYEAqylCZaxfIfGH5FvZglAX
+0LEPpw8sujf1C2jJkfC0tZHpSI2vhRmJGlDkjg8QKOZ9M+iC+bdsksRg+DkdOjv
k2TMWZxJTH9d5/DrS0XFH6qmqizzpQGoSHU7ho8an5TngA8oEcIYQ1U8BJqrQNNu
wFAR85yUs44k95ZDBZIb5MECgYEAll6RVx1XlGu+kNGcnfT3EgmBFpUCIEV6vCAP
BZggR8u1hyrmnp5JCr1aF3JmcJiVqSO98ii4cT/Lil8PAMA4gsj4b9YV5biMPsgz
5IXpBEr5WUCkKSYu44PgImWTrki++s5lSJJKBmHM2NLpzYD6gokprwwQM/3fcRVr
iWzEbGcCgYBtM3zw+a9YZOOA6TVHWlcH1F3nEVXwjBd/DfeIfmOSrrke+nzUxEKb
91i0MZ8G77yrEOpLTOYHkrN6GZOezk4VcIy4LUmXXfQjKdfrB/Noj/ZKsQlVzVrc
FnKAyeCwskveC+js4QOFcXNKce9h6ySg3Oiin5iax4e89NipSsfeYQ==
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_HARVESTER_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUPBezel0+gYQ75ULotR1tSq2zBDowDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDPpF301kklnHmr9lz++7iRev33RBfOlOugCeTFDnyjgaR3
+PQoAm5yUyVa/Xp8VCe6b8Ksrcj+2xslCwhTtei4TEL19yvybdlQb8ZkZ84WBB0w
22xneZur9ZAcNSx6+SjBVrmXb6f6UA/NXibDheYnGrNwoCJEy08+j1wSscu/9wok
Lq0b/ftD9iHh59xwkuQEm+zzlp6KwdqTKWqzm8+Qpx/9d9yG4EesEGBTq2FIr6kK
BYsVMURKJqmbiG4S2ZDTr2aaVTb3wluV4l8tRPk7vfU8NKJ50/hgIYf5PrjOEN+m
uwlRojkSIPKQi6xWfFaknToV1+BmKPvbjIfRC6tLAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQA4Q1eMPOUARR6xJ9GiU7mn
shWCKbiENr/p16ZjEoFa+uDT+HYkoQc9QnXH7tAkXDnzYmogGnkd/8G0QrHN/812
+yaN2agSlF852Hm3sODQ6VYyP9GjDY5OPTixZ1ZZnezENESGOdrO9FLlJPcEqgLR
MUq82Opri/m3iceHOAz/Zl45dtPkzGhTLwoSmkaBebXPMBfEQyDuGZtahdoHL34o
CKMdDUIe+PI7dKGgzlxTR56azOlxq08U4NkUzflyw8i/lo0nqWblS9i3fOcqxTUF
F0Wo/vgqW34MtM1jmK7J89ob0Shyk4tp+lTFpKyZGYx6gsSZhPsvUKgFaWv5vO7w
-----END CERTIFICATE-----
"""

SSL_TEST_HARVESTER_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEAz6Rd9NZJJZx5q/Zc/vu4kXr990QXzpTroAnkxQ58o4Gkd/j0
KAJuclMlWv16fFQnum/CrK3I/tsbJQsIU7XouExC9fcr8m3ZUG/GZGfOFgQdMNts
Z3mbq/WQHDUsevkowVa5l2+n+lAPzV4mw4XmJxqzcKAiRMtPPo9cErHLv/cKJC6t
G/37Q/Yh4efccJLkBJvs85aeisHakylqs5vPkKcf/XfchuBHrBBgU6thSK+pCgWL
FTFESiapm4huEtmQ069mmlU298JbleJfLUT5O731PDSiedP4YCGH+T64zhDfprsJ
UaI5EiDykIusVnxWpJ06FdfgZij724yH0QurSwIDAQABAoIBAQCse7UwtPB+NcU8
e6I0jZV8A9cDl6TPDhXNE2rd5nLSv9aHcl4e/iYRAdB+DSG6NY3q6PulxZa5Xfjm
hIuePfXbYj4dY/01nNpwXulGbpRa5X/6HcunCPWvmoTjaahRXrfkIs54ZV/8JpV5
20QIFypgQWXdoKfWzaw+I4gwX1EJXAR2Vu3NoSt6Do81cClPoG3/2oMnHHpmAXba
hGD7AR40XIWXP1g+2beV48dZFIUTgY6nSi4lUKw+yKPV2iMYRfZl5zetRQrvvRqw
0cvEWLM9HomankYSJKik99Ew8oVkUXik3RG0TLTAjCpu4kB62BHAlDP8DROMUyzU
ukPb9WfBAoGBAP0B0Fkr2SWOLlbAwaHAxQL/rpmYdGtlwxhyIXgp3I7rrK7f6Vpu
5uUofc6F0BPAFm08zNupTwFXwppYV8cq1rZIqKz/lrBYOQ9DySTvKQ8PFA1POqaG
6fGpQuMOS0eaFHWFq71D4iOCeWfOIVWELTB/fxepBMWMt1yLZH5kFvk/AoGBANIZ
LGMwqn3HQ44q0gr8gIHNn8ckD37vmFXhGbmV0CNQK91UKCXvQsAAAG5hpSO9AkG9
1Me51jdJkdSCD8Sm4xJQgG9l3eZETyCKnpETsOO21NXwLLrApE3rkoLeXl1FLvOi
yD2jmIqSBSmmQQk58rJP00m8dbxEg7/atn+r9l71AoGAdQ4OyALKb16tv8NkkqKu
Q6nprPRlAH9PWRMMuxybB6Cv/yRnQUjIGMO4GQG9+DVJ6vZTTVmyzMs+pMw0qFgA
iMtrBsclf6cBlzdka2sfEvGF9HZwizUmLYGDCavGVnVJ++mBQIqUqJsn8kfjEj6Z
7bM/KMIxC/kDqJRxRSynmF8CgYA1ggnG/Vb+j6hJ5ef2ulY0ON1Ph4SWzIicnqbB
mEEZLUrupIdRGf2DDr0NX7sCPhn8Ck/NyiufixIj8Df52pMMUApV1LjN82m/nO5D
fqNXa3ILRo7txTBTVALNZ7rawErz6+pZoHeyGvxLnjcEFYkVFrV2GxRVLZHsY9SE
8ZPenQKBgQDmsvYTyWnU55e/dLPxu/FdVxrCWHN7j+thboZiK/kOnAyf8jc3Ttxk
08ey5OgL6v6YZm3sbxgXoHjIHsBa7YUEzwzDImSb+jbGTCYZmXolv0GXWcRjEJP9
OdcwpREJMzZpGZ/QtbiTyXxHEk8vAgGur0HK1SCo6RBuJjtK2XkiXA==
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_TIMELORD_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUePevwBw17O1QxSIuitVeEUltmhUwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOVoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDYCl75++jqmfKWVWn7sgEgX4IsHVVLdX37I7jnY4nrpdND
ZFHl+JIUHeMPzlnlEHBZHn4yoimLXhuia8VPrss8YH2WcaKyxuTfeoKjxEbs8oqi
rbjkmwTN8SkcMYoO6wXJqBPZaS2L1sQo2HJGPa7l0K6KjDdepuVJiEuoBro5zHLF
Le7g1V0aRHbqDGB6WjbgFh36Jisw6yPuNgcYDDl6fVWN5PT/UxOg1qVNrxPmNUrg
uGlZsCrI3/UwK/cxEv00H0S5qUcLNMb/9wVAvuPhG0UoTIkFPxelB5hSkhujaGRa
5S9N4Wr+k2dNMtdO1swClRjWGj0skG0DWiKb0COLAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQAmIlBN0D+64rjkWlN5bk5L
7MVKhgRJmhbEgHW5JmbBkAMEUqf6reyuz+L0TbR2yAkujw69HTAMNvDuvoNewa8Q
x0qtlTJLOgYz/D1lS+Ti/EHcmkrHIKK9O+lBin2RpPoQZHaYLSpRXt1QMqmfn/ct
9Jjx8AnKzTC/nOGDYOYskkPCiQ7JIaVyNFAmFBNoQHVlztDw6pR/Gpnt/jJ4Eyz/
JZSXHTTfqGm+4rRDfwM7yREwoG19ab6T/6Cbbfi5BbalJG5RqH9V7gLQspQIvwlk
ByC8qnXsqTu5UNh6GweiYWSb4zIol6a6i1SGY4lZ8ejkQKezCX5mU1GKkZhX7rOw
-----END CERTIFICATE-----
"""

SSL_TEST_TIMELORD_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEA2Ape+fvo6pnyllVp+7IBIF+CLB1VS3V9+yO452OJ66XTQ2RR
5fiSFB3jD85Z5RBwWR5+MqIpi14bomvFT67LPGB9lnGissbk33qCo8RG7PKKoq24
5JsEzfEpHDGKDusFyagT2Wkti9bEKNhyRj2u5dCuiow3XqblSYhLqAa6OcxyxS3u
4NVdGkR26gxgelo24BYd+iYrMOsj7jYHGAw5en1VjeT0/1MToNalTa8T5jVK4Lhp
WbAqyN/1MCv3MRL9NB9EualHCzTG//cFQL7j4RtFKEyJBT8XpQeYUpIbo2hkWuUv
TeFq/pNnTTLXTtbMApUY1ho9LJBtA1oim9AjiwIDAQABAoIBADaHt51lpk7A+OhV
OuEa9wnS0aqM5OCdEWV4iM1u5HskagTSp5FigoytVojAzxPwaEs2VSg2okHZSSRg
C3enaQkeWdTFbgmuBksaQytO+ZAJUFDrhIW/UyEgQ/4nyu8lB1r1H716vNJVLUIa
EA4Gt8SuPcAe0EMF4U4EJwSMXikEKLgehaHMFJpwP1stdj3m3b8Z2z02LeCNlhkZ
i6fo3HmFJp/MtmzcdONPiA3eN/CqkOIx7hqmtop6TNcVbhh7ygir8be0x8uMtRnm
A7mkkzT7fmTszCo/vLQuLXSIhCMxveHLbjAR3/kNu3m+g8k2dFn7U5lwB/yItZpH
UGbM+xECgYEA/8sAm1IQ6hpoAkAd1mXZ6OoKFz/VT3pqdQi8owDTJuvrkHXJEVHx
g0qT9/Ejyw741Y38qSLUkcBKs7Do8/SOsOEa0WXn7Z6yA57AGhyvMeDSziLDebib
R2BGkaGlUCPY2ur0G4TVbfZRiBCDt/24W8MiCxHMQqNhc4LfB6GSli8CgYEA2Dch
4NDCrcSzG5Ayy+SSMzBF3pEwtrbp0L8pZotYU9vUh7PeA4hxOqqwkJ3o0q9ky4rV
/jttDS0XwgxdlO0wmOQmPlf0+/H3SRyjv/gDPVoIGG1j5qy+8XsXWwXjKJrcntT6
ZNrnppH/AjB9REb2jHgIpiEaXEWi8qV0KWYtjWUCgYEA5ZCKBdt8JK34RHTIkn8s
N3W8AqueREsoOcrSiTQXTOoUqrya7HlvWBZZf/8h0X6pYfehFyaUQg6ivGv8Vd1Z
JnOQPvfHU6HGP6t82rJ9pTUR3ECk0JwWd9N24gKBu8/AMpEE7aGkbqDPAV9Y54rE
p04UQn0BpZi9wA5nHZErXq8CgYEApdZrsr8MgzHa2upYO9E+XerJvHOKdKqJXmYx
QAjjH7b+JXPNdtJBXukq8lFcEbUbz47WoeA8vZ13movXf3IKE0rzFmBqdJBbxoUk
CAsXmQgkK3BD7ttSvW9XM6apXCyjyWO8Bn1srnNOsAQ3IJZ5ZrZuk9XSHTIAULKd
DV/k0o0CgYBDU4fugQUVcIlKPdNTp38QzzU70MJOYYorGnght+AmdahW7HA3DkKp
MFaBOlZh1yLwsP+0C1PSFTq3X/8WODHMejvSqVZ7wPGVRVNNOdaoeB0xZbRppeaf
5sBZcuEojNjNtxF0uDd3GWGuZEJ6dDTDkhRaZTpioFS5o6+3nMBx8w==
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_TIMELORD_PUBLIC_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUBqJvyp77D14V2vRSpQEMvr4LaOYwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOVoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDaaW7GUi71V+7SiknlTlrVQBokOlO7l6AJ8gjq3UOvYs3c
jd+XGO+hG0QpWhGMFwAD7Vfw2QLqA4MmgkE3xFvL+YadaOI1SkyoLMj+gM1W3NIu
wYn3dWQiB9i+u/Z5nv9rOYtfJAEllQLxY3j+7B+HQsU2bP0oRdUOCgDS7BR+SVuV
6f4YkHxxdcwW8AE4fyqZ3R/ZUXi/Op4H+/9e8nqGPFd1TkARJtXErvS7zRwF7PT+
h0q3m46OyYqYLVmVTAHCNWgDJEnyzYCzzocixe/9EBwVcqE/DS0h4yfxwmJovhWe
BMg+M+vaBQYOtz+23xbQfEW4/dxQecl7DKqmfCPzAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQAD9AGq5YOxEAr2veRCi7n3
t+zdX1PCsjIyYRPCsLpjBXklC8lKTUK7QI3P1zAZFkYVxGGn/YZf3gVUMNufFc+n
KY9glRor3qH8AlC7dhRkwj2QOkIS9442w9wgAiYUdofsY9JLV7gx04DK4UuBWuhZ
usDZo7sw6IlvynHP8KOr60SLMlgXb4uKYNAokiC14PL4kCk31i8XaXPqcgvx2WLF
8z5OqUpJrcoIZPQwkAI7ajD2crsJbJ1IFv3ZDNTXfnjoGJBfCtWLk+mkZHpimMBH
7bEpyOLEL90M7zNkEyGmXN7HcxgyLY9LNnRoDvHTCuYMDzJEeN5EBJqi5lc6S3LD
-----END CERTIFICATE-----
"""

SSL_TEST_TIMELORD_PUBLIC_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEA2mluxlIu9Vfu0opJ5U5a1UAaJDpTu5egCfII6t1Dr2LN3I3f
lxjvoRtEKVoRjBcAA+1X8NkC6gODJoJBN8Rby/mGnWjiNUpMqCzI/oDNVtzSLsGJ
93VkIgfYvrv2eZ7/azmLXyQBJZUC8WN4/uwfh0LFNmz9KEXVDgoA0uwUfklblen+
GJB8cXXMFvABOH8qmd0f2VF4vzqeB/v/XvJ6hjxXdU5AESbVxK70u80cBez0/odK
t5uOjsmKmC1ZlUwBwjVoAyRJ8s2As86HIsXv/RAcFXKhPw0tIeMn8cJiaL4VngTI
PjPr2gUGDrc/tt8W0HxFuP3cUHnJewyqpnwj8wIDAQABAoIBAAgcllKjRr8x7tq+
7BOqH1GAOfhSej2jWfxEkvQ2F4dyAfdJBZ7U/EFdg+CNSc2MJSIANJ3rUWJjkwFh
IQ7zKf2fXNKHzK7x/XrZfvWEzxuExAGcioU4ROnGpNQgovzNXOscjgKQkcJVkHBq
o1nEZz9cyXpF/fj67vjAgltTDUfLXBmymdyAwafnOQDwYmk6HwbYpN2gMpRPp3K0
iI0psoGgJNcsrBoOXGrt+XpwNup17nmrGKOJg3UhhJ8/0pBXlXvl1mr9fh5LMrtV
tgBVavWSR65x/U7gZ8Fv5y5dzaEqItN8IC8NYFr2Sj6KaKjYda3xgES++Ancib2w
Odok8vECgYEA9at5cmAgi+19iRrCle31fpRwOleGAiEJRrAYpXwxFBK0wgE2WWQO
kTAI5ptunzDVskh5dZVOp/N603H9Q6/kt4TwPvgpgGXs9kMySY02+/yrYfk7tsiG
WyJcyTC6exH+oCxcuAR1snToEqLwK7lvj8vmG2xSkyBHduXmDDpq9dUCgYEA45iJ
1aYSUMm29O1Og0/zc/cn7lA/+j7gzRp3hJ6TKEIHOCDYPXm4TASD718WMapMzWBG
YqKZkz0tS5PyDQtAZJaIBPxDqc811hxsI7TAdeVWgGWplZl5tj+CGRlGluHQ60gd
5WZiXLLgPgJSaoYML3fyRxeHp1bXSAqID42jrqcCgYBRQu0jJiTd5Cq4JT/kNBgW
TKdqxPSw05ir5yff5VqelylvWtNNcklXQRpQnW1T/uVQ2aRDfqXL7D/enT/3GcF/
6ttFGPUxYYWmV8fUiH99jrBp8fUMmvvGPXGPoyyr9OHSvZnVzLPZ2c3CDF+BNw9t
8zYkMAqIXA8Hhzg3J0OmGQKBgQC18jP+gI0T2K5Sj4O8piXSBzv84FrFC49baFUz
yDGMdXyf+mHkg3I6E0IGEtYDTkmnxdd+8s4jTzzuto9CrkQEJfMHSUQudGBtkrtA
sDdMCNNmQd+YoOEUOSEgP3yXEl+aOETuJzNrN/Tcna6VSiRjrlgZTU49EMd56DFt
faXk8QKBgQDnISLybHaESDgFLCMspkHERh7RC3yvLb0qQLmxWGCPH4W+b7Zh9y3T
a2b5saiWvXYtGEQ1Qrfwn6vmchq7sm7igqJsowyvJe4co3n7p0dtNyXJklp1jbQv
e73hgWSXlZswTNdec8tnY3Gh6UmGMKpgll+m8bfDPpb5Taf2YfB56Q==
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_CRAWLER_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUHjQDXM3yOfjn9cS6/zkVXNvAcPYwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOVoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDlFI0WeUh9oK/l1+Kpa4JzUd0pgEWwyCKrtaDIMzuTPJEB
XA7OVfkemL9q1UAZ9FljEYXzFXzTk+OXwN3I1S4lFgitdc8xBQLljv5HelEq5H0P
ypUAE+HELxFQ0uWQqLoNvIqrEXh12mgdFvyIqyJj86Cvqv0J4DOEgtsXjxGYO+/4
ioJ+oO0yfe9BbL7rED+gfqUKabloEeL0s51sWrqlMOyOyq1MX1rP3cMmSDjQFcfV
AqY1IF+dgzw/WXbUjvFJPO4IN2lY/v5x9N+i946k1pFvecfXIseyZRym41kpP+Xd
JwxPasgMqcT/3d0BAaQCGpgNkpe3etisK4+0pXp3AgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQAcxIApP9qu9aa898tHtK5n
bnG1d5B0Tktv5BXkwS+BQ8uytnm6QtOVe4frxUdKUJEK3085Lr0B/j3bJNyJ4m96
fjpjScWHwk7N0egLq761XyBv7D1BjgDAsAYyYNOA4mgBqzfrJtLoDjnf8SMKpocr
ZLDXb6Qy8LGj0D5oaO3kWLzhWEdhFBcEVN/PnnBcuj/vT+MhhLGGJJk8WMp7dajB
I2+E8WW/RxHoXFHnmF3B7TYPhcEZI90AxmIWGnNW1cLArrvnCodpPvc03om+U6qX
kkNKgCYDEIJ5dTJWyjA+hwB6XjRgqshNNcc6FwXb2TAyQm8JQkIlRX4R6bjm5KEr
-----END CERTIFICATE-----
"""

SSL_TEST_CRAWLER_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpQIBAAKCAQEA5RSNFnlIfaCv5dfiqWuCc1HdKYBFsMgiq7WgyDM7kzyRAVwO
zlX5Hpi/atVAGfRZYxGF8xV805Pjl8DdyNUuJRYIrXXPMQUC5Y7+R3pRKuR9D8qV
ABPhxC8RUNLlkKi6DbyKqxF4ddpoHRb8iKsiY/Ogr6r9CeAzhILbF48RmDvv+IqC
fqDtMn3vQWy+6xA/oH6lCmm5aBHi9LOdbFq6pTDsjsqtTF9az93DJkg40BXH1QKm
NSBfnYM8P1l21I7xSTzuCDdpWP7+cfTfoveOpNaRb3nH1yLHsmUcpuNZKT/l3ScM
T2rIDKnE/93dAQGkAhqYDZKXt3rYrCuPtKV6dwIDAQABAoIBACaxx/sp1WLTB6Mt
Si4P6OMNyuu9rQflZmXWsife74s5Sc0O/blAsGnIp1ymVMONkKO5TAT5djJO4ry5
aT0JNVgyVpjmM282R3z7IFx2fstd+Pn34B85PINk5+ndFcXPmk3DsTCgLNbLY5a2
UUXMkmyLtHK30TIf2tPhPpzGH9S0yS63lhqgKY5FSv5ROA8gPxiXyZTXfnwllWj0
cAVbSPj2Zh5JeWnaJ5zckOFu28SRKVhqw9xYxyeVxi/3SANSX5KL6Y0SUDV2Nj4B
YhgqeF9d8HN1dUR5OI+XU8Ri2lRS4Qhad14fQps7mA/CGqyza6oPcaPX6JGrRqWC
8wXIflECgYEA/9/CklcL7A/19EhcQZ4k1WWEg/7QPfMR+UCbgYY5wh6Xna4ievek
FReZ3Iy5eAtTeXOeDltkWu0+U/1nGEc2OMedjJ7RuLpnf0fie63FPxy0lgA5976U
LbjKgGkTMdQtIrkrBEfc3Qt//8iGIj4+jXm+AqwyEUHY+tCPoOMjhY0CgYEA5TFq
QrU+O63GEP9ZguONbXLT98ZWM9KEkkbCkW2Fmwvovhh91mY5dTI5r2wZSypOFECl
A7qL4eY5BqaOJMVBV93QIH+lq//k+WarYwsPbMQo1OAC+dDnGioT5UdTlTKibuOF
oop6DzUw6MxcKN25acAdaVHNBVYL3e6grzTGFRMCgYEA1JNig8+bkQgFj2hadQ1c
joljmYGmIgGCCjDZwLbx30spLQ/Rh5WblQZhhnDHWWsXc4xRYJ4e9sepA4BAGVR4
QJ3eT0hW9gCvIXhoOdY5S+CeSwxvHFsRoham7h3dm4Up5nItwcWuQVpfgfJnNaW+
tCzTZkcG5X1uNZn16XUrOrUCgYEArPu279JLvk9SBtIzhQ6qV8krIv+XYFWrqZuu
joTTKkq6x39EuuU8yK+3yicUkB7F/5Mq+L17hmx07ggXkLkwsFjCFUVCjab98x8C
jCZ05erHxqOsm1CInL7VdV94PkuRPZCL9DoAFipXV92mUVHH94EAsdUDoNto6YCJ
1rNGLfECgYEAlkga+K/PXl9uWtSRUUO+tUdV69gflR0S5c8Zwx7qi87nm8ZWBK2M
tv4KY0Id4rhk3bV3L4iiL5ZGW9gaFP90A7OR4MG/fWCb+UsTc5G7uOqj7BvO98G/
i/DjkOB53i786L34LxzhW7O/8XlAJeVIGqrl0/aC1X6CPUQ0kXhYgrM=
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_DAEMON_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUaYsQ6TR2c6krUj5rtYG1fdLox7wwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOVoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQCyZrrU3fBR88wFzlro0PddB8t5ILWZW5T4BqfGtJZUXgFR
wC5beEeufrlXk16bFIxLV55QWEn9PUIn7W5dz6sVmcj82/v52QOs0BctIhkc2ZIn
fj3++1C7hyNcYVAk0aPWXLvSnLx3jY2qGunO2vT13APoi00HeBxYv66ac06m2n82
xBQ/pZNWwCg9XRjPADDk9c6DJTPJGHgeeTmXOAI/obAP5V0FnrkPpC2lXXWdXd+B
Yco12V8sEFxocUQrRhP1tN6B9Aw0w1qTrTdD1jBF91zE+OCeQ+lgupGMZG7LuvOi
wvKocJugWQA3PoblbM4NYz+Q1QR9Z+sjLr/SdRLTAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQBs4EPhrB7jb6BH6cMCtKTr
28TcvWfC4eBd2NZKCI5ZsyerfwjIg1F/G7WgmToNE+FR5izQISSTzwzZhbEQlrt/
+bUeuUWHM6wk7on7cmSw7qVt7U7yRBFW5DKVcKKjO7s2Bs5JmVH953RTmRA7pM/0
khHIqJL6GfpxeD4zWO4P0Gipxr7ugqKNTViqZQOlIpj1vZpOOGWSpm/oPmp+XphI
TiwUtjaIW88JOy8oh3v6vM7FOxiAoqVhnkk7jnvYBR2+zrglMY1kNz/xQHYsBBDt
b+RUdVHiPgcPvlD+RhNxhqjfCeoAA8BQ5w9AOpum473CA7efRTKhusjP3dMz7hop
-----END CERTIFICATE-----
"""

SSL_TEST_DAEMON_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpQIBAAKCAQEAsma61N3wUfPMBc5a6ND3XQfLeSC1mVuU+AanxrSWVF4BUcAu
W3hHrn65V5NemxSMS1eeUFhJ/T1CJ+1uXc+rFZnI/Nv7+dkDrNAXLSIZHNmSJ349
/vtQu4cjXGFQJNGj1ly70py8d42Nqhrpztr09dwD6ItNB3gcWL+umnNOptp/NsQU
P6WTVsAoPV0YzwAw5PXOgyUzyRh4Hnk5lzgCP6GwD+VdBZ65D6QtpV11nV3fgWHK
NdlfLBBcaHFEK0YT9bTegfQMNMNak603Q9YwRfdcxPjgnkPpYLqRjGRuy7rzosLy
qHCboFkANz6G5WzODWM/kNUEfWfrIy6/0nUS0wIDAQABAoIBAQCkNUC1HI/OryvV
EErHT0jQnZMhLAVbYlutLFsT4Dca8ugd+lnxIbZBHC23OVN0RSaZVs3GwG1iJheV
rsrOmI6uND422LmHEMR8NZGPnnok/vKzXVTtNR/rbIkUyBG/wiMe9X6AbnpfRLDn
qRwapZmiD+CdFFNTk6/r1It/aLE1jBA1ikuxa0DxsKc3V8EGd6A3HlG2kusnUmyT
uZcmpal45cBb4lfNxPb3wvzYwcuWuLGeTWkUhPSK0GepKWd/1A2UOb5Hv8bogwSI
mSDtSTJc82+OQ+q95SCdokC0+DkxBWMvIrNOSGVKQhprlTGNPZYBksQ+GPOPjsHv
BJhJNblZAoGBAOMaSJIQpBzzJ8i+Sd4dF2Y2yeFH8I7C4EIRnkrOwBk+MF4j4y+/
EJ9USTV6mn01+PjYFpp2E2vwzle9W0mbSKrKGbSg4LCHPDW/jMr62gyp211XsjVM
//NKh4vLWMVqfukqK7aezzv4/ld0B932h8O+ydr/5qIJrFoG5aIXNQSPAoGBAMka
CBjKHzBaf4e05CE6EuVhGxRVlAv/5NILQBWxk/lb4+gE1yITYCZIMOOro7Riz0Mw
eNTYxgAzmi1m/2pyEHoBH8Si8zibDgvWoLCzxHUZWYBVjRQSnVAQshS+CGFnUVXk
7octxIxZ4sF0VdNxEsQxtmRC21b6l2p3myxhxxd9AoGBAJkT61zBaM22vNdcJJBa
DZ6xSYzXn/eqtdzy4T7Nc8DQIWwjRoy1fi/XcW1h0eaaTdL2XO4KrvawH3YWxgSs
bePpTEBrvGhhD52vGujrA7VLfPjDFvVEjLXiQ9Ff5NtU+GyUJ1j2XPuwz0gFhxP/
i+3OmWNL4JrSaGFaisiiWfNTAoGAQDPiX9+vo2VRD8dNidobjEW7Onuz3zzcFb7n
v8VIw7R1h258LHHZIX+stBLpGb7uArpn6TQnyoqu703Rv28ZGYpgVQvc8LZ3/omQ
Fbnev6/PJiyp9L2YoaYi5IJi9I3iprSG/ns7I7/SqAZ3HCEeZ6qjW1EI67TnQGNd
FAdHY5UCgYEArkoftye658CBQzwXlpnGs9OiMkOD5Aadx42IOKwB6lMTlZfWIF6l
q5zjv7JT3jR7CNmCEY9oxKfc36GHOLg821+hkHH0E258r0TGoblzTyBJIWYDUnw4
MZHc7XIwanFXwpIjK/gPn0xtVph4JpKPTaHePskAkN4lPeg/KtEjVt4=
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_INTRODUCER_PUBLIC_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUdIXMFTtdGFLVnK0Pm2qmcKCMWlYwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDcyOTE5MDgwOVoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQCe22gtTak2xx5y3fPpK9vwCsVWGODc6AxfKo3cGe3Lw7xu
e8R8QPUCGEW9tzGwLvVuPzFnRgnMC+H2sGzh7eSm6Q7+3EyllNxmcVeWcrVUQUZc
zxV+POfQLyWbMzdY90HJRrxVXhK2U5fwYjzvZcjFVm0xwH/tWiFEXRZX8oMP0Szz
A0IwirL/pJc3Y+yvu149pKiGqX6m3id7S5jC6O6Q9b68iBurj9Ypvl30pZbuERTG
uOsMDFhIeQp8imZIqBBXSA0zJgWbc/cYhVkOY9D2RdHf41jqQHDab1J+Ce2f00I0
4OPLJARHVed8XsKD+yjiLQxcuF3/hgSSsQNBKLZ1AgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQA7j2j+OoDypCwsAdu3I5le
TezpdosGIbOC9Xcpuu6kjJoCVDbALWNId00pVc7TWLiskoOkaHVsMh/Wz4KkRnah
eNsMM9kmHWLhv+zdECFIbm9IOuLgnbVkwmTaCIhDur7vSPDbwLrkU8ppk9eyYdEj
hl4tnpH9I0PY2GOky7POSgV9kXMEuoW+uV/i23GiBGVuyrklgDtaUXoUuCq3cI4Q
dDz2c0kOQqYlkyanIEGLXalKjaSc73z1bbY2Yqe0OXbwh1qxHqW84gAd3WiK/TAV
DToeMdyg4Z07yaKBG0n2Za3tcNVSH9bVF3jIWvIGyDNq+IRvevPZbKQJgy4lZypZ
-----END CERTIFICATE-----
"""

SSL_TEST_INTRODUCER_PUBLIC_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAnttoLU2pNscect3z6Svb8ArFVhjg3OgMXyqN3Bnty8O8bnvE
fED1AhhFvbcxsC71bj8xZ0YJzAvh9rBs4e3kpukO/txMpZTcZnFXlnK1VEFGXM8V
fjzn0C8lmzM3WPdByUa8VV4StlOX8GI872XIxVZtMcB/7VohRF0WV/KDD9Es8wNC
MIqy/6SXN2Psr7tePaSohql+pt4ne0uYwujukPW+vIgbq4/WKb5d9KWW7hEUxrjr
DAxYSHkKfIpmSKgQV0gNMyYFm3P3GIVZDmPQ9kXR3+NY6kBw2m9Sfgntn9NCNODj
yyQER1XnfF7Cg/so4i0MXLhd/4YEkrEDQSi2dQIDAQABAoIBAFUvw0XIB53gWWuv
qkS3yFGI9RmebbcImNnPsBL/tI04gaudo1geXUSqT3vU6UhZZjOgn5WJ8WcCWEI5
ian+MEdxYyE4ccG5ehdVYGrFSIfOGTVbt2Litkzle30mkd+beCO/Uu9UayRq32eA
xrx6EJgCKLwtRmy5kG1gncK7wZyQ7sqVafhI2F0giPB5SsJ7QYoLB2bU5q6qjOCW
SBEGHnPmZRGD9QS5/VW9uuEIw016ExMwrYBIThrSr0mxlGZNEYY9oT/yw0WW/8DJ
L2zE0jSjo8VKFyZ2ws4JTXaO+pVm4bRGWq6yut4+X4LJJZ31mXw3DCyORHJJMuiX
xnbebwECgYEA03hqEPmSI9E1+tdTeKVWc6wMjLMfRBxm+ehAqySOGP45JD1wOvsk
is07Z+/mE1s5fK2jtCbgCWWvbGc3rOkS+U/W1Ffya8aF7VgzZg4fE0Eh6LNdqOYg
A1IPzFHBAMWSw6kOg0loCz2SxGgGzBi1NOHk94/6qQKjd9AQAOFATu0CgYEAwE7N
uQLaXk3G+BeN8/vl9w8ke0Q2KBmjF1lT8QDpByZtZr/ULGH9I/hBzjneMWtZ5FdK
8EJtpUjDIjfQuj+UN5SW3wO7PZSi8Qte8WJfNITJwdTzgLFzSPWqjmQFb1sHKKBk
a6ifSlLyLlFpF6Rb3k9MRjC469H0K0/s1NBXjKkCgYAmGr95+Y50t8DzRxN6XMh8
JVdJjwkmO7vucgQdEoQsSrq/JfbwGcrW2+5FZzvc5wYvyhYCYzLhXppCfULR/Fl3
5QP/1S0QcIA3aToxvH+q94SFqrRIt9ay4nmIRn7xhahRa0f7LiH6TIkCrP/0jMsy
ETYxyZbQF3KfJzWLAwuUbQKBgBhepCFK5ENJFerAiWOey1CYoaXhSf+g6BMAZevH
ksAuNXTs4EVTkZGqVyR8HtUiZnzSNxPlRyHH67p9Wj0P64dzk/7k6ShvmopzR2/N
aIUuQqcGhmkLk7FuOFPdBvdKkMisMq2V68GuazRFasztLhso72CBsh1oOMn4UenS
ERYpAoGBAI9ue+8d1W+eBv8C9uTica+/FczcZpTrvKsa5C4MpYS7J2aNHUYAgkq5
GMfmzPpqDCDNy5xupTdyD2APTKU4MRtQ3Jcl8mTbh5x4pvcc+3cyfx8GCcVbpRzN
OSm2k0N8WDFHaQF2jNOFr/ZMKXg1/GsWnJP+ikyQaOhUbJMT6B+F
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_PRIVATE_CA_CERT_AND_KEY_8: Tuple[bytes, bytes] = (SSL_TEST_PRIVATE_CA_CRT, SSL_TEST_PRIVATE_CA_KEY)

SSL_TEST_NODE_CERTS_AND_KEYS_8: Dict[str, Dict[str, Dict[str, bytes]]] = {
    "full_node": {
        "private": {"crt": SSL_TEST_FULLNODE_PRIVATE_CRT, "key": SSL_TEST_FULLNODE_PRIVATE_KEY},
        "public": {"crt": SSL_TEST_FULLNODE_PUBLIC_CRT, "key": SSL_TEST_FULLNODE_PUBLIC_KEY},
    },
    "wallet": {
        "private": {"crt": SSL_TEST_WALLET_PRIVATE_CRT, "key": SSL_TEST_WALLET_PRIVATE_KEY},
        "public": {"crt": SSL_TEST_WALLET_PUBLIC_CRT, "key": SSL_TEST_WALLET_PUBLIC_KEY},
    },
    "farmer": {
        "private": {"crt": SSL_TEST_FARMER_PRIVATE_CRT, "key": SSL_TEST_FARMER_PRIVATE_KEY},
        "public": {"crt": SSL_TEST_FARMER_PUBLIC_CRT, "key": SSL_TEST_FARMER_PUBLIC_KEY},
    },
    "harvester": {
        "private": {"crt": SSL_TEST_HARVESTER_PRIVATE_CRT, "key": SSL_TEST_HARVESTER_PRIVATE_KEY},
    },
    "timelord": {
        "private": {"crt": SSL_TEST_TIMELORD_PRIVATE_CRT, "key": SSL_TEST_TIMELORD_PRIVATE_KEY},
        "public": {"crt": SSL_TEST_TIMELORD_PUBLIC_CRT, "key": SSL_TEST_TIMELORD_PUBLIC_KEY},
    },
    "crawler": {
        "private": {"crt": SSL_TEST_CRAWLER_PRIVATE_CRT, "key": SSL_TEST_CRAWLER_PRIVATE_KEY},
    },
    "daemon": {
        "private": {"crt": SSL_TEST_DAEMON_PRIVATE_CRT, "key": SSL_TEST_DAEMON_PRIVATE_KEY},
    },
    "introducer": {
        "public": {"crt": SSL_TEST_INTRODUCER_PUBLIC_CRT, "key": SSL_TEST_INTRODUCER_PUBLIC_KEY},
    },
}
