import numpy as np
import matplotlib.pyplot as plt
import pandas as pd
import scipy
import folium
from scipy.stats import lognorm
"""
The tcra vulnerability module contains function to estimate damage states
to buildings/structures through stochastic simulation.

"""

class FragilityAnalysis:
    """ 
    This is FragilityAnalysis class. This class estimates probabilities of various damage states and assign damage states.
    
    Parameters
    ----------
    fragility_curves : 
        fragility curves defined for building structural archetypes
    mu :
        median value of lognormal distribution
    sigma : 
        standard deviation of lognormal distribution

    Returns
    -------
    damage_state :
        damage state
    """
    def __init__(self, fragility_curves):
        self.fragility_curves = fragility_curves

    def generate_fragility_curve(self, mu, sigma, intensity):
        """
        Generate the fragility curve using log-normal distribution.
        
        Parameters
        ----------
        fragility_curves : 
            fragility curves defined for building structural archetypes
        mu :
            median value of lognormal distribution
        sigma : 
            standard deviation of lognormal distribution
    
        Returns
        -------
        lognorm.cdf :
            lognormal cumulative distribution function        
        """
        return lognorm.cdf(intensity, s=sigma, scale=mu)

    def estimate_damage(self, building_data):
        """
        Estimate damage probabilities for all buildings.
        
        Parameters
        ----------
        building_data : 
            building inventory
        mph :
            wind speed
        type : 
            structural archetype defined in HAZUS
        mu :
            median value of lognormal distribution
        sigma : 
            standard deviation of lognormal distribution
        id :
            ids
        x :
            longitude
        y :
            latitude
        Occupancy :
            building occupancy type
            
        Returns
        -------
        pd.DataFrame :
            dataframe inventory with probability of failure           
        """
        results = []
        for _, row in building_data.iterrows():
            building_type = row['type']
            intensity = row['mph']
            
            fragility_curves_building = self.fragility_curves[building_type]
            building_probabilities = {}
            for damage_state, fragility_params in fragility_curves_building.items():
                fragility_curve = self.generate_fragility_curve(fragility_params['mu'], fragility_params['sigma'], intensity)
                building_probabilities[damage_state] = fragility_curve
            building_probabilities['id'] = row['id']
            building_probabilities['x'] = row['x']
            building_probabilities['y'] = row['y']
            building_probabilities['mph'] = row['mph']
            building_probabilities['type'] = row['type']
            building_probabilities['Occupancy'] = row['Occupancy']
            results.append(building_probabilities)
        return pd.DataFrame(results)
    
    def estimate_epn_damage_state(self, epn_data):
        """
        Estimate damage probabilities for all electrical poles.
        Parameters
        ----------
        epn_data : 
            electrical system inventory
        mph :
            wind speed
        type : 
            structural archetype defined in HAZUS
        mu :
            median value of lognormal distribution
        sigma : 
            standard deviation of lognormal distribution
        id :
            ids
        x :
            longitude
        y :
            latitude
        Occupancy :
            building occupancy type
            
        Returns
        -------
        pd.DataFrame :
            dataframe inventory with probability of failure             
        
        """
        results = []
        for _, row in epn_data.iterrows():
            epn_type = row['type']
            intensity = row['mph']
            
            fragility_curves_epn = self.fragility_curves[epn_type]
            epn_probabilities = {}
            for damage_state, fragility_params in fragility_curves_epn.items():
                fragility_curve = self.generate_fragility_curve(fragility_params['mu'], fragility_params['sigma'], intensity)
                epn_probabilities[damage_state] = fragility_curve
            epn_probabilities['id'] = row['id']
            epn_probabilities['x'] = row['x']
            epn_probabilities['y'] = row['y']
            epn_probabilities['mph'] = row['mph']
            epn_probabilities['type'] = row['type']
            results.append(epn_probabilities)
        return pd.DataFrame(results)

    def sample_damage_state(self, Pr, DStates, seed=None):
        """ Generate damage states.
    
        Parameters
        ----------
        Pr: 
            building invetory with damage state probabilities
        DStates:
            damage states

        Returns
        -------
        damage_state :
            damage state           
                
        """
        
        self.DStates = DStates
        np.random.seed(seed)  # Set the random seed
        
        id=Pr.id
        p = pd.Series(data=np.random.uniform(size=Pr.shape[0]), index=Pr.index)
        damage_state = pd.Series(data=[None] * Pr.shape[0], index=Pr.index)

        for DS_name in DStates:
            damage_state[p < Pr[DS_name]] = DS_name

        return damage_state
