# pardoc

Yet another docstring parser for python, using [`lark-parser`][1]

## Installation
```python
pip install pardoc
```

## A quick look
```python console
>>> from pardoc import google_parser, pretty

>>> docstring = """This is an example of a module level function.

Function parameters should be documented in the ``Args`` section. The name
of each parameter is required. The type and description of each parameter
is optional, but should be included if not obvious.

If \*args or \*\*kwargs are accepted,
they should be listed as ``*args`` and ``**kwargs``.

The format for a parameter is::

    name (type): description
        The description may span multiple lines. Following
        lines should be indented. The "(type)" is optional.

        Multiple paragraphs are supported in parameter
        descriptions.

Args:
    param1 (int): The first parameter.
    param2 (:obj:`str`, optional): The second parameter. Defaults to None.
        Second line of description should be indented.
    *args: Variable length argument list.
    **kwargs: Arbitrary keyword arguments.

Returns:
    bool: True if successful, False otherwise.

    The return type is optional and may be specified at the beginning of
    the ``Returns`` section followed by a colon.

    The ``Returns`` section may span multiple lines and paragraphs.
    Following lines should be indented to match the first line.

    The ``Returns`` section supports any reStructuredText formatting,
    including literal blocks::

        {
            'param1': param1,
            'param2': param2
        }

Raises:
    AttributeError: The ``Raises`` section is a list of all exceptions
        that are relevant to the interface.
    ValueError: If `param2` is equal to `param1`.

"""

>>> parsed = google_parser.parse(docstring)
>>> pretty(parsed, print_=True)

ParsedSection(title=SUMMARY)
   ParsedPara(lines=1)
      This is an example of a module level function.
   ParsedPara(lines=3)
      Function parameters should be documented in the ``Args`` section. The name
      of each parameter is required. The type and description of each parameter
      is optional, but should be included if not obvious.
   ParsedPara(lines=2)
      If \*args or \*\*kwargs are accepted,
      they should be listed as ``*args`` and ``**kwargs``.
   ParsedPara(lines=1)
      The format for a parameter is::
   ParsedPara(lines=2)
      ParsedPara(lines=1)
         name (type): description
      ParsedPara(lines=2)
         ParsedPara(lines=2)
            The description may span multiple lines. Following
            lines should be indented. The "(type)" is optional.
         ParsedPara(lines=2)
            Multiple paragraphs are supported in parameter
            descriptions.

ParsedSection(title=Args)
   ParsedItem(name=param1, type=int, desc=The first parameter.)
   ParsedItem(name=param2, type=:obj:`str`, optional, desc=The second parameter. Defaults to None.)
      ParsedPara(lines=1)
         Second line of description should be indented.
   ParsedItem(name=*args, type=None, desc=Variable length argument list.)
   ParsedItem(name=**kwargs, type=None, desc=Arbitrary keyword arguments.)

ParsedSection(title=Returns)
   ParsedItem(name=bool, type=None, desc=True if successful, False otherwise.)
   ParsedPara(lines=2)
      The return type is optional and may be specified at the beginning of
      the ``Returns`` section followed by a colon.
   ParsedPara(lines=2)
      The ``Returns`` section may span multiple lines and paragraphs.
      Following lines should be indented to match the first line.
   ParsedPara(lines=2)
      The ``Returns`` section supports any reStructuredText formatting,
      including literal blocks::
   ParsedPara(lines=2)
      ParsedPara(lines=1)
         {
      ParsedPara(lines=1)
         ParsedPara(lines=2)
            'param1': param1,
            'param2': param2
   ParsedPara(lines=1)
      ParsedPara(lines=1)
         }

ParsedSection(title=Raises)
   ParsedItem(name=AttributeError, type=None, desc=The ``Raises`` section is a list of all exceptions)
      ParsedPara(lines=1)
         that are relevant to the interface.
   ParsedItem(name=ValueError, type=None, desc=If `param2` is equal to `param1`.)

```

## Usage

### Parsing a known style docstring

```python
from pardoc import google_parser, numpy_parser
parsed = google_parser(docstring)
# or
parsed = google_parser(docstring)
```

### Parsing an unknown style docstring

```python
from pardoc import auto_parser

parser = auto_parser(docstring)
# parsing results from auto_parser is cached and reused.
parsed = parser.parse(docstring)
```

### Parsed object

There are 6 types of parsed objects, include the final `Parsed` object that
attaches all sections

The first 5 are all `namedtuple`s:
```python
ParsedItem = namedtuple('ParsedItem',
                        ['name', 'type', 'desc', 'more'])
ParsedTodo = namedtuple('ParsedTodo', ['todo', 'more'])
ParsedSection = namedtuple('ParsedSection', ['title', 'section'])
ParsedPara = namedtuple('ParsedPara', ['lines'])
ParsedCode = namedtuple('ParsedCode', ['lang', 'codes'])

```

The `Parsed` is an ordered dictionary (`OrderedDiot`) from [`diot`][2], which
allows dot access to keys:
```python
from diot import OrderedDiot

class Parsed(OrderedDiot):
    """Parsed object"""
```

### Formatting a parsed object to the original style

```python console
>>> from pardoc import google_parser
>>> docstring = """Example function with types documented in the docstring.


Args:


    param0: No type

    param1 (int): The first parameter.
    param2 (str): The second parameter.

Returns:


    bool: The return value. True for success, False otherwise.

"""
>>> # note the arbitrary empty lines
>>> reformatted = google_parser.format(docstring)
>>> # or
>>> reformatted = google_parser.format(google_parser.parse(docstring))
>>> print(reformatted)
Example function with types documented in the docstring.

Args:
    param0: No type
    param1 (int): The first parameter.
    param2 (str): The second parameter.

Returns:
    bool: The return value. True for success, False otherwise.
```

### Pretty printing the parsed objects

See `A quick look`


[1]: https://github.com/lark-parser/lark
[2]: https://github.com/pwwang/diot
