import pandas as pd
import os
import glob
import subprocess
from pynextgen.logging_config import get_logger

logger = get_logger(__file__, __name__)


def clean_sample_name(sample_path, extra_name_rm=[]):
    """ Clean sample names in feature count tables
    """

    new_name = os.path.basename(sample_path)
    new_name = os.path.splitext(new_name)[0]

    for pattern in extra_name_rm:
        new_name = new_name.replace(pattern, "")

    return new_name


def clean_feature_counts(feature_counts, extra_name_rm=[], drop_loc=True):
    """ Cleanup a featureCount count table
    - extra_name_rm: list of strings to remove from the header names
    - drop_loc: whether removing the 'Chr, Start, End, Strand, Length'
    columns
    """

    counts_df = pd.read_csv(feature_counts, sep="\t", comment="#", index_col="Geneid")

    # Simplify the sample names
    counts_df.columns = [
        clean_sample_name(x, extra_name_rm=extra_name_rm) for x in counts_df.columns
    ]

    if drop_loc:
        counts_df.drop(
            ["Chr", "Start", "End", "Strand", "Length"], axis=1, inplace=True
        )

    counts_df.sort_index(inplace=True)

    return counts_df


def clean_kallisto(kallisto_dir, csv_out="", make_round=False):
    """Cleanup the results from a kallisto run which creates results for
    each libraries in separate directory. The name of the libraries in
    the final table are corresponding to the names of last output directories

    OBSOLETE ! NOW USING bioconductor tximport which is
    importing the data as generated by kallisto.

    """

    all_abundances = glob.glob(os.path.join(kallisto_dir, "*", "abundance.tsv"))

    all_dfs = []

    for abundance in all_abundances:
        lib_name = os.path.split(os.path.split(abundance)[0])[1]
        df = pd.read_csv(
            abundance, sep="\t", index_col=0, usecols=["target_id", "est_counts"]
        )
        df.columns = [lib_name]
        all_dfs.append(df)

    all_dfs = pd.concat(all_dfs, axis=1)

    if make_round:
        all_dfs = all_dfs.round()

    if csv_out:
        all_dfs.to_csv(csv_out)

    return all_dfs


def run_feature_counts(bam, annotation, supp_args=""):
    """ Wrapper to run feature counts from within python

    bam: path to the bam file
    annotation: path to the annotation file
    supp_args: supplementary arguments to pass to featureCounts
    """
    outfile = (
        os.path.splitext(bam)[0]
        + "_"
        + os.path.splitext(os.path.basename(annotation))[0]
        + ".fcounts"
    )

    cmd = f"featureCounts {supp_args} -a {annotation} -o {outfile} {bam}"

    try:
        logger.info(f"Launching: {cmd}")
        output = subprocess.check_output(
            cmd, shell=True, stderr=subprocess.STDOUT, universal_newlines=True
        )
        print(output)
        logger.info(f"Feature counts results written to: {outfile}")
        return pd.read_csv(outfile, sep="\t", comment="#")

    except subprocess.CalledProcessError as exc:
        print(exc.output)
