from pydantic import BaseModel
import numpy as np

Pos = list | tuple | np.ndarray


class Point(BaseModel):
    """
    Represent a point in 2D
    """

    x: float
    y: float

    @classmethod
    def from_list(cls, point: Pos) -> "Point":
        """
        Build an instance of Point from a list
        """
        return Point(x=point[0], y=point[1])

    @property
    def coord(self) -> np.ndarray:
        """
        Return the point as coordinate (int dtype)
        """
        return np.array([self.x, self.y], dtype=int)

    @property
    def pos(self) -> np.ndarray:
        """
        Return the point as position (float dtype)
        """
        return np.array([self.x, self.y], dtype=float)


class GameConfig(BaseModel):
    """
    Global configuration of the game
    """

    dim: Point
    """
    dimension of the map (unit: coord)
    """
    n_player: int
    """
    number of players in the game
    """
    initial_money: int
    """
    money players start with
    """
    initial_n_probes: int
    """
    initial number of probes to start with (must be smaller
    than `factory_max_probe`)
    """
    base_income: float
    """
    base income that each player receive unconditionally
    """
    building_occupation_min: int
    """
    minimal occupation value on tile required to build a building (factory/turret)
    """
    factory_price: int
    """
    amount to pay to build a new factory
    """
    factory_max_probe: int
    """
    maximal number of probe generated by a factory
    """
    factory_build_probe_delay: float
    """
    delay to wait to build a probe from the factory (sec)
    """
    max_occupation: int
    """
    maximal occupation value that can be reached
    """
    probe_speed: float
    """
    speed of the probe in coordinate/sec
    """
    probe_price: int
    """
    amount to pay to produce one probe
    """
    probe_claim_delay: float
    """
    delay to wait claim a tile, the probe can be manually moved but not claim
    another tile during the delay (see Probe `is_claiming` flag for details)
    """
    probe_maintenance_costs: float
    """
    Costs of possessing one probe (computed in the player's income)
    """
    turret_price: int
    """
    amount to pay to build a new turret
    """
    turret_fire_delay: float
    """
    delay to wait for the turret between two fires (sec)
    """
    turret_scope: float
    """
    scope of the turret (unit: coord)
    """
    turret_maintenance_costs: float
    """
    Costs of possessing one turret (computed in the player's income)
    """
    income_rate: float
    """
    factor of how the occupation level of a tile reflects on its income,
    as `income = occupation * rate`
    """
    deprecate_rate: float
    """
    probability that a tile with maximum occupation lose 1 occupation
    """


class Response(BaseModel):
    """
    Represent a response from the server
    """

    success: bool = True
    msg: str = ""


class StartGame(Response):
    """
    Represents the start game event
    """

    gid: str


class QueueInvitation(Response):
    """
    Represents an invitation to a queue
    """

    qid: str
