import { Stack } from "aws-cdk-lib";
import { IAutoScalingGroup } from "aws-cdk-lib/aws-autoscaling";
import { IMachineImage, InstanceType, ISecurityGroup } from "aws-cdk-lib/aws-ec2";
import { CfnInstanceProfile, IRole } from "aws-cdk-lib/aws-iam";
import { IBucket } from "aws-cdk-lib/aws-s3";
import { Construct } from "constructs";
import { AutoscalingConfiguration, DockerConfiguration, GlobalConfiguration, MachineConfiguration } from "../runner-configuration";
import { CacheProps } from "./cache";
import { Network, NetworkProps } from "./network";
/**
 * This is a AWS CDK Construct that may be used to deploy a GitLab runner with Docker executor and auto-scaling.
 *
 * @remarks
 * The `cdk-gitlab-runner` defines the {@link GitlabRunnerAutoscalingProps} interface and {@link GitlabRunnerAutoscaling} construct class,
 * which are used to provision a the runner.
 *
 * @packageDocumentation
 */
/**
 * Properties of the Gitlab Runner. You have to provide at least the GitLab's Runner's authentication token.
 */
export interface GitlabRunnerAutoscalingProps extends GlobalConfiguration {
    /**
     * The GitLab Runner’s authentication token, which is obtained during runner registration.
     * @see https://docs.gitlab.com/ee/api/runners.html#registration-and-authentication-tokens
     */
    readonly gitlabToken: string;
    /**
     * GitLab instance URL.
     * @default "https://gitlab.com"
     */
    readonly gitlabUrl?: string;
    readonly cache?: GitlabRunnerAutoscalingCacheProps;
    /**
     * The network configuration for the Runner. If not set, the defaults will be used.
     * @link NetworkProps
     */
    readonly network?: NetworkProps;
    /**
     * The manager EC2 instance configuration. If not set, the defaults will be used.
     * @link GitlabRunnerAutoscalingManagerProps
     */
    readonly manager?: GitlabRunnerAutoscalingManagerProps;
    readonly runners?: GitlabRunnerAutoscalingRunnerProps;
}
/**
 * The distributed GitLab runner S3 cache. Either pass an existing bucket or override default options.
 * @see https://docs.gitlab.com/runner/configuration/advanced-configuration.html#the-runnerscaches3-section
 */
export interface GitlabRunnerAutoscalingCacheProps {
    /**
     * An existing S3 bucket used as runner's cache.
     */
    readonly bucket?: IBucket;
    /**
     * If no existing S3 bucket is provided, a S3 bucket will be created.
     */
    readonly options?: CacheProps;
}
export interface GitlabRunnerAutoscalingManagerProps {
    /**
     * An Amazon Machine Image ID for the Manager EC2 instance. If empty the latest Amazon 2 Image will be looked up.
     *
     * Should be RHEL flavor like Amazon Linux 2 with yum available for instance initialization.
     *
     * @see https://cloudinit.readthedocs.io/en/latest/
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-init.html
     */
    readonly machineImage?: IMachineImage;
    /**
     * Instance type for manager EC2 instance. It's a combination of a class and size.
     * @default InstanceType.of(InstanceClass.T3, InstanceSize.NANO)
     */
    readonly instanceType?: InstanceType;
    /**
     * A set of security credentials that you use to prove your identity when connecting to an Amazon EC2 instance. You won't be able to ssh into an instance without the Key Pair.
     */
    readonly keyPairName?: string;
}
/**
 * The runner EC2 instances configuration. If not set, the defaults will be used.
 * @link GitlabRunnerAutoscalingProps
 */
export interface GitlabRunnerAutoscalingRunnerProps {
    /**
     * Instance type for runner EC2 instances. It's a combination of a class and size.
     * @default InstanceType.of(InstanceClass.T3, InstanceSize.MICRO)
     */
    readonly instanceType?: InstanceType;
    /**
     * An Amazon Machine Image ID for the Runners EC2 instances. If empty the latest Ubuntu 20.04 focal will be looked up.
     *
     * Any operating system supported by Dcoker Machine's provisioner.
     *
     * @see https://cloud-images.ubuntu.com/locator/ec2/
     * @see https://gitlab.com/gitlab-org/ci-cd/docker-machine/-/tree/main/libmachine/provision
     */
    readonly machineImage?: IMachineImage;
    /**
     * Optionally pass an IAM role, that get's assigned to the EC2 runner instances.
     */
    readonly role?: IRole;
    /**
     * Limit how many jobs can be handled concurrently by this registered runner. 0 (default) means do not limit.
     * @default 10
     */
    readonly limit?: number;
    /**
     * Maximum build log size in kilobytes. Default is 4096 (4MB).
     * @default 52428800 (50GB)
     */
    readonly outputLimit?: number;
    /**
     * Append or overwrite environment variables.
     * @default ["DOCKER_DRIVER=overlay2", "DOCKER_TLS_CERTDIR=/certs"]
     */
    readonly environment?: string[];
    /**
     * Optional docker configuration
     */
    readonly docker?: DockerConfiguration;
    /**
     * Optional docker machine configuration
     */
    readonly machine?: MachineConfiguration;
    /**
     * Optional autoscaling configuration
     */
    readonly autoscaling?: AutoscalingConfiguration[];
}
/**
 * The Gitlab Runner autoscaling on EC2 by Docker Machine.
 *
 * @example <caption>Provisioning a basic Runner</caption>
 * const app = new cdk.App();
 * const stack = new cdk.Stack(app, "RunnerStack", {
 *   env: {
 *     account: "000000000000",
 *     region: "us-east-1",
 *   }
 * });
 *
 * new GitlabRunnerAutoscaling(scope, "GitlabRunner", {
 *   gitlabToken: "xxxxxxxxxxxxxxxxxxxx",
 * });
 */
export declare class GitlabRunnerAutoscaling extends Construct {
    readonly network: Network;
    readonly cacheBucket: IBucket;
    readonly runners: GitlabRunnerAutoscalingRunners;
    readonly manager: GitlabRunnerAutoscalingManager;
    constructor(scope: Stack, id: string, props: GitlabRunnerAutoscalingProps);
}
export interface GitlabRunnerAutoscalingRunners {
    readonly securityGroupName: string;
    readonly securityGroup: ISecurityGroup;
    readonly role: IRole;
    readonly instanceProfile: CfnInstanceProfile;
    readonly instanceType: InstanceType;
    readonly machineImage: IMachineImage;
}
export interface GitlabRunnerAutoscalingManager {
    readonly securityGroup: ISecurityGroup;
    readonly instanceType: InstanceType;
    readonly machineImage: IMachineImage;
    readonly autoScalingGroup: IAutoScalingGroup;
    readonly role: IRole;
}
