"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderCredentials = exports.defaultDeletionProtection = exports.applyRemovalPolicy = exports.engineDescription = exports.setupS3ImportExport = exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const database_secret_1 = require("../database-secret");
const props_1 = require("../props");
/**
 * The default set of characters we exclude from generated passwords for database users.
 * It's a combination of characters that have a tendency to cause problems in shell scripts,
 * some engine-specific characters (for example, Oracle doesn't like '@' in its passwords),
 * and some that trip up other services, like DMS.
 *
 * This constant is private to the RDS module.
 */
exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = " %+~`#$&*()|[]{}:;<>?!'/@\"\\";
/**
 * Validates the S3 import/export props and returns the import/export roles, if any.
 * If `combineRoles` is true, will reuse the import role for export (or vice versa) if possible.
 *
 * Notably, `combineRoles` is (by default) set to true for instances, but false for clusters.
 * This is because the `combineRoles` functionality is most applicable to instances and didn't exist
 * for the initial clusters implementation. To maintain backwards compatibility, it is set to false for clusters.
 */
function setupS3ImportExport(scope, props, combineRoles) {
    let s3ImportRole = props.s3ImportRole;
    let s3ExportRole = props.s3ExportRole;
    if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
        if (props.s3ImportRole) {
            throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
        }
        s3ImportRole = (combineRoles && s3ExportRole) ? s3ExportRole : new iam.Role(scope, 'S3ImportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ImportBuckets) {
            bucket.grantRead(s3ImportRole);
        }
    }
    if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
        if (props.s3ExportRole) {
            throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
        }
        s3ExportRole = (combineRoles && s3ImportRole) ? s3ImportRole : new iam.Role(scope, 'S3ExportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ExportBuckets) {
            bucket.grantReadWrite(s3ExportRole);
        }
    }
    return { s3ImportRole, s3ExportRole };
}
exports.setupS3ImportExport = setupS3ImportExport;
function engineDescription(engine) {
    var _a;
    return engine.engineType + (((_a = engine.engineVersion) === null || _a === void 0 ? void 0 : _a.fullVersion) ? `-${engine.engineVersion.fullVersion}` : '');
}
exports.engineDescription = engineDescription;
function applyRemovalPolicy(cfnDatabase, removalPolicy) {
    if (!removalPolicy) {
        // the default DeletionPolicy is 'Snapshot', which is fine,
        // but we should also make it 'Snapshot' for UpdateReplace policy
        cfnDatabase.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
    }
    else {
        // just apply whatever removal policy the customer explicitly provided
        cfnDatabase.applyRemovalPolicy(removalPolicy);
    }
}
exports.applyRemovalPolicy = applyRemovalPolicy;
/**
 * By default, deletion protection is disabled.
 * Enable if explicitly provided or if the RemovalPolicy has been set to RETAIN
 */
function defaultDeletionProtection(deletionProtection, removalPolicy) {
    return deletionProtection !== null && deletionProtection !== void 0 ? deletionProtection : (removalPolicy === core_1.RemovalPolicy.RETAIN ? true : undefined);
}
exports.defaultDeletionProtection = defaultDeletionProtection;
/**
 * Renders the credentials for an instance or cluster
 */
function renderCredentials(scope, engine, credentials) {
    var _a;
    let renderedCredentials = credentials !== null && credentials !== void 0 ? credentials : props_1.Credentials.fromUsername((_a = engine.defaultUsername) !== null && _a !== void 0 ? _a : 'admin'); // For backwards compatibilty
    if (!renderedCredentials.secret && !renderedCredentials.password) {
        renderedCredentials = props_1.Credentials.fromSecret(new database_secret_1.DatabaseSecret(scope, 'Secret', {
            username: renderedCredentials.username,
            encryptionKey: renderedCredentials.encryptionKey,
            excludeCharacters: renderedCredentials.excludeCharacters,
            // if username must be referenced as a string we can safely replace the
            // secret when customization options are changed without risking a replacement
            replaceOnPasswordCriteriaChanges: credentials === null || credentials === void 0 ? void 0 : credentials.usernameAsString,
        }), 
        // pass username if it must be referenced as a string
        (credentials === null || credentials === void 0 ? void 0 : credentials.usernameAsString) ? renderedCredentials.username : undefined);
    }
    return renderedCredentials;
}
exports.renderCredentials = renderCredentials;
//# sourceMappingURL=data:application/json;base64,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