# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mat_dp_core', 'mat_dp_core.maths_core']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.23.0,<2.0.0', 'scipy>=1.7.1,<2.0.0']

setup_kwargs = {
    'name': 'mat-dp-core',
    'version': '0.1.0',
    'description': '',
    'long_description': '# Mat-dp core\n## *MATerial Demand Projections Model*\n\nWelcome to the Mat-dp core. This repo represents the core of the MAT-dp project, which aims to deliver user-friendly and open-access software to study the environmental implications of materials used for building low-carbon systems. \n\n# Installation and launch\n\nPlease install poetry - see [here](https://github.com/python-poetry/poetry)\n\nTo install all the project dependencies\n\n`poetry install`\n\nThen go the examples folder\n\n`cd examples`\n\nThen run the pizza box example\n\n`poetry run python3 test.py`\n\n# Concepts\n\n## Definitions\n\nThe following terms will be used frequently:\n\nResource - A resource to be produced or consumed, such as steel or aluminium.\n\nProcess - A process which produces and/or consumes resources.\n\nConstraint - A condition the system is placed under.\n\n* Run Ratio Constraint - A constraint that fixes the ratio of runs between two processes - e.g. wind and solar will run at a ratio of 1:2.\n\n* Resource Constraint - A constraint on the amount of resource produced, e.g. we must produce at least 10 energy.\n\n* Run Eq Constraint - A constraint that specifies the number of runs a process must make.\n\nObjective - The objective function is the property of the system which will be minimised. This could be something like the number of runs of the system, or the total cost.\n\nMeasurement - a measurement taken of the solved system, determining the \n\n# Usage - High Level\n\n## Introduction\n\nThe below describes a practical example of using MAT-dp. Imagine...\n\n* Pizza boxes are made from cardboard and recycled cardboard. *(process/resource)*\n* There are different processes for making them, which have different ratios of `cardboard:recycled_cardboard` . *(process)*\n* We wish to priorites the process that uses the most recycled cardboard, but not so as to eliminate the less efficient version. *(ratio constraint)*\n* We then, rather inefficiently, burn them to produce energy. *(process)*\n* We must produce at least 8 kWh of energy to survive the frosty winters. *(resource constraint)*\n* We wish to only generate the minimum amount of cardboard and pizza boxes. *(objective)*\n* How many pizza boxes must we burn to survive? *(measurement)*\n## Step 1: Define resources\n\nFirstly we must define all the resources we wish to use, with their name and units.\n\n```py\nfrom mat_dp_core import Resources\n\nresources = Resources()\ncardboard = resources.create("cardboard", unit="m2")\nrecycled_cardboard = resources.create("recycled_cardboard", unit="m2")\npizza_box = resources.create("pizza_box")\nenergy = resources.create("energy", unit="kWh")\n```\n\n## Step 2: Define processes\n\nWe must now take these resources and use them to define our processes. These are defined by a name and the resources that they produce and consume.\n\n```py\nfrom mat_dp_core import Processes\nprocesses = Processes()\ncardboard_producer = processes.create("cardboard producer", (cardboard, +1))\nrecycled_cardboard_producer = processes.create(\n    "recycled cardboard producer", (recycled_cardboard, +1)\n)\npizza_box_producer = processes.create(\n    "pizza box producer",\n    (recycled_cardboard, -0.5),\n    (cardboard, -2),\n    (pizza_box, 1),\n)\nrecycled_pizza_box_producer = processes.create(\n    "recycled pizza box producer",\n    (recycled_cardboard, -3),\n    (cardboard, -1),\n    (pizza_box, 1),\n)\npower_plant = processes.create("power plant", (pizza_box, -1), (energy, 4))\nenergy_grid = processes.create("energy grid", (energy, -2))\n```\n\n\n\n## Step 3: Define constraints\n\nNow we need to define the constraints of the problem. We want to specify we take equal amounts of pizza boxes from each producer *(Run ratio constraint)*, and that we only require 8 kWh of energy *(Resource constraint)*:\n\n```py\nfrom mat_dp_core import EqConstraint\nconstraints = [\n    EqConstraint(\n        "recycled pizza box ratio",\n        pizza_box_producer - recycled_pizza_box_producer,\n        0,\n    ),\n    EqConstraint("required energy", energy_grid, 8),\n]\n```\n\n## Step 4: Define an objective function\n\nOnce we\'ve established all of our constraints, we must define an objective function. The below example specifies we minimise the total number of runs:\n\n```py\n# Minimise total number of runs\nobjective = (\n    cardboard_producer\n    + recycled_cardboard_producer\n    + pizza_box_producer\n    + recycled_pizza_box_producer\n    + power_plant\n    + energy_grid\n)\n```\n\n## Step 5: Make a measurement\n\nWe must now measure the number of pizza boxes to burn.\n\n```py\nfrom mat_dp_core import Measure\n\nmeasurement = Measure(resources, processes, constraints, objective)\nprint(measurement.resource(pizza_box))\n```\n\n\n# Visualising the documentation\n\n\nTo view the documentation in html format, go to [this website](https://client.dreamingspires.dev/mat_dp_core/) \nor run the documentation through mkdocs using the following command at the root of the repository:\n\n`poetry run mkdocs serve`\n\n# Contributing to Mat-dp\n\n\nContributions are welcome! \nIf you see something that needs to be improved, open an issue in the [respective section of the repository](https://github.com/Mat-dp/mat-dp-core/issues).\nIf you have questions, need assistance or need better instructions for contributing, please \n[get in touch via e-mail](mailto:refficiency-enquiries@eng.cam.ac.uk) mentioning "Mat-dp" in the subject.\n\n\nFor any questions on how to use the software, please refer to the [documentation](https://github.com/Mat-dp/mat-dp-core/tree/master/docs). \nIt contains useful definitions and examples of using the software. Please contact us by e-mail for any other support requried.\n\n',
    'author': 'Edd Salkield',
    'author_email': 'edd@salkield.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<3.10',
}


setup(**setup_kwargs)
