import numpy as np


def lda_slater_x(n, alpha=2 / 3):
    '''Slater exchange functional (spin-paired).
    Thesis: Eq. 2.11
    '''
    rs = (3 / (4 * np.pi * n))**(1 / 3)

    f = -9 / 8 * (3 / (2 * np.pi))**(2 / 3)

    ex = f * alpha / rs
    vx = 4 / 3 * ex
    return ex, vx


def lda_chachiyo_c(n):
    '''Chachiyo parametrization of the correlation functional (spin-paired).'''
    rs = (3 / (4 * np.pi * n))**(1 / 3)

    a = (np.log(2) - 1) / (2 * np.pi**2)
    b = 20.4562557

    ec = a * np.log(1 + b / rs + b / rs**2)
    vc = ec + a * b * (2 + rs) / (3 * (b + b * rs + rs**2))
    return ec, vc


def lda_vwn_c(n):
    '''Vosko-Wilk-Nusair parametrization of the correlation functional (spin-paired).
    Not used, only for reference.
    Thesis: Eq. 2.12 ff.
    '''
    rs = (3 / (4 * np.pi * n))**(1 / 3)

    a = 0.0310907
    b = 3.72744
    c = 12.9352
    x0 = -0.10498

    rs12 = np.sqrt(rs)
    q = np.sqrt(4 * c - b * b)
    qx = np.arctan(q / (2 * rs12 + b))
    f1 = 2 * b / q
    f2 = b * x0 / (x0 * x0 + b * x0 + c)
    f3 = 2 * (2 * x0 + b) / q
    fx = rs + b * rs12 + c

    ec = a * (np.log(rs / fx) + f1 * qx - f2 * (np.log((rs12 - x0)**2 / fx) + f3 * qx))
    tx = 2 * rs12 + b
    tt = tx * tx + q * q
    vc = ec - rs12 * a / 6 * (
        2 / rs12 - tx / fx - 4 * b / tt - f2 * (2 / (rs12 - x0) - tx / fx - 4 * (2 * x0 + b) / tt))
    return ec, vc
