"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterFromSnapshot = exports.DatabaseCluster = exports.DatabaseClusterBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const util_1 = require("./private/util");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Add a new db proxy to this cluster.
     */
    addProxy(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseProxyOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addProxy);
            }
            throw error;
        }
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromCluster(this),
            ...options,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
exports.DatabaseClusterBase = DatabaseClusterBase;
_a = JSII_RTTI_SYMBOL_1;
DatabaseClusterBase[_a] = { fqn: "@aws-cdk/aws-rds.DatabaseClusterBase", version: "1.158.0" };
/**
 * Abstract base for ``DatabaseCluster`` and ``DatabaseClusterFromSnapshot``
 */
class DatabaseClusterNew extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w;
        super(scope, id);
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            core_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_d = props.subnetGroup) !== null && _d !== void 0 ? _d : new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: props.instanceProps.vpc,
            vpcSubnets: props.instanceProps.vpcSubnets,
            removalPolicy: util_1.renderUnless(util_1.helperRemovalPolicy(props.removalPolicy), core_1.RemovalPolicy.DESTROY),
        });
        this.securityGroups = (_e = props.instanceProps.securityGroups) !== null && _e !== void 0 ? _e : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: props.instanceProps.vpc,
            }),
        ];
        const combineRoles = (_f = props.engine.combineImportAndExportRoles) !== null && _f !== void 0 ? _f : false;
        let { s3ImportRole, s3ExportRole } = util_1.setupS3ImportExport(this, props, combineRoles);
        if (props.parameterGroup && props.parameters) {
            throw new Error('You cannot specify both parameterGroup and parameters');
        }
        const parameterGroup = (_g = props.parameterGroup) !== null && _g !== void 0 ? _g : (props.parameters
            ? new parameter_group_1.ParameterGroup(this, 'ParameterGroup', {
                engine: props.engine,
                parameters: props.parameters,
            })
            : undefined);
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            s3ImportRole,
            s3ExportRole,
            parameterGroup,
        });
        const clusterAssociatedRoles = [];
        if (s3ImportRole) {
            clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn, featureName: (_h = clusterEngineBindConfig.features) === null || _h === void 0 ? void 0 : _h.s3Import });
        }
        if (s3ExportRole &&
            // only add the second associated Role if it's different than the first
            // (duplicates in the associated Roles array are not allowed by the RDS service)
            (s3ExportRole !== s3ImportRole ||
                ((_j = clusterEngineBindConfig.features) === null || _j === void 0 ? void 0 : _j.s3Import) !== ((_k = clusterEngineBindConfig.features) === null || _k === void 0 ? void 0 : _k.s3Export))) {
            clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn, featureName: (_l = clusterEngineBindConfig.features) === null || _l === void 0 ? void 0 : _l.s3Export });
        }
        const clusterParameterGroup = (_m = props.parameterGroup) !== null && _m !== void 0 ? _m : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        this.engine = props.engine;
        const clusterIdentifier = core_1.FeatureFlags.of(this).isEnabled(cxapi.RDS_LOWERCASE_DB_IDENTIFIER) && !core_1.Token.isUnresolved(props.clusterIdentifier)
            ? (_o = props.clusterIdentifier) === null || _o === void 0 ? void 0 : _o.toLowerCase() : props.clusterIdentifier;
        this.newCfnProps = {
            // Basic
            engine: props.engine.engineType,
            engineVersion: (_p = props.engine.engineVersion) === null || _p === void 0 ? void 0 : _p.fullVersion,
            dbClusterIdentifier: clusterIdentifier,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            vpcSecurityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            port: (_q = props.port) !== null && _q !== void 0 ? _q : clusterEngineBindConfig.port,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            enableIamDatabaseAuthentication: props.iamAuthentication,
            // Admin
            backtrackWindow: (_r = props.backtrackWindow) === null || _r === void 0 ? void 0 : _r.toSeconds(),
            backupRetentionPeriod: (_t = (_s = props.backup) === null || _s === void 0 ? void 0 : _s.retention) === null || _t === void 0 ? void 0 : _t.toDays(),
            preferredBackupWindow: (_u = props.backup) === null || _u === void 0 ? void 0 : _u.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            enableCloudwatchLogsExports: props.cloudwatchLogsExports,
            // Encryption
            kmsKeyId: (_v = props.storageEncryptionKey) === null || _v === void 0 ? void 0 : _v.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
            // Tags
            copyTagsToSnapshot: (_w = props.copyTagsToSnapshot) !== null && _w !== void 0 ? _w : true,
        };
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     */
    addRotationSingleUser(options = {}) {
        if (!this.secret) {
            throw new Error('Cannot add a single user rotation for a cluster without a secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...util_1.applyDefaultRotationOptions(options, this.vpcSubnets),
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add a multi user rotation for a cluster without a secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...util_1.applyDefaultRotationOptions(options, this.vpcSubnets),
            secret: options.secret,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            target: this,
        });
    }
}
/**
 * Represents an imported database cluster.
 */
class ImportedDatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.clusterIdentifier = attrs.clusterIdentifier;
        const defaultPort = attrs.port ? ec2.Port.tcp(attrs.port) : undefined;
        this.connections = new ec2.Connections({
            securityGroups: attrs.securityGroups,
            defaultPort,
        });
        this.engine = attrs.engine;
        this._clusterEndpoint = (attrs.clusterEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
        this._clusterReadEndpoint = (attrs.readerEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
        this._instanceIdentifiers = attrs.instanceIdentifiers;
        this._instanceEndpoints = (attrs.instanceEndpointAddresses && attrs.port)
            ? attrs.instanceEndpointAddresses.map(addr => new endpoint_1.Endpoint(addr, attrs.port))
            : undefined;
    }
    get clusterEndpoint() {
        if (!this._clusterEndpoint) {
            throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
        }
        return this._clusterEndpoint;
    }
    get clusterReadEndpoint() {
        if (!this._clusterReadEndpoint) {
            throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
        }
        return this._clusterReadEndpoint;
    }
    get instanceIdentifiers() {
        if (!this._instanceIdentifiers) {
            throw new Error('Cannot access `instanceIdentifiers` of an imported cluster without provided instanceIdentifiers');
        }
        return this._instanceIdentifiers;
    }
    get instanceEndpoints() {
        if (!this._instanceEndpoints) {
            throw new Error('Cannot access `instanceEndpoints` of an imported cluster without instanceEndpointAddresses and port');
        }
        return this._instanceEndpoints;
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterNew {
    constructor(scope, id, props) {
        var _d, _e;
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            // Admin
            masterUsername: credentials.username,
            masterUserPassword: (_d = credentials.password) === null || _d === void 0 ? void 0 : _d.unsafeUnwrap(),
        });
        this.clusterIdentifier = cluster.ref;
        if (secret) {
            this.secret = secret.attach(this);
        }
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy((_e = props.removalPolicy) !== null && _e !== void 0 ? _e : core_1.RemovalPolicy.SNAPSHOT);
        setLogRetention(this, props);
        const createdInstances = createInstances(this, props, this.subnetGroup);
        this.instanceIdentifiers = createdInstances.instanceIdentifiers;
        this.instanceEndpoints = createdInstances.instanceEndpoints;
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDatabaseClusterAttributes);
            }
            throw error;
        }
        return new ImportedDatabaseCluster(scope, id, attrs);
    }
}
exports.DatabaseCluster = DatabaseCluster;
_b = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_b] = { fqn: "@aws-cdk/aws-rds.DatabaseCluster", version: "1.158.0" };
/**
 * A database cluster restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseClusterFromSnapshot extends DatabaseClusterNew {
    constructor(scope, id, props) {
        var _d;
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseClusterFromSnapshotProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            snapshotIdentifier: props.snapshotIdentifier,
        });
        this.clusterIdentifier = cluster.ref;
        if (secret) {
            this.secret = secret.attach(this);
        }
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy((_d = props.removalPolicy) !== null && _d !== void 0 ? _d : core_1.RemovalPolicy.SNAPSHOT);
        setLogRetention(this, props);
        const createdInstances = createInstances(this, props, this.subnetGroup);
        this.instanceIdentifiers = createdInstances.instanceIdentifiers;
        this.instanceEndpoints = createdInstances.instanceEndpoints;
    }
}
exports.DatabaseClusterFromSnapshot = DatabaseClusterFromSnapshot;
_c = JSII_RTTI_SYMBOL_1;
DatabaseClusterFromSnapshot[_c] = { fqn: "@aws-cdk/aws-rds.DatabaseClusterFromSnapshot", version: "1.158.0" };
/**
 * Sets up CloudWatch log retention if configured.
 * A function rather than protected member to prevent exposing ``DatabaseClusterBaseProps``.
 */
function setLogRetention(cluster, props) {
    if (props.cloudwatchLogsExports) {
        const unsupportedLogTypes = props.cloudwatchLogsExports.filter(logType => !props.engine.supportedLogTypes.includes(logType));
        if (unsupportedLogTypes.length > 0) {
            throw new Error(`Unsupported logs for the current engine type: ${unsupportedLogTypes.join(',')}`);
        }
        if (props.cloudwatchLogsRetention) {
            for (const log of props.cloudwatchLogsExports) {
                new logs.LogRetention(cluster, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/cluster/${cluster.clusterIdentifier}/${log}`,
                    retention: props.cloudwatchLogsRetention,
                    role: props.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * Creates the instances for the cluster.
 * A function rather than a protected method on ``DatabaseClusterNew`` to avoid exposing
 * ``DatabaseClusterNew`` and ``DatabaseClusterBaseProps`` in the API.
 */
function createInstances(cluster, props, subnetGroup) {
    var _d, _e, _f, _g, _h;
    const instanceCount = props.instances != null ? props.instances : 2;
    if (core_1.Token.isUnresolved(instanceCount)) {
        throw new Error('The number of instances an RDS Cluster consists of cannot be provided as a deploy-time only value!');
    }
    if (instanceCount < 1) {
        throw new Error('At least one instance is required');
    }
    const instanceIdentifiers = [];
    const instanceEndpoints = [];
    const portAttribute = cluster.clusterEndpoint.port;
    const instanceProps = props.instanceProps;
    // Get the actual subnet objects so we can depend on internet connectivity.
    const internetConnected = instanceProps.vpc.selectSubnets(instanceProps.vpcSubnets).internetConnectivityEstablished;
    let monitoringRole;
    if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
        monitoringRole = props.monitoringRole || new aws_iam_1.Role(cluster, 'MonitoringRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
            ],
        });
    }
    const enablePerformanceInsights = instanceProps.enablePerformanceInsights
        || instanceProps.performanceInsightRetention !== undefined || instanceProps.performanceInsightEncryptionKey !== undefined;
    if (enablePerformanceInsights && instanceProps.enablePerformanceInsights === false) {
        throw new Error('`enablePerformanceInsights` disabled, but `performanceInsightRetention` or `performanceInsightEncryptionKey` was set');
    }
    const instanceType = (_d = instanceProps.instanceType) !== null && _d !== void 0 ? _d : ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MEDIUM);
    if (instanceProps.parameterGroup && instanceProps.parameters) {
        throw new Error('You cannot specify both parameterGroup and parameters');
    }
    const instanceParameterGroup = (_e = instanceProps.parameterGroup) !== null && _e !== void 0 ? _e : (instanceProps.parameters
        ? new parameter_group_1.ParameterGroup(cluster, 'InstanceParameterGroup', {
            engine: props.engine,
            parameters: instanceProps.parameters,
        })
        : undefined);
    const instanceParameterGroupConfig = instanceParameterGroup === null || instanceParameterGroup === void 0 ? void 0 : instanceParameterGroup.bindToInstance({});
    for (let i = 0; i < instanceCount; i++) {
        const instanceIndex = i + 1;
        const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
            props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                undefined;
        const instance = new rds_generated_1.CfnDBInstance(cluster, `Instance${instanceIndex}`, {
            // Link to cluster
            engine: props.engine.engineType,
            engineVersion: (_f = props.engine.engineVersion) === null || _f === void 0 ? void 0 : _f.fullVersion,
            dbClusterIdentifier: cluster.clusterIdentifier,
            dbInstanceIdentifier: instanceIdentifier,
            // Instance properties
            dbInstanceClass: databaseInstanceType(instanceType),
            publiclyAccessible: (_g = instanceProps.publiclyAccessible) !== null && _g !== void 0 ? _g : (instanceProps.vpcSubnets && instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC),
            enablePerformanceInsights: enablePerformanceInsights || instanceProps.enablePerformanceInsights,
            performanceInsightsKmsKeyId: (_h = instanceProps.performanceInsightEncryptionKey) === null || _h === void 0 ? void 0 : _h.keyArn,
            performanceInsightsRetentionPeriod: enablePerformanceInsights
                ? (instanceProps.performanceInsightRetention || props_1.PerformanceInsightRetention.DEFAULT)
                : undefined,
            // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
            dbSubnetGroupName: subnetGroup.subnetGroupName,
            dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            autoMinorVersionUpgrade: props.instanceProps.autoMinorVersionUpgrade,
            allowMajorVersionUpgrade: props.instanceProps.allowMajorVersionUpgrade,
            deleteAutomatedBackups: props.instanceProps.deleteAutomatedBackups,
        });
        // For instances that are part of a cluster:
        //
        //  Cluster DESTROY or SNAPSHOT -> DESTROY (snapshot is good enough to recreate)
        //  Cluster RETAIN              -> RETAIN (otherwise cluster state will disappear)
        instance.applyRemovalPolicy(util_1.helperRemovalPolicy(props.removalPolicy));
        // We must have a dependency on the NAT gateway provider here to create
        // things in the right order.
        instance.node.addDependency(internetConnected);
        instanceIdentifiers.push(instance.ref);
        instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
    }
    return { instanceEndpoints, instanceIdentifiers };
}
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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