"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssociationManager = void 0;
const cdk = require("@aws-cdk/core");
const servicecatalog_generated_1 = require("../servicecatalog.generated");
const util_1 = require("./util");
const validation_1 = require("./validation");
class AssociationManager {
    static associateProductWithPortfolio(portfolio, product, options) {
        validation_1.InputValidator.validateLength(this.prettyPrintAssociation(portfolio, product), 'description', 0, 2000, options === null || options === void 0 ? void 0 : options.description);
        const associationKey = util_1.hashValues(portfolio.node.addr, product.node.addr, product.stack.node.addr);
        const constructId = `PortfolioProductAssociation${associationKey}`;
        const existingAssociation = portfolio.node.tryFindChild(constructId);
        const cfnAssociation = existingAssociation
            ? existingAssociation
            : new servicecatalog_generated_1.CfnPortfolioProductAssociation(portfolio, constructId, {
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
            });
        return {
            associationKey: associationKey,
            cfnPortfolioProductAssociation: cfnAssociation,
        };
    }
    static constrainTagUpdates(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `ResourceUpdateConstraint${association.associationKey}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnResourceUpdateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                tagUpdateOnProvisionedProduct: options.allow === false ? 'NOT_ALLOWED' : 'ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot have multiple tag update constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static notifyOnStackEvents(portfolio, product, topic, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchNotificationConstraint${util_1.hashValues(topic.node.addr, topic.stack.node.addr, association.associationKey)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchNotificationConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                notificationArns: [topic.topicArn],
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Topic ${topic.node.path} is already subscribed to association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static constrainCloudFormationParameters(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchTemplateConstraint${util_1.hashValues(association.associationKey, options.rule.ruleName)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchTemplateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                rules: this.formatTemplateRule(portfolio.stack, options.rule),
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Provisioning rule ${options.rule.ruleName} already configured on association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static setLaunchRole(portfolio, product, launchRole, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a stackset deployment constraint has already been configured.
        if (portfolio.node.tryFindChild(this.stackSetConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot set launch role when a StackSet rule is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.launchRoleConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchRoleConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                roleArn: launchRole.roleArn,
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot set multiple launch roles for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static deployWithStackSets(portfolio, product, options) {
        var _a;
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a launch role has already been set.
        if (portfolio.node.tryFindChild(this.launchRoleConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot configure StackSet deployment when a launch role is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.stackSetConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnStackSetConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: (_a = options.description) !== null && _a !== void 0 ? _a : '',
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                accountList: options.accounts,
                regionList: options.regions,
                adminRole: options.adminRole.roleArn,
                executionRole: options.executionRoleName,
                stackInstanceControl: options.allowStackSetInstanceOperations ? 'ALLOWED' : 'NOT_ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot configure multiple StackSet deployment constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static associateTagOptions(portfolio, tagOptions) {
        const portfolioStack = cdk.Stack.of(portfolio);
        for (const [key, tagOptionsList] of Object.entries(tagOptions.tagOptionsMap)) {
            validation_1.InputValidator.validateLength(portfolio.node.addr, 'TagOption key', 1, 128, key);
            tagOptionsList.forEach((value) => {
                validation_1.InputValidator.validateLength(portfolio.node.addr, 'TagOption value', 1, 256, value);
                const tagOptionKey = util_1.hashValues(key, value, portfolioStack.node.addr);
                const tagOptionConstructId = `TagOption${tagOptionKey}`;
                let cfnTagOption = portfolioStack.node.tryFindChild(tagOptionConstructId);
                if (!cfnTagOption) {
                    cfnTagOption = new servicecatalog_generated_1.CfnTagOption(portfolioStack, tagOptionConstructId, {
                        key: key,
                        value: value,
                        active: true,
                    });
                }
                const tagAssocationKey = util_1.hashValues(key, value, portfolio.node.addr);
                const tagAssocationConstructId = `TagOptionAssociation${tagAssocationKey}`;
                if (!portfolio.node.tryFindChild(tagAssocationConstructId)) {
                    new servicecatalog_generated_1.CfnTagOptionAssociation(portfolio, tagAssocationConstructId, {
                        resourceId: portfolio.portfolioId,
                        tagOptionId: cfnTagOption.ref,
                    });
                }
            });
        }
        ;
    }
    static stackSetConstraintLogicalId(associationKey) {
        return `StackSetConstraint${associationKey}`;
    }
    static launchRoleConstraintLogicalId(associationKey) {
        return `LaunchRoleConstraint${associationKey}`;
    }
    static prettyPrintAssociation(portfolio, product) {
        return `- Portfolio: ${portfolio.node.path} | Product: ${product.node.path}`;
    }
    static formatTemplateRule(stack, rule) {
        return JSON.stringify({
            [rule.ruleName]: {
                Assertions: this.formatAssertions(stack, rule.assertions),
                RuleCondition: rule.condition ? stack.resolve(rule.condition) : undefined,
            },
        });
    }
    static formatAssertions(stack, assertions) {
        return assertions.reduce((formattedAssertions, assertion) => {
            formattedAssertions.push({
                Assert: stack.resolve(assertion.assert),
                AssertDescription: assertion.description,
            });
            return formattedAssertions;
        }, new Array());
    }
    ;
}
exports.AssociationManager = AssociationManager;
//# sourceMappingURL=data:application/json;base64,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