# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['token_throttler', 'token_throttler.storage']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'token-throttler',
    'version': '1.0.0',
    'description': 'Simple token throttle package',
    'long_description': '# Token throttler\n\n![Coverage](https://img.shields.io/gitlab/coverage/vojko.pribudic/token-throttler/main?job_name=tests)\n![Version](https://img.shields.io/pypi/pyversions/token-throttler)\n![Downloads](https://pepy.tech/badge/token-throttler)\n![License](https://img.shields.io/pypi/l/token-throttler)\n\n**Token throttler** is an extendable rate-limiting library somewhat based on a [token bucket algorithm](https://en.wikipedia.org/wiki/Token_bucket).\n\n## Table of contents\n\n1. [ Installation ](#installation)\n2. [ Features ](#features)\n3. [ Usage ](#usage)\n    1. [ Manual usage example ](#usage-manual)\n    2. [ Decorator usage example ](#usage-decorator)\n4. [ Storage ](#storage)\n   1. [ Redis storage example ](#storage-redis)\n\n<a name="installation"></a>\n## 1. Installation\n\nToken throttler is available on PyPI:\n```console \n$ python -m pip install token-throttler\n```\nToken throttler officially supports Python >= 3.7.\n\n<a name="features"></a>\n## 2. Features\n\n- Configurable token throttler cost and identifier\n- Multiple buckets per throttler per identifier\n- Buckets can be added manually or by a `dict` configuration\n- Manual usage or usage via decorator\n- Decorator usage supports async code too\n- Custom decorator can be written\n- Extendable storage engine (eg. Redis)\n\n<a name="usage"></a>\n## 3. Usage\n\nToken throttler supports both manual usage and via decorator.\n\nDecorator usage supports both async and sync.\n\n<a name="usage-manual"></a>\n### 1) Manual usage example:\n\n```python\nfrom token_throttler import TokenBucket, TokenThrottler\nfrom token_throttler.storage import RuntimeStorage\n\nthrottler: TokenThrottler = TokenThrottler(cost=1, storage=RuntimeStorage())\nthrottler.add_bucket(identifier="hello_world", bucket=TokenBucket(replenish_time=10, max_tokens=10))\nthrottler.add_bucket(identifier="hello_world", bucket=TokenBucket(replenish_time=30, max_tokens=20))\n\n\ndef hello_world() -> None:\n    print("Hello World")\n\n\nfor i in range(10):\n    throttler.consume(identifier="hello_world")\n    hello_world()\n\nif throttler.consume(identifier="hello_world"):\n    hello_world()\nelse:\n    print("bucket_one ran out of tokens")\n```\n\n<a name="usage-decorator"></a>\n### 2) Decorator usage example:\n\n```python\nfrom token_throttler import TokenBucket, TokenThrottler, TokenThrottlerException\nfrom token_throttler.storage import RuntimeStorage\n\nthrottler: TokenThrottler = TokenThrottler(1, RuntimeStorage())\nthrottler.add_bucket("hello_world", TokenBucket(10, 10))\nthrottler.add_bucket("hello_world", TokenBucket(30, 20))\n\n\n@throttler.enable("hello_world")\ndef hello_world() -> None:\n    print("Hello World")\n\n\nfor i in range(10):\n    hello_world()\n\ntry:\n    hello_world()\nexcept TokenThrottlerException:\n    print("bucket_one ran out of tokens")\n```\n\nFor other examples see [**examples**](https://gitlab.com/vojko.pribudic/token-throttler/-/tree/main/examples) directory.\n\n<a name="storage"></a>\n## 4. Storage\n\nCurrently, token throttler supports `RuntimeStorage` but is very easy to extend.\nIf you want your own storage engine, feel free to extend the `token_throttler.storage.BucketStorage` class.\n\n<a name="storage-redis"></a>\n### 1) Redis storage example:\n\n```python\nimport pickle\nfrom datetime import timedelta\nfrom typing import Union\n\nfrom redis import StrictRedis\n\nfrom token_throttler import TokenBucket, TokenThrottler\nfrom token_throttler.storage import BucketStorage\n\n\nclass RedisStorage(BucketStorage):\n    def __init__(self, connection_string: str, delimiter: str) -> None:\n        super().__init__()\n        self.redis: StrictRedis = StrictRedis.from_url(url=connection_string)\n        self.delimiter: str = delimiter\n\n    def _save_bucket(self, cache_key: str, bucket: TokenBucket) -> None:\n        self.redis.setex(\n            cache_key,\n            timedelta(seconds=bucket.replenish_time),\n            pickle.dumps(bucket),\n        )\n\n    def _get_bucket(self, cache_key: str):\n        bucket: Union[bytes, None] = self.redis.get(cache_key)\n        if not bucket:\n            bucket_info: list[str] = cache_key.split(self.delimiter)\n            token_bucket: TokenBucket = TokenBucket(\n                int(bucket_info[1]), int(bucket_info[-1])\n            )\n            token_bucket.cost = int(bucket_info[2])\n            token_bucket.identifier = bucket_info[0]\n            return token_bucket\n        return pickle.loads(bucket)\n\n    def add_bucket(self, bucket: TokenBucket) -> None:\n        cache_key: str = f"{self.delimiter}".join(\n            map(\n                str,\n                [\n                    bucket.identifier,\n                    bucket.replenish_time,\n                    bucket.cost,\n                    bucket.max_tokens,\n                ],\n            )\n        )\n        self[bucket.identifier][str(bucket.replenish_time)] = cache_key\n        self._save_bucket(cache_key, self._get_bucket(cache_key))\n\n    def replenish(self, bucket: TokenBucket) -> None:\n        pass\n\n    def consume(self, identifier: str, bucket_key: str) -> bool:\n        cache_key: str = self[identifier][str(bucket_key)]\n        bucket: TokenBucket = self._get_bucket(cache_key)\n        bucket_state: bool = bucket.consume()\n        self._save_bucket(cache_key, bucket)\n        return bucket_state\n\n\nthrottler: TokenThrottler = TokenThrottler(1, RedisStorage(connection_string="connection-string-to-redis", delimiter="||"))\n...\n```',
    'author': 'Vojko Pribudić',
    'author_email': 'dmanthing@gmail.com',
    'maintainer': 'Vojko Pribudić',
    'maintainer_email': 'dmanthing@gmail.com',
    'url': 'https://gitlab.com/vojko.pribudic/token-throttler',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7',
}


setup(**setup_kwargs)
