from functools import reduce
from hashlib import sha1
from operator import add, or_

from django import forms

from feincms3_forms.models import FormFieldBase


def short_prefix(obj, part=""):
    """
    Stupid, stable prefix for the form

    May be used to support more than one form on a single page and also to
    discourage styling based on element IDs :-)
    """
    identifier = f"{obj._meta.label}:{obj.pk}:{part}".encode()
    return "form-" + sha1(identifier).hexdigest()[:5]


class FormMixin:
    def get_form_fields(self, plugin):
        """
        Return the form fields generated by the passed plugin instance
        """
        return self._f3f_plugin_fields[plugin]

    def clean(self):
        data = super().clean()
        for hook in self._f3f_cleaners:
            data = hook(self, data)
        return data


def create_form(plugins, *, form_class=forms.Form, form_kwargs):
    field_plugins = [plugin for plugin in plugins if isinstance(plugin, FormFieldBase)]
    plugin_fields = {plugin: plugin.get_fields() for plugin in field_plugins}
    all_fields = reduce(or_, plugin_fields.values(), {})
    all_names = set(all_fields)

    initial = reduce(
        or_,
        (plugin.get_initial() for plugin in field_plugins),
        {},
    )
    form_kwargs["initial"] = initial | form_kwargs.get("initial", {})

    form = type("Form", (FormMixin, form_class), all_fields)(**form_kwargs)
    form._f3f_plugin_fields = {
        plugin: {name: form[name] for name in fields}
        for plugin, fields in plugin_fields.items()
    } | {None: {name: form[name] for name in form.fields if name not in all_names}}
    form._f3f_cleaners = reduce(
        add, (plugin.get_cleaners() for plugin in field_plugins), []
    )

    return form
