import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IDatabaseCluster } from './cluster-ref';
import { IDatabaseInstance } from './instance';
/**
 * SessionPinningFilter.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 * @stability stable
 */
export declare class SessionPinningFilter {
    /**
     * Filter name
     */
    readonly filterName: string;
    /**
     * You can opt out of session pinning for the following kinds of application statements:.
     *
     * - Setting session variables and configuration settings.
     *
     * @stability stable
     */
    static readonly EXCLUDE_VARIABLE_SETS: SessionPinningFilter;
    /**
     * custom filter.
     *
     * @stability stable
     */
    static of(filterName: string): SessionPinningFilter;
    private constructor();
}
/**
 * Proxy target: Instance or Cluster.
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 *
 * @stability stable
 */
export declare class ProxyTarget {
    private readonly dbInstance;
    private readonly dbCluster;
    /**
     * From instance.
     *
     * @param instance RDS database instance.
     * @stability stable
     */
    static fromInstance(instance: IDatabaseInstance): ProxyTarget;
    /**
     * From cluster.
     *
     * @param cluster RDS database cluster.
     * @stability stable
     */
    static fromCluster(cluster: IDatabaseCluster): ProxyTarget;
    private constructor();
    /**
     * Bind this target to the specified database proxy.
     *
     * @stability stable
     */
    bind(_: DatabaseProxy): ProxyTargetConfig;
}
/**
 * The result of binding a `ProxyTarget` to a `DatabaseProxy`.
 *
 * @stability stable
 */
export interface ProxyTargetConfig {
    /**
     * The engine family of the database instance or cluster this proxy connects with.
     *
     * @stability stable
     */
    readonly engineFamily: string;
    /**
     * The database instances to which this proxy connects.
     *
     * Either this or `dbClusters` will be set and the other `undefined`.
     *
     * @default - `undefined` if `dbClusters` is set.
     * @stability stable
     */
    readonly dbInstances?: IDatabaseInstance[];
    /**
     * The database clusters to which this proxy connects.
     *
     * Either this or `dbInstances` will be set and the other `undefined`.
     *
     * @default - `undefined` if `dbInstances` is set.
     * @stability stable
     */
    readonly dbClusters?: IDatabaseCluster[];
}
/**
 * Options for a new DatabaseProxy.
 *
 * @stability stable
 */
export interface DatabaseProxyOptions {
    /**
     * The identifier for the proxy.
     *
     * This name must be unique for all proxies owned by your AWS account in the specified AWS Region.
     * An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens;
     * it can't end with a hyphen or contain two consecutive hyphens.
     *
     * @default - Generated by CloudFormation (recommended)
     * @stability stable
     */
    readonly dbProxyName?: string;
    /**
     * The duration for a proxy to wait for a connection to become available in the connection pool.
     *
     * Only applies when the proxy has opened its maximum number of connections and all connections are busy with client
     * sessions.
     *
     * Value must be between 1 second and 1 hour, or `Duration.seconds(0)` to represent unlimited.
     *
     * @default cdk.Duration.seconds(120)
     * @stability stable
     */
    readonly borrowTimeout?: cdk.Duration;
    /**
     * One or more SQL statements for the proxy to run when opening each new database connection.
     *
     * Typically used with SET statements to make sure that each connection has identical settings such as time zone
     * and character set.
     * For multiple statements, use semicolons as the separator.
     * You can also include multiple variables in a single SET statement, such as SET x=1, y=2.
     *
     * not currently supported for PostgreSQL.
     *
     * @default - no initialization query
     * @stability stable
     */
    readonly initQuery?: string;
    /**
     * The maximum size of the connection pool for each target in a target group.
     *
     * For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB
     * cluster used by the target group.
     *
     * 1-100
     *
     * @default 100
     * @stability stable
     */
    readonly maxConnectionsPercent?: number;
    /**
     * Controls how actively the proxy closes idle database connections in the connection pool.
     *
     * A high value enables the proxy to leave a high percentage of idle connections open.
     * A low value causes the proxy to close idle client connections and return the underlying database connections
     * to the connection pool.
     * For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance
     * or Aurora DB cluster used by the target group.
     *
     * between 0 and MaxConnectionsPercent
     *
     * @default 50
     * @stability stable
     */
    readonly maxIdleConnectionsPercent?: number;
    /**
     * Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection.
     *
     * Including an item in the list exempts that class of SQL operations from the pinning behavior.
     *
     * @default - no session pinning filters
     * @stability stable
     */
    readonly sessionPinningFilters?: SessionPinningFilter[];
    /**
     * Whether the proxy includes detailed information about SQL statements in its logs.
     *
     * This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections.
     * The debug information includes the text of SQL statements that you submit through the proxy.
     * Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive
     * information that appears in the logs.
     *
     * @default false
     * @stability stable
     */
    readonly debugLogging?: boolean;
    /**
     * Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy.
     *
     * @default false
     * @stability stable
     */
    readonly iamAuth?: boolean;
    /**
     * The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it.
     *
     * You can set this value higher or lower than the connection timeout limit for the associated database.
     *
     * @default cdk.Duration.minutes(30)
     * @stability stable
     */
    readonly idleClientTimeout?: cdk.Duration;
    /**
     * A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy.
     *
     * By enabling this setting, you can enforce encrypted TLS connections to the proxy.
     *
     * @default true
     * @stability stable
     */
    readonly requireTLS?: boolean;
    /**
     * IAM role that the proxy uses to access secrets in AWS Secrets Manager.
     *
     * @default - A role will automatically be created
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster.
     *
     * These secrets are stored within Amazon Secrets Manager.
     * One or more secrets are required.
     *
     * @stability stable
     */
    readonly secrets: secretsmanager.ISecret[];
    /**
     * One or more VPC security groups to associate with the new proxy.
     *
     * @default - No security groups
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The subnets used by the proxy.
     *
     * @default - the VPC default strategy if not specified.
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * The VPC to associate with the new proxy.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
}
/**
 * Construction properties for a DatabaseProxy.
 *
 * @stability stable
 */
export interface DatabaseProxyProps extends DatabaseProxyOptions {
    /**
     * DB proxy target: Instance or Cluster.
     *
     * @stability stable
     */
    readonly proxyTarget: ProxyTarget;
}
/**
 * Properties that describe an existing DB Proxy.
 *
 * @stability stable
 */
export interface DatabaseProxyAttributes {
    /**
     * DB Proxy Name.
     *
     * @stability stable
     */
    readonly dbProxyName: string;
    /**
     * DB Proxy ARN.
     *
     * @stability stable
     */
    readonly dbProxyArn: string;
    /**
     * Endpoint.
     *
     * @stability stable
     */
    readonly endpoint: string;
    /**
     * The security groups of the instance.
     *
     * @stability stable
     */
    readonly securityGroups: ec2.ISecurityGroup[];
}
/**
 * DB Proxy.
 *
 * @stability stable
 */
export interface IDatabaseProxy extends cdk.IResource {
    /**
     * DB Proxy Name.
     *
     * @stability stable
     * @attribute true
     */
    readonly dbProxyName: string;
    /**
     * DB Proxy ARN.
     *
     * @stability stable
     * @attribute true
     */
    readonly dbProxyArn: string;
    /**
     * Endpoint.
     *
     * @stability stable
     * @attribute true
     */
    readonly endpoint: string;
    /**
     * Grant the given identity connection access to the proxy.
     *
     * @stability stable
     */
    grantConnect(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Represents an RDS Database Proxy.
 *
 */
declare abstract class DatabaseProxyBase extends cdk.Resource implements IDatabaseProxy {
    abstract readonly dbProxyName: string;
    abstract readonly dbProxyArn: string;
    abstract readonly endpoint: string;
    /**
     * Grant the given identity connection access to the proxy.
     *
     * @stability stable
     */
    grantConnect(grantee: iam.IGrantable): iam.Grant;
}
/**
 * RDS Database Proxy.
 *
 * @stability stable
 * @resource AWS::RDS::DBProxy
 */
export declare class DatabaseProxy extends DatabaseProxyBase implements ec2.IConnectable, secretsmanager.ISecretAttachmentTarget {
    /**
     * Import an existing database proxy.
     *
     * @stability stable
     */
    static fromDatabaseProxyAttributes(scope: Construct, id: string, attrs: DatabaseProxyAttributes): IDatabaseProxy;
    /**
     * DB Proxy Name.
     *
     * @stability stable
     * @attribute true
     */
    readonly dbProxyName: string;
    /**
     * DB Proxy ARN.
     *
     * @stability stable
     * @attribute true
     */
    readonly dbProxyArn: string;
    /**
     * Endpoint.
     *
     * @stability stable
     * @attribute true
     */
    readonly endpoint: string;
    /**
     * Access to network connections.
     *
     * @stability stable
     */
    readonly connections: ec2.Connections;
    private readonly resource;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: DatabaseProxyProps);
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget(): secretsmanager.SecretAttachmentTargetProps;
}
export {};
