import * as ec2 from '@aws-cdk/aws-ec2';
import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as logs from '@aws-cdk/aws-logs';
import * as s3 from '@aws-cdk/aws-s3';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { Duration, IResource, RemovalPolicy, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Endpoint } from './endpoint';
import { IInstanceEngine } from './instance-engine';
import { IOptionGroup } from './option-group';
import { IParameterGroup } from './parameter-group';
import { Credentials, PerformanceInsightRetention, RotationMultiUserOptions, RotationSingleUserOptions, SnapshotCredentials } from './props';
import { DatabaseProxy, DatabaseProxyOptions } from './proxy';
import { CfnDBInstanceProps } from './rds.generated';
import { ISubnetGroup } from './subnet-group';
/**
 * A database instance.
 *
 * @stability stable
 */
export interface IDatabaseInstance extends IResource, ec2.IConnectable, secretsmanager.ISecretAttachmentTarget {
    /**
     * The instance identifier.
     *
     * @stability stable
     */
    readonly instanceIdentifier: string;
    /**
     * The instance arn.
     *
     * @stability stable
     */
    readonly instanceArn: string;
    /**
     * The instance endpoint address.
     *
     * @stability stable
     * @attribute EndpointAddress
     */
    readonly dbInstanceEndpointAddress: string;
    /**
     * The instance endpoint port.
     *
     * @stability stable
     * @attribute EndpointPort
     */
    readonly dbInstanceEndpointPort: string;
    /**
     * The instance endpoint.
     *
     * @stability stable
     */
    readonly instanceEndpoint: Endpoint;
    /**
     * The engine of this database Instance.
     *
     * May be not known for imported Instances if it wasn't provided explicitly,
     * or for read replicas.
     *
     * @stability stable
     */
    readonly engine?: IInstanceEngine;
    /**
     * Add a new db proxy to this instance.
     *
     * @stability stable
     */
    addProxy(id: string, options: DatabaseProxyOptions): DatabaseProxy;
    /**
     * Grant the given identity connection access to the database.
     *
     * @stability stable
     */
    grantConnect(grantee: iam.IGrantable): iam.Grant;
    /**
     * Defines a CloudWatch event rule which triggers for instance events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
}
/**
 * Properties that describe an existing instance.
 *
 * @stability stable
 */
export interface DatabaseInstanceAttributes {
    /**
     * The instance identifier.
     *
     * @stability stable
     */
    readonly instanceIdentifier: string;
    /**
     * The endpoint address.
     *
     * @stability stable
     */
    readonly instanceEndpointAddress: string;
    /**
     * The database port.
     *
     * @stability stable
     */
    readonly port: number;
    /**
     * The security groups of the instance.
     *
     * @stability stable
     */
    readonly securityGroups: ec2.ISecurityGroup[];
    /**
     * The engine of the existing database Instance.
     *
     * @default - the imported Instance's engine is unknown
     * @stability stable
     */
    readonly engine?: IInstanceEngine;
}
/**
 * A new or imported database instance.
 *
 * @stability stable
 */
export declare abstract class DatabaseInstanceBase extends Resource implements IDatabaseInstance {
    /**
     * Import an existing database instance.
     *
     * @stability stable
     */
    static fromDatabaseInstanceAttributes(scope: Construct, id: string, attrs: DatabaseInstanceAttributes): IDatabaseInstance;
    /**
     * The instance identifier.
     *
     * @stability stable
     */
    abstract readonly instanceIdentifier: string;
    /**
     * The instance endpoint address.
     *
     * @stability stable
     */
    abstract readonly dbInstanceEndpointAddress: string;
    /**
     * The instance endpoint port.
     *
     * @stability stable
     */
    abstract readonly dbInstanceEndpointPort: string;
    /**
     * The instance endpoint.
     *
     * @stability stable
     */
    abstract readonly instanceEndpoint: Endpoint;
    /**
     * The engine of this database Instance.
     *
     * May be not known for imported Instances if it wasn't provided explicitly,
     * or for read replicas.
     *
     * @stability stable
     */
    abstract readonly engine?: IInstanceEngine;
    /**
     * @stability stable
     */
    protected abstract enableIamAuthentication?: boolean;
    /**
     * Access to network connections.
     *
     * @stability stable
     */
    abstract readonly connections: ec2.Connections;
    /**
     * Add a new db proxy to this instance.
     *
     * @stability stable
     */
    addProxy(id: string, options: DatabaseProxyOptions): DatabaseProxy;
    /**
     * Grant the given identity connection access to the database.
     *
     * @stability stable
     */
    grantConnect(grantee: iam.IGrantable): iam.Grant;
    /**
     * Defines a CloudWatch event rule which triggers for instance events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * The instance arn.
     *
     * @stability stable
     */
    get instanceArn(): string;
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget(): secretsmanager.SecretAttachmentTargetProps;
}
/**
 * The license model.
 *
 * @stability stable
 */
export declare enum LicenseModel {
    /**
     * License included.
     *
     * @stability stable
     */
    LICENSE_INCLUDED = "license-included",
    /**
     * Bring your own licencse.
     *
     * @stability stable
     */
    BRING_YOUR_OWN_LICENSE = "bring-your-own-license",
    /**
     * General public license.
     *
     * @stability stable
     */
    GENERAL_PUBLIC_LICENSE = "general-public-license"
}
/**
 * The processor features.
 *
 * @stability stable
 */
export interface ProcessorFeatures {
    /**
     * The number of CPU core.
     *
     * @default - the default number of CPU cores for the chosen instance class.
     * @stability stable
     */
    readonly coreCount?: number;
    /**
     * The number of threads per core.
     *
     * @default - the default number of threads per core for the chosen instance class.
     * @stability stable
     */
    readonly threadsPerCore?: number;
}
/**
 * The type of storage.
 *
 * @stability stable
 */
export declare enum StorageType {
    /**
     * Standard.
     *
     * @stability stable
     */
    STANDARD = "standard",
    /**
     * General purpose (SSD).
     *
     * @stability stable
     */
    GP2 = "gp2",
    /**
     * Provisioned IOPS (SSD).
     *
     * @stability stable
     */
    IO1 = "io1"
}
/**
 * Construction properties for a DatabaseInstanceNew.
 *
 * @stability stable
 */
export interface DatabaseInstanceNewProps {
    /**
     * Specifies if the database instance is a multiple Availability Zone deployment.
     *
     * @default false
     * @stability stable
     */
    readonly multiAz?: boolean;
    /**
     * The name of the Availability Zone where the DB instance will be located.
     *
     * @default - no preference
     * @stability stable
     */
    readonly availabilityZone?: string;
    /**
     * The storage type.
     *
     * Storage types supported are gp2, io1, standard.
     *
     * @default GP2
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#Concepts.Storage.GeneralSSD
     * @stability stable
     */
    readonly storageType?: StorageType;
    /**
     * The number of I/O operations per second (IOPS) that the database provisions.
     *
     * The value must be equal to or greater than 1000.
     *
     * @default - no provisioned iops
     * @stability stable
     */
    readonly iops?: number;
    /**
     * The number of CPU cores and the number of threads per core.
     *
     * @default - the default number of CPU cores and threads per core for the
     * chosen instance class.
     *
     * See https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html#USER_ConfigureProcessor
     * @stability stable
     */
    readonly processorFeatures?: ProcessorFeatures;
    /**
     * A name for the DB instance.
     *
     * If you specify a name, AWS CloudFormation
     * converts it to lowercase.
     *
     * @default - a CloudFormation generated name
     * @stability stable
     */
    readonly instanceIdentifier?: string;
    /**
     * The VPC network where the DB subnet group should be created.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
    /**
     * (deprecated) The type of subnets to add to the created DB subnet group.
     *
     * @default - private subnets
     * @deprecated use `vpcSubnets`
     */
    readonly vpcPlacement?: ec2.SubnetSelection;
    /**
     * The type of subnets to add to the created DB subnet group.
     *
     * @default - private subnets
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * The security groups to assign to the DB instance.
     *
     * @default - a new security group is created
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The port for the instance.
     *
     * @default - the default port for the chosen engine.
     * @stability stable
     */
    readonly port?: number;
    /**
     * The option group to associate with the instance.
     *
     * @default - no option group
     * @stability stable
     */
    readonly optionGroup?: IOptionGroup;
    /**
     * Whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.
     *
     * @default false
     * @stability stable
     */
    readonly iamAuthentication?: boolean;
    /**
     * The number of days during which automatic DB snapshots are retained.
     *
     * Set to zero to disable backups.
     * When creating a read replica, you must enable automatic backups on the source
     * database instance by setting the backup retention to a value other than zero.
     *
     * @default Duration.days(1)
     * @stability stable
     */
    readonly backupRetention?: Duration;
    /**
     * The daily time range during which automated backups are performed.
     *
     * Constraints:
     * - Must be in the format `hh24:mi-hh24:mi`.
     * - Must be in Universal Coordinated Time (UTC).
     * - Must not conflict with the preferred maintenance window.
     * - Must be at least 30 minutes.
     *
     * @default - a 30-minute window selected at random from an 8-hour block of
     * time for each AWS Region. To see the time blocks available, see
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
     * @stability stable
     */
    readonly preferredBackupWindow?: string;
    /**
     * Indicates whether to copy all of the user-defined tags from the DB instance to snapshots of the DB instance.
     *
     * @default true
     * @stability stable
     */
    readonly copyTagsToSnapshot?: boolean;
    /**
     * Indicates whether automated backups should be deleted or retained when you delete a DB instance.
     *
     * @default false
     * @stability stable
     */
    readonly deleteAutomatedBackups?: boolean;
    /**
     * The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instance.
     *
     * @default - no enhanced monitoring
     * @stability stable
     */
    readonly monitoringInterval?: Duration;
    /**
     * Role that will be used to manage DB instance monitoring.
     *
     * @default - A role is automatically created for you
     * @stability stable
     */
    readonly monitoringRole?: iam.IRole;
    /**
     * Whether to enable Performance Insights for the DB instance.
     *
     * @default - false, unless ``performanceInsightRentention`` or ``performanceInsightEncryptionKey`` is set.
     * @stability stable
     */
    readonly enablePerformanceInsights?: boolean;
    /**
     * The amount of time, in days, to retain Performance Insights data.
     *
     * @default 7
     * @stability stable
     */
    readonly performanceInsightRetention?: PerformanceInsightRetention;
    /**
     * The AWS KMS key for encryption of Performance Insights data.
     *
     * @default - default master key
     * @stability stable
     */
    readonly performanceInsightEncryptionKey?: kms.IKey;
    /**
     * The list of log types that need to be enabled for exporting to CloudWatch Logs.
     *
     * @default - no log exports
     * @stability stable
     */
    readonly cloudwatchLogsExports?: string[];
    /**
     * The number of days log events are kept in CloudWatch Logs.
     *
     * When updating
     * this property, unsetting it doesn't remove the log retention policy. To
     * remove the retention policy, set the value to `Infinity`.
     *
     * @default - logs never expire
     * @stability stable
     */
    readonly cloudwatchLogsRetention?: logs.RetentionDays;
    /**
     * The IAM role for the Lambda function associated with the custom resource that sets the retention policy.
     *
     * @default - a new role is created.
     * @stability stable
     */
    readonly cloudwatchLogsRetentionRole?: iam.IRole;
    /**
     * Indicates that minor engine upgrades are applied automatically to the DB instance during the maintenance window.
     *
     * @default true
     * @stability stable
     */
    readonly autoMinorVersionUpgrade?: boolean;
    /**
     * The weekly time range (in UTC) during which system maintenance can occur.
     *
     * Format: `ddd:hh24:mi-ddd:hh24:mi`
     * Constraint: Minimum 30-minute window
     *
     * @default - a 30-minute window selected at random from an 8-hour block of
     * time for each AWS Region, occurring on a random day of the week. To see
     * the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
     * @stability stable
     */
    readonly preferredMaintenanceWindow?: string;
    /**
     * Indicates whether the DB instance should have deletion protection enabled.
     *
     * @default - true if ``removalPolicy`` is RETAIN, false otherwise
     * @stability stable
     */
    readonly deletionProtection?: boolean;
    /**
     * The CloudFormation policy to apply when the instance is removed from the stack or replaced during an update.
     *
     * @default - RemovalPolicy.SNAPSHOT (remove the resource, but retain a snapshot of the data)
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Upper limit to which RDS can scale the storage in GiB(Gibibyte).
     *
     * @default - No autoscaling of RDS instance
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling
     * @stability stable
     */
    readonly maxAllocatedStorage?: number;
    /**
     * The Active Directory directory ID to create the DB instance in.
     *
     * @default - Do not join domain
     * @stability stable
     */
    readonly domain?: string;
    /**
     * The IAM role to be used when making API calls to the Directory Service.
     *
     * The role needs the AWS-managed policy
     * AmazonRDSDirectoryServiceAccess or equivalent.
     *
     * @default - The role will be created for you if {@link DatabaseInstanceNewProps#domain} is specified
     * @stability stable
     */
    readonly domainRole?: iam.IRole;
    /**
     * Existing subnet group for the instance.
     *
     * @default - a new subnet group will be created.
     * @stability stable
     */
    readonly subnetGroup?: ISubnetGroup;
    /**
     * Role that will be associated with this DB instance to enable S3 import.
     *
     * This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines.
     *
     * This property must not be used if `s3ImportBuckets` is used.
     *
     * For Microsoft SQL Server:
     *
     * @default - New role is created if `s3ImportBuckets` is set, no role is defined otherwise
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html
     * @stability stable
     */
    readonly s3ImportRole?: iam.IRole;
    /**
     * S3 buckets that you want to load data from.
     *
     * This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines.
     *
     * This property must not be used if `s3ImportRole` is used.
     *
     * For Microsoft SQL Server:
     *
     * @default - None
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html
     * @stability stable
     */
    readonly s3ImportBuckets?: s3.IBucket[];
    /**
     * Role that will be associated with this DB instance to enable S3 export.
     *
     * This feature is only supported by the Microsoft SQL Server and Oracle engines.
     *
     * This property must not be used if `s3ExportBuckets` is used.
     *
     * For Microsoft SQL Server:
     *
     * @default - New role is created if `s3ExportBuckets` is set, no role is defined otherwise
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
     * @stability stable
     */
    readonly s3ExportRole?: iam.IRole;
    /**
     * S3 buckets that you want to load data into.
     *
     * This feature is only supported by the Microsoft SQL Server and Oracle engines.
     *
     * This property must not be used if `s3ExportRole` is used.
     *
     * For Microsoft SQL Server:
     *
     * @default - None
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
     * @stability stable
     */
    readonly s3ExportBuckets?: s3.IBucket[];
    /**
     * Indicates whether the DB instance is an internet-facing instance.
     *
     * @default - `true` if `vpcSubnets` is `subnetType: SubnetType.PUBLIC`, `false` otherwise
     * @stability stable
     */
    readonly publiclyAccessible?: boolean;
}
/**
 * A new database instance.
 */
declare abstract class DatabaseInstanceNew extends DatabaseInstanceBase implements IDatabaseInstance {
    /**
     * The VPC where this database instance is deployed.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
    /**
     * Access to network connections.
     *
     * @stability stable
     */
    readonly connections: ec2.Connections;
    protected abstract readonly instanceType: ec2.InstanceType;
    /**
     * @stability stable
     */
    protected readonly vpcPlacement?: ec2.SubnetSelection;
    /**
     * @stability stable
     */
    protected readonly newCfnProps: CfnDBInstanceProps;
    private readonly cloudwatchLogsExports?;
    private readonly cloudwatchLogsRetention?;
    private readonly cloudwatchLogsRetentionRole?;
    private readonly domainId?;
    private readonly domainRole?;
    /**
     * @stability stable
     */
    protected enableIamAuthentication?: boolean;
    constructor(scope: Construct, id: string, props: DatabaseInstanceNewProps);
    /**
     * @stability stable
     */
    protected setLogRetention(): void;
}
/**
 * Construction properties for a DatabaseInstanceSource.
 *
 * @stability stable
 */
export interface DatabaseInstanceSourceProps extends DatabaseInstanceNewProps {
    /**
     * The database engine.
     *
     * @stability stable
     */
    readonly engine: IInstanceEngine;
    /**
     * The name of the compute and memory capacity for the instance.
     *
     * @default - m5.large (or, more specifically, db.m5.large)
     * @stability stable
     */
    readonly instanceType?: ec2.InstanceType;
    /**
     * The license model.
     *
     * @default - RDS default license model
     * @stability stable
     */
    readonly licenseModel?: LicenseModel;
    /**
     * Whether to allow major version upgrades.
     *
     * @default false
     * @stability stable
     */
    readonly allowMajorVersionUpgrade?: boolean;
    /**
     * The time zone of the instance.
     *
     * This is currently supported only by Microsoft Sql Server.
     *
     * @default - RDS default timezone
     * @stability stable
     */
    readonly timezone?: string;
    /**
     * The allocated storage size, specified in gigabytes (GB).
     *
     * @default 100
     * @stability stable
     */
    readonly allocatedStorage?: number;
    /**
     * The name of the database.
     *
     * @default - no name
     * @stability stable
     */
    readonly databaseName?: string;
    /**
     * The DB parameter group to associate with the instance.
     *
     * @default - no parameter group
     * @stability stable
     */
    readonly parameterGroup?: IParameterGroup;
}
/**
 * A new source database instance (not a read replica)
 */
declare abstract class DatabaseInstanceSource extends DatabaseInstanceNew implements IDatabaseInstance {
    /**
     * The engine of this database Instance.
     *
     * May be not known for imported Instances if it wasn't provided explicitly,
     * or for read replicas.
     *
     * @stability stable
     */
    readonly engine?: IInstanceEngine;
    /**
     * The AWS Secrets Manager secret attached to the instance.
     */
    abstract readonly secret?: secretsmanager.ISecret;
    /**
     * @stability stable
     */
    protected readonly sourceCfnProps: CfnDBInstanceProps;
    /**
     * @stability stable
     */
    protected readonly instanceType: ec2.InstanceType;
    private readonly singleUserRotationApplication;
    private readonly multiUserRotationApplication;
    constructor(scope: Construct, id: string, props: DatabaseInstanceSourceProps);
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param options the options for the rotation, if you want to override the defaults.
     * @stability stable
     */
    addRotationSingleUser(options?: RotationSingleUserOptions): secretsmanager.SecretRotation;
    /**
     * Adds the multi user rotation to this instance.
     *
     * @stability stable
     */
    addRotationMultiUser(id: string, options: RotationMultiUserOptions): secretsmanager.SecretRotation;
}
/**
 * Construction properties for a DatabaseInstance.
 *
 * @stability stable
 */
export interface DatabaseInstanceProps extends DatabaseInstanceSourceProps {
    /**
     * Credentials for the administrative user.
     *
     * @default - A username of 'admin' (or 'postgres' for PostgreSQL) and SecretsManager-generated password
     * @stability stable
     */
    readonly credentials?: Credentials;
    /**
     * For supported engines, specifies the character set to associate with the DB instance.
     *
     * @default - RDS default character set name
     * @stability stable
     */
    readonly characterSetName?: string;
    /**
     * Indicates whether the DB instance is encrypted.
     *
     * @default - true if storageEncryptionKey has been provided, false otherwise
     * @stability stable
     */
    readonly storageEncrypted?: boolean;
    /**
     * The KMS key that's used to encrypt the DB instance.
     *
     * @default - default master key if storageEncrypted is true, no key otherwise
     * @stability stable
     */
    readonly storageEncryptionKey?: kms.IKey;
}
/**
 * A database instance.
 *
 * @stability stable
 * @resource AWS::RDS::DBInstance
 */
export declare class DatabaseInstance extends DatabaseInstanceSource implements IDatabaseInstance {
    /**
     * The instance identifier.
     *
     * @stability stable
     */
    readonly instanceIdentifier: string;
    /**
     * The instance endpoint address.
     *
     * @stability stable
     */
    readonly dbInstanceEndpointAddress: string;
    /**
     * The instance endpoint port.
     *
     * @stability stable
     */
    readonly dbInstanceEndpointPort: string;
    /**
     * The instance endpoint.
     *
     * @stability stable
     */
    readonly instanceEndpoint: Endpoint;
    /**
     * The AWS Secrets Manager secret attached to the instance.
     *
     * @stability stable
     */
    readonly secret?: secretsmanager.ISecret;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: DatabaseInstanceProps);
}
/**
 * Construction properties for a DatabaseInstanceFromSnapshot.
 *
 * @stability stable
 */
export interface DatabaseInstanceFromSnapshotProps extends DatabaseInstanceSourceProps {
    /**
     * The name or Amazon Resource Name (ARN) of the DB snapshot that's used to restore the DB instance.
     *
     * If you're restoring from a shared manual DB
     * snapshot, you must specify the ARN of the snapshot.
     *
     * @stability stable
     */
    readonly snapshotIdentifier: string;
    /**
     * Master user credentials.
     *
     * Note - It is not possible to change the master username for a snapshot;
     * however, it is possible to provide (or generate) a new password.
     *
     * @default - The existing username and password from the snapshot will be used.
     * @stability stable
     */
    readonly credentials?: SnapshotCredentials;
}
/**
 * A database instance restored from a snapshot.
 *
 * @stability stable
 * @resource AWS::RDS::DBInstance
 */
export declare class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource implements IDatabaseInstance {
    /**
     * The instance identifier.
     *
     * @stability stable
     */
    readonly instanceIdentifier: string;
    /**
     * The instance endpoint address.
     *
     * @stability stable
     */
    readonly dbInstanceEndpointAddress: string;
    /**
     * The instance endpoint port.
     *
     * @stability stable
     */
    readonly dbInstanceEndpointPort: string;
    /**
     * The instance endpoint.
     *
     * @stability stable
     */
    readonly instanceEndpoint: Endpoint;
    /**
     * The AWS Secrets Manager secret attached to the instance.
     *
     * @stability stable
     */
    readonly secret?: secretsmanager.ISecret;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: DatabaseInstanceFromSnapshotProps);
}
/**
 * Construction properties for a DatabaseInstanceReadReplica.
 *
 * @stability stable
 */
export interface DatabaseInstanceReadReplicaProps extends DatabaseInstanceNewProps {
    /**
     * The name of the compute and memory capacity classes.
     *
     * @stability stable
     */
    readonly instanceType: ec2.InstanceType;
    /**
     * The source database instance.
     *
     * Each DB instance can have a limited number of read replicas. For more
     * information, see https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html.
     *
     * @stability stable
     */
    readonly sourceDatabaseInstance: IDatabaseInstance;
    /**
     * Indicates whether the DB instance is encrypted.
     *
     * @default - true if storageEncryptionKey has been provided, false otherwise
     * @stability stable
     */
    readonly storageEncrypted?: boolean;
    /**
     * The KMS key that's used to encrypt the DB instance.
     *
     * @default - default master key if storageEncrypted is true, no key otherwise
     * @stability stable
     */
    readonly storageEncryptionKey?: kms.IKey;
}
/**
 * A read replica database instance.
 *
 * @stability stable
 * @resource AWS::RDS::DBInstance
 */
export declare class DatabaseInstanceReadReplica extends DatabaseInstanceNew implements IDatabaseInstance {
    /**
     * The instance identifier.
     *
     * @stability stable
     */
    readonly instanceIdentifier: string;
    /**
     * The instance endpoint address.
     *
     * @stability stable
     */
    readonly dbInstanceEndpointAddress: string;
    /**
     * The instance endpoint port.
     *
     * @stability stable
     */
    readonly dbInstanceEndpointPort: string;
    /**
     * The instance endpoint.
     *
     * @stability stable
     */
    readonly instanceEndpoint: Endpoint;
    /**
     * The engine of this database Instance.
     *
     * May be not known for imported Instances if it wasn't provided explicitly,
     * or for read replicas.
     *
     * @stability stable
     */
    readonly engine?: IInstanceEngine;
    /**
     * @stability stable
     */
    protected readonly instanceType: ec2.InstanceType;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: DatabaseInstanceReadReplicaProps);
}
export {};
