"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 * @stability stable
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter.
     *
     * @stability stable
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
/**
 * You can opt out of session pinning for the following kinds of application statements:.
 *
 * - Setting session variables and configuration settings.
 *
 * @stability stable
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster.
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 *
 * @stability stable
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance.
     *
     * @param instance RDS database instance.
     * @stability stable
     */
    static fromInstance(instance) {
        return new ProxyTarget(instance, undefined);
    }
    /**
     * From cluster.
     *
     * @param cluster RDS database cluster.
     * @stability stable
     */
    static fromCluster(cluster) {
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     *
     * @stability stable
     */
    bind(_) {
        var _a, _b, _c, _d;
        const engine = (_b = (_a = this.dbInstance) === null || _a === void 0 ? void 0 : _a.engine) !== null && _b !== void 0 ? _b : (_c = this.dbCluster) === null || _c === void 0 ? void 0 : _c.engine;
        if (!engine) {
            const errorResource = (_d = this.dbCluster) !== null && _d !== void 0 ? _d : this.dbInstance;
            throw new Error(`Could not determine engine for proxy target '${errorResource === null || errorResource === void 0 ? void 0 : errorResource.node.path}'. ` +
                'Please provide it explicitly when importing the resource');
        }
        const engineFamily = engine.engineFamily;
        if (!engineFamily) {
            throw new Error(`Engine '${util_1.engineDescription(engine)}' does not support proxies`);
        }
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
/**
 * Represents an RDS Database Proxy.
 *
 */
class DatabaseProxyBase extends cdk.Resource {
    /**
     * Grant the given identity connection access to the proxy.
     *
     * @stability stable
     */
    grantConnect(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [this.dbProxyArn],
        });
    }
}
/**
 * RDS Database Proxy.
 *
 * @stability stable
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends DatabaseProxyBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id, { physicalName: props.dbProxyName || id });
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        this.connections = new ec2.Connections({ securityGroups: props.securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_a = props.idleClientTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            requireTls: (_b = props.requireTLS) !== null && _b !== void 0 ? _b : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: (_c = props.securityGroups) === null || _c === void 0 ? void 0 : _c.map(_ => _.securityGroupId),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        const proxyTargetGroup = new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
        (_d = bindResult.dbClusters) === null || _d === void 0 ? void 0 : _d.forEach((c) => proxyTargetGroup.node.addDependency(c));
    }
    /**
     * Import an existing database proxy.
     *
     * @stability stable
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        class Import extends DatabaseProxyBase {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
}
exports.DatabaseProxy = DatabaseProxy;
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _a, _b;
    return {
        connectionBorrowTimeout: (_a = props.borrowTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_b = props.sessionPinningFilters) === null || _b === void 0 ? void 0 : _b.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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