"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OptionGroup = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * An option group.
 *
 * @stability stable
 */
class OptionGroup extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * The connections object for the options.
         *
         * @stability stable
         */
        this.optionConnections = {};
        this.configurations = [];
        const majorEngineVersion = (_a = props.engine.engineVersion) === null || _a === void 0 ? void 0 : _a.majorVersion;
        if (!majorEngineVersion) {
            throw new Error("OptionGroup cannot be used with an engine that doesn't specify a version");
        }
        props.configurations.forEach(config => this.addConfiguration(config));
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.engineType,
            majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.engineType} ${majorEngineVersion}`,
            optionConfigurations: core_1.Lazy.any({ produce: () => this.renderConfigurations(this.configurations) }),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     *
     * @stability stable
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
            addConfiguration(_) { return false; }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a configuration to this OptionGroup.
     *
     * This method is a no-op for an imported OptionGroup.
     *
     * @stability stable
     */
    addConfiguration(configuration) {
        this.configurations.push(configuration);
        if (configuration.port) {
            if (!configuration.vpc) {
                throw new Error('`port` and `vpc` must be specified together.');
            }
            const securityGroups = configuration.securityGroups && configuration.securityGroups.length > 0
                ? configuration.securityGroups
                : [new ec2.SecurityGroup(this, `SecurityGroup${configuration.name}`, {
                        description: `Security group for ${configuration.name} option`,
                        vpc: configuration.vpc,
                    })];
            this.optionConnections[configuration.name] = new ec2.Connections({
                securityGroups: securityGroups,
                defaultPort: ec2.Port.tcp(configuration.port),
            });
        }
        return true;
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            const securityGroups = config.vpc
                ? this.optionConnections[config.name].securityGroups.map(sg => sg.securityGroupId)
                : undefined;
            configs.push({
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
                port: config.port,
                vpcSecurityGroupMemberships: securityGroups,
            });
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,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