#!/usr/bin/env python3
# -*- coding: utf-8 -*-

'''
Test script for sino module

Latest version can be found at https://github.com/letuananh/chirptext

:copyright: (c) 2012 Le Tuan Anh <tuananh.ke@gmail.com>
:license: MIT, see LICENSE for more details.
'''

import os
import unittest
from chirptext.sino import Radical, KangxiMap

# -------------------------------------------------------------------------------
# Configuration
# -------------------------------------------------------------------------------

TEST_DIR = os.path.dirname(os.path.realpath(__file__))
TEST_DATA = os.path.join(TEST_DIR, 'data')


# -------------------------------------------------------------------------------
# Data Structures
# -------------------------------------------------------------------------------

class TestSino(unittest.TestCase):

    def test_sino_model(self):
        km = KangxiMap()
        self.assertEqual(len(km), 0)

    def test_rad_string(self):
        rads = Radical.kangxi()
        黃 = rads['黃']
        self.assertEqual(str(黃), '黃-yellow[sc:12]')
        self.assertEqual(repr(黃), '黃-yellow[sc:12]')
        
    def test_sino(self):
        rads = Radical.kangxi()
        rads2 = Radical.kangxi()
        self.assertIs(rads, rads2)
        kiiro = rads[200]
        self.assertEqual(len(rads), 214)
        self.assertIn('黃', rads)
        黃 = rads['黃']
        黃_expected_json = {'idseq': 201, 'radical': '黃', 'variants': '', 'strokes': 12, 'meaning': 'yellow', 'pinyin': 'huáng', 'hanviet': 'hoàng', 'hiragana': 'きいろ ', 'romaji': ' kiiro', 'hangeul': '누를황 ', 'romaja': ' nureulhwang', 'frequency': 42, 'simplified': '', 'examples': '黈 䵍 黌'}
        黃_actual_json = 黃.to_json()
        self.assertEqual(黃_expected_json, 黃_actual_json)
        self.assertEqual(rads['黃'], kiiro)
        self.assertEqual(rads['201'], kiiro)
        self.assertEqual(rads.all, ['一', '丨', '丶', '丿', '乙', '亅', '二', '亠', '人', '儿', '入', '八', '冂', '冖', '冫', '几', '凵', '刀', '力', '勹', '匕', '匚', '匸', '十', '卜', '卩', '厂', '厶', '又', '口', '囗', '土', '士', '夂', '夊', '夕', '大', '女', '子', '宀', '寸', '小', '尢', '尸', '屮', '山', '巛', '工', '己', '巾', '干', '幺', '广', '廴', '廾', '弋', '弓', '彐', '彡', '彳', '心', '戈', '戶', '手', '支', '攴', '文', '斗', '斤', '方', '无', '日', '曰', '月', '木', '欠', '止', '歹', '殳', '毋', '比', '毛', '氏', '气', '水', '火', '爪', '父', '爻', '爿', '片', '牙', '牛', '犬', '玄', '玉', '瓜', '瓦', '甘', '生', '用', '田', '疋', '疒', '癶', '白', '皮', '皿', '目', '矛', '矢', '石', '示', '禸', '禾', '穴', '立', '竹', '米', '糸', '缶', '网', '羊', '羽', '老', '而', '耒', '耳', '聿', '肉', '臣', '自', '至', '臼', '舌', '舛', '舟', '艮', '色', '艸', '虍', '虫', '血', '行', '衣', '襾', '見', '角', '言', '谷', '豆', '豕', '豸', '貝', '赤', '走', '足', '身', '車', '辛', '辰', '辵', '邑', '酉', '釆', '里', '金', '長', '門', '阜', '隶', '隹', '雨', '青', '非', '面', '革', '韋', '韭', '音', '頁', '風', '飛', '食', '首', '香', '馬', '骨', '高', '髟', '鬥', '鬯', '鬲', '鬼', '魚', '鳥', '鹵', '鹿', '麥', '麻', '黃', '黍', '黑', '黹', '黽', '鼎', '鼓', '鼠', '鼻', '齊', '齒', '龍', '龜', '龠'])
        self.assertEqual(rads.strokes, {1: ['一', '丨', '丶', '丿', '乙', '亅'], 2: ['二', '亠', '人', '儿', '入', '八', '冂', '冖', '冫', '几', '凵', '刀', '力', '勹', '匕', '匚', '匸', '十', '卜', '卩', '厂', '厶', '又'], 3: ['口', '囗', '土', '士', '夂', '夊', '夕', '大', '女', '子', '宀', '寸', '小', '尢', '尸', '屮', '山', '巛', '工', '己', '巾', '干', '幺', '广', '廴', '廾', '弋', '弓', '彐', '彡', '彳'], 4: ['心', '戈', '戶', '手', '支', '攴', '文', '斗', '斤', '方', '无', '日', '曰', '月', '木', '欠', '止', '歹', '殳', '毋', '比', '毛', '氏', '气', '水', '火', '爪', '父', '爻', '爿', '片', '牙', '牛', '犬'], 5: ['玄', '玉', '瓜', '瓦', '甘', '生', '用', '田', '疋', '疒', '癶', '白', '皮', '皿', '目', '矛', '矢', '石', '示', '禸', '禾', '穴', '立'], 6: ['竹', '米', '糸', '缶', '网', '羊', '羽', '老', '而', '耒', '耳', '聿', '肉', '臣', '自', '至', '臼', '舌', '舛', '舟', '艮', '色', '艸', '虍', '虫', '血', '行', '衣', '襾'], 7: ['見', '角', '言', '谷', '豆', '豕', '豸', '貝', '赤', '走', '足', '身', '車', '辛', '辰', '辵', '邑', '酉', '釆', '里'], 8: ['金', '長', '門', '阜', '隶', '隹', '雨', '青', '非'], 9: ['面', '革', '韋', '韭', '音', '頁', '風', '飛', '食', '首', '香'], 10: ['馬', '骨', '高', '髟', '鬥', '鬯', '鬲', '鬼'], 11: ['魚', '鳥', '鹵', '鹿', '麥', '麻'], 12: ['黃', '黍', '黑', '黹'], 13: ['黽', '鼎', '鼓', '鼠'], 14: ['鼻', '齊'], 15: ['齒'], 16: ['龍', '龜'], 17: ['龠']})


# -------------------------------------------------------------------------------
# Main
# -------------------------------------------------------------------------------

if __name__ == "__main__":
    unittest.main()
