#!/usr/bin/env python
from __future__ import annotations

import functools
import os
import shlex
import subprocess  # nosec:B404
import sys
from collections.abc import AsyncGenerator, Callable, Generator
from contextlib import AbstractAsyncContextManager, asynccontextmanager, contextmanager
from datetime import datetime
from pathlib import Path
from typing import TYPE_CHECKING, Annotated, Any

import anyio
import typer
from rich.progress import Progress, SpinnerColumn, TaskID

from .client import CdnHostBuilder, HttpSniff

if TYPE_CHECKING:
    if sys.version_info >= (3, 11):
        from typing import Self
    else:
        from typing_extensions import Self


app = typer.Typer()


def load_bool(name: str) -> bool:
    v = os.getenv(name)
    if not v:
        return False
    return v.lower() in ("1", "true", "t", "y", "yes", "on")


def run_shell(cmd: str) -> None:
    """Run cmd by subprocess

    Example::
        run_shell('PYTHONPATH=. python main.py')

    Will be convert to:
        subprocess.run(['python', 'main.py'], env={**os.environ, 'PYTHONPATH': '.'})
    """
    typer.echo(f"--> {cmd}")
    command = shlex.split(cmd)
    cmd_env = None
    index = 0
    for i, c in enumerate(command):
        if "=" not in c:
            index = i
            break
        name, value = c.split("=")
        if cmd_env is None:
            cmd_env = {**os.environ, name: value}
        else:
            cmd_env[name] = value
    if cmd_env is not None:
        command = command[index:]
    subprocess.run(command, env=cmd_env)  # nosec:B603


TEMPLATE = """
#!/usr/bin/env python
'''This file is auto generated by fastapi_cdn_host.
Feel free to change or remove it.
'''
import subprocess
import sys

import fastapi_cdn_host

from {} import app

fastapi_cdn_host.patch_docs(app)

def _runserver() -> int:
    r = subprocess.run(["fastapi", "dev", __file__, *sys.argv[2:]])
    return int(bool(r.returncode))

if __name__ == "__main__":
    sys.exit(_runserver())
"""


def write_app(dest: Path, from_path: str | Path) -> None:
    module = Path(from_path).stem
    size = dest.write_text(TEMPLATE.format(module).strip())
    typer.echo(f"Create {dest} with {size=}")


@contextmanager
def patch_app(path: str | Path, remove: bool = True) -> Generator[Path, None, None]:
    ident = f"{datetime.now():%Y%m%d%H%M%S}"
    app_file = Path(f"app_{ident}.py")
    write_app(app_file, path)
    try:
        yield app_file
    finally:
        if remove:
            app_file.unlink()
            typer.echo(f"Auto remove temp file: {app_file}")


class PercentBar(AbstractAsyncContextManager):
    default_threshold: Annotated[int, "0-100"] = 80
    total_seconds: Annotated[int, "Cancel task if reach this value"] = 5

    def __init__(
        self, msg: str, seconds: int | None = None, color: str = "", **kw: Any
    ) -> None:
        self.seconds = seconds or self.total_seconds
        self.prompt = self.build_prompt(msg, color)
        self.progress = Progress(**kw)

    @staticmethod
    def build_prompt(msg: str, color: str | None, suffix: str = ":") -> str:
        prompt = f"{msg}{suffix}"
        if color:
            prompt = f"[{color}]" + prompt
        return prompt

    async def play(
        self, progress: Progress, task: TaskID, threshold: int | None = None
    ) -> None:
        expected = 1 / 2
        progress_forward = functools.partial(progress.advance, task)

        # Play quickly: run 80% progress in 1/2 total seconds
        threshold = threshold or self.default_threshold
        cost = self.seconds * expected
        await self.percent_play(cost, threshold, progress_forward)

        # Play slow
        slow = 100 - threshold
        cost = self.seconds - cost
        await self.percent_play(cost, slow, progress_forward)

    @staticmethod
    async def percent_play(cost: float, percent: int, forward: Callable) -> None:
        delay = cost / percent
        for _ in range(percent):
            await anyio.sleep(delay)
            forward()

    async def __aenter__(self) -> Self:
        self.progress.start()
        self.progress_task = self.progress.add_task(self.prompt)
        self.task_group = await anyio.create_task_group().__aenter__()
        self.task_group.start_soon(self.play, self.progress, self.progress_task)
        return self

    async def __aexit__(self, *args, **kw) -> None:
        self.task_group.cancel_scope.cancel()
        self.progress.update(self.progress_task, completed=100)
        await self.task_group.__aexit__(*args, **kw)
        self.progress.__exit__(*args, **kw)


@asynccontextmanager
async def percentbar(msg: str, **kwargs: Any) -> AsyncGenerator[None, None]:
    """Progressbar with custom font color

    :param msg: prompt message.
    """
    async with PercentBar(msg, **kwargs):
        yield


class SpinnerProgress(Progress):
    def __init__(self, msg: str, color: str | None = None, **kwargs: Any) -> None:
        self.prompt = PercentBar.build_prompt(msg, color, "...")
        kwargs.setdefault("transient", True)
        super().__init__(SpinnerColumn(), *Progress.get_default_columns(), **kwargs)

    def start(self) -> None:
        super().start()
        self.add_task(self.prompt, total=None)


@contextmanager
def spinnerbar(
    msg: str, color: str | None = None, **kwargs: Any
) -> Generator[None, None, None]:
    with SpinnerProgress(msg, color, **kwargs):
        yield


async def download_offline_assets(dirname: str | Path, timeout: float = 30) -> None:
    cwd = await anyio.Path.cwd()
    static_root = (
        cwd / dirname if isinstance(dirname, str) else anyio.Path(dirname.resolve())
    )
    if not await static_root.exists():
        await static_root.mkdir(parents=True)
        typer.echo(f"Directory {static_root} created.")
    else:
        async for p in static_root.glob("swagger-ui*.js"):
            relative_path = p.relative_to(cwd)
            typer.echo(f"{relative_path} already exists. abort!")
            return
    async with percentbar("Comparing cdn hosts response speed"):
        urls = await CdnHostBuilder.sniff_the_fastest()
    typer.echo(f"Result: {urls}")
    with spinnerbar("Fetching files from cdn", color="yellow"):
        url_list = [urls.js, urls.css, urls.redoc]
        contents = await HttpSniff.bulk_fetch(
            url_list, get_content=True, total_seconds=timeout
        )
        for url, content in zip(url_list, contents):
            if not content:
                red_head = typer.style("ERROR:", fg=typer.colors.RED)
                typer.echo(red_head + f" Failed to fetch content from {url}")
            else:
                path = static_root / Path(url).name
                size = await path.write_bytes(content)
                typer.echo(f"Write to {path} with {size=}")
    typer.secho("Done.", fg=typer.colors.GREEN)


def handle_cache() -> None:
    exists, cache_path = CdnHostBuilder.get_cache_file()
    if not exists:
        typer.echo("Cache not create yet.")
        return
    if "--remove" in sys.argv:
        cache_path.unlink()
        typer.secho(
            f"Success to remove cache file({cache_path})", fg=typer.colors.GREEN
        )
    else:
        typer.echo(f"Content of cache file({cache_path}):\n{cache_path.read_text()}")
        if "--reload" in sys.argv:
            cache_path.unlink()
            with spinnerbar("Refreshing cache", color="yellow"):
                CdnHostBuilder(cache=True).run()
            typer.echo(f"Cache file updated:\n{cache_path.read_text()}")


@app.command()
def dev(
    path: Annotated[
        Path,
        typer.Argument(
            help=(
                "A path to a Python file or package directory"
                " (with [blue]__init__.py[/blue] file)"
                " containing a [bold]FastAPI[/bold] app."
                " If not provided, a default set of paths will be tried."
            )
        ),
    ],
    port: Annotated[
        int,
        typer.Option(
            help=(
                "The port to serve on."
                " You would normally have a termination proxy on top (another program)"
                " handling HTTPS on port [blue]443[/blue] and HTTP on port [blue]80[/blue],"
                " transferring the communication to your app."
            )
        ),
    ] = 0,
    remove: Annotated[
        bool,
        typer.Option(
            help="Whether remove the temp app_<time>.py file after server stopped."
        ),
    ] = True,
    prod: Annotated[
        bool,
        typer.Option(help="Whether enable production mode."),
    ] = False,
    reload: Annotated[
        bool,
        typer.Option(help="Enable auto-reload of the server when (code) files change."),
    ] = True,
) -> None:
    if str(path) == "offline":
        anyio.run(download_offline_assets, "static")
        return
    elif str(path) == "cache":
        handle_cache()
        return
    with patch_app(path, remove) as file:
        runserver(file, prod, reload, port)


def runserver(file: Path, prod: bool, reload: bool, port: int) -> None:
    if load_bool("FASTCDN_UVICORN"):
        module = file.stem
        if file.parent != Path() and file.parent.resolve() != Path.cwd():
            os.chdir(file.parent)
        cmd = f"PYTHONPATH=. uvicorn {module}:app"
        if reload and not load_bool("FASTCDN_NORELOAD"):
            cmd += " --reload"
    else:
        mode = "run" if prod else "dev"
        cmd = f"PYTHONPATH=. fastapi {mode} {file}"
        if (not reload and not prod) or load_bool("FASTCDN_NORELOAD"):
            cmd += " --no-reload"
    if port:
        cmd += f" --{port=}"
    run_shell(cmd)


def main() -> None:
    app()


if __name__ == "__main__":
    main()
