"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DnsValidatedCertificate = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
/**
 * A certificate managed by AWS Certificate Manager.  Will be automatically
 * validated using DNS validation against the specified Route 53 hosted zone.
 *
 * @resource AWS::CertificateManager::Certificate
 * @experimental
 */
class DnsValidatedCertificate extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.domainName = props.domainName;
        this.normalizedZoneName = props.hostedZone.zoneName;
        // Remove trailing `.` from zone name
        if (this.normalizedZoneName.endsWith('.')) {
            this.normalizedZoneName = this.normalizedZoneName.substring(0, this.normalizedZoneName.length - 1);
        }
        // Remove any `/hostedzone/` prefix from the Hosted Zone ID
        this.hostedZoneId = props.hostedZone.hostedZoneId.replace(/^\/hostedzone\//, '');
        const requestorFunction = new lambda.Function(this, 'CertificateRequestorFunction', {
            code: lambda.Code.fromAsset(path.resolve(__dirname, '..', 'lambda-packages', 'dns_validated_certificate_handler', 'lib')),
            handler: 'index.certificateRequestHandler',
            runtime: lambda.Runtime.NODEJS_10_X,
            timeout: cdk.Duration.minutes(15),
            role: props.customResourceRole,
        });
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['acm:RequestCertificate', 'acm:DescribeCertificate', 'acm:DeleteCertificate'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:GetChange'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:changeResourceRecordSets'],
            resources: [`arn:${cdk.Stack.of(requestorFunction).partition}:route53:::hostedzone/${this.hostedZoneId}`],
        }));
        const certificate = new cdk.CustomResource(this, 'CertificateRequestorResource', {
            serviceToken: requestorFunction.functionArn,
            properties: {
                DomainName: props.domainName,
                SubjectAlternativeNames: cdk.Lazy.listValue({ produce: () => props.subjectAlternativeNames }, { omitEmpty: true }),
                HostedZoneId: this.hostedZoneId,
                Region: props.region,
                Route53Endpoint: props.route53Endpoint,
            },
        });
        this.certificateArn = certificate.getAtt('Arn').toString();
    }
    validate() {
        const errors = [];
        // Ensure the zone name is a parent zone of the certificate domain name
        if (!cdk.Token.isUnresolved(this.normalizedZoneName) &&
            this.domainName !== this.normalizedZoneName &&
            !this.domainName.endsWith('.' + this.normalizedZoneName)) {
            errors.push(`DNS zone ${this.normalizedZoneName} is not authoritative for certificate domain name ${this.domainName}`);
        }
        return errors;
    }
}
exports.DnsValidatedCertificate = DnsValidatedCertificate;
//# sourceMappingURL=data:application/json;base64,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