"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const core_1 = require("@aws-cdk/core");
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * How to validate a certificate
 */
class CertificateValidation {
    /** @param props Certification validation properties */
    constructor(props) {
        var _a;
        this.props = props;
        this.method = (_a = props.method) !== null && _a !== void 0 ? _a : ValidationMethod.EMAIL;
    }
    /**
     * Validate the certifcate with DNS
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created
     */
    static fromDns(hostedZone) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * Validate the certifcate with automatically created DNS records in multiple
     * Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created
     * for the domains in the certificate
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * Validate the certifcate with Email
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *  admin@domain.com
     *  administrator@domain.com
     *  hostmaster@domain.com
     *  postmaster@domain.com
     *  webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
}
exports.CertificateValidation = CertificateValidation;
/**
 * A certificate managed by AWS Certificate Manager
 */
class Certificate extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
        });
        this.certificateArn = cert.ref;
    }
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    var _a, _b, _c;
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of getUniqueDnsDomainNames(domainNames)) {
                const hostedZone = (_b = (_a = validation.props.hostedZones) === null || _a === void 0 ? void 0 : _a[domainName]) !== null && _b !== void 0 ? _b : validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = (_c = validation.props.validationDomains) === null || _c === void 0 ? void 0 : _c[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain !== null && validationDomain !== void 0 ? validationDomain : util_1.apexDomain(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
/**
 * Removes wildcard domains (*.example.com) where the base domain (example.com) is present.
 * This is because the DNS validation treats them as the same thing, and the automated CloudFormation
 * DNS validation errors out with the duplicate records.
 */
function getUniqueDnsDomainNames(domainNames) {
    return domainNames.filter(domain => {
        return core_1.Token.isUnresolved(domain) || !domain.startsWith('*.') || !domainNames.includes(domain.replace('*.', ''));
    });
}
//# sourceMappingURL=data:application/json;base64,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