# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['xarray_dataclasses']

package_data = \
{'': ['*']}

install_requires = \
['morecopy>=0.2,<0.3',
 'numpy>=1.19,<2.0',
 'typing-extensions>=3.10,<4.0',
 'xarray>=0.15,<1.0']

setup_kwargs = {
    'name': 'xarray-dataclasses',
    'version': '0.6.1',
    'description': 'xarray extension for typed DataArray and Dataset creation',
    'long_description': '# xarray-dataclasses\n\n[![PyPI](https://img.shields.io/pypi/v/xarray-dataclasses.svg?label=PyPI&style=flat-square)](https://pypi.org/project/xarray-dataclasses/)\n[![Python](https://img.shields.io/pypi/pyversions/xarray-dataclasses.svg?label=Python&color=yellow&style=flat-square)](https://pypi.org/project/xarray-dataclasses/)\n[![Test](https://img.shields.io/github/workflow/status/astropenguin/xarray-dataclasses/Test?logo=github&label=Test&style=flat-square)](https://github.com/astropenguin/xarray-dataclasses/actions)\n[![License](https://img.shields.io/badge/license-MIT-blue.svg?label=License&style=flat-square)](LICENSE)\n[![DOI](https://img.shields.io/badge/DOI-10.5281/zenodo.4624819-blue?style=flat-square)](https://doi.org/10.5281/zenodo.4624819)\n\nxarray extension for typed DataArray and Dataset creation\n\n\n## Overview\n\nxarray-dataclasses is a Python package that makes it easy to create typed DataArray and Dataset objects of [xarray] using [the Python\'s dataclass].\n\n```python\nfrom dataclasses import dataclass\nfrom typing import Literal\nfrom xarray_dataclasses import AsDataArray, Coord, Data\n\n\n@dataclass\nclass Image(AsDataArray):\n    """Specifications of images."""\n\n    data: Data[tuple[Literal["x"], Literal["y"]], float]\n    x: Coord[Literal["x"], int] = 0\n    y: Coord[Literal["y"], int] = 0\n\n\n# create an image as DataArray\nimage = Image.new([[0, 1], [2, 3]], x=[0, 1], y=[0, 1])\n\n# create an image filled with ones\nones = Image.ones((2, 2), x=[0, 1], y=[0, 1])\n```\n\n### Features\n\n- DataArray and Dataset objects with fixed dimensions, data type, and coordinates can easily be created.\n- NumPy-like special functions such as ``ones()`` are provided as class methods.\n- Compatible with [the Python\'s dataclass].\n- Compatible with static type check by [Pyright].\n\n### Installation\n\n```shell\n$ pip install xarray-dataclasses\n```\n\n\n## Background\n\n[xarray] is useful for handling labeled multi-dimensional data, but it is a bit troublesome to create DataArray and Dataset objects with fixed dimensions, data type, or coordinates (typed DataArray and typed Dataset).\nFor example, let us think about the following specifications of images as DataArray.\n\n- Dimensions of data must be `("x", "y")`.\n- Data type of data must be `float`.\n- Data type of dimensions must be `int`.\n- Default value of dimensions must be `0`.\n\nThen a function to create a typed DataArray object is something like this.\n\n```python\nimport numpy as np\nimport xarray as xr\n\n\ndef create_image(data, x=0, y=0):\n    """Specifications of images."""\n    data = np.array(data)\n\n    if x == 0:\n        x = np.full(data.shape[0], x)\n    else:\n        x = np.array(x)\n\n    if y == 0:\n        y = np.full(data.shape[1], y)\n    else:\n        y = np.array(y)\n\n    return xr.DataArray(\n        data=data.astype(float),\n        dims=("x", "y"),\n        coords={\n            "x": ("x", x.astype(int)),\n            "y": ("y", y.astype(int)),\n        },\n    )\n\n\nimage = create_image([[0, 1], [2, 3]])\n```\n\nThe issues are\n\n- It is not easy to figure out the specifications from the code.\n- It is not easy to reuse the code, for example, to add new coordinates.\n\n[xarray-dataclasses](#xarray-dataclasses) resolves them by defining the specifications as a dataclass.\n\n```python\nfrom dataclasses import dataclass\nfrom xarray_dataclasses import AsDataArray, Coord, Data\n\n\n@dataclass\nclass Image(AsDataArray):\n    """Specifications of 2D images."""\n\n    data: Data[tuple[Literal["x"], Literal["y"]], float]\n    x: Coord[Literal["x"], int] = 0\n    y: Coord[Literal["y"], int] = 0\n\n\nimage = Image.new([[0, 1], [2, 3]])\n```\n\nNow the specifications become much easier to read.\n\n- The type hints have complete information for DataArray creation.\n- The default values are given as class variables.\n- The mix-in class `AsDataArray` provides class methods such as `new()`.\n- The extension of the specifications is easy by class inheritance.\n\n## Basic usage\n\nxarray-dataclasses uses [the Python\'s dataclass].\nPlease learn how to use it before proceeding.\nData (or data variables), coordinates, attributes, and a name of a DataArray or a Dataset object are defined as dataclass fields with the following type hints.\nNote that the following imports are supposed in the examples below.\n\n```python\nfrom dataclasses import dataclass\nfrom typing import Literal\nfrom xarray_dataclasses import AsDataArray, AsDataset\nfrom xarray_dataclasses import Attr, Coord, Data, Name\n```\n\n### Data field\n\nThe data field is a field whose value will become the data of a DataArray object or a data variable of a Dataset object.\nThe type hint `Data[TDims, TDtype]` fixes the dimensions and the data type of the object.\nHere are some examples of how to specify them.\n\nType hint | Inferred dimensions\n--- | ---\n`Data[Literal[()], ...]` | `()`\n`Data[Literal["x"], ...]` | `("x",)`\n`Data[tuple[Literal["x"], Literal["y"]], ...]` | `("x", "y")`\n\nType hint | Inferred data type\n--- | ---\n`Data[..., Any]` | `None`\n`Data[..., None]` | `None`\n`Data[..., float]` | `numpy.dtype("float64")`\n`Data[..., numpy.float128]` | `numpy.dtype("float128")`\n| `Data[..., Literal["datetime64[ns]"]]` | `numpy.dtype("<M8[ns]")`\n\n### Coordinate field\n\nThe coordinate field is a field whose value will become a coordinate of a DataArray or a Dataset object.\nThe type hint `Coord[TDims, TDtype]` fixes the dimensions and the data type of the object.\n\n### Attribute field\n\nThe attribute field is a field whose value will become an attribute of a DataArray or a Dataset object.\nThe type hint `Attr[T]` specifies the type of the value, which is used only for static type check.\n\n### Name field\n\nThe name field is a field whose value will become the name of a DataArray object.\nThe type hint `Name[T]` specifies the type of the value, which is used only for static type check.\n\n### DataArray class\n\nThe DataArray class is a dataclass that defines typed DataArray specifications.\nExactly one data field is allowed in a DataArray class.\nThe second and subsequent data fields are just ignored in DataArray creation.\n\n```python\n@dataclass\nclass Image(AsDataArray):\n    """Specifications of images."""\n\n    data: Data[tuple[Literal["x"], Literal["y"]], float]\n    x: Coord[Literal["x"], int] = 0\n    y: Coord[Literal["y"], int] = 0\n    units: Attr[str] = "cd / m^2"\n    name: Name[str] = "luminance"\n```\n\nA DataArray object is created by the shorthand method `new()`.\n\n```python\nImage.new([[0, 1], [2, 3]], x=[0, 1], y=[0, 1])\n\n<xarray.DataArray "luminance" (x: 2, y: 2)>\narray([[0., 1.],\n       [2., 3.]])\nCoordinates:\n  * x        (x) int64 0 1\n  * y        (y) int64 0 1\nAttributes:\n    units:    cd / m^2\n```\n\nNumPy-like `empty()`, `zeros()`, `ones()`, `full()` methods are available.\n\n```python\nImage.ones((3, 3))\n\n<xarray.DataArray "luminance" (x: 3, y: 3)>\narray([[1., 1., 1.],\n       [1., 1., 1.],\n       [1., 1., 1.]])\nCoordinates:\n  * x        (x) int64 0 0 0\n  * y        (y) int64 0 0 0\nAttributes:\n    units:    cd / m^2\n```\n\n### Dataset class\n\nThe Dataset class is a dataclass that defines typed Dataset specifications.\nMultiple data fields are allowed to define the data variables of the object.\n\n```python\n@dataclass\nclass ColorImage(AsDataset):\n    """Specifications of color images."""\n\n    red: Data[tuple[Literal["x"], Literal["y"]], float]\n    green: Data[tuple[Literal["x"], Literal["y"]], float]\n    blue: Data[tuple[Literal["x"], Literal["y"]], float]\n    x: Coord[Literal["x"], int] = 0\n    y: Coord[Literal["y"], int] = 0\n    units: Attr[str] = "cd / m^2"\n```\n\nA Dataset object is created by the shorthand method `new()`.\n\n```python\nColorImage.new(\n    [[0, 0], [0, 0]],  # red\n    [[1, 1], [1, 1]],  # green\n    [[2, 2], [2, 2]],  # blue\n)\n\n<xarray.Dataset>\nDimensions:  (x: 2, y: 2)\nCoordinates:\n  * x        (x) int64 0 0\n  * y        (y) int64 0 0\nData variables:\n    red      (x, y) float64 0.0 0.0 0.0 0.0\n    green    (x, y) float64 1.0 1.0 1.0 1.0\n    blue     (x, y) float64 2.0 2.0 2.0 2.0\nAttributes:\n    units:    cd / m^2\n```\n\n## Advanced usage\n\n### Coordof and Dataof type hints\n\n[xarray-dataclasses] provides advanced type hints, `Coordof[T]` and `Dataof[T]`.\nUnlike `Data` and `Coord`, they specify a dataclass that defines a DataArray class.\nThis is useful, for example, when users want to add metadata to dimensions for [plotting].\n\n```python\nfrom xarray_dataclasses import Coordof\n\n\n@dataclass\nclass XAxis:\n    data: Data[Literal["x"], int]\n    long_name: Attr[str] = "x axis"\n    units: Attr[str] = "pixel"\n\n\n@dataclass\nclass YAxis:\n    data: Data[Literal["y"], int]\n    long_name: Attr[str] = "y axis"\n    units: Attr[str] = "pixel"\n\n\n@dataclass\nclass Image(AsDataArray):\n    """Specifications of images."""\n\n    data: Data[tuple[Literal["x"], Literal["y"]], float]\n    x: Coordof[XAxis] = 0\n    y: Coordof[YAxis] = 0\n```\n\n### Custom DataArray and Dataset factories\n\nFor customization, users can use a function or a class to create an initial DataArray or Dataset object by specifying a special class attribute, `__dataarray_factory__` or `__dataset_factory__`, respectively.\n\n```python\nimport xarray as xr\n\n\nclass Custom(xr.DataArray):\n    """Custom DataArray."""\n\n    __slots__ = ()\n\n    def custom_method(self) -> None:\n        print("Custom method!")\n\n\n@dataclass\nclass Image(AsDataArray):\n    """Specifications of images."""\n\n    data: Data[tuple[Literal["x"], Literal["y"]], float]\n    x: Coord[Literal["x"], int] = 0\n    y: Coord[Literal["y"], int] = 0\n    __dataarray_factory__ = Custom\n\n\nimage = Image.ones([3, 3])\nisinstance(image, Custom) # True\nimage.custom_method() # Custom method!\n```\n\n### DataArray and Dataset creation without shorthands\n\n[xarray-dataclasses] provides functions, `asdataarray` and `asdataset`.\nThis is useful, for example, users do not want to inherit the mix-in class (`AsDataArray` or `AsDataset`) in a DataArray or Dataset dataclass.\n\n```python\nfrom xarray_dataclasses import asdataarray\n\n\n@dataclass\nclass Image:\n    """Specifications of images."""\n\n    data: Data[tuple[Literal["x"], Literal["y"]], float]\n    x: Coord[Literal["x"], int] = 0\n    y: Coord[Literal["y"], int] = 0\n\n\nimage = asdataarray(Image([[0, 1], [2, 3]], x=[0, 1], y=[0, 1]))\n```\n\n\n<!-- References -->\n[Pyright]: https://github.com/microsoft/pyright\n[the Python\'s dataclass]: https://docs.python.org/3/library/dataclasses.html\n[xarray]: https://xarray.pydata.org/en/stable/index.html\n[plotting]: https://xarray.pydata.org/en/stable/user-guide/plotting.html#simple-example\n',
    'author': 'Akio Taniguchi',
    'author_email': 'taniguchi@a.phys.nagoya-u.ac.jp',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/astropenguin/xarray-dataclasses/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<3.10',
}


setup(**setup_kwargs)
