from abc import abstractmethod

from secml.array import CArray
from secml.ml.classifiers import CClassifier
from secml_malware.models import End2EndModel, CClassifierEnd2EndMalware, CClassifierEmber


class CWrapperPhi:
	"""
	Abstract class that encapsulates a model for being used in a black-box way.
	"""

	def __init__(self, model: CClassifier):
		"""
		Creates the wrapper.

		Parameters
		----------
		model : CClassifier
		The model to wrap
		"""
		self.classifier = model

	@abstractmethod
	def extract_features(self, x: CArray):
		"""
		It maps the input sample inside the feature space of the wrapped model.

		Parameters
		----------
		x : CArray
			The sample in the input space.
		Returns
		-------
		CArray
			The feature space representation of the input sample.
		"""
		raise NotImplementedError("This method is abstract, you should implement it somewhere else!")

	def predict(self, x: CArray, return_decision_function: bool = True):
		"""
		Returns the prediction of the sample (in input space).

		Parameters
		----------
		x : CArray
			The input sample in input space.
		return_decision_function : bool, default True
			If True, it also returns the decision function value, rather than only the label.
			Default is True.
		Returns
		-------
		CArray, (CArray)
			Returns the label of the sample.
			If return_decision_function is True, it also returns the output of the decision function.
		"""
		padding_position = x.find(x == 256)
		if padding_position:
			x = x[0, :padding_position[0]]
		feature_vector = self.extract_features(x)
		return self.classifier.predict(feature_vector, return_decision_function=return_decision_function)


class CEmberWrapperPhi(CWrapperPhi):
	"""
	Class that wraps a GBDT classifier with EMBER feature set.
	"""

	def __init__(self, model: CClassifierEmber):
		"""
		Creates the wrapper of a CClassifierEmber.

		Parameters
		----------
		model : CClassifierEmber
		The GBDT model to wrap
		"""
		if not isinstance(model, CClassifierEmber):
			raise ValueError(f"Input model is {type(model)} and not CClassifierEmber")
		super().__init__(model)

	def extract_features(self, x):
		"""
		It extracts the EMBER hand-crafted features

		Parameters
		----------
		x : CArray
			The sample in the input space.
		Returns
		-------
		CArray
			The feature space representation of the input sample.
		"""
		padding_positions = x.find(x == 256)
		clf : CClassifierEmber = self.classifier
		if padding_positions:
			return clf.extract_features(x[0, :padding_positions[0]])
		else:
			return clf.extract_features(x)


class CEnd2EndWrapperPhi(CWrapperPhi):
	"""
	Class that wraps an end-to-end model
	"""

	def __init__(self, model: CClassifierEnd2EndMalware):
		"""
		Creates the wrapper of a CClassifierEnd2EndMalware.

		Parameters
		----------
		model : CClassifierEnd2EndMalware
		The end to end model to wrap
		"""
		if not isinstance(model, CClassifierEnd2EndMalware):
			raise ValueError(f"Input model is {type(model)} and not CClassifierEnd2EndMalware")
		super().__init__(model)

	def extract_features(self, x):
		"""
		Crops and pads the input sample for being passed to the network.

		Parameters
		----------
		x : CArray
			The sample in the input space.
		Returns
		-------
		CArray
			The feature space representation of the input sample.
		"""
		clf: CClassifierEnd2EndMalware = self.classifier
		padded_x = CArray.zeros((1, clf.get_input_max_length())) + clf.get_embedding_value()
		length = min(x.shape[-1], clf.get_input_max_length())
		padded_x[0, :length] = x[0, :length] + clf.get_is_shifting_values()
		return padded_x
