"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ifw = require(".");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_INTEG_REGION || process.env.CDK_DEFAULT_REGION || 'us-east-1',
            account: process.env.CDK_INTEG_ACCOUNT || process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'integ-stack', { env });
        const database = new aws_cdk_lib_1.aws_timestream.CfnDatabase(stack, 'Database', {
            databaseName: 'FleetWise',
        });
        const table = new aws_cdk_lib_1.aws_timestream.CfnTable(stack, 'Table', {
            databaseName: 'FleetWise',
            tableName: 'FleetWise',
        });
        table.node.addDependency(database);
        const role = new aws_cdk_lib_1.aws_iam.Role(stack, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('iotfleetwise.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AdministratorAccess'),
            ],
        });
        const signalCatalog = new ifw.SignalCatalog(stack, 'SignalCatalog', {
            database,
            table,
            role,
            description: 'my signal catalog',
            nodes: [
                new ifw.SignalCatalogBranch('Vehicle', 'Vehicle'),
                new ifw.SignalCatalogSensor('EngineTorque', 'Vehicle.EngineTorque', 'DOUBLE'),
            ],
        });
        const model_a = new ifw.VehicleModel(stack, 'ModelA', {
            signalCatalog,
            name: 'modelA',
            description: 'Model A vehicle',
            networkInterfaces: [new ifw.CanVehicleInterface('1', 'vcan0')],
            signals: [
                new ifw.CanVehicleSignal('EngineTorque', 'Vehicle.EngineTorque', '1', 401, // messageId
                1.0, // factor
                true, // isBigEndian
                false, // isSigned
                8, // length
                0.0, // offset
                9),
            ],
        });
        const vin100 = new ifw.Vehicle(stack, 'vin100', {
            vehicleId: 'vin100',
            vehicleModel: model_a,
            createIotThing: true,
        });
        const vpc = aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(stack, 'VPC', { isDefault: true });
        const securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(stack, 'SecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        securityGroup.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.anyIpv4(), aws_cdk_lib_1.aws_ec2.Port.tcp(22), 'SSH access');
        securityGroup.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.anyIpv4(), aws_cdk_lib_1.aws_ec2.Port.allIcmp(), 'ping');
        // EC2 role
        const ec2_role = new aws_cdk_lib_1.aws_iam.Role(stack, 'ec2Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AdministratorAccess'),
            ],
        });
        // Ubuntu 18.04 for Arm64
        const machineImage = aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter('/aws/service/canonical/ubuntu/server/18.04/stable/current/arm64/hvm/ebs-gp2/ami-id', { os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX });
        // Create the Vehicle simulator
        const keyName = stack.node.tryGetContext('key_name');
        const instance = new aws_cdk_lib_1.aws_ec2.Instance(stack, 'VehicleSim', {
            vpc: vpc,
            instanceType: new aws_cdk_lib_1.aws_ec2.InstanceType('m6g.xlarge'),
            machineImage,
            securityGroup,
            keyName,
            role: ec2_role,
            vpcSubnets: {
                subnetGroupName: 'Public',
            },
            resourceSignalTimeout: cdk.Duration.minutes(30),
        });
        const sourceUrl = 'https://github.com/aws/aws-iot-fleetwise-edge.git';
        const userData = `\
    set -euo pipefail

    # Wait for any existing package install to finish
    i=0
    while true; do
        if sudo fuser /var/{lib/{dpkg,apt/lists},cache/apt/archives}/lock >/dev/null 2>&1; then
            i=0
        else
            i=\`expr $i + 1\`
            if expr $i \\>= 10 > /dev/null; then
                break
            fi
        fi
        sleep 1
    done

    # Upgrade system and reboot if required
    apt update && apt upgrade -y
    if [ -f /var/run/reboot-required ]; then
      # Delete the UserData info file so that we run again after reboot
      rm -f /var/lib/cloud/instances/*/sem/config_scripts_user
      reboot
      exit
    fi

    # Install helper scripts:
    apt update && apt install -y python3-setuptools
    mkdir -p /opt/aws/bin
    wget https://s3.amazonaws.com/cloudformation-examples/aws-cfn-bootstrap-py3-latest.tar.gz
    python3 -m easy_install --script-dir /opt/aws/bin aws-cfn-bootstrap-py3-latest.tar.gz
    rm aws-cfn-bootstrap-py3-latest.tar.gz

    # On error, signal back to cfn:
    error_handler() {
      /opt/aws/bin/cfn-signal --success false --stack ${stack.stackName} --resource ${instance.instance.logicalId} --region ${stack.region}
    }
    trap error_handler ERR

    # Increase pid_max:
    echo 1048576 > /proc/sys/kernel/pid_max
    # Disable syslog:
    systemctl stop syslog.socket rsyslog.service
    # Remove journald rate limiting and set max size:
    printf "RateLimitBurst=0\nSystemMaxUse=1G\n" >> /etc/systemd/journald.conf

    # Install packages
    apt update && apt install -y git ec2-instance-connect htop jq unzip

    # Install AWS CLI:
    curl "https://awscli.amazonaws.com/awscli-exe-linux-aarch64.zip" -o "awscliv2.zip"
    unzip awscliv2.zip
    ./aws/install
    rm awscliv2.zip

    # Download source
    cd /home/ubuntu
    if echo ${sourceUrl} | grep -q 's3://'; then
      sudo -u ubuntu aws s3 cp ${sourceUrl} aws-iot-fleetwise-edge.zip
      sudo -u ubuntu unzip aws-iot-fleetwise-edge.zip -d aws-iot-fleetwise-edge
    else
      sudo -u ubuntu git clone ${sourceUrl} aws-iot-fleetwise-edge
    fi
    cd aws-iot-fleetwise-edge
    
    # Install SocketCAN modules:
    ./tools/install-socketcan.sh --bus-count 1
    
    # Install CAN Simulator
    ./tools/install-cansim.sh --bus-count 1
    
    # Install FWE credentials and config file
    mkdir /etc/aws-iot-fleetwise
    echo -n "${vin100.certificatePem}" > /etc/aws-iot-fleetwise/certificate.pem
    echo -n "${vin100.privateKey}" > /etc/aws-iot-fleetwise/private-key.key
    ./tools/configure-fwe.sh \
      --input-config-file "configuration/static-config.json" \
      --output-config-file "/etc/aws-iot-fleetwise/config-0.json" \
      --vehicle-id vin100 \
      --endpoint-url "${vin100.endpointAddress}" \
      --topic-prefix '$aws/iotfleetwise/' \
      --can-bus0 "vcan0"

    # Install source deps
    ./tools/install-deps-native.sh

    # Build source
    sudo -u ubuntu ./tools/build-fwe-native.sh

    # Install FWE
    ./tools/install-fwe.sh
    
    # Signal init complete:
    /opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${instance.instance.logicalId} --region ${stack.region}`;
        instance.addUserData(userData);
        new cdk.CfnOutput(stack, 'Vehicle ssh command', { value: `ssh -i ${keyName}.pem ubuntu@${instance.instancePublicIp}` });
        new ifw.Campaign(stack, 'Campaign', {
            name: 'FwTimeBasedCampaign',
            target: vin100,
            collectionScheme: new ifw.TimeBasedCollectionScheme(cdk.Duration.seconds(10)),
            signals: [
                new ifw.CampaignSignal('Vehicle.EngineTorque'),
            ],
            autoApprove: true,
        });
        new ifw.Fleet(stack, 'Fleet', {
            fleetId: 'fleet',
            signalCatalog,
            vehicles: [vin100],
        });
        this.stack = [stack];
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuZnVsbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9pbnRlZy5mdWxsLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLG1DQUFtQztBQUNuQyw2Q0FJcUI7QUFDckIseUJBQXlCO0FBR3pCLE1BQWEsWUFBWTtJQUV2QjtRQUNFLE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBRTFCLE1BQU0sR0FBRyxHQUFHO1lBQ1YsTUFBTSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsSUFBSSxXQUFXO1lBQ3JGLE9BQU8sRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLGlCQUFpQixJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMsbUJBQW1CO1NBQzFFLENBQUM7UUFFRixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFFekQsTUFBTSxRQUFRLEdBQUcsSUFBSSw0QkFBRSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQ3JELFlBQVksRUFBRSxXQUFXO1NBQzFCLENBQUMsQ0FBQztRQUVILE1BQU0sS0FBSyxHQUFHLElBQUksNEJBQUUsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtZQUM1QyxZQUFZLEVBQUUsV0FBVztZQUN6QixTQUFTLEVBQUUsV0FBVztTQUN2QixDQUFDLENBQUM7UUFFSCxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUVuQyxNQUFNLElBQUksR0FBRyxJQUFJLHFCQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7WUFDdkMsU0FBUyxFQUFFLElBQUkscUJBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyw0QkFBNEIsQ0FBQztZQUNqRSxlQUFlLEVBQUU7Z0JBQ2YscUJBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMscUJBQXFCLENBQUM7YUFDbEU7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtZQUNsRSxRQUFRO1lBQ1IsS0FBSztZQUNMLElBQUk7WUFDSixXQUFXLEVBQUUsbUJBQW1CO1lBQ2hDLEtBQUssRUFBRTtnQkFDTCxJQUFJLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO2dCQUNqRCxJQUFJLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxjQUFjLEVBQUUsc0JBQXNCLEVBQUUsUUFBUSxDQUFDO2FBQzlFO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDcEQsYUFBYTtZQUNiLElBQUksRUFBRSxRQUFRO1lBQ2QsV0FBVyxFQUFFLGlCQUFpQjtZQUM5QixpQkFBaUIsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLG1CQUFtQixDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztZQUM5RCxPQUFPLEVBQUU7Z0JBQ1AsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxFQUFFLHNCQUFzQixFQUFFLEdBQUcsRUFDbEUsR0FBRyxFQUFFLFlBQVk7Z0JBQ2pCLEdBQUcsRUFBRSxTQUFTO2dCQUNkLElBQUksRUFBRSxjQUFjO2dCQUNwQixLQUFLLEVBQUUsV0FBVztnQkFDbEIsQ0FBQyxFQUFFLFNBQVM7Z0JBQ1osR0FBRyxFQUFFLFNBQVM7Z0JBQ2QsQ0FBQyxDQUFDO2FBQ0w7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM5QyxTQUFTLEVBQUUsUUFBUTtZQUNuQixZQUFZLEVBQUUsT0FBTztZQUNyQixjQUFjLEVBQUUsSUFBSTtTQUNyQixDQUFDLENBQUM7UUFFSCxNQUFNLEdBQUcsR0FBRyxxQkFBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRWxFLE1BQU0sYUFBYSxHQUFHLElBQUkscUJBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtZQUNsRSxHQUFHO1lBQ0gsZ0JBQWdCLEVBQUUsSUFBSTtTQUN2QixDQUFDLENBQUM7UUFDSCxhQUFhLENBQUMsY0FBYyxDQUFDLHFCQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLHFCQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUNqRixhQUFhLENBQUMsY0FBYyxDQUFDLHFCQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLHFCQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRTdFLFdBQVc7UUFDWCxNQUFNLFFBQVEsR0FBRyxJQUFJLHFCQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7WUFDOUMsU0FBUyxFQUFFLElBQUkscUJBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztZQUN4RCxlQUFlLEVBQUU7Z0JBQ2YscUJBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsOEJBQThCLENBQUM7Z0JBQzFFLHFCQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLHFCQUFxQixDQUFDO2FBQ2xFO1NBQ0YsQ0FBQyxDQUFDO1FBRUgseUJBQXlCO1FBQ3pCLE1BQU0sWUFBWSxHQUFHLHFCQUFHLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUNwRCxvRkFBb0YsRUFDcEYsRUFBRSxFQUFFLEVBQUUscUJBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsQ0FDdEMsQ0FBQztRQUVGLCtCQUErQjtRQUMvQixNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNyRCxNQUFNLFFBQVEsR0FBRyxJQUFJLHFCQUFHLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDckQsR0FBRyxFQUFFLEdBQUc7WUFDUixZQUFZLEVBQUUsSUFBSSxxQkFBRyxDQUFDLFlBQVksQ0FBQyxZQUFZLENBQUM7WUFDaEQsWUFBWTtZQUNaLGFBQWE7WUFDYixPQUFPO1lBQ1AsSUFBSSxFQUFFLFFBQVE7WUFDZCxVQUFVLEVBQUU7Z0JBQ1YsZUFBZSxFQUFFLFFBQVE7YUFDMUI7WUFDRCxxQkFBcUIsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7U0FDaEQsQ0FBQyxDQUFDO1FBRUgsTUFBTSxTQUFTLEdBQUcsbURBQW1ELENBQUM7UUFDdEUsTUFBTSxRQUFRLEdBQUc7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O3dEQW1DbUMsS0FBSyxDQUFDLFNBQVMsZUFBZSxRQUFRLENBQUMsUUFBUSxDQUFDLFNBQVMsYUFBYSxLQUFLLENBQUMsTUFBTTs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztjQXNCNUgsU0FBUztpQ0FDVSxTQUFTOzs7aUNBR1QsU0FBUzs7Ozs7Ozs7Ozs7O2VBWTNCLE1BQU0sQ0FBQyxjQUFjO2VBQ3JCLE1BQU0sQ0FBQyxVQUFVOzs7Ozt3QkFLUixNQUFNLENBQUMsZUFBZTs7Ozs7Ozs7Ozs7Ozs7c0NBY1IsS0FBSyxDQUFDLFNBQVMsZUFBZSxRQUFRLENBQUMsUUFBUSxDQUFDLFNBQVMsYUFBYSxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7UUFFdkgsUUFBUSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUMvQixJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFLEVBQUUsS0FBSyxFQUFFLFVBQVUsT0FBTyxlQUFlLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUV4SCxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUNsQyxJQUFJLEVBQUUscUJBQXFCO1lBQzNCLE1BQU0sRUFBRSxNQUFNO1lBQ2QsZ0JBQWdCLEVBQUUsSUFBSSxHQUFHLENBQUMseUJBQXlCLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDN0UsT0FBTyxFQUFFO2dCQUNQLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxzQkFBc0IsQ0FBQzthQUMvQztZQUNELFdBQVcsRUFBRSxJQUFJO1NBQ2xCLENBQUMsQ0FBQztRQUVILElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO1lBQzVCLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLGFBQWE7WUFDYixRQUFRLEVBQUUsQ0FBQyxNQUFNLENBQUM7U0FDbkIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLEtBQUssR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3ZCLENBQUM7Q0FDRjtBQTVORCxvQ0E0TkM7QUFFRCxJQUFJLFlBQVksRUFBRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7XG4gIGF3c190aW1lc3RyZWFtIGFzIHRzLFxuICBhd3NfaWFtIGFzIGlhbSxcbiAgYXdzX2VjMiBhcyBlYzIsXG59IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGlmdyBmcm9tICcuJztcblxuXG5leHBvcnQgY2xhc3MgSW50ZWdUZXN0aW5nIHtcbiAgcmVhZG9ubHkgc3RhY2s6IGNkay5TdGFja1tdO1xuICBjb25zdHJ1Y3RvcigpIHtcbiAgICBjb25zdCBhcHAgPSBuZXcgY2RrLkFwcCgpO1xuXG4gICAgY29uc3QgZW52ID0ge1xuICAgICAgcmVnaW9uOiBwcm9jZXNzLmVudi5DREtfSU5URUdfUkVHSU9OIHx8IHByb2Nlc3MuZW52LkNES19ERUZBVUxUX1JFR0lPTiB8fCAndXMtZWFzdC0xJyxcbiAgICAgIGFjY291bnQ6IHByb2Nlc3MuZW52LkNES19JTlRFR19BQ0NPVU5UIHx8IHByb2Nlc3MuZW52LkNES19ERUZBVUxUX0FDQ09VTlQsXG4gICAgfTtcblxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayhhcHAsICdpbnRlZy1zdGFjaycsIHsgZW52IH0pO1xuXG4gICAgY29uc3QgZGF0YWJhc2UgPSBuZXcgdHMuQ2ZuRGF0YWJhc2Uoc3RhY2ssICdEYXRhYmFzZScsIHtcbiAgICAgIGRhdGFiYXNlTmFtZTogJ0ZsZWV0V2lzZScsXG4gICAgfSk7XG5cbiAgICBjb25zdCB0YWJsZSA9IG5ldyB0cy5DZm5UYWJsZShzdGFjaywgJ1RhYmxlJywge1xuICAgICAgZGF0YWJhc2VOYW1lOiAnRmxlZXRXaXNlJyxcbiAgICAgIHRhYmxlTmFtZTogJ0ZsZWV0V2lzZScsXG4gICAgfSk7XG5cbiAgICB0YWJsZS5ub2RlLmFkZERlcGVuZGVuY3koZGF0YWJhc2UpO1xuXG4gICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnaW90ZmxlZXR3aXNlLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FkbWluaXN0cmF0b3JBY2Nlc3MnKSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBzaWduYWxDYXRhbG9nID0gbmV3IGlmdy5TaWduYWxDYXRhbG9nKHN0YWNrLCAnU2lnbmFsQ2F0YWxvZycsIHtcbiAgICAgIGRhdGFiYXNlLFxuICAgICAgdGFibGUsXG4gICAgICByb2xlLFxuICAgICAgZGVzY3JpcHRpb246ICdteSBzaWduYWwgY2F0YWxvZycsXG4gICAgICBub2RlczogW1xuICAgICAgICBuZXcgaWZ3LlNpZ25hbENhdGFsb2dCcmFuY2goJ1ZlaGljbGUnLCAnVmVoaWNsZScpLFxuICAgICAgICBuZXcgaWZ3LlNpZ25hbENhdGFsb2dTZW5zb3IoJ0VuZ2luZVRvcnF1ZScsICdWZWhpY2xlLkVuZ2luZVRvcnF1ZScsICdET1VCTEUnKSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBtb2RlbF9hID0gbmV3IGlmdy5WZWhpY2xlTW9kZWwoc3RhY2ssICdNb2RlbEEnLCB7XG4gICAgICBzaWduYWxDYXRhbG9nLFxuICAgICAgbmFtZTogJ21vZGVsQScsXG4gICAgICBkZXNjcmlwdGlvbjogJ01vZGVsIEEgdmVoaWNsZScsXG4gICAgICBuZXR3b3JrSW50ZXJmYWNlczogW25ldyBpZncuQ2FuVmVoaWNsZUludGVyZmFjZSgnMScsICd2Y2FuMCcpXSxcbiAgICAgIHNpZ25hbHM6IFtcbiAgICAgICAgbmV3IGlmdy5DYW5WZWhpY2xlU2lnbmFsKCdFbmdpbmVUb3JxdWUnLCAnVmVoaWNsZS5FbmdpbmVUb3JxdWUnLCAnMScsXG4gICAgICAgICAgNDAxLCAvLyBtZXNzYWdlSWRcbiAgICAgICAgICAxLjAsIC8vIGZhY3RvclxuICAgICAgICAgIHRydWUsIC8vIGlzQmlnRW5kaWFuXG4gICAgICAgICAgZmFsc2UsIC8vIGlzU2lnbmVkXG4gICAgICAgICAgOCwgLy8gbGVuZ3RoXG4gICAgICAgICAgMC4wLCAvLyBvZmZzZXRcbiAgICAgICAgICA5KSwgLy8gc3RhcnRCaXRcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBjb25zdCB2aW4xMDAgPSBuZXcgaWZ3LlZlaGljbGUoc3RhY2ssICd2aW4xMDAnLCB7XG4gICAgICB2ZWhpY2xlSWQ6ICd2aW4xMDAnLFxuICAgICAgdmVoaWNsZU1vZGVsOiBtb2RlbF9hLFxuICAgICAgY3JlYXRlSW90VGhpbmc6IHRydWUsXG4gICAgfSk7XG5cbiAgICBjb25zdCB2cGMgPSBlYzIuVnBjLmZyb21Mb29rdXAoc3RhY2ssICdWUEMnLCB7IGlzRGVmYXVsdDogdHJ1ZSB9KTtcblxuICAgIGNvbnN0IHNlY3VyaXR5R3JvdXAgPSBuZXcgZWMyLlNlY3VyaXR5R3JvdXAoc3RhY2ssICdTZWN1cml0eUdyb3VwJywge1xuICAgICAgdnBjLFxuICAgICAgYWxsb3dBbGxPdXRib3VuZDogdHJ1ZSxcbiAgICB9KTtcbiAgICBzZWN1cml0eUdyb3VwLmFkZEluZ3Jlc3NSdWxlKGVjMi5QZWVyLmFueUlwdjQoKSwgZWMyLlBvcnQudGNwKDIyKSwgJ1NTSCBhY2Nlc3MnKTtcbiAgICBzZWN1cml0eUdyb3VwLmFkZEluZ3Jlc3NSdWxlKGVjMi5QZWVyLmFueUlwdjQoKSwgZWMyLlBvcnQuYWxsSWNtcCgpLCAncGluZycpO1xuXG4gICAgLy8gRUMyIHJvbGVcbiAgICBjb25zdCBlYzJfcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ2VjMlJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblNTTU1hbmFnZWRJbnN0YW5jZUNvcmUnKSxcbiAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBZG1pbmlzdHJhdG9yQWNjZXNzJyksXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgLy8gVWJ1bnR1IDE4LjA0IGZvciBBcm02NFxuICAgIGNvbnN0IG1hY2hpbmVJbWFnZSA9IGVjMi5NYWNoaW5lSW1hZ2UuZnJvbVNzbVBhcmFtZXRlcihcbiAgICAgICcvYXdzL3NlcnZpY2UvY2Fub25pY2FsL3VidW50dS9zZXJ2ZXIvMTguMDQvc3RhYmxlL2N1cnJlbnQvYXJtNjQvaHZtL2Vicy1ncDIvYW1pLWlkJyxcbiAgICAgIHsgb3M6IGVjMi5PcGVyYXRpbmdTeXN0ZW1UeXBlLkxJTlVYIH0sXG4gICAgKTtcblxuICAgIC8vIENyZWF0ZSB0aGUgVmVoaWNsZSBzaW11bGF0b3JcbiAgICBjb25zdCBrZXlOYW1lID0gc3RhY2subm9kZS50cnlHZXRDb250ZXh0KCdrZXlfbmFtZScpO1xuICAgIGNvbnN0IGluc3RhbmNlID0gbmV3IGVjMi5JbnN0YW5jZShzdGFjaywgJ1ZlaGljbGVTaW0nLCB7XG4gICAgICB2cGM6IHZwYyxcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IGVjMi5JbnN0YW5jZVR5cGUoJ202Zy54bGFyZ2UnKSxcbiAgICAgIG1hY2hpbmVJbWFnZSxcbiAgICAgIHNlY3VyaXR5R3JvdXAsXG4gICAgICBrZXlOYW1lLFxuICAgICAgcm9sZTogZWMyX3JvbGUsXG4gICAgICB2cGNTdWJuZXRzOiB7XG4gICAgICAgIHN1Ym5ldEdyb3VwTmFtZTogJ1B1YmxpYycsXG4gICAgICB9LFxuICAgICAgcmVzb3VyY2VTaWduYWxUaW1lb3V0OiBjZGsuRHVyYXRpb24ubWludXRlcygzMCksXG4gICAgfSk7XG5cbiAgICBjb25zdCBzb3VyY2VVcmwgPSAnaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtaW90LWZsZWV0d2lzZS1lZGdlLmdpdCc7XG4gICAgY29uc3QgdXNlckRhdGEgPSBgXFxcbiAgICBzZXQgLWV1byBwaXBlZmFpbFxuXG4gICAgIyBXYWl0IGZvciBhbnkgZXhpc3RpbmcgcGFja2FnZSBpbnN0YWxsIHRvIGZpbmlzaFxuICAgIGk9MFxuICAgIHdoaWxlIHRydWU7IGRvXG4gICAgICAgIGlmIHN1ZG8gZnVzZXIgL3Zhci97bGliL3tkcGtnLGFwdC9saXN0c30sY2FjaGUvYXB0L2FyY2hpdmVzfS9sb2NrID4vZGV2L251bGwgMj4mMTsgdGhlblxuICAgICAgICAgICAgaT0wXG4gICAgICAgIGVsc2VcbiAgICAgICAgICAgIGk9XFxgZXhwciAkaSArIDFcXGBcbiAgICAgICAgICAgIGlmIGV4cHIgJGkgXFxcXD49IDEwID4gL2Rldi9udWxsOyB0aGVuXG4gICAgICAgICAgICAgICAgYnJlYWtcbiAgICAgICAgICAgIGZpXG4gICAgICAgIGZpXG4gICAgICAgIHNsZWVwIDFcbiAgICBkb25lXG5cbiAgICAjIFVwZ3JhZGUgc3lzdGVtIGFuZCByZWJvb3QgaWYgcmVxdWlyZWRcbiAgICBhcHQgdXBkYXRlICYmIGFwdCB1cGdyYWRlIC15XG4gICAgaWYgWyAtZiAvdmFyL3J1bi9yZWJvb3QtcmVxdWlyZWQgXTsgdGhlblxuICAgICAgIyBEZWxldGUgdGhlIFVzZXJEYXRhIGluZm8gZmlsZSBzbyB0aGF0IHdlIHJ1biBhZ2FpbiBhZnRlciByZWJvb3RcbiAgICAgIHJtIC1mIC92YXIvbGliL2Nsb3VkL2luc3RhbmNlcy8qL3NlbS9jb25maWdfc2NyaXB0c191c2VyXG4gICAgICByZWJvb3RcbiAgICAgIGV4aXRcbiAgICBmaVxuXG4gICAgIyBJbnN0YWxsIGhlbHBlciBzY3JpcHRzOlxuICAgIGFwdCB1cGRhdGUgJiYgYXB0IGluc3RhbGwgLXkgcHl0aG9uMy1zZXR1cHRvb2xzXG4gICAgbWtkaXIgLXAgL29wdC9hd3MvYmluXG4gICAgd2dldCBodHRwczovL3MzLmFtYXpvbmF3cy5jb20vY2xvdWRmb3JtYXRpb24tZXhhbXBsZXMvYXdzLWNmbi1ib290c3RyYXAtcHkzLWxhdGVzdC50YXIuZ3pcbiAgICBweXRob24zIC1tIGVhc3lfaW5zdGFsbCAtLXNjcmlwdC1kaXIgL29wdC9hd3MvYmluIGF3cy1jZm4tYm9vdHN0cmFwLXB5My1sYXRlc3QudGFyLmd6XG4gICAgcm0gYXdzLWNmbi1ib290c3RyYXAtcHkzLWxhdGVzdC50YXIuZ3pcblxuICAgICMgT24gZXJyb3IsIHNpZ25hbCBiYWNrIHRvIGNmbjpcbiAgICBlcnJvcl9oYW5kbGVyKCkge1xuICAgICAgL29wdC9hd3MvYmluL2Nmbi1zaWduYWwgLS1zdWNjZXNzIGZhbHNlIC0tc3RhY2sgJHtzdGFjay5zdGFja05hbWV9IC0tcmVzb3VyY2UgJHtpbnN0YW5jZS5pbnN0YW5jZS5sb2dpY2FsSWR9IC0tcmVnaW9uICR7c3RhY2sucmVnaW9ufVxuICAgIH1cbiAgICB0cmFwIGVycm9yX2hhbmRsZXIgRVJSXG5cbiAgICAjIEluY3JlYXNlIHBpZF9tYXg6XG4gICAgZWNobyAxMDQ4NTc2ID4gL3Byb2Mvc3lzL2tlcm5lbC9waWRfbWF4XG4gICAgIyBEaXNhYmxlIHN5c2xvZzpcbiAgICBzeXN0ZW1jdGwgc3RvcCBzeXNsb2cuc29ja2V0IHJzeXNsb2cuc2VydmljZVxuICAgICMgUmVtb3ZlIGpvdXJuYWxkIHJhdGUgbGltaXRpbmcgYW5kIHNldCBtYXggc2l6ZTpcbiAgICBwcmludGYgXCJSYXRlTGltaXRCdXJzdD0wXFxuU3lzdGVtTWF4VXNlPTFHXFxuXCIgPj4gL2V0Yy9zeXN0ZW1kL2pvdXJuYWxkLmNvbmZcblxuICAgICMgSW5zdGFsbCBwYWNrYWdlc1xuICAgIGFwdCB1cGRhdGUgJiYgYXB0IGluc3RhbGwgLXkgZ2l0IGVjMi1pbnN0YW5jZS1jb25uZWN0IGh0b3AganEgdW56aXBcblxuICAgICMgSW5zdGFsbCBBV1MgQ0xJOlxuICAgIGN1cmwgXCJodHRwczovL2F3c2NsaS5hbWF6b25hd3MuY29tL2F3c2NsaS1leGUtbGludXgtYWFyY2g2NC56aXBcIiAtbyBcImF3c2NsaXYyLnppcFwiXG4gICAgdW56aXAgYXdzY2xpdjIuemlwXG4gICAgLi9hd3MvaW5zdGFsbFxuICAgIHJtIGF3c2NsaXYyLnppcFxuXG4gICAgIyBEb3dubG9hZCBzb3VyY2VcbiAgICBjZCAvaG9tZS91YnVudHVcbiAgICBpZiBlY2hvICR7c291cmNlVXJsfSB8IGdyZXAgLXEgJ3MzOi8vJzsgdGhlblxuICAgICAgc3VkbyAtdSB1YnVudHUgYXdzIHMzIGNwICR7c291cmNlVXJsfSBhd3MtaW90LWZsZWV0d2lzZS1lZGdlLnppcFxuICAgICAgc3VkbyAtdSB1YnVudHUgdW56aXAgYXdzLWlvdC1mbGVldHdpc2UtZWRnZS56aXAgLWQgYXdzLWlvdC1mbGVldHdpc2UtZWRnZVxuICAgIGVsc2VcbiAgICAgIHN1ZG8gLXUgdWJ1bnR1IGdpdCBjbG9uZSAke3NvdXJjZVVybH0gYXdzLWlvdC1mbGVldHdpc2UtZWRnZVxuICAgIGZpXG4gICAgY2QgYXdzLWlvdC1mbGVldHdpc2UtZWRnZVxuICAgIFxuICAgICMgSW5zdGFsbCBTb2NrZXRDQU4gbW9kdWxlczpcbiAgICAuL3Rvb2xzL2luc3RhbGwtc29ja2V0Y2FuLnNoIC0tYnVzLWNvdW50IDFcbiAgICBcbiAgICAjIEluc3RhbGwgQ0FOIFNpbXVsYXRvclxuICAgIC4vdG9vbHMvaW5zdGFsbC1jYW5zaW0uc2ggLS1idXMtY291bnQgMVxuICAgIFxuICAgICMgSW5zdGFsbCBGV0UgY3JlZGVudGlhbHMgYW5kIGNvbmZpZyBmaWxlXG4gICAgbWtkaXIgL2V0Yy9hd3MtaW90LWZsZWV0d2lzZVxuICAgIGVjaG8gLW4gXCIke3ZpbjEwMC5jZXJ0aWZpY2F0ZVBlbX1cIiA+IC9ldGMvYXdzLWlvdC1mbGVldHdpc2UvY2VydGlmaWNhdGUucGVtXG4gICAgZWNobyAtbiBcIiR7dmluMTAwLnByaXZhdGVLZXl9XCIgPiAvZXRjL2F3cy1pb3QtZmxlZXR3aXNlL3ByaXZhdGUta2V5LmtleVxuICAgIC4vdG9vbHMvY29uZmlndXJlLWZ3ZS5zaCBcXFxuICAgICAgLS1pbnB1dC1jb25maWctZmlsZSBcImNvbmZpZ3VyYXRpb24vc3RhdGljLWNvbmZpZy5qc29uXCIgXFxcbiAgICAgIC0tb3V0cHV0LWNvbmZpZy1maWxlIFwiL2V0Yy9hd3MtaW90LWZsZWV0d2lzZS9jb25maWctMC5qc29uXCIgXFxcbiAgICAgIC0tdmVoaWNsZS1pZCB2aW4xMDAgXFxcbiAgICAgIC0tZW5kcG9pbnQtdXJsIFwiJHt2aW4xMDAuZW5kcG9pbnRBZGRyZXNzfVwiIFxcXG4gICAgICAtLXRvcGljLXByZWZpeCAnJGF3cy9pb3RmbGVldHdpc2UvJyBcXFxuICAgICAgLS1jYW4tYnVzMCBcInZjYW4wXCJcblxuICAgICMgSW5zdGFsbCBzb3VyY2UgZGVwc1xuICAgIC4vdG9vbHMvaW5zdGFsbC1kZXBzLW5hdGl2ZS5zaFxuXG4gICAgIyBCdWlsZCBzb3VyY2VcbiAgICBzdWRvIC11IHVidW50dSAuL3Rvb2xzL2J1aWxkLWZ3ZS1uYXRpdmUuc2hcblxuICAgICMgSW5zdGFsbCBGV0VcbiAgICAuL3Rvb2xzL2luc3RhbGwtZndlLnNoXG4gICAgXG4gICAgIyBTaWduYWwgaW5pdCBjb21wbGV0ZTpcbiAgICAvb3B0L2F3cy9iaW4vY2ZuLXNpZ25hbCAtLXN0YWNrICR7c3RhY2suc3RhY2tOYW1lfSAtLXJlc291cmNlICR7aW5zdGFuY2UuaW5zdGFuY2UubG9naWNhbElkfSAtLXJlZ2lvbiAke3N0YWNrLnJlZ2lvbn1gO1xuXG4gICAgaW5zdGFuY2UuYWRkVXNlckRhdGEodXNlckRhdGEpO1xuICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHN0YWNrLCAnVmVoaWNsZSBzc2ggY29tbWFuZCcsIHsgdmFsdWU6IGBzc2ggLWkgJHtrZXlOYW1lfS5wZW0gdWJ1bnR1QCR7aW5zdGFuY2UuaW5zdGFuY2VQdWJsaWNJcH1gIH0pO1xuXG4gICAgbmV3IGlmdy5DYW1wYWlnbihzdGFjaywgJ0NhbXBhaWduJywge1xuICAgICAgbmFtZTogJ0Z3VGltZUJhc2VkQ2FtcGFpZ24nLFxuICAgICAgdGFyZ2V0OiB2aW4xMDAsXG4gICAgICBjb2xsZWN0aW9uU2NoZW1lOiBuZXcgaWZ3LlRpbWVCYXNlZENvbGxlY3Rpb25TY2hlbWUoY2RrLkR1cmF0aW9uLnNlY29uZHMoMTApKSxcbiAgICAgIHNpZ25hbHM6IFtcbiAgICAgICAgbmV3IGlmdy5DYW1wYWlnblNpZ25hbCgnVmVoaWNsZS5FbmdpbmVUb3JxdWUnKSxcbiAgICAgIF0sXG4gICAgICBhdXRvQXBwcm92ZTogdHJ1ZSxcbiAgICB9KTtcblxuICAgIG5ldyBpZncuRmxlZXQoc3RhY2ssICdGbGVldCcsIHtcbiAgICAgIGZsZWV0SWQ6ICdmbGVldCcsXG4gICAgICBzaWduYWxDYXRhbG9nLFxuICAgICAgdmVoaWNsZXM6IFt2aW4xMDBdLFxuICAgIH0pO1xuXG4gICAgdGhpcy5zdGFjayA9IFtzdGFja107XG4gIH1cbn1cblxubmV3IEludGVnVGVzdGluZygpOyJdfQ==