"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PgStacDatabase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const bootstrapper_1 = require("../bootstrapper");
const instanceSizes = require("./instance-memory.json");
/**
 * An RDS instance with pgSTAC installed. This is a wrapper around the
 * `rds.DatabaseInstance` higher-level construct making use
 * of the BootstrapPgStac construct.
 */
class PgStacDatabase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const defaultParameters = this.getParameters(props.instanceType?.toString() || "m5.large", props.parameters);
        const parameterGroup = new aws_cdk_lib_1.aws_rds.ParameterGroup(this, "parameterGroup", {
            engine: props.engine,
            parameters: {
                shared_buffers: defaultParameters.sharedBuffers,
                effective_cache_size: defaultParameters.effectiveCacheSize,
                work_mem: defaultParameters.workMem,
                maintenance_work_mem: defaultParameters.maintenanceWorkMem,
                max_locks_per_transaction: defaultParameters.maxLocksPerTransaction,
                temp_buffers: defaultParameters.tempBuffers,
                seq_page_cost: defaultParameters.seqPageCost,
                random_page_cost: defaultParameters.randomPageCost,
                ...props.parameters,
            },
        });
        this.db = new aws_cdk_lib_1.aws_rds.DatabaseInstance(this, "db", {
            instanceIdentifier: aws_cdk_lib_1.Stack.of(this).stackName,
            parameterGroup,
            ...props,
        });
        const bootstrap = new bootstrapper_1.BootstrapPgStac(this, "bootstrap-pgstac-instance", {
            vpc: props.vpc,
            database: this.db,
            dbSecret: this.db.secret,
            pgstacDbName: props.pgstacDbName,
            pgstacVersion: props.pgstacVersion,
            pgstacUsername: props.pgstacUsername,
            secretsPrefix: props.secretsPrefix,
        });
        this.pgstacSecret = bootstrap.secret;
    }
    getParameters(instanceType, parameters) {
        // https://github.com/aws/aws-cli/issues/1279#issuecomment-909318236
        const memory_in_kb = instanceSizes[instanceType] * 1024;
        // It's only necessary to consider passed in parameters for any value that used to
        // derive subsequent values. Values that don't have dependencies will be overriden
        // when we unpack the passed-in user parameters
        const maxConnections = parameters?.maxConnections
            ? Number.parseInt(parameters.maxConnections)
            : // https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.MaxConnections
                Math.min(Math.round((memory_in_kb * 1024) / 9531392), 5000);
        const sharedBuffers = parameters?.sharedBufers
            ? Number.parseInt(parameters.sharedBufers)
            : Math.round(0.25 * memory_in_kb);
        const effectiveCacheSize = Math.round(0.75 * memory_in_kb);
        const workMem = Math.floor(sharedBuffers / maxConnections);
        const maintenanceWorkMem = Math.round(0.25 * sharedBuffers);
        const tempBuffers = 128 * 1024;
        const seqPageCost = 1;
        const randomPageCost = 1.1;
        return {
            maxConnections: `${maxConnections}`,
            sharedBuffers: `${sharedBuffers}`,
            effectiveCacheSize: `${effectiveCacheSize}`,
            workMem: `${workMem}`,
            maintenanceWorkMem: `${maintenanceWorkMem}`,
            maxLocksPerTransaction: "1024",
            tempBuffers: `${tempBuffers}`,
            seqPageCost: `${seqPageCost}`,
            randomPageCost: `${randomPageCost}`,
        };
    }
}
exports.PgStacDatabase = PgStacDatabase;
_a = JSII_RTTI_SYMBOL_1;
PgStacDatabase[_a] = { fqn: "cdk-pgstac.PgStacDatabase", version: "2.2.0" };
//# sourceMappingURL=data:application/json;base64,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