"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkTransformer = void 0;
const aws_appsync_1 = require("@aws-cdk/aws-appsync");
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphqlTypeStatements = ['Query', 'Mutation', 'Subscription'];
class CdkTransformer extends graphql_transformer_core_1.Transformer {
    constructor() {
        super('CdkTransformer', 'directive @nullable on FIELD_DEFINITION');
        this.after = (ctx) => {
            this.buildResources(ctx);
            // TODO: Improve this iteration
            Object.keys(this.tables).forEach(tableName => {
                let table = this.tables[tableName];
                Object.keys(this.resolverTableMap).forEach(resolverName => {
                    if (this.resolverTableMap[resolverName] === tableName)
                        table.resolvers.push(resolverName);
                });
                Object.keys(this.gsiResolverTableMap).forEach(resolverName => {
                    if (this.gsiResolverTableMap[resolverName] === tableName)
                        table.gsiResolvers.push(resolverName);
                });
            });
            // @ts-ignore - we are overloading the use of outputs here...
            ctx.setOutput('cdkTables', this.tables);
            // @ts-ignore - we are overloading the use of outputs here...
            ctx.setOutput('noneResolvers', this.noneDataSources);
            // @ts-ignore - we are overloading the use of outputs here...
            ctx.setOutput('functionResolvers', this.functionResolvers);
            const query = ctx.getQuery();
            if (query) {
                const queryFields = graphql_transformer_core_1.getFieldArguments(query);
                ctx.setOutput('queries', queryFields);
            }
            const mutation = ctx.getMutation();
            if (mutation) {
                const mutationFields = graphql_transformer_core_1.getFieldArguments(mutation);
                ctx.setOutput('mutations', mutationFields);
            }
            const subscription = ctx.getSubscription();
            if (subscription) {
                const subscriptionFields = graphql_transformer_core_1.getFieldArguments(subscription);
                ctx.setOutput('subscriptions', subscriptionFields);
            }
        };
        this.tables = {};
        this.noneDataSources = {};
        this.functionResolvers = {};
        this.resolverTableMap = {};
        this.gsiResolverTableMap = {};
    }
    buildResources(ctx) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        const templateResources = ctx.template.Resources;
        if (!templateResources)
            return;
        for (const [resourceName, resource] of Object.entries(templateResources)) {
            if (resource.Type === 'AWS::DynamoDB::Table') {
                this.buildTablesFromResource(resourceName, ctx);
            }
            else if (resource.Type === 'AWS::AppSync::Resolver') {
                if (((_a = resource.Properties) === null || _a === void 0 ? void 0 : _a.DataSourceName) === 'NONE') {
                    this.noneDataSources[resource.Properties.FieldName] = {
                        typeName: resource.Properties.TypeName,
                        fieldName: resource.Properties.FieldName,
                    };
                }
                else if (((_b = resource.Properties) === null || _b === void 0 ? void 0 : _b.Kind) === 'PIPELINE') {
                    // Inspired by:
                    // https://github.com/aws-amplify/amplify-cli/blob/master/packages/graphql-function-transformer/src/__tests__/FunctionTransformer.test.ts#L20
                    const dependsOn = (_c = resource.DependsOn) !== null && _c !== void 0 ? _c : '';
                    const functionConfiguration = templateResources[dependsOn];
                    const functionDependsOn = (_d = functionConfiguration.DependsOn) !== null && _d !== void 0 ? _d : '';
                    const functionDataSource = templateResources[functionDependsOn];
                    const functionArn = (_g = (_f = (_e = functionDataSource.Properties) === null || _e === void 0 ? void 0 : _e.LambdaConfig) === null || _f === void 0 ? void 0 : _f.LambdaFunctionArn) === null || _g === void 0 ? void 0 : _g.payload[1].payload[0];
                    const functionName = functionArn.split(':').slice(-1)[0];
                    const fieldName = resource.Properties.FieldName;
                    const typeName = resource.Properties.TypeName;
                    if (!this.functionResolvers[functionName])
                        this.functionResolvers[functionName] = [];
                    this.functionResolvers[functionName].push({
                        typeName: typeName,
                        fieldName: fieldName,
                        defaultRequestMappingTemplate: aws_appsync_1.MappingTemplate.lambdaRequest().renderTemplate(),
                        defaultResponseMappingTemplate: (_h = functionConfiguration.Properties) === null || _h === void 0 ? void 0 : _h.ResponseMappingTemplate,
                    });
                }
                else { // Should be a table/model resolver -> Maybe not true when we add in @searchable, etc
                    let typeName = (_j = resource.Properties) === null || _j === void 0 ? void 0 : _j.TypeName;
                    let fieldName = (_k = resource.Properties) === null || _k === void 0 ? void 0 : _k.FieldName;
                    let tableName = (_m = (_l = resource.Properties) === null || _l === void 0 ? void 0 : _l.DataSourceName) === null || _m === void 0 ? void 0 : _m.payload[0];
                    tableName = tableName.replace('DataSource', 'Table');
                    if (graphqlTypeStatements.indexOf(typeName) >= 0) {
                        this.resolverTableMap[fieldName] = tableName;
                    }
                    else { // this is a GSI
                        this.gsiResolverTableMap[`${typeName}${fieldName}`] = tableName;
                    }
                }
            }
        }
    }
    buildTablesFromResource(resourceName, ctx) {
        var _a, _b, _c, _d;
        const tableResource = ctx.template.Resources ? ctx.template.Resources[resourceName] : undefined;
        const attributeDefinitions = (_a = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _a === void 0 ? void 0 : _a.AttributeDefinitions;
        const keySchema = (_b = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _b === void 0 ? void 0 : _b.KeySchema;
        const keys = this.parseKeySchema(keySchema, attributeDefinitions);
        let table = {
            tableName: resourceName,
            partitionKey: keys.partitionKey,
            sortKey: keys.sortKey,
            ttl: (_c = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _c === void 0 ? void 0 : _c.TimeToLiveSpecification,
            globalSecondaryIndexes: [],
            resolvers: [],
            gsiResolvers: [],
        };
        const gsis = (_d = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _d === void 0 ? void 0 : _d.GlobalSecondaryIndexes;
        if (gsis) {
            gsis.forEach((gsi) => {
                const gsiKeys = this.parseKeySchema(gsi.KeySchema, attributeDefinitions);
                const gsiDefinition = {
                    indexName: gsi.IndexName,
                    projection: gsi.Projection,
                    partitionKey: gsiKeys.partitionKey,
                    sortKey: gsiKeys.sortKey,
                };
                table.globalSecondaryIndexes.push(gsiDefinition);
            });
        }
        this.tables[resourceName] = table;
    }
    parseKeySchema(keySchema, attributeDefinitions) {
        let partitionKey = {};
        let sortKey = {};
        keySchema.forEach((key) => {
            const keyType = key.KeyType;
            const attributeName = key.AttributeName;
            const attribute = attributeDefinitions.find((attr) => attr.AttributeName === attributeName);
            if (keyType === 'HASH') {
                partitionKey = {
                    name: attribute.AttributeName,
                    type: attribute.AttributeType,
                };
            }
            else if (keyType === 'RANGE') {
                sortKey = {
                    name: attribute.AttributeName,
                    type: attribute.AttributeType,
                };
            }
        });
        return { partitionKey, sortKey };
    }
}
exports.CdkTransformer = CdkTransformer;
//# sourceMappingURL=data:application/json;base64,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