"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const runner_configuration_1 = require("../runner-configuration");
const cache_1 = require("./cache");
const network_1 = require("./network");
/**
 * The Gitlab Runner autoscaling on EC2 by Docker Machine.
 *
 * @stability stable
 * @example
 *
 * <caption>Provisioning a basic Runner</caption>
 * const app = new cdk.App();
 * const stack = new cdk.Stack(app, "RunnerStack", {
 * env: {
 * account: "000000000000",
 * region: "us-east-1",
 * }
 * });
 *
 * new GitlabRunnerAutoscaling(scope, "GitlabRunner", {
 * gitlabToken: "xxxxxxxxxxxxxxxxxxxx",
 * });
 */
class GitlabRunnerAutoscaling extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id);
        const { manager, cache, runners, network } = props;
        /**
         * S3 Bucket for Runners' cache
         */
        this.cacheBucket = (cache === null || cache === void 0 ? void 0 : cache.bucket) || new cache_1.Cache(scope, "Cache", cache === null || cache === void 0 ? void 0 : cache.options).bucket;
        /**
         * Network
         */
        this.network = new network_1.Network(scope, "Network", network);
        /**
         * IAM
         */
        const ec2ServicePrincipal = new aws_iam_1.ServicePrincipal("ec2.amazonaws.com", {});
        const ec2ManagedPolicyForSSM = aws_iam_1.ManagedPolicy.fromManagedPolicyArn(scope, "AmazonEC2RoleforSSM", "arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM");
        /**
         * GitLab Runners
         */
        const runnersSecurityGroupName = `${scope.stackName}-RunnersSecurityGroup`;
        const runnersSecurityGroup = new aws_ec2_1.SecurityGroup(scope, "RunnersSecurityGroup", {
            securityGroupName: runnersSecurityGroupName,
            description: "Security group for GitLab Runners.",
            vpc: this.network.vpc,
        });
        const runnersRole = (runners === null || runners === void 0 ? void 0 : runners.role) ||
            new aws_iam_1.Role(scope, "RunnersRole", {
                assumedBy: ec2ServicePrincipal,
                managedPolicies: [ec2ManagedPolicyForSSM],
            });
        const runnersInstanceProfile = new aws_iam_1.CfnInstanceProfile(scope, "RunnersInstanceProfile", {
            roles: [runnersRole.roleName],
        });
        const runnersInstanceType = (runners === null || runners === void 0 ? void 0 : runners.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
        const runnersLookupMachineImage = new aws_ec2_1.LookupMachineImage({
            name: "ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64-server-*",
            owners: ["099720109477"],
            filters: {
                architecture: ["x86_64"],
                "image-type": ["machine"],
                state: ["available"],
                "root-device-type": ["ebs"],
                "virtualization-type": ["hvm"],
            },
        }).getImage(this);
        const runnersMachineImage = (runners === null || runners === void 0 ? void 0 : runners.machineImage) ||
            aws_ec2_1.MachineImage.genericLinux({
                [scope.region]: runnersLookupMachineImage.imageId,
            });
        /**
         * GitLab Manager
         */
        const managerSecurityGroup = new aws_ec2_1.SecurityGroup(scope, "ManagerSecurityGroup", {
            vpc: this.network.vpc,
            description: "Security group for GitLab Runners Manager.",
        });
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(22), "SSH traffic from Manager");
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(2376), "SSH traffic from Docker");
        const managerInstanceType = (runners === null || runners === void 0 ? void 0 : runners.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.NANO);
        const managerMachineImage = (runners === null || runners === void 0 ? void 0 : runners.machineImage) ||
            aws_ec2_1.MachineImage.latestAmazonLinux({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                edition: aws_ec2_1.AmazonLinuxEdition.STANDARD,
                virtualization: aws_ec2_1.AmazonLinuxVirt.HVM,
                storage: aws_ec2_1.AmazonLinuxStorage.EBS,
                cpuType: aws_ec2_1.AmazonLinuxCpuType.X86_64,
            });
        const managerRole = new aws_iam_1.Role(scope, "ManagerRole", {
            assumedBy: ec2ServicePrincipal,
            managedPolicies: [ec2ManagedPolicyForSSM],
            inlinePolicies: {
                Cache: aws_iam_1.PolicyDocument.fromJson({
                    Version: "2012-10-17",
                    Statement: [
                        {
                            Effect: "Allow",
                            Action: ["s3:ListObjects*", "s3:GetObject*", "s3:DeleteObject*", "s3:PutObject*"],
                            Resource: [`${this.cacheBucket.bucketArn}/*`],
                        },
                        {
                            Effect: "Allow",
                            Action: ["s3:ListBucket"],
                            Resource: [`${this.cacheBucket.bucketArn}`],
                        },
                    ],
                }),
                Runners: aws_iam_1.PolicyDocument.fromJson({
                    Version: "2012-10-17",
                    Statement: [
                        {
                            Effect: "Allow",
                            Action: ["ec2:CreateKeyPair", "ec2:DeleteKeyPair", "ec2:ImportKeyPair", "ec2:Describe*"],
                            Resource: ["*"],
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:CreateTags", "ssm:UpdateInstanceInformation"],
                            Resource: ["*"],
                            Condition: {
                                StringLike: {
                                    "aws:RequestTag/Name": "*gitlab-runner-*",
                                },
                                "ForAllValues:StringEquals": {
                                    "aws:TagKeys": ["Name"],
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:RequestSpotInstances", "ec2:CancelSpotInstanceRequests"],
                            Resource: ["*"],
                            Condition: {
                                StringEqualsIfExists: {
                                    "ec2:Region": `${scope.region}`,
                                },
                                ArnEqualsIfExists: {
                                    "ec2:Vpc": `${this.network.vpc.vpcArn}`,
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:RunInstances"],
                            Resource: ["*"],
                            Condition: {
                                StringEquals: {
                                    "ec2:InstanceType": [`${runnersInstanceType.toString()}`],
                                    "ec2:InstanceProfile": `${runnersInstanceProfile.ref}`,
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:TerminateInstances", "ec2:StopInstances", "ec2:StartInstances", "ec2:RebootInstances"],
                            Resource: ["*"],
                            Condition: {
                                StringLike: {
                                    "ec2:ResourceTag/Name": "*gitlab-runner-*",
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["iam:PassRole"],
                            Resource: [`${runnersRole.roleArn}`],
                        },
                    ],
                }),
            },
        });
        const userData = aws_ec2_1.UserData.forLinux({});
        userData.addCommands(`yum update -y aws-cfn-bootstrap` // !/bin/bash -xe
        );
        const gitlabRunnerConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        gitlabRunnerConfigRestartHandle._addFile("/etc/gitlab-runner/config.toml");
        const rsyslogConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        rsyslogConfigRestartHandle._addFile("/etc/rsyslog.d/25-gitlab-runner.conf");
        /**
         * Config set keys
         * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html#aws-resource-init-configsets
         */
        const REPOSITORIES = "repositories";
        const PACKAGES = "packages";
        const CONFIG = "config";
        const RESTART = "restart";
        const initConfig = aws_ec2_1.CloudFormationInit.fromConfigSets({
            configSets: {
                default: [REPOSITORIES, PACKAGES, CONFIG, RESTART],
            },
            configs: {
                [REPOSITORIES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://packages.gitlab.com/install/repositories/runner/gitlab-runner/script.rpm.sh | bash", { key: "10-gitlab-runner" }),
                ]),
                [PACKAGES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitPackage.yum("docker"),
                    aws_ec2_1.InitPackage.yum("gitlab-runner"),
                    aws_ec2_1.InitPackage.yum("tzdata"),
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://gitlab-docker-machine-downloads.s3.amazonaws.com/v0.16.2-gitlab.12/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine", 
                    //"curl -L https://github.com/docker/machine/releases/download/v0.16.2/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine",
                    { key: "10-docker-machine" }),
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner start", {
                        key: "20-gitlab-runner-start",
                    }),
                ]),
                [CONFIG]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/config.toml", runner_configuration_1.ConfigurationMapper.withDefaults({
                        globalConfiguration: {
                            concurrent: (_b = props === null || props === void 0 ? void 0 : props.concurrent) !== null && _b !== void 0 ? _b : 10,
                            checkInterval: (_c = props === null || props === void 0 ? void 0 : props.checkInterval) !== null && _c !== void 0 ? _c : 0,
                            logFormat: (props === null || props === void 0 ? void 0 : props.logFormat) || "runner",
                            logLevel: (props === null || props === void 0 ? void 0 : props.logLevel) || "info",
                        },
                        runnerConfiguration: {
                            token: props.gitlabToken,
                            url: props.gitlabUrl || "https://gitlab.com",
                            limit: (_d = runners === null || runners === void 0 ? void 0 : runners.limit) !== null && _d !== void 0 ? _d : 10,
                            outputLimit: (_e = runners === null || runners === void 0 ? void 0 : runners.outputLimit) !== null && _e !== void 0 ? _e : 52428800,
                            environment: (runners === null || runners === void 0 ? void 0 : runners.environment) || ["DOCKER_DRIVER=overlay2", "DOCKER_TLS_CERTDIR=/certs"],
                        },
                        dockerConfiguration: {
                            ...runners === null || runners === void 0 ? void 0 : runners.docker,
                        },
                        machineConfiguration: {
                            ...runners === null || runners === void 0 ? void 0 : runners.machine,
                            machineOptions: {
                                ...(_f = runners === null || runners === void 0 ? void 0 : runners.machine) === null || _f === void 0 ? void 0 : _f.machineOptions,
                                instanceType: runnersInstanceType.toString(),
                                ami: runnersMachineImage.getImage(scope).imageId,
                                region: scope.region,
                                vpcId: this.network.vpc.vpcId,
                                zone: this.network.availabilityZone.slice(-1),
                                subnetId: this.network.subnet.subnetId,
                                securityGroup: `${runnersSecurityGroupName}`,
                                usePrivateAddress: true,
                                iamInstanceProfile: `${runnersInstanceProfile.ref}`,
                            },
                        },
                        autoscalingConfigurations: (runners === null || runners === void 0 ? void 0 : runners.autoscaling) || [],
                        cacheConfiguration: {
                            s3: {
                                serverAddress: `s3.${scope.urlSuffix}`,
                                bucketName: `${this.cacheBucket.bucketName}`,
                                bucketLocation: `${scope.region}`,
                            },
                        },
                    }).toToml(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/rsyslog.d/25-gitlab-runner.conf", `:programname, isequal, "gitlab-runner" /var/log/gitlab-runner.log`, {
                        owner: "root",
                        group: "root",
                        mode: "000644",
                    }),
                    aws_ec2_1.InitService.enable("gitlab-runner", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: gitlabRunnerConfigRestartHandle,
                    }),
                    aws_ec2_1.InitService.enable("rsyslog", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: rsyslogConfigRestartHandle,
                    }),
                ]),
                [RESTART]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner restart", {
                        key: "10-gitlab-runner-restart",
                    }),
                ]),
            },
        });
        const managerAutoScalingGroup = new aws_autoscaling_1.AutoScalingGroup(scope, "ManagerAutoscalingGroup", {
            vpc: this.network.vpc,
            vpcSubnets: {
                subnets: [this.network.subnet],
            },
            instanceType: managerInstanceType,
            machineImage: managerMachineImage,
            keyName: manager === null || manager === void 0 ? void 0 : manager.keyPairName,
            securityGroup: managerSecurityGroup,
            role: managerRole,
            userData: userData,
            init: initConfig,
            initOptions: {
                ignoreFailures: false,
            },
            maxCapacity: 1,
            minCapacity: 1,
            desiredCapacity: 1,
            signals: aws_autoscaling_1.Signals.waitForCount(1, { timeout: core_1.Duration.minutes(15) }),
        });
        this.runners = {
            securityGroupName: runnersSecurityGroupName,
            securityGroup: runnersSecurityGroup,
            role: runnersRole,
            instanceProfile: runnersInstanceProfile,
            instanceType: runnersInstanceType,
            machineImage: runnersMachineImage,
        };
        this.manager = {
            securityGroup: managerSecurityGroup,
            instanceType: managerInstanceType,
            machineImage: managerMachineImage,
            autoScalingGroup: managerAutoScalingGroup,
            role: managerRole,
        };
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscaling", version: "0.0.105" };
//# sourceMappingURL=data:application/json;base64,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