// Copyright 2021 Silicon Labs, Inc.
//
// This file, and derivatives thereof are licensed under the
// Solderpad License, Version 2.0 (the "License").
//
// Use of this file means you agree to the terms and conditions
// of the license and are in full compliance with the License.
//
// You may obtain a copy of the License at:
//
//     https://solderpad.org/licenses/SHL-2.0/
//
// Unless required by applicable law or agreed to in writing, software
// and hardware implementations thereof distributed under the License
// is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED.
//
// See the License for the specific language governing permissions and
// limitations under the License.

////////////////////////////////////////////////////////////////////////////////
// Engineer:       Arjan Bink - arjan.bink@silabs.com                         //
//                                                                            //
// Additional contributions by:                                               //
//                 Øystein Knauserud - oystein.knauserud@silabs.com          //
//                 Michael Platzer - michael.platzer@tuwien.ac.at             //
//                                                                            //
// Design Name:    Write Back stage                                           //
// Project Name:   CV32E40X                                                   //
// Language:       SystemVerilog                                              //
//                                                                            //
// Description:    Write back stage: Hosts write back from load/store unit    //
//                 and combined write back from ALU/MULT/DIV/CSR.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

module cv32e40x_wb_stage import cv32e40x_pkg::*;
(
  input  logic          clk,            // Not used in RTL; only used by assertions
  input  logic          rst_n,          // Not used in RTL; only used by assertions

  // EX/WB pipeline 
  input  ex_wb_pipe_t   ex_wb_pipe_i,

  // Controller
  input  ctrl_fsm_t     ctrl_fsm_i,

  // LSU
  input  logic [31:0]   lsu_rdata_i,
  input  mpu_status_e   lsu_mpu_status_i,

  // Register file interface
  output logic          rf_we_wb_o,     // Register file write enable
  output rf_addr_t      rf_waddr_wb_o,  // Register file write address
  output logic [31:0]   rf_wdata_wb_o,  // Register file write data

  // LSU handshake interface
  input  logic          lsu_valid_i,
  output logic          lsu_ready_o,
  output logic          lsu_valid_o,
  input  logic          lsu_ready_i,

  // WB stalled by LSU
  output logic          data_stall_o,

  // Stage ready/valid
  output logic          wb_ready_o,
  output logic          wb_valid_o,

  // eXtension interface
  if_xif.cpu_result     xif_result_if
);

  logic                 instr_valid;
  logic                 wb_valid;
  logic                 lsu_exception;

  // eXtension interface signals
  logic                 xif_waiting;
  logic                 xif_exception;

  assign instr_valid = ex_wb_pipe_i.instr_valid && !ctrl_fsm_i.kill_wb && !ctrl_fsm_i.halt_wb;

  assign lsu_exception = (lsu_mpu_status_i != MPU_OK);


  //////////////////////////////////////////////////////////////////////////////
  // Register file interface
  //
  // Note that write back is not suppressed during bus errors (in order to prevent
  // a timing path from the late arriving data_err_i into the register file).
  //
  // Note that the register file is only written for the last part of a split misaligned load.
  // (rf_we suppressed in ex_stage for the first part, lsu aggregates data for second part)
  //
  // Note that the register file is written multiple times in case waited loads (in
  // order to prevent a timing path from the late arriving data_rvalid_i into the
  // register file.
  //
  // In case of MPU/PMA error, the register file should not be written.
  // rf_we_wb_o is deasserted if lsu_mpu_status is not equal to MPU_OK

  // TODO: Could use result interface.we into account if out of order completion is allowed.
  assign rf_we_wb_o     = ex_wb_pipe_i.rf_we && !lsu_exception && !xif_waiting && !xif_exception && instr_valid;
  // TODO: Could use result interface.rd into account if out of order completion is allowed.
  assign rf_waddr_wb_o  = ex_wb_pipe_i.rf_waddr;
  // TODO: Could use result interface.rd into account if out of order completion is allowed.
  assign rf_wdata_wb_o  = ex_wb_pipe_i.lsu_en ? lsu_rdata_i : (ex_wb_pipe_i.xif_en ? xif_result_if.result.data : ex_wb_pipe_i.rf_wdata);

  //////////////////////////////////////////////////////////////////////////////
  // LSU inputs are valid when LSU is enabled; LSU outputs need to remain valid until downstream stage is ready

  // Does not depend on local instr_valid (i.e. kept high for stalls and kills)
  // Ok, as controller will never kill ongoing LSU instructions, and thus
  // the lsu valid_1_o which lsu_valid_o factors into should not be affected.
  assign lsu_valid_o = ex_wb_pipe_i.lsu_en && ex_wb_pipe_i.instr_valid;
  assign lsu_ready_o = 1'b1; // Always ready (there is no downstream stage)

  //////////////////////////////////////////////////////////////////////////////
  // Stage ready/valid

  assign wb_ready_o = ctrl_fsm_i.kill_wb || (lsu_ready_i && !xif_waiting && !ctrl_fsm_i.halt_wb);

  // wb_valid
  //
  // - Will be 0 for interrupted instruction and debug entry
  // - Will be 1 for synchronous exceptions (which is easier to deal with for RVFI); this implies that wb_valid
  //   cannot be used to increment the minstret CSR (as that should not increment for e.g. ecall, ebreak, etc.)
  // - Will be 1 only for the second phase of a split misaligned load/store that completes without MPU errors.
  //   If an MPU error occurs, wb_valid will be 1 due to lsu_exception (for any phase where the error occurs)

  assign wb_valid = ((!ex_wb_pipe_i.lsu_en && !xif_waiting) ||    // Non-LSU instructions have valid result in WB, also for exceptions, unless we are waiting for a coprocessor
                     ( ex_wb_pipe_i.lsu_en && lsu_valid_i)  ||    // LSU instructions have valid result based on data_rvalid_i
                     ( ex_wb_pipe_i.lsu_en && lsu_exception)      // LSU instruction had an exception
                    ) && instr_valid;

  assign wb_valid_o = wb_valid;

  // Export signal indicating WB stage stalled by load/store
  assign data_stall_o = (ex_wb_pipe_i.lsu_en && !lsu_valid_i) && !wb_valid && instr_valid;


  //---------------------------------------------------------------------------
  // eXtension interface
  //---------------------------------------------------------------------------

  // TODO: How to handle conflicting values of ex_wb_pipe_i.rf_waddr and xif_result_if.result.rd?
  // TODO: How to handle conflicting values of ex_wb_pipe_i.rf_we (based on xif_issue_if.issue_resp.writeback in ID) and xif_result_if.result.we?
  // TODO: Check whether result IDs match the instruction IDs propagated along the pipeline
  // TODO: Implement writeback to extension context status into mstatus (ecswe, ecsdata)

  // Need to wait for the result
  assign xif_waiting = ex_wb_pipe_i.instr_valid && ex_wb_pipe_i.xif_en && !xif_result_if.result_valid;

  // Coprocessor signals a synchronous exception
  // TODO: Maybe do something when an exception occurs (other than just inhibiting writeback)
  assign xif_exception = ex_wb_pipe_i.instr_valid && ex_wb_pipe_i.xif_en && xif_result_if.result_valid && xif_result_if.result.exc;

  // CV32E40X is ready to receive the result as soon as an offloaded instruction has reached WB
  assign xif_result_if.result_ready = ex_wb_pipe_i.instr_valid && ex_wb_pipe_i.xif_en;

endmodule // cv32e40x_wb_stage
