"""
Command-line interface for MAL toolbox operations

Usage:
    maltoolbox compile <lang_file> <output_file>
    maltoolbox generate-attack-graph [--graphviz] <model_file> <lang_file>
    maltoolbox upgrade-model <model_file> <lang_file> <output_file>
    maltoolbox visualize-model <model_file> <lang_file>

Arguments:
    <model_file>    Path to JSON instance model file.
    <lang_file>     Path to .mar or .mal file containing MAL spec.
    <output_file>   Path to write the result of the compilation (yml/json).

Options:
  -h --help         Show this screen.
  -g --graphviz     Visualize with graphviz

Notes:
    - <lang_file> can be either a .mar file (generated by the older MAL
      compiler) or a .mal file containing the DSL written in MAL.
"""

import logging
import json
import docopt

from . import log_configs
from .attackgraph import create_attack_graph, AttackGraph
from .language.compiler import MalCompiler
from .language.languagegraph import LanguageGraph
from .translators.updater import load_model_from_older_version
from .visualization.graphviz_utils import render_model, render_attack_graph
from .model import Model

logger = logging.getLogger(__name__)

def generate_attack_graph(
        model_file: str,
        lang_file: str
    ) -> AttackGraph:
    """Create an attack graph

    Args:
    model_file      - path to the model file
    lang_file       - path to the language file
    """
    attack_graph = create_attack_graph(lang_file, model_file)
    if log_configs['attackgraph_file']:
        attack_graph.save_to_file(
            log_configs['attackgraph_file']
        )
    return attack_graph

def compile(lang_file: str, output_file: str) -> None:
    """Compile language and dump into output file"""
    compiler = MalCompiler()
    with open(output_file, "w") as f:
        json.dump(compiler.compile(lang_file), f, indent=2)


def upgrade_model(model_file: str, lang_file: str, output_file: str):
    lang_graph = LanguageGraph.load_from_file(lang_file)

    if log_configs['langspec_file']:
        lang_graph.save_to_file(log_configs['langspec_file'])

    model = load_model_from_older_version(model_file, lang_graph)
    model.save_to_file(output_file)


def main():
    args = docopt.docopt(__doc__)

    if args['generate-attack-graph']:
        attack_graph = generate_attack_graph(
            args['<model_file>'], args['<lang_file>']
        )
        if args['--graphviz']:
            render_attack_graph(attack_graph)

    elif args['compile']:
        compile(
            args['<lang_file>'], args['<output_file>']
        )
    elif args['upgrade-model']:
        upgrade_model(
            args['<model_file>'], args['<lang_file>'], args['<output_file>']
        )
    elif args['visualize-model']:
        lang_graph = LanguageGraph.load_from_file(args['<lang_file>'])
        model = Model.load_from_file(args['<model_file>'], lang_graph)
        render_model(model)

if __name__ == "__main__":
    main()
