import * as iam from '@aws-cdk/aws-iam';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { IEngine } from './engine';
import { IOptionGroup } from './option-group';
import { Construct } from '@aws-cdk/core';
/**
 * The options passed to {@link IInstanceEngine.bind}.
 */
export interface InstanceEngineBindOptions {
    /**
     * The Active Directory directory ID to create the DB instance in.
     *
     * @default - none (it's an optional field)
     */
    readonly domain?: string;
    /**
     * The timezone of the database, set by the customer.
     *
     * @default - none (it's an optional field)
     */
    readonly timezone?: string;
    /**
     * The role used for S3 importing.
     *
     * @default - none
     */
    readonly s3ImportRole?: iam.IRole;
    /**
     * The role used for S3 exporting.
     *
     * @default - none
     */
    readonly s3ExportRole?: iam.IRole;
    /**
     * The option group of the database
     *
     * @default - none
     */
    readonly optionGroup?: IOptionGroup;
}
/**
 * The type returned from the {@link IInstanceEngine.bind} method.
 */
export interface InstanceEngineConfig {
    /**
     * Features supported by the database engine.
     *
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html
     *
     * @default - no features
     */
    readonly features?: InstanceEngineFeatures;
    /**
     * Option group of the database.
     *
     * @default - none
     */
    readonly optionGroup?: IOptionGroup;
}
/**
 * Represents Database Engine features
 */
export interface InstanceEngineFeatures {
    /**
     * Feature name for the DB instance that the IAM role to access the S3 bucket for import
     * is to be associated with.
     *
     * @default - no s3Import feature name
     */
    readonly s3Import?: string;
    /**
     * Feature name for the DB instance that the IAM role to export to S3 bucket is to be
     * associated with.
     *
     * @default - no s3Export feature name
     */
    readonly s3Export?: string;
}
/**
 * Interface representing a database instance (as opposed to cluster) engine.
 */
export interface IInstanceEngine extends IEngine {
    /** The application used by this engine to perform rotation for a single-user scenario. */
    readonly singleUserRotationApplication: secretsmanager.SecretRotationApplication;
    /** The application used by this engine to perform rotation for a multi-user scenario. */
    readonly multiUserRotationApplication: secretsmanager.SecretRotationApplication;
    /**
     * Whether this engine supports automatic backups of a read replica instance.
     *
     * @default false
     */
    readonly supportsReadReplicaBackups?: boolean;
    /**
     * Method called when the engine is used to create a new instance.
     */
    bindToInstance(scope: Construct, options: InstanceEngineBindOptions): InstanceEngineConfig;
}
/**
 * The versions for the MariaDB instance engines
 * (those returned by {@link DatabaseInstanceEngine.mariaDb}).
 */
export declare class MariaDbEngineVersion {
    /**
     * Version "10.0" (only a major version, without a specific minor version).
     * @deprecated MariaDB 10.0 will reach end of life on May 18, 2021
    */
    static readonly VER_10_0: MariaDbEngineVersion;
    /**
     * Version "10.0.17".
     * @deprecated MariaDB 10.0 will reach end of life on May 18, 2021
    */
    static readonly VER_10_0_17: MariaDbEngineVersion;
    /**
     * Version "10.0.24".
     * @deprecated MariaDB 10.0 will reach end of life on May 18, 2021
    */
    static readonly VER_10_0_24: MariaDbEngineVersion;
    /**
     * Version "10.0.28".
     * @deprecated MariaDB 10.0 will reach end of life on May 18, 2021
    */
    static readonly VER_10_0_28: MariaDbEngineVersion;
    /**
     * Version "10.0.31".
     * @deprecated MariaDB 10.0 will reach end of life on May 18, 2021
    */
    static readonly VER_10_0_31: MariaDbEngineVersion;
    /**
     * Version "10.0.32".
     * @deprecated MariaDB 10.0 will reach end of life on May 18, 2021
    */
    static readonly VER_10_0_32: MariaDbEngineVersion;
    /**
     * Version "10.0.34".
     * @deprecated MariaDB 10.0 will reach end of life on May 18, 2021
    */
    static readonly VER_10_0_34: MariaDbEngineVersion;
    /**
     * Version "10.0.35".
     * @deprecated MariaDB 10.0 will reach end of life on May 18, 2021
    */
    static readonly VER_10_0_35: MariaDbEngineVersion;
    /**
     * Version "10.1" (only a major version, without a specific minor version).
     * @deprecated MariaDB 10.1 will reach end of life on May 18, 2021
    */
    static readonly VER_10_1: MariaDbEngineVersion;
    /**
     * Version "10.1.14".
     * @deprecated MariaDB 10.1 will reach end of life on May 18, 2021
    */
    static readonly VER_10_1_14: MariaDbEngineVersion;
    /**
     * Version "10.1.19".
     * @deprecated MariaDB 10.1 will reach end of life on May 18, 2021
    */
    static readonly VER_10_1_19: MariaDbEngineVersion;
    /**
     * Version "10.1.23".
     * @deprecated MariaDB 10.1 will reach end of life on May 18, 2021
    */
    static readonly VER_10_1_23: MariaDbEngineVersion;
    /**
     * Version "10.1.26".
     * @deprecated MariaDB 10.1 will reach end of life on May 18, 2021
    */
    static readonly VER_10_1_26: MariaDbEngineVersion;
    /**
     * Version "10.1.31".
     * @deprecated MariaDB 10.1 will reach end of life on May 18, 2021
    */
    static readonly VER_10_1_31: MariaDbEngineVersion;
    /**
     * Version "10.1.34".
     * @deprecated MariaDB 10.1 will reach end of life on May 18, 2021
    */
    static readonly VER_10_1_34: MariaDbEngineVersion;
    /** Version "10.2" (only a major version, without a specific minor version). */
    static readonly VER_10_2: MariaDbEngineVersion;
    /** Version "10.2.11". */
    static readonly VER_10_2_11: MariaDbEngineVersion;
    /** Version "10.2.12". */
    static readonly VER_10_2_12: MariaDbEngineVersion;
    /** Version "10.2.15". */
    static readonly VER_10_2_15: MariaDbEngineVersion;
    /** Version "10.2.21". */
    static readonly VER_10_2_21: MariaDbEngineVersion;
    /** Version "10.2.32". */
    static readonly VER_10_2_32: MariaDbEngineVersion;
    /** Version "10.2.37". */
    static readonly VER_10_2_37: MariaDbEngineVersion;
    /** Version "10.2.39". */
    static readonly VER_10_2_39: MariaDbEngineVersion;
    /** Version "10.2.40". */
    static readonly VER_10_2_40: MariaDbEngineVersion;
    /** Version "10.2.41". */
    static readonly VER_10_2_41: MariaDbEngineVersion;
    /** Version "10.3" (only a major version, without a specific minor version). */
    static readonly VER_10_3: MariaDbEngineVersion;
    /** Version "10.3.8". */
    static readonly VER_10_3_8: MariaDbEngineVersion;
    /** Version "10.3.13". */
    static readonly VER_10_3_13: MariaDbEngineVersion;
    /** Version "10.3.20". */
    static readonly VER_10_3_20: MariaDbEngineVersion;
    /** Version "10.3.23". */
    static readonly VER_10_3_23: MariaDbEngineVersion;
    /** Version "10.3.28". */
    static readonly VER_10_3_28: MariaDbEngineVersion;
    /** Version "10.3.31". */
    static readonly VER_10_3_31: MariaDbEngineVersion;
    /** Version "10.3.32". */
    static readonly VER_10_3_32: MariaDbEngineVersion;
    /** Version "10.4" (only a major version, without a specific minor version). */
    static readonly VER_10_4: MariaDbEngineVersion;
    /** Version "10.4.8". */
    static readonly VER_10_4_8: MariaDbEngineVersion;
    /** Version "10.4.13". */
    static readonly VER_10_4_13: MariaDbEngineVersion;
    /** Version "10.4.18". */
    static readonly VER_10_4_18: MariaDbEngineVersion;
    /** Version "10.4.21". */
    static readonly VER_10_4_21: MariaDbEngineVersion;
    /** Version "10.4.22". */
    static readonly VER_10_4_22: MariaDbEngineVersion;
    /** Version "10.5" (only a major version, without a specific minor version). */
    static readonly VER_10_5: MariaDbEngineVersion;
    /** Version "10.5.8". */
    static readonly VER_10_5_8: MariaDbEngineVersion;
    /** Version "10.5.9". */
    static readonly VER_10_5_9: MariaDbEngineVersion;
    /** Version "10.5.12". */
    static readonly VER_10_5_12: MariaDbEngineVersion;
    /** Version "10.5.13". */
    static readonly VER_10_5_13: MariaDbEngineVersion;
    /**
     * Create a new MariaDbEngineVersion with an arbitrary version.
     *
     * @param mariaDbFullVersion the full version string,
     *   for example "10.5.28"
     * @param mariaDbMajorVersion the major version of the engine,
     *   for example "10.5"
     */
    static of(mariaDbFullVersion: string, mariaDbMajorVersion: string): MariaDbEngineVersion;
    /** The full version string, for example, "10.5.28". */
    readonly mariaDbFullVersion: string;
    /** The major version of the engine, for example, "10.5". */
    readonly mariaDbMajorVersion: string;
    private constructor();
}
/**
 * Properties for MariaDB instance engines.
 * Used in {@link DatabaseInstanceEngine.mariaDb}.
 */
export interface MariaDbInstanceEngineProps {
    /** The exact version of the engine to use. */
    readonly version: MariaDbEngineVersion;
}
/**
 * The versions for the MySQL instance engines
 * (those returned by {@link DatabaseInstanceEngine.mysql}).
 */
export declare class MysqlEngineVersion {
    /**
     * Version "5.5" (only a major version, without a specific minor version).
     * @deprecated MySQL 5.5 will reach end of life on May 25, 2021
    */
    static readonly VER_5_5: MysqlEngineVersion;
    /**
     * Version "5.5.46".
     * @deprecated MySQL 5.5 will reach end of life on May 25, 2021
    */
    static readonly VER_5_5_46: MysqlEngineVersion;
    /**
     * Version "5.5.53".
     * @deprecated MySQL 5.5 will reach end of life on May 25, 2021
    */
    static readonly VER_5_5_53: MysqlEngineVersion;
    /**
     * Version "5.5.57".
     * @deprecated MySQL 5.5 will reach end of life on May 25, 2021
    */
    static readonly VER_5_5_57: MysqlEngineVersion;
    /**
     * Version "5.5.59".
     * @deprecated MySQL 5.5 will reach end of life on May 25, 2021
    */
    static readonly VER_5_5_59: MysqlEngineVersion;
    /**
     * Version "5.5.61".
     * @deprecated MySQL 5.5 will reach end of life on May 25, 2021
    */
    static readonly VER_5_5_61: MysqlEngineVersion;
    /**
     * Version "5.6" (only a major version, without a specific minor version).
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6: MysqlEngineVersion;
    /**
     * Version "5.6.34".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_34: MysqlEngineVersion;
    /**
     * Version "5.6.35".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_35: MysqlEngineVersion;
    /**
     * Version "5.6.37".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_37: MysqlEngineVersion;
    /**
     * Version "5.6.39".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_39: MysqlEngineVersion;
    /**
     * Version "5.6.40".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_40: MysqlEngineVersion;
    /**
     * Version "5.6.41".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_41: MysqlEngineVersion;
    /**
     * Version "5.6.43".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_43: MysqlEngineVersion;
    /**
     * Version "5.6.44".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_44: MysqlEngineVersion;
    /**
     * Version "5.6.46".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_46: MysqlEngineVersion;
    /**
     * Version "5.6.48".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_48: MysqlEngineVersion;
    /**
     * Version "5.6.49".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_49: MysqlEngineVersion;
    /**
     * Version "5.6.51".
     * @deprecated MySQL 5.6 will reach end of life on August 3, 2021
    */
    static readonly VER_5_6_51: MysqlEngineVersion;
    /** Version "5.7" (only a major version, without a specific minor version). */
    static readonly VER_5_7: MysqlEngineVersion;
    /** Version "5.7.16". */
    static readonly VER_5_7_16: MysqlEngineVersion;
    /** Version "5.7.17". */
    static readonly VER_5_7_17: MysqlEngineVersion;
    /** Version "5.7.19". */
    static readonly VER_5_7_19: MysqlEngineVersion;
    /** Version "5.7.21". */
    static readonly VER_5_7_21: MysqlEngineVersion;
    /** Version "5.7.22". */
    static readonly VER_5_7_22: MysqlEngineVersion;
    /** Version "5.7.23". */
    static readonly VER_5_7_23: MysqlEngineVersion;
    /** Version "5.7.24". */
    static readonly VER_5_7_24: MysqlEngineVersion;
    /** Version "5.7.25". */
    static readonly VER_5_7_25: MysqlEngineVersion;
    /** Version "5.7.26". */
    static readonly VER_5_7_26: MysqlEngineVersion;
    /** Version "5.7.28". */
    static readonly VER_5_7_28: MysqlEngineVersion;
    /** Version "5.7.30". */
    static readonly VER_5_7_30: MysqlEngineVersion;
    /** Version "5.7.31". */
    static readonly VER_5_7_31: MysqlEngineVersion;
    /** Version "5.7.33". */
    static readonly VER_5_7_33: MysqlEngineVersion;
    /** Version "5.7.34". */
    static readonly VER_5_7_34: MysqlEngineVersion;
    /** Version "5.7.35". */
    static readonly VER_5_7_35: MysqlEngineVersion;
    /** Version "5.7.36". */
    static readonly VER_5_7_36: MysqlEngineVersion;
    /** Version "5.7.37". */
    static readonly VER_5_7_37: MysqlEngineVersion;
    /** Version "8.0" (only a major version, without a specific minor version). */
    static readonly VER_8_0: MysqlEngineVersion;
    /** Version "8.0.11". */
    static readonly VER_8_0_11: MysqlEngineVersion;
    /** Version "8.0.13". */
    static readonly VER_8_0_13: MysqlEngineVersion;
    /** Version "8.0.15". */
    static readonly VER_8_0_15: MysqlEngineVersion;
    /** Version "8.0.16". */
    static readonly VER_8_0_16: MysqlEngineVersion;
    /** Version "8.0.17". */
    static readonly VER_8_0_17: MysqlEngineVersion;
    /** Version "8.0.19". */
    static readonly VER_8_0_19: MysqlEngineVersion;
    /** Version "8.0.20 ". */
    static readonly VER_8_0_20: MysqlEngineVersion;
    /** Version "8.0.21 ". */
    static readonly VER_8_0_21: MysqlEngineVersion;
    /** Version "8.0.23". */
    static readonly VER_8_0_23: MysqlEngineVersion;
    /** Version "8.0.25". */
    static readonly VER_8_0_25: MysqlEngineVersion;
    /** Version "8.0.26". */
    static readonly VER_8_0_26: MysqlEngineVersion;
    /** Version "8.0.27". */
    static readonly VER_8_0_27: MysqlEngineVersion;
    /** Version "8.0.28". */
    static readonly VER_8_0_28: MysqlEngineVersion;
    /**
     * Create a new MysqlEngineVersion with an arbitrary version.
     *
     * @param mysqlFullVersion the full version string,
     *   for example "8.1.43"
     * @param mysqlMajorVersion the major version of the engine,
     *   for example "8.1"
     */
    static of(mysqlFullVersion: string, mysqlMajorVersion: string): MysqlEngineVersion;
    /** The full version string, for example, "10.5.28". */
    readonly mysqlFullVersion: string;
    /** The major version of the engine, for example, "10.5". */
    readonly mysqlMajorVersion: string;
    private constructor();
}
/**
 * Properties for MySQL instance engines.
 * Used in {@link DatabaseInstanceEngine.mysql}.
 */
export interface MySqlInstanceEngineProps {
    /** The exact version of the engine to use. */
    readonly version: MysqlEngineVersion;
}
/**
 * Features supported by the Postgres database engine
 */
export interface PostgresEngineFeatures {
    /**
     * Whether this version of the Postgres engine supports the S3 data import feature.
     *
     * @default false
     */
    readonly s3Import?: boolean;
    /**
     * Whether this version of the Postgres engine supports the S3 data export feature.
     *
     * @default false
     */
    readonly s3Export?: boolean;
}
/**
 * The versions for the PostgreSQL instance engines
 * (those returned by {@link DatabaseInstanceEngine.postgres}).
 */
export declare class PostgresEngineVersion {
    /**
     * Version "9.5" (only a major version, without a specific minor version).
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5: PostgresEngineVersion;
    /**
     * Version "9.5.2".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_2: PostgresEngineVersion;
    /**
     * Version "9.5.4".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_4: PostgresEngineVersion;
    /**
     * Version "9.5.6".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_6: PostgresEngineVersion;
    /**
     * Version "9.5.7".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_7: PostgresEngineVersion;
    /**
     * Version "9.5.9".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_9: PostgresEngineVersion;
    /**
     * Version "9.5.10".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_10: PostgresEngineVersion;
    /**
     * Version "9.5.12".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_12: PostgresEngineVersion;
    /**
     * Version "9.5.13".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_13: PostgresEngineVersion;
    /**
     * Version "9.5.14".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_14: PostgresEngineVersion;
    /**
     * Version "9.5.15".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_15: PostgresEngineVersion;
    /**
     * Version "9.5.16".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_16: PostgresEngineVersion;
    /**
     * Version "9.5.18".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_18: PostgresEngineVersion;
    /**
     * Version "9.5.19".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_19: PostgresEngineVersion;
    /**
     * Version "9.5.20".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_20: PostgresEngineVersion;
    /**
     * Version "9.5.21".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_21: PostgresEngineVersion;
    /**
     * Version "9.5.22".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_22: PostgresEngineVersion;
    /**
     * Version "9.5.23".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_23: PostgresEngineVersion;
    /**
     * Version "9.5.24".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_24: PostgresEngineVersion;
    /**
     * Version "9.5.25".
     * @deprecated PostgreSQL 9.5 will reach end of life on February 16, 2021
    */
    static readonly VER_9_5_25: PostgresEngineVersion;
    /**
     * Version "9.6" (only a major version, without a specific minor version).
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6: PostgresEngineVersion;
    /**
     * Version "9.6.1".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_1: PostgresEngineVersion;
    /**
     * Version "9.6.2".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_2: PostgresEngineVersion;
    /**
     * Version "9.6.3".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_3: PostgresEngineVersion;
    /**
     * Version "9.6.5".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_5: PostgresEngineVersion;
    /**
     * Version "9.6.6".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_6: PostgresEngineVersion;
    /**
     * Version "9.6.8".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_8: PostgresEngineVersion;
    /**
     * Version "9.6.9".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_9: PostgresEngineVersion;
    /**
     * Version "9.6.10".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_10: PostgresEngineVersion;
    /**
     * Version "9.6.11".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_11: PostgresEngineVersion;
    /**
     * Version "9.6.12".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_12: PostgresEngineVersion;
    /**
     * Version "9.6.14".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_14: PostgresEngineVersion;
    /**
     * Version "9.6.15".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_15: PostgresEngineVersion;
    /**
     * Version "9.6.16".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_16: PostgresEngineVersion;
    /**
     * Version "9.6.17".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_17: PostgresEngineVersion;
    /**
     * Version "9.6.18".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_18: PostgresEngineVersion;
    /**
     * Version "9.6.19".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_19: PostgresEngineVersion;
    /**
     * Version "9.6.20".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_20: PostgresEngineVersion;
    /**
     * Version "9.6.21".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_21: PostgresEngineVersion;
    /**
     * Version "9.6.22".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_22: PostgresEngineVersion;
    /**
     * Version "9.6.23".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_23: PostgresEngineVersion;
    /**
     * Version "9.6.24".
     * @deprecated PostgreSQL 9.6 will reach end of life in November 2021
    */
    static readonly VER_9_6_24: PostgresEngineVersion;
    /** Version "10" (only a major version, without a specific minor version). */
    static readonly VER_10: PostgresEngineVersion;
    /** Version "10.1". */
    static readonly VER_10_1: PostgresEngineVersion;
    /** Version "10.3". */
    static readonly VER_10_3: PostgresEngineVersion;
    /** Version "10.4". */
    static readonly VER_10_4: PostgresEngineVersion;
    /** Version "10.5". */
    static readonly VER_10_5: PostgresEngineVersion;
    /** Version "10.6". */
    static readonly VER_10_6: PostgresEngineVersion;
    /** Version "10.7". */
    static readonly VER_10_7: PostgresEngineVersion;
    /** Version "10.9". */
    static readonly VER_10_9: PostgresEngineVersion;
    /** Version "10.10". */
    static readonly VER_10_10: PostgresEngineVersion;
    /** Version "10.11". */
    static readonly VER_10_11: PostgresEngineVersion;
    /** Version "10.12". */
    static readonly VER_10_12: PostgresEngineVersion;
    /** Version "10.13". */
    static readonly VER_10_13: PostgresEngineVersion;
    /** Version "10.14". */
    static readonly VER_10_14: PostgresEngineVersion;
    /** Version "10.15". */
    static readonly VER_10_15: PostgresEngineVersion;
    /** Version "10.16". */
    static readonly VER_10_16: PostgresEngineVersion;
    /** Version "10.17". */
    static readonly VER_10_17: PostgresEngineVersion;
    /** Version "10.18". */
    static readonly VER_10_18: PostgresEngineVersion;
    /** Version "10.19". */
    static readonly VER_10_19: PostgresEngineVersion;
    /** Version "10.20". */
    static readonly VER_10_20: PostgresEngineVersion;
    /** Version "11" (only a major version, without a specific minor version). */
    static readonly VER_11: PostgresEngineVersion;
    /** Version "11.1". */
    static readonly VER_11_1: PostgresEngineVersion;
    /** Version "11.2". */
    static readonly VER_11_2: PostgresEngineVersion;
    /** Version "11.4". */
    static readonly VER_11_4: PostgresEngineVersion;
    /** Version "11.5". */
    static readonly VER_11_5: PostgresEngineVersion;
    /** Version "11.6". */
    static readonly VER_11_6: PostgresEngineVersion;
    /** Version "11.7". */
    static readonly VER_11_7: PostgresEngineVersion;
    /** Version "11.8". */
    static readonly VER_11_8: PostgresEngineVersion;
    /** Version "11.9". */
    static readonly VER_11_9: PostgresEngineVersion;
    /** Version "11.10". */
    static readonly VER_11_10: PostgresEngineVersion;
    /** Version "11.11". */
    static readonly VER_11_11: PostgresEngineVersion;
    /** Version "11.12". */
    static readonly VER_11_12: PostgresEngineVersion;
    /** Version "11.13". */
    static readonly VER_11_13: PostgresEngineVersion;
    /** Version "11.14". */
    static readonly VER_11_14: PostgresEngineVersion;
    /** Version "11.15". */
    static readonly VER_11_15: PostgresEngineVersion;
    /** Version "12" (only a major version, without a specific minor version). */
    static readonly VER_12: PostgresEngineVersion;
    /** Version "12.2". */
    static readonly VER_12_2: PostgresEngineVersion;
    /** Version "12.3". */
    static readonly VER_12_3: PostgresEngineVersion;
    /** Version "12.4". */
    static readonly VER_12_4: PostgresEngineVersion;
    /** Version "12.5". */
    static readonly VER_12_5: PostgresEngineVersion;
    /** Version "12.6". */
    static readonly VER_12_6: PostgresEngineVersion;
    /** Version "12.7". */
    static readonly VER_12_7: PostgresEngineVersion;
    /** Version "12.8". */
    static readonly VER_12_8: PostgresEngineVersion;
    /** Version "12.9". */
    static readonly VER_12_9: PostgresEngineVersion;
    /** Version "12.10". */
    static readonly VER_12_10: PostgresEngineVersion;
    /** Version "13" (only a major version, without a specific minor version). */
    static readonly VER_13: PostgresEngineVersion;
    /** Version "13.1". */
    static readonly VER_13_1: PostgresEngineVersion;
    /** Version "13.2". */
    static readonly VER_13_2: PostgresEngineVersion;
    /** Version "13.3". */
    static readonly VER_13_3: PostgresEngineVersion;
    /** Version "13.4". */
    static readonly VER_13_4: PostgresEngineVersion;
    /** Version "13.5". */
    static readonly VER_13_5: PostgresEngineVersion;
    /** Version "13.6". */
    static readonly VER_13_6: PostgresEngineVersion;
    /** Version "14" (only a major version, without a specific minor version). */
    static readonly VER_14: PostgresEngineVersion;
    /** Version "14.1". */
    static readonly VER_14_1: PostgresEngineVersion;
    /** Version "14.2". */
    static readonly VER_14_2: PostgresEngineVersion;
    /**
     * Create a new PostgresEngineVersion with an arbitrary version.
     *
     * @param postgresFullVersion the full version string,
     *   for example "13.11"
     * @param postgresMajorVersion the major version of the engine,
     *   for example "13"
     */
    static of(postgresFullVersion: string, postgresMajorVersion: string, postgresFeatures?: PostgresEngineFeatures): PostgresEngineVersion;
    /** The full version string, for example, "13.11". */
    readonly postgresFullVersion: string;
    /** The major version of the engine, for example, "13". */
    readonly postgresMajorVersion: string;
    /**
     * The supported features for the DB engine
     * @internal
     */
    readonly _features: InstanceEngineFeatures;
    private constructor();
}
/**
 * Properties for PostgreSQL instance engines.
 * Used in {@link DatabaseInstanceEngine.postgres}.
 */
export interface PostgresInstanceEngineProps {
    /** The exact version of the engine to use. */
    readonly version: PostgresEngineVersion;
}
/**
 * The versions for the legacy Oracle instance engines
 * (those returned by {@link DatabaseInstanceEngine.oracleSe}
 * and {@link DatabaseInstanceEngine.oracleSe1}).
 * Note: RDS will stop allowing creating new databases with this version in August 2020.
 *
 * @deprecated instances can no longer be created with these engine versions. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
export declare class OracleLegacyEngineVersion {
    /** Version "11.2" (only a major version, without a specific minor version). */
    static readonly VER_11_2: OracleLegacyEngineVersion;
    /** Version "11.2.0.2.v2". */
    static readonly VER_11_2_0_2_V2: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v1". */
    static readonly VER_11_2_0_4_V1: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v3". */
    static readonly VER_11_2_0_4_V3: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v4". */
    static readonly VER_11_2_0_4_V4: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v5". */
    static readonly VER_11_2_0_4_V5: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v6". */
    static readonly VER_11_2_0_4_V6: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v7". */
    static readonly VER_11_2_0_4_V7: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v8". */
    static readonly VER_11_2_0_4_V8: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v9". */
    static readonly VER_11_2_0_4_V9: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v10". */
    static readonly VER_11_2_0_4_V10: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v11". */
    static readonly VER_11_2_0_4_V11: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v12". */
    static readonly VER_11_2_0_4_V12: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v13". */
    static readonly VER_11_2_0_4_V13: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v14". */
    static readonly VER_11_2_0_4_V14: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v15". */
    static readonly VER_11_2_0_4_V15: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v16". */
    static readonly VER_11_2_0_4_V16: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v17". */
    static readonly VER_11_2_0_4_V17: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v18". */
    static readonly VER_11_2_0_4_V18: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v19". */
    static readonly VER_11_2_0_4_V19: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v20". */
    static readonly VER_11_2_0_4_V20: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v21". */
    static readonly VER_11_2_0_4_V21: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v22". */
    static readonly VER_11_2_0_4_V22: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v23". */
    static readonly VER_11_2_0_4_V23: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v24". */
    static readonly VER_11_2_0_4_V24: OracleLegacyEngineVersion;
    /** Version "11.2.0.4.v25". */
    static readonly VER_11_2_0_4_V25: OracleLegacyEngineVersion;
    private static of;
    /** The full version string, for example, "11.2.0.4.v24". */
    readonly oracleLegacyFullVersion: string;
    /** The major version of the engine, for example, "11.2". */
    readonly oracleLegacyMajorVersion: string;
    private constructor();
}
/**
 * The versions for the Oracle instance engines
 * (those returned by {@link DatabaseInstanceEngine.oracleSe2} and
 * {@link DatabaseInstanceEngine.oracleEe}).
 */
export declare class OracleEngineVersion {
    /** Version "12.1" (only a major version, without a specific minor version). */
    static readonly VER_12_1: OracleEngineVersion;
    /** Version "12.1.0.2.v1". */
    static readonly VER_12_1_0_2_V1: OracleEngineVersion;
    /** Version "12.1.0.2.v2". */
    static readonly VER_12_1_0_2_V2: OracleEngineVersion;
    /** Version "12.1.0.2.v3". */
    static readonly VER_12_1_0_2_V3: OracleEngineVersion;
    /** Version "12.1.0.2.v4". */
    static readonly VER_12_1_0_2_V4: OracleEngineVersion;
    /** Version "12.1.0.2.v5". */
    static readonly VER_12_1_0_2_V5: OracleEngineVersion;
    /** Version "12.1.0.2.v6". */
    static readonly VER_12_1_0_2_V6: OracleEngineVersion;
    /** Version "12.1.0.2.v7". */
    static readonly VER_12_1_0_2_V7: OracleEngineVersion;
    /** Version "12.1.0.2.v8". */
    static readonly VER_12_1_0_2_V8: OracleEngineVersion;
    /** Version "12.1.0.2.v9". */
    static readonly VER_12_1_0_2_V9: OracleEngineVersion;
    /** Version "12.1.0.2.v10". */
    static readonly VER_12_1_0_2_V10: OracleEngineVersion;
    /** Version "12.1.0.2.v11". */
    static readonly VER_12_1_0_2_V11: OracleEngineVersion;
    /** Version "12.1.0.2.v12". */
    static readonly VER_12_1_0_2_V12: OracleEngineVersion;
    /** Version "12.1.0.2.v13". */
    static readonly VER_12_1_0_2_V13: OracleEngineVersion;
    /** Version "12.1.0.2.v14". */
    static readonly VER_12_1_0_2_V14: OracleEngineVersion;
    /** Version "12.1.0.2.v15". */
    static readonly VER_12_1_0_2_V15: OracleEngineVersion;
    /** Version "12.1.0.2.v16". */
    static readonly VER_12_1_0_2_V16: OracleEngineVersion;
    /** Version "12.1.0.2.v17". */
    static readonly VER_12_1_0_2_V17: OracleEngineVersion;
    /** Version "12.1.0.2.v18". */
    static readonly VER_12_1_0_2_V18: OracleEngineVersion;
    /** Version "12.1.0.2.v19". */
    static readonly VER_12_1_0_2_V19: OracleEngineVersion;
    /** Version "12.1.0.2.v20". */
    static readonly VER_12_1_0_2_V20: OracleEngineVersion;
    /** Version "12.1.0.2.v21". */
    static readonly VER_12_1_0_2_V21: OracleEngineVersion;
    /** Version "12.1.0.2.v22". */
    static readonly VER_12_1_0_2_V22: OracleEngineVersion;
    /** Version "12.1.0.2.v23". */
    static readonly VER_12_1_0_2_V23: OracleEngineVersion;
    /** Version "12.1.0.2.v24". */
    static readonly VER_12_1_0_2_V24: OracleEngineVersion;
    /** Version "12.2" (only a major version, without a specific minor version). */
    static readonly VER_12_2: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2018-10.rur-2018-10.r1". */
    static readonly VER_12_2_0_1_2018_10_R1: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2019-01.rur-2019-01.r1". */
    static readonly VER_12_2_0_1_2019_01_R1: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2019-04.rur-2019-04.r1". */
    static readonly VER_12_2_0_1_2019_04_R1: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2019-07.rur-2019-07.r1". */
    static readonly VER_12_2_0_1_2019_07_R1: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2019-10.rur-2019-10.r1". */
    static readonly VER_12_2_0_1_2019_10_R1: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2020-01.rur-2020-01.r1". */
    static readonly VER_12_2_0_1_2020_01_R1: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2020-04.rur-2020-04.r1". */
    static readonly VER_12_2_0_1_2020_04_R1: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2020-07.rur-2020-07.r1". */
    static readonly VER_12_2_0_1_2020_07_R1: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2021-10.rur-2020-10.r1". */
    static readonly VER_12_2_0_1_2020_10_R1: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2021-01.rur-2021-01.r1". */
    static readonly VER_12_2_0_1_2021_01_R1: OracleEngineVersion;
    /** Version "12.2.0.1.ru-2021-04.rur-2021-04.r1". */
    static readonly VER_12_2_0_1_2021_04_R1: OracleEngineVersion;
    /** Version "18" (only a major version, without a specific minor version). */
    static readonly VER_18: OracleEngineVersion;
    /** Version "18.0.0.0.ru-2019-07.rur-2019-07.r1". */
    static readonly VER_18_0_0_0_2019_07_R1: OracleEngineVersion;
    /** Version "18.0.0.0.ru-2019-10.rur-2019-10.r1". */
    static readonly VER_18_0_0_0_2019_10_R1: OracleEngineVersion;
    /** Version "18.0.0.0.ru-2020-01.rur-2020-01.r1". */
    static readonly VER_18_0_0_0_2020_01_R1: OracleEngineVersion;
    /** Version "18.0.0.0.ru-2020-04.rur-2020-04.r1". */
    static readonly VER_18_0_0_0_2020_04_R1: OracleEngineVersion;
    /** Version "18.0.0.0.ru-2020-07.rur-2020-07.r1". */
    static readonly VER_18_0_0_0_2020_07_R1: OracleEngineVersion;
    /** Version "19" (only a major version, without a specific minor version). */
    static readonly VER_19: OracleEngineVersion;
    /** Version "19.0.0.0.ru-2019-07.rur-2019-07.r1". */
    static readonly VER_19_0_0_0_2019_07_R1: OracleEngineVersion;
    /** Version "19.0.0.0.ru-2019-10.rur-2019-10.r1". */
    static readonly VER_19_0_0_0_2019_10_R1: OracleEngineVersion;
    /** Version "19.0.0.0.ru-2020-01.rur-2020-01.r1". */
    static readonly VER_19_0_0_0_2020_01_R1: OracleEngineVersion;
    /** Version "19.0.0.0.ru-2020-04.rur-2020-04.r1". */
    static readonly VER_19_0_0_0_2020_04_R1: OracleEngineVersion;
    /** Version "19.0.0.0.ru-2020-07.rur-2020-07.r1". */
    static readonly VER_19_0_0_0_2020_07_R1: OracleEngineVersion;
    /** Version "19.0.0.0.ru-2020-07.rur-2020-10.r1". */
    static readonly VER_19_0_0_0_2020_10_R1: OracleEngineVersion;
    /** Version "19.0.0.0.ru-2021-01.rur-2021-01.r1". */
    static readonly VER_19_0_0_0_2021_01_R1: OracleEngineVersion;
    /** Version "19.0.0.0.ru-2021-01.rur-2021-01.r2". */
    static readonly VER_19_0_0_0_2021_01_R2: OracleEngineVersion;
    /** Version "19.0.0.0.ru-2021-01.rur-2021-04.r1". */
    static readonly VER_19_0_0_0_2021_04_R1: OracleEngineVersion;
    /**
     * Creates a new OracleEngineVersion with an arbitrary version.
     *
     * @param oracleFullVersion the full version string,
     *   for example "19.0.0.0.ru-2019-10.rur-2019-10.r1"
     * @param oracleMajorVersion the major version of the engine,
     *   for example "19"
     */
    static of(oracleFullVersion: string, oracleMajorVersion: string): OracleEngineVersion;
    /** The full version string, for example, "19.0.0.0.ru-2019-10.rur-2019-10.r1". */
    readonly oracleFullVersion: string;
    /** The major version of the engine, for example, "19". */
    readonly oracleMajorVersion: string;
    private constructor();
}
interface OracleInstanceEngineProps {
    /** The exact version of the engine to use. */
    readonly version: OracleEngineVersion;
}
/**
 * Properties for Oracle Standard Edition instance engines.
 * Used in {@link DatabaseInstanceEngine.oracleSe}.
 *
 * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
export interface OracleSeInstanceEngineProps {
    /** The exact version of the engine to use. */
    readonly version: OracleLegacyEngineVersion;
}
/**
 * Properties for Oracle Standard Edition 1 instance engines.
 * Used in {@link DatabaseInstanceEngine.oracleSe1}.
 *
 * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
export interface OracleSe1InstanceEngineProps {
    /** The exact version of the engine to use. */
    readonly version: OracleLegacyEngineVersion;
}
/**
 * Properties for Oracle Standard Edition 2 instance engines.
 * Used in {@link DatabaseInstanceEngine.oracleSe2}.
 */
export interface OracleSe2InstanceEngineProps extends OracleInstanceEngineProps {
}
/**
 * Properties for Oracle Enterprise Edition instance engines.
 * Used in {@link DatabaseInstanceEngine.oracleEe}.
 */
export interface OracleEeInstanceEngineProps extends OracleInstanceEngineProps {
}
/**
 * The versions for the SQL Server instance engines
 * (those returned by {@link DatabaseInstanceEngine.sqlServerSe},
 * {@link DatabaseInstanceEngine.sqlServerEx}, {@link DatabaseInstanceEngine.sqlServerWeb}
 * and {@link DatabaseInstanceEngine.sqlServerEe}).
 */
export declare class SqlServerEngineVersion {
    /** Version "11.00" (only a major version, without a specific minor version). */
    static readonly VER_11: SqlServerEngineVersion;
    /** Version "11.00.5058.0.v1". */
    static readonly VER_11_00_5058_0_V1: SqlServerEngineVersion;
    /** Version "11.00.6020.0.v1". */
    static readonly VER_11_00_6020_0_V1: SqlServerEngineVersion;
    /** Version "11.00.6594.0.v1". */
    static readonly VER_11_00_6594_0_V1: SqlServerEngineVersion;
    /** Version "11.00.7462.6.v1". */
    static readonly VER_11_00_7462_6_V1: SqlServerEngineVersion;
    /** Version "11.00.7493.4.v1". */
    static readonly VER_11_00_7493_4_V1: SqlServerEngineVersion;
    /** Version "12.00" (only a major version, without a specific minor version). */
    static readonly VER_12: SqlServerEngineVersion;
    /** Version "12.00.5000.0.v1". */
    static readonly VER_12_00_5000_0_V1: SqlServerEngineVersion;
    /** Version "12.00.5546.0.v1". */
    static readonly VER_12_00_5546_0_V1: SqlServerEngineVersion;
    /** Version "12.00.5571.0.v1". */
    static readonly VER_12_00_5571_0_V1: SqlServerEngineVersion;
    /** Version "12.00.6293.0.v1". */
    static readonly VER_12_00_6293_0_V1: SqlServerEngineVersion;
    /** Version "12.00.6329.1.v1". */
    static readonly VER_12_00_6329_1_V1: SqlServerEngineVersion;
    /** Version "13.00" (only a major version, without a specific minor version). */
    static readonly VER_13: SqlServerEngineVersion;
    /** Version "13.00.2164.0.v1". */
    static readonly VER_13_00_2164_0_V1: SqlServerEngineVersion;
    /** Version "13.00.4422.0.v1". */
    static readonly VER_13_00_4422_0_V1: SqlServerEngineVersion;
    /** Version "13.00.4451.0.v1". */
    static readonly VER_13_00_4451_0_V1: SqlServerEngineVersion;
    /** Version "13.00.4466.4.v1". */
    static readonly VER_13_00_4466_4_V1: SqlServerEngineVersion;
    /** Version "13.00.4522.0.v1". */
    static readonly VER_13_00_4522_0_V1: SqlServerEngineVersion;
    /** Version "13.00.5216.0.v1". */
    static readonly VER_13_00_5216_0_V1: SqlServerEngineVersion;
    /** Version "13.00.5292.0.v1". */
    static readonly VER_13_00_5292_0_V1: SqlServerEngineVersion;
    /** Version "13.00.5366.0.v1". */
    static readonly VER_13_00_5366_0_V1: SqlServerEngineVersion;
    /** Version "13.00.5426.0.v1". */
    static readonly VER_13_00_5426_0_V1: SqlServerEngineVersion;
    /** Version "13.00.5598.27.v1". */
    static readonly VER_13_00_5598_27_V1: SqlServerEngineVersion;
    /** Version "13.00.5820.21.v1". */
    static readonly VER_13_00_5820_21_V1: SqlServerEngineVersion;
    /** Version "13.00.5850.14.v1". */
    static readonly VER_13_00_5850_14_V1: SqlServerEngineVersion;
    /** Version "13.00.5882.1.v1". */
    static readonly VER_13_00_5882_1_V1: SqlServerEngineVersion;
    /** Version "14.00" (only a major version, without a specific minor version). */
    static readonly VER_14: SqlServerEngineVersion;
    /** Version "14.00.1000.169.v1". */
    static readonly VER_14_00_1000_169_V1: SqlServerEngineVersion;
    /** Version "14.00.3015.40.v1". */
    static readonly VER_14_00_3015_40_V1: SqlServerEngineVersion;
    /** Version "14.00.3035.2.v1". */
    static readonly VER_14_00_3035_2_V1: SqlServerEngineVersion;
    /** Version "14.00.3049.1.v1". */
    static readonly VER_14_00_3049_1_V1: SqlServerEngineVersion;
    /**
     * Version "14.00.3192.2.v1".
     * @deprecated SQL Server version 14.00.3192.2.v1 reached end of life
    */
    static readonly VER_14_00_3192_2_V1: SqlServerEngineVersion;
    /** Version "14.00.3223.3.v1". */
    static readonly VER_14_00_3223_3_V1: SqlServerEngineVersion;
    /** Version "14.00.3281.6.v1". */
    static readonly VER_14_00_3281_6_V1: SqlServerEngineVersion;
    /** Version "14.00.3294.2.v1". */
    static readonly VER_14_00_3294_2_V1: SqlServerEngineVersion;
    /** Version "14.00.3356.20.v1". */
    static readonly VER_14_00_3356_20_V1: SqlServerEngineVersion;
    /** Version "14.00.3381.3.v1". */
    static readonly VER_14_00_3381_3_V1: SqlServerEngineVersion;
    /** Version "15.00" (only a major version, without a specific minor version). */
    static readonly VER_15: SqlServerEngineVersion;
    /** Version "15.00.4043.16.v1". */
    static readonly VER_15_00_4043_16_V1: SqlServerEngineVersion;
    /**
     * Version "15.00.4043.23.v1".
     * @deprecated This version is erroneous. You might be looking for {@link SqlServerEngineVersion.VER_15_00_4073_23_V1}, instead.
     */
    static readonly VER_15_00_4043_23_V1: SqlServerEngineVersion;
    /** Version "15.00.4073.23.v1". */
    static readonly VER_15_00_4073_23_V1: SqlServerEngineVersion;
    /**
     * Create a new SqlServerEngineVersion with an arbitrary version.
     *
     * @param sqlServerFullVersion the full version string,
     *   for example "15.00.3049.1.v1"
     * @param sqlServerMajorVersion the major version of the engine,
     *   for example "15.00"
     */
    static of(sqlServerFullVersion: string, sqlServerMajorVersion: string): SqlServerEngineVersion;
    /** The full version string, for example, "15.00.3049.1.v1". */
    readonly sqlServerFullVersion: string;
    /** The major version of the engine, for example, "15.00". */
    readonly sqlServerMajorVersion: string;
    private constructor();
}
interface SqlServerInstanceEngineProps {
    /** The exact version of the engine to use. */
    readonly version: SqlServerEngineVersion;
}
/**
 * Properties for SQL Server Standard Edition instance engines.
 * Used in {@link DatabaseInstanceEngine.sqlServerSe}.
 */
export interface SqlServerSeInstanceEngineProps extends SqlServerInstanceEngineProps {
}
/**
 * Properties for SQL Server Express Edition instance engines.
 * Used in {@link DatabaseInstanceEngine.sqlServerEx}.
 */
export interface SqlServerExInstanceEngineProps extends SqlServerInstanceEngineProps {
}
/**
 * Properties for SQL Server Web Edition instance engines.
 * Used in {@link DatabaseInstanceEngine.sqlServerWeb}.
 */
export interface SqlServerWebInstanceEngineProps extends SqlServerInstanceEngineProps {
}
/**
 * Properties for SQL Server Enterprise Edition instance engines.
 * Used in {@link DatabaseInstanceEngine.sqlServerEe}.
 */
export interface SqlServerEeInstanceEngineProps extends SqlServerInstanceEngineProps {
}
/**
 * A database instance engine. Provides mapping to DatabaseEngine used for
 * secret rotation.
 */
export declare class DatabaseInstanceEngine {
    /**
     * The unversioned 'mariadb' instance engine.
     *
     * NOTE: using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link mariaDb()} method
     */
    static readonly MARIADB: IInstanceEngine;
    /**
     * The unversioned 'mysql' instance engine.
     *
     * NOTE: using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link mysql()} method
     */
    static readonly MYSQL: IInstanceEngine;
    /**
     * The unversioned 'oracle-ee' instance engine.
     *
     * NOTE: using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link oracleEe()} method
     */
    static readonly ORACLE_EE: IInstanceEngine;
    /**
     * The unversioned 'oracle-se2' instance engine.
     *
     * NOTE: using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link oracleSe2()} method
     */
    static readonly ORACLE_SE2: IInstanceEngine;
    /**
     * The unversioned 'oracle-se1' instance engine.
     *
     * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
     */
    static readonly ORACLE_SE1: IInstanceEngine;
    /**
     * The unversioned 'oracle-se' instance engine.
     *
     * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
     */
    static readonly ORACLE_SE: IInstanceEngine;
    /**
     * The unversioned 'postgres' instance engine.
     *
     * NOTE: using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link postgres()} method
     */
    static readonly POSTGRES: IInstanceEngine;
    /**
     * The unversioned 'sqlserver-ee' instance engine.
     *
     * NOTE: using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link sqlServerEe()} method
     */
    static readonly SQL_SERVER_EE: IInstanceEngine;
    /**
     * The unversioned 'sqlserver-se' instance engine.
     *
     * NOTE: using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link sqlServerSe()} method
     */
    static readonly SQL_SERVER_SE: IInstanceEngine;
    /**
     * The unversioned 'sqlserver-ex' instance engine.
     *
     * NOTE: using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link sqlServerEx()} method
     */
    static readonly SQL_SERVER_EX: IInstanceEngine;
    /**
     * The unversioned 'sqlserver-web' instance engine.
     *
     * NOTE: using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link sqlServerWeb()} method
     */
    static readonly SQL_SERVER_WEB: IInstanceEngine;
    /** Creates a new MariaDB instance engine. */
    static mariaDb(props: MariaDbInstanceEngineProps): IInstanceEngine;
    /** Creates a new MySQL instance engine. */
    static mysql(props: MySqlInstanceEngineProps): IInstanceEngine;
    /** Creates a new PostgreSQL instance engine. */
    static postgres(props: PostgresInstanceEngineProps): IInstanceEngine;
    /**
     * Creates a new Oracle Standard Edition instance engine.
     * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
     */
    static oracleSe(props: OracleSeInstanceEngineProps): IInstanceEngine;
    /**
     * Creates a new Oracle Standard Edition 1 instance engine.
     * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
     */
    static oracleSe1(props: OracleSe1InstanceEngineProps): IInstanceEngine;
    /** Creates a new Oracle Standard Edition 1 instance engine. */
    static oracleSe2(props: OracleSe2InstanceEngineProps): IInstanceEngine;
    /** Creates a new Oracle Enterprise Edition instance engine. */
    static oracleEe(props: OracleEeInstanceEngineProps): IInstanceEngine;
    /** Creates a new SQL Server Standard Edition instance engine. */
    static sqlServerSe(props: SqlServerSeInstanceEngineProps): IInstanceEngine;
    /** Creates a new SQL Server Express Edition instance engine. */
    static sqlServerEx(props: SqlServerExInstanceEngineProps): IInstanceEngine;
    /** Creates a new SQL Server Web Edition instance engine. */
    static sqlServerWeb(props: SqlServerWebInstanceEngineProps): IInstanceEngine;
    /** Creates a new SQL Server Enterprise Edition instance engine. */
    static sqlServerEe(props: SqlServerEeInstanceEngineProps): IInstanceEngine;
}
export {};
