"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ifw = require(".");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_INTEG_REGION || process.env.CDK_DEFAULT_REGION || 'us-east-1',
            account: process.env.CDK_INTEG_ACCOUNT || process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'integ-stack', { env });
        const databaseName = 'FleetWise';
        const tableName = 'FleetWise';
        const database = new aws_cdk_lib_1.aws_timestream.CfnDatabase(stack, 'Database', {
            databaseName,
        });
        const table = new aws_cdk_lib_1.aws_timestream.CfnTable(stack, 'Table', {
            databaseName,
            tableName,
        });
        table.node.addDependency(database);
        const role = new aws_cdk_lib_1.aws_iam.Role(stack, 'Role', {
            roleName: 'iotfleetwise-role',
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('iotfleetwise.amazonaws.com'),
        });
        role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                'timestream:WriteRecords',
                'timestream:Select',
            ],
            resources: ['*'],
        }));
        role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                'timestream:DescribeEndpoints',
            ],
            resources: ['*'],
        }));
        const signalCatalog = new ifw.SignalCatalog(stack, 'SignalCatalog', {
            database,
            table,
            role,
            description: 'my signal catalog',
            nodes: [
                new ifw.SignalCatalogBranch('Vehicle'),
                new ifw.SignalCatalogSensor('Vehicle.EngineTorque', 'DOUBLE'),
            ],
        });
        const model_a = new ifw.VehicleModel(stack, 'ModelA', {
            signalCatalog,
            name: 'modelA',
            description: 'Model A vehicle',
            networkInterfaces: [new ifw.CanVehicleInterface('1', 'vcan0')],
            signals: [
                new ifw.CanVehicleSignal('Vehicle.EngineTorque', '1', 401, // messageId
                1.0, // factor
                true, // isBigEndian
                false, // isSigned
                8, // length
                0.0, // offset
                9),
            ],
        });
        const vin100 = new ifw.Vehicle(stack, 'vin100', {
            vehicleName: 'vin100',
            vehicleModel: model_a,
            createIotThing: true,
        });
        const vpc = aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(stack, 'VPC', { isDefault: true });
        const securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(stack, 'SecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        securityGroup.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.anyIpv4(), aws_cdk_lib_1.aws_ec2.Port.tcp(22), 'SSH access');
        securityGroup.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.anyIpv4(), aws_cdk_lib_1.aws_ec2.Port.allIcmp(), 'ping');
        // EC2 role
        const ec2_role = new aws_cdk_lib_1.aws_iam.Role(stack, 'ec2Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            ],
        });
        ec2_role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                's3:List*',
                's3:Get*',
            ],
            resources: ['arn:aws:s3:::*'],
        }));
        // Ubuntu 18.04 for Arm64
        const machineImage = aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter('/aws/service/canonical/ubuntu/server/18.04/stable/current/arm64/hvm/ebs-gp2/ami-id', { os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX });
        // Create the Vehicle simulator
        const keyName = stack.node.tryGetContext('key_name');
        const instance = new aws_cdk_lib_1.aws_ec2.Instance(stack, 'VehicleSim', {
            vpc: vpc,
            instanceType: new aws_cdk_lib_1.aws_ec2.InstanceType('m6g.xlarge'),
            machineImage,
            securityGroup,
            keyName,
            role: ec2_role,
            vpcSubnets: {
                subnetGroupName: 'Public',
            },
            resourceSignalTimeout: cdk.Duration.minutes(30),
        });
        const sourceUrl = 'https://github.com/aws/aws-iot-fleetwise-edge.git';
        const userData = `\
    set -euo pipefail

    # Wait for any existing package install to finish
    i=0
    while true; do
        if sudo fuser /var/{lib/{dpkg,apt/lists},cache/apt/archives}/lock >/dev/null 2>&1; then
            i=0
        else
            i=\`expr $i + 1\`
            if expr $i \\>= 10 > /dev/null; then
                break
            fi
        fi
        sleep 1
    done

    # Upgrade system and reboot if required
    apt update && apt upgrade -y
    if [ -f /var/run/reboot-required ]; then
      # Delete the UserData info file so that we run again after reboot
      rm -f /var/lib/cloud/instances/*/sem/config_scripts_user
      reboot
      exit
    fi

    # Install helper scripts:
    apt update && apt install -y python3-setuptools
    mkdir -p /opt/aws/bin
    wget https://s3.amazonaws.com/cloudformation-examples/aws-cfn-bootstrap-py3-latest.tar.gz
    python3 -m easy_install --script-dir /opt/aws/bin aws-cfn-bootstrap-py3-latest.tar.gz
    rm aws-cfn-bootstrap-py3-latest.tar.gz

    # On error, signal back to cfn:
    error_handler() {
      /opt/aws/bin/cfn-signal --success false --stack ${stack.stackName} --resource ${instance.instance.logicalId} --region ${stack.region}
    }
    trap error_handler ERR

    # Increase pid_max:
    echo 1048576 > /proc/sys/kernel/pid_max
    # Disable syslog:
    systemctl stop syslog.socket rsyslog.service
    # Remove journald rate limiting and set max size:
    printf "RateLimitBurst=0\nSystemMaxUse=1G\n" >> /etc/systemd/journald.conf

    # Install packages
    apt update && apt install -y git ec2-instance-connect htop jq unzip

    # Install AWS CLI:
    curl "https://awscli.amazonaws.com/awscli-exe-linux-aarch64.zip" -o "awscliv2.zip"
    unzip awscliv2.zip
    ./aws/install
    rm awscliv2.zip

    # Download source
    cd /home/ubuntu
    if echo ${sourceUrl} | grep -q 's3://'; then
      sudo -u ubuntu aws s3 cp ${sourceUrl} aws-iot-fleetwise-edge.zip
      sudo -u ubuntu unzip aws-iot-fleetwise-edge.zip -d aws-iot-fleetwise-edge
    else
      sudo -u ubuntu git clone ${sourceUrl} aws-iot-fleetwise-edge
    fi
    cd aws-iot-fleetwise-edge
    
    # Install SocketCAN modules:
    ./tools/install-socketcan.sh --bus-count 1
    
    # Install CAN Simulator
    ./tools/install-cansim.sh --bus-count 1
    
    # Install FWE credentials and config file
    mkdir /etc/aws-iot-fleetwise
    echo -n "${vin100.certificatePem}" > /etc/aws-iot-fleetwise/certificate.pem
    echo -n "${vin100.privateKey}" > /etc/aws-iot-fleetwise/private-key.key
    ./tools/configure-fwe.sh \
      --input-config-file "configuration/static-config.json" \
      --output-config-file "/etc/aws-iot-fleetwise/config-0.json" \
      --vehicle-name vin100 \
      --endpoint-url "${vin100.endpointAddress}" \
      --topic-prefix '$aws/iotfleetwise/' \
      --can-bus0 "vcan0"

    # Install source deps
    ./tools/install-deps-native.sh

    # Build source
    sudo -u ubuntu ./tools/build-fwe-native.sh

    # Install FWE
    ./tools/install-fwe.sh
    
    # Signal init complete:
    /opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${instance.instance.logicalId} --region ${stack.region}`;
        instance.addUserData(userData);
        new cdk.CfnOutput(stack, 'Vehicle Sim ssh command', { value: `ssh -i ${keyName}.pem ubuntu@${instance.instancePublicIp}` });
        new ifw.Campaign(stack, 'Campaign', {
            name: 'FwTimeBasedCampaign',
            target: vin100,
            collectionScheme: new ifw.TimeBasedCollectionScheme(cdk.Duration.seconds(10)),
            signals: [
                new ifw.CampaignSignal('Vehicle.EngineTorque'),
            ],
            autoApprove: true,
        });
        new ifw.Fleet(stack, 'Fleet', {
            fleetId: 'fleet',
            signalCatalog,
            vehicles: [vin100],
        });
        this.stack = [stack];
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,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