"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ifw = require(".");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_INTEG_REGION || process.env.CDK_DEFAULT_REGION || 'us-east-1',
            account: process.env.CDK_INTEG_ACCOUNT || process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'integ-stack', { env });
        const databaseName = 'FleetWise';
        const tableName = 'FleetWise';
        const database = new aws_cdk_lib_1.aws_timestream.CfnDatabase(stack, 'Database', {
            databaseName,
        });
        const table = new aws_cdk_lib_1.aws_timestream.CfnTable(stack, 'Table', {
            databaseName,
            tableName,
        });
        table.node.addDependency(database);
        const role = new aws_cdk_lib_1.aws_iam.Role(stack, 'Role', {
            roleName: 'iotfleetwise-role',
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('iotfleetwise.amazonaws.com'),
        });
        role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                'timestream:WriteRecords',
                'timestream:Select',
            ],
            resources: ['*'],
        }));
        role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                'timestream:DescribeEndpoints',
            ],
            resources: ['*'],
        }));
        const signalCatalog = new ifw.SignalCatalog(stack, 'SignalCatalog', {
            database,
            table,
            role,
            description: 'my signal catalog',
            nodes: [
                new ifw.SignalCatalogBranch('Vehicle'),
                new ifw.SignalCatalogSensor('Vehicle.EngineTorque', 'DOUBLE'),
            ],
        });
        const model_a = new ifw.VehicleModel(stack, 'ModelA', {
            signalCatalog,
            name: 'modelA',
            description: 'Model A vehicle',
            networkInterfaces: [new ifw.CanVehicleInterface('1', 'vcan0')],
            signals: [
                new ifw.CanVehicleSignal('Vehicle.EngineTorque', '1', 401, // messageId
                1.0, // factor
                true, // isBigEndian
                false, // isSigned
                8, // length
                0.0, // offset
                9),
            ],
        });
        const vin100 = new ifw.Vehicle(stack, 'vin100', {
            vehicleName: 'vin100',
            vehicleModel: model_a,
            createIotThing: true,
        });
        const vin200 = new ifw.Vehicle(stack, 'vin200', {
            vehicleName: 'vin200',
            vehicleModel: model_a,
            createIotThing: true,
        });
        new ifw.Campaign(stack, 'Campaign1', {
            name: 'FwTimeBasedCampaign1',
            target: vin100,
            collectionScheme: new ifw.TimeBasedCollectionScheme(cdk.Duration.seconds(10)),
            signals: [
                new ifw.CampaignSignal('Vehicle.EngineTorque'),
            ],
            autoApprove: true,
        });
        new ifw.Fleet(stack, 'Fleet1', {
            fleetId: 'fleet1',
            signalCatalog,
            vehicles: [vin100, vin200],
        });
        new ifw.Fleet(stack, 'Fleet2', {
            fleetId: 'fleet2',
            signalCatalog,
            vehicles: [vin100],
        });
        this.stack = [stack];
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,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