"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const dynamodb_1 = require("./rules/dynamodb");
const ec2_1 = require("./rules/ec2");
const efs_1 = require("./rules/efs");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const redshift_1 = require("./rules/redshift");
/**
 * Check for NIST 800-53 compliance.
 *
 * Based on the NIST 800-53 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_4.html
 *
 * @stability stable
 */
class NIST80053Checks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkDynamoDB(node, ignores);
            this.checkEC2(node, ignores);
            this.checkEFS(node, ignores);
            this.checkEMR(node, ignores);
            this.checkIAM(node, ignores);
            this.checkRedshift(node, ignores);
        }
    }
    /**
     * CheckDynamoDB Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-DynamoDBPITREnabled') &&
            !dynamodb_1.nist80053DynamoDBPITREnabled(node)) {
            const ruleId = 'NIST.800.53-DynamoDBPITREnabled';
            const info = 'The DynamoDB table does not have Point-in-time Recovery enabled (Control IDs: CP-9(b), CP-10, SI-12).';
            const explanation = 'The recovery maintains continuous backups of your table for the last 35 days.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EC2 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckDetailedMonitoring') &&
            !ec2_1.nist80053EC2CheckDetailedMonitoring(node)) {
            const ruleId = 'NIST.800.53-EC2CheckDetailedMonitoring';
            const info = 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: CA-7(a)(b), SI-4(2), SI-4(a)(b)(c)).';
            const explanation = 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckInsideVPC') &&
            !ec2_1.nist80053EC2CheckInsideVPC(node)) {
            const ruleId = 'NIST.800.53-EC2CheckInsideVPC';
            const info = 'The EC2 instance is not within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckNoPublicIPs') &&
            !ec2_1.nist80053EC2CheckNoPublicIPs(node)) {
            const ruleId = 'NIST.800.53-EC2CheckNoPublicIPs';
            const info = 'The EC2 instance is associated with a public IP address - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)). ';
            const explanation = 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckSSHRestricted') &&
            !ec2_1.nist80053EC2CheckSSHRestricted(node)) {
            const ruleId = 'NIST.800.53-EC2CheckSSHRestricted';
            const info = 'The Security Group allows unrestricted SSH access - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EFS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEFS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EFSEncrypted') &&
            !efs_1.nist80053EFSEncrypted(node)) {
            const ruleId = 'NIST.800.53-EFSEncrypted';
            const info = 'The EFS does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EMR Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEMR(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EMRKerberosEnabled') &&
            !emr_1.nist80053EMRKerberosEnabled(node)) {
            const ruleId = 'NIST.800.53-EMRKerberosEnabled';
            const info = 'The EMR cluster does not have Kerberos enabled - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).';
            const explanation = 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check IAM Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkIAM(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMGroupMembershipCheck') &&
            !iam_1.nist80053IamGroupMembership(node)) {
            const ruleId = 'NIST.800.53-IAMGroupMembershipCheck';
            const info = 'The IAM user does not belong to any group(s) - (Control IDs: AC-2(1), AC-2(j), AC-3, and AC-6).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMUserNoPoliciesCheck') &&
            !iam_1.nist80053IamUserNoPolicies(node)) {
            const ruleId = 'NIST.800.53-IAMUserNoPoliciesCheck';
            const info = 'The IAM policy is attached at the user level - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).';
            const explanation = 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMNoInlinePolicyCheck') &&
            !iam_1.nist80053IamNoInlinePolicy(node)) {
            const ruleId = 'NIST.800.53-IAMNoInlinePolicyCheck';
            const info = 'The IAM Group, User, or Role contains an inline policy - (Control ID: AC-6).';
            const explanation = 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess') &&
            !iam_1.nist80053IamPolicyNoStatementsWithAdminAccess(node)) {
            const ruleId = 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess';
            const info = 'The IAM policy grants admin access - (Control IDs AC-2(1), AC-2(j), AC-3, AC-6).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Redshift Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-RedshiftClusterConfiguration') &&
            !redshift_1.nist80053RedshiftClusterConfiguration(node)) {
            const ruleId = 'NIST.800.53-RedshiftClusterConfiguration';
            const info = 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c), SC-13).';
            const explanation = 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RedshiftClusterPublicAccess') &&
            !redshift_1.nist80053RedshiftClusterPublicAccess(node)) {
            const ruleId = 'NIST.800.53-RedshiftClusterPublicAccess';
            const info = 'The Redshift cluster allows public access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.NIST80053Checks = NIST80053Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053Checks[_a] = { fqn: "monocdk-nag.NIST80053Checks", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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