"""
Generates a 3-D small intestine mesh along the central line,
with variable numbers of elements around, along and through
wall, with variable radius and thickness along.
"""

import copy

from opencmiss.zinc.node import Node
from scaffoldmaker.annotation.annotationgroup import AnnotationGroup
from scaffoldmaker.annotation.smallintestine_terms import get_smallintestine_term
from scaffoldmaker.meshtypes.meshtype_1d_path1 import MeshType_1d_path1, extractPathParametersFromRegion
from scaffoldmaker.meshtypes.scaffold_base import Scaffold_base
from scaffoldmaker.scaffoldpackage import ScaffoldPackage
from scaffoldmaker.utils import interpolation as interp
from scaffoldmaker.utils import tubemesh
from scaffoldmaker.utils import vector
from scaffoldmaker.utils.tubemesh import CylindricalSegmentTubeMeshInnerPoints
from scaffoldmaker.utils.zinc_utils import exnodeStringFromNodeValues


class MeshType_3d_smallintestine1(Scaffold_base):
    '''
    Generates a 3-D small intestine mesh with variable numbers
    of elements around, along the central line, and through wall.
    The small intestine is created by a function that generates
    a small intestine segment and uses tubemesh to map the segment
    along a central line profile.
    '''

    centralPathDefaultScaffoldPackages = {
        'Cattle 1' : ScaffoldPackage(MeshType_1d_path1, {
            'scaffoldSettings' : {
                'D2 derivatives': True,
                'Coordinate dimensions' : 3,
                'Length' : 1.0,
                'Number of elements' : 536
                },
            'meshEdits' : exnodeStringFromNodeValues(
                [ Node.VALUE_LABEL_VALUE, Node.VALUE_LABEL_D_DS1, Node.VALUE_LABEL_D_DS2, Node.VALUE_LABEL_D2_DS1DS2  ], [
                [ [ -502.3,  732.1,   92.0 ], [  -49.8,  -90.7,    0.0 ], [   0.0,   0.0,  57.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -543.1,  654.8,   92.0 ], [  -31.7,  -64.0,    0.0 ], [   0.0,   0.0,  57.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -566.2,  605.0,   92.0 ], [  -43.4,  -40.2,    0.0 ], [   0.0,   0.0,  57.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -623.9,  583.6,   92.0 ], [  -58.9,    0.0,    0.0 ], [   0.0,   0.0,  57.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -674.4,  602.3,   92.0 ], [  -25.8,   24.9,    0.0 ], [   0.0,   0.0,  57.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -676.3,  613.4,   92.0 ], [   21.7,   17.1,   -7.5 ], [  11.8,   9.3,  55.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -638.2,  613.2,   80.5 ], [   47.9,    1.9,  -11.3 ], [  13.2,   0.5,  56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -586.8,  617.6,   72.1 ], [    3.3,   41.0,  -47.1 ], [   3.5,  43.3,  37.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -644.6,  633.7,   57.5 ], [  -52.2,    4.6,   -6.7 ], [  -7.2,   0.7,  57.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -677.2,  630.2,   57.5 ], [    0.1,  -43.7,  -12.1 ], [  -0.0, -15.3,  55.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -632.9,  609.9,   50.6 ], [   51.9,  -16.2,  -11.7 ], [  11.6,  -3.6,  56.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -581.3,  601.0,   35.4 ], [    4.9,  -48.0,  -34.5 ], [   3.3, -33.3,  46.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -631.7,  589.3,   35.9 ], [  -57.5,   19.9,   -5.3 ], [  -4.7,   1.6,  57.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -667.4,  632.0,   27.3 ], [  -60.0,  -18.8,  -39.1 ], [  -5.8,  14.4,   1.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -666.4,  558.0,    7.7 ], [  -33.7,  -54.6,   46.3 ], [  -5.9, -12.1, -18.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -698.0,  570.7,   66.5 ], [  -52.9,    5.4,   48.5 ], [ -19.2,  -6.4, -20.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -760.2,  567.8,   94.3 ], [  -13.6,   33.4,   46.8 ], [ -23.4, -15.1,   3.9 ], [ -10.7, -0.4,  -0.3 ] ],
                [ [ -724.5,  582.1,   93.5 ], [   45.9,    1.1,   -0.9 ], [   1.1,  -0.4,  57.5 ], [   1.1, -0.7,  -0.1 ] ],
                [ [ -681.8,  570.0,   92.8 ], [   28.8,  -21.0,   -0.7 ], [   1.4,   0.1,  57.5 ], [   1.0, -0.0,   0.1 ] ],
                [ [ -667.9,  548.9,   92.3 ], [    3.9,  -28.5,   -1.0 ], [   0.2,  -2.1,  57.5 ], [  -1.8,  0.7,   0.1 ] ],
                [ [ -677.1,  520.9,   91.0 ], [  -27.3,  -11.7,    0.6 ], [   1.0,   0.5,  57.5 ], [   0.0, -2.8,  -0.2 ] ],
                [ [ -700.5,  532.4,   92.9 ], [  -14.4,   20.0,   -0.2 ], [  -3.2,  -1.7,  57.4 ], [   0.6,  0.5,   0.1 ] ],
                [ [ -703.5,  552.3,   91.1 ], [  -16.6,   15.0,   -1.8 ], [  -3.6,   3.1,  57.3 ], [   0.0,  3.2,   0.2 ] ],
                [ [ -726.5,  553.8,   90.2 ], [  -21.7,   -2.9,   -3.0 ], [  -7.8,  -1.0,  56.9 ], [ -10.1, -9.2, -23.9 ] ],
                [ [ -739.6,  548.1,   86.4 ], [   15.3,  -13.2,  -28.2 ], [ -10.7, -26.0,   6.4 ], [ -10.9, -6.4, -43.9 ] ],
                [ [ -694.5,  550.1,   70.6 ], [   44.7,   -4.7,  -16.3 ], [ -13.4,  12.8, -40.5 ], [   6.6,  7.5, -36.2 ] ],
                [ [ -652.7,  539.3,   54.7 ], [   26.4,  -22.5,   -3.7 ], [  -4.8,   3.7, -57.2 ], [   4.0, -5.5,  -6.6 ] ],
                [ [ -644.9,  520.2,   58.6 ], [   -8.4,  -26.0,    7.1 ], [  -4.5, -13.8, -55.7 ], [  -3.8, -1.6,   0.3 ] ],
                [ [ -670.5,  503.3,   65.2 ], [  -49.2,   -7.4,    8.4 ], [  -9.4,  -1.4, -56.7 ], [  -1.2, -1.4,   0.3 ] ],
                [ [ -723.2,  522.1,   69.8 ], [   13.5,   42.0,  -69.3 ], [ -11.1, -47.2, -30.8 ], [  27.5, -1.4,  -1.5 ] ],
                [ [ -641.7,  515.0,   30.8 ], [   50.0,  -30.4,  -15.0 ], [ -12.3,   7.2, -55.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -631.4,  498.1,   31.4 ], [   -4.5,  -28.1,    4.1 ], [  -0.1,  -8.3, -56.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -652.6,  477.3,   37.0 ], [  -50.0,   -6.3,    6.6 ], [  -7.4,  -0.1, -57.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -711.3,  508.4,   47.4 ], [  -65.2,   35.3,   20.7 ], [ -14.8,   5.3, -55.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -756.9,  547.9,   71.4 ], [  -13.4,  -32.8,   30.6 ], [  -7.5,   0.3,  -2.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -764.2,  524.8,   82.7 ], [    2.4,  -34.0,   -3.0 ], [  -9.0,   0.2,  -9.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -733.1,  495.7,   71.4 ], [   -9.6,  -50.8,   23.3 ], [  -7.4,   3.2,   4.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -714.7,  462.8,   83.0 ], [   19.7,   -9.4,    0.0 ], [  -2.7,  -5.6,  19.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -703.0,  463.2,   92.2 ], [   -6.9,   19.3,   27.8 ], [ -20.2,  10.7, -12.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -722.3,  486.2,   90.2 ], [  -30.4,   26.1,   -0.9 ], [   1.1,  -0.8, -57.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -745.1,  520.9,   90.7 ], [  -10.1,   19.9,    1.1 ], [  -1.3,   2.6, -57.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -724.5,  528.3,   92.0 ], [   61.8,   -0.9,    2.8 ], [   2.5,  -0.1, -57.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -691.0,  492.2,   92.5 ], [   65.6,  -18.5,  -13.5 ], [ -10.7,   3.1, -56.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -640.0,  474.1,   57.5 ], [   42.1,  -24.2,  -15.1 ], [ -14.8,   8.5, -55.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -611.1,  454.0,   57.3 ], [   -5.2,  -35.6,    0.0 ], [   0.0,   0.0, -57.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -640.7,  443.8,   57.5 ], [  -30.1,   -5.2,    0.1 ], [  -0.2,  -0.1, -57.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -667.6,  443.1,   57.5 ], [  -25.6,    5.8,    0.0 ], [   0.0,   0.0, -57.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -702.1,  461.2,   42.9 ], [  -35.1,    4.2,   -7.1 ], [  11.3,  -1.4, -56.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -751.1,  480.0,   41.4 ], [   -2.7,   -4.2,   52.0 ], [ -21.8, -25.9,  -3.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -740.7,  460.8,   81.1 ], [   28.5,  -35.4,   18.5 ], [  12.1, -26.7, -49.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -715.8,  438.1,   60.6 ], [   47.5,    3.9,  -28.6 ], [ -29.5,  -2.4, -49.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -690.6,  420.7,   23.6 ], [   72.6,  -48.0,   -9.8 ], [  -5.4,   3.6, -57.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -634.4,  437.8,    2.6 ], [  -11.7,   62.0,  -18.7 ], [   3.0, -16.1, -55.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -685.4,  456.3,   14.8 ], [  -80.3,   28.9,   -4.4 ], [   2.8,  -1.0, -57.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -713.8,  509.7,   13.1 ], [  -37.4,   33.3,    7.4 ], [  -6.3,   5.5, -56.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -762.9,  511.8,   46.3 ], [  -17.4,  -20.3,   31.3 ], [ -28.4, -33.1, -37.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -778.2,  487.3,   79.1 ], [   -7.6,  -16.3,   20.8 ], [ -18.3, -39.5, -37.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -800.5,  488.7,   56.5 ], [   -2.6,   28.5,  -35.3 ], [  32.0,  30.5,  22.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -767.6,  546.3,   28.5 ], [   27.5,   29.9,   28.5 ], [  10.5, -22.4,  13.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -730.3,  549.5,   39.8 ], [   38.0,  -16.9,  -12.3 ], [ -16.1, -27.0, -12.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -695.5,  539.6,   -1.8 ], [  -25.6,  -41.7,  -52.6 ], [  25.8, -21.2,   4.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -753.6,  538.1,   17.5 ], [  -40.4,  -47.5,    4.7 ], [  14.5, -14.6, -23.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -716.7,  466.2,   -1.6 ], [  -40.4,  -47.8,   46.2 ], [  10.5, -26.4, -18.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -774.9,  492.5,   14.3 ], [    4.6,  -28.4,   48.2 ], [   8.2, -27.8, -17.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -755.0,  488.6,   92.3 ], [  -28.5,    3.5,    3.8 ], [  -5.6, -29.6, -15.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -793.8,  484.4,  104.2 ], [  -32.9,    3.6,   -1.3 ], [  -2.5, -29.1, -16.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -790.7,  455.5,   85.3 ], [   -2.3,  -19.4,  -39.7 ], [  17.9,  -2.2,   0.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -801.7,  464.2,   17.7 ], [   26.1,  -25.9,  -35.6 ], [  15.2,   9.4,   4.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -762.4,  459.9,   21.7 ], [   25.4,    2.0,   17.8 ], [  -1.1,  21.1,  -0.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -762.8,  454.4,   83.5 ], [   -1.6,   18.9,   65.1 ], [  18.3,  10.4,  -2.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -767.2,  479.0,  107.6 ], [  -24.8,   -1.9,    8.7 ], [   5.8,  14.8,  19.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -795.1,  432.4,   92.8 ], [   -9.1,  -19.7,  -41.4 ], [ -16.7,  -2.3,   4.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -769.2,  411.9,   25.9 ], [   68.4,  -16.9,  -33.0 ], [   2.1, -18.1,  13.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -679.4,  419.0,   45.0 ], [   55.5,   10.7,   -0.2 ], [  -1.0,   5.8,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -624.8,  428.1,   30.9 ], [   37.0,    1.1,   -8.3 ], [   3.5,   5.1,  16.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -596.1,  415.9,   17.9 ], [   13.7,  -50.5,  -10.3 ], [   1.5,  -2.0,  12.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -633.7,  383.4,   10.5 ], [  -53.9,  -32.7,    2.9 ], [  -4.0,   8.0,  15.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -721.0,  400.4,   -2.4 ], [  -84.8,  146.0,    5.6 ], [  -7.1,  -4.7,  15.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -838.9,  413.2,   12.2 ], [   -5.8,    0.0,    0.0 ], [  -0.0,   3.7,  17.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -841.9,  390.3,    9.1 ], [   23.7,  -40.8,   14.1 ], [  -8.9,   0.1,  15.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -785.0,  366.4,   25.0 ], [   68.1,  -25.4,   20.5 ], [  -4.8,   0.8,  16.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -715.1,  372.1,   29.9 ], [   45.5,    7.3,    0.6 ], [  -1.1,   5.5,  16.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -676.7,  381.7,   46.0 ], [   33.5,   -1.6,    1.7 ], [  -0.8,   3.1,  17.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -624.7,  387.7,   43.5 ], [   20.2,   30.9,   -2.5 ], [  -8.3,   6.6,  13.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -620.0,  414.7,   50.8 ], [  -23.2,   24.5,    6.1 ], [  -3.4,  -7.1,  15.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -667.6,  418.5,   51.4 ], [  -25.9,   -0.2,    2.6 ], [   1.7,   4.2,  16.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -705.4,  416.8,   61.8 ], [  -55.1,  -17.1,    0.0 ], [  -2.1,   6.6,  16.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -749.8,  368.9,   72.2 ], [  -21.4,  -38.2,   -0.9 ], [  -9.5,   5.0,  13.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -794.9,  366.1,   65.9 ], [  -29.1,   -4.3,    9.3 ], [   2.5,  17.3,  15.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -808.2,  363.3,   93.3 ], [   18.8,  -10.2,   29.8 ], [   6.8,  16.8,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -772.3,  355.1,   95.7 ], [   43.7,   -4.2,   -0.2 ], [   6.4,  17.1,  14.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -723.7,  356.5,   92.0 ], [   34.8,   14.4,   -6.2 ], [  -3.2,  15.3,  17.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -704.1,  380.2,   92.3 ], [   -4.5,   25.2,    1.1 ], [  11.3,   1.1,  20.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -724.4,  396.0,   92.7 ], [  -30.0,   -4.7,   -0.5 ], [  -2.9,  17.3,  15.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -761.4,  390.0,   92.0 ], [  -26.8,   -4.4,   -1.3 ], [  -3.6,  17.2,  15.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -796.4,  394.2,   92.0 ], [  -19.9,   14.6,   -1.1 ], [   9.8,  14.4,  15.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -808.0,  419.2,   92.0 ], [   22.2,    4.7,  -19.8 ], [   7.4,  18.2,  12.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -772.4,  406.2,   98.3 ], [   40.5,   -7.1,    9.5 ], [  -0.0,  18.8,  14.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -743.7,  434.9,   88.3 ], [  -18.9,  -10.8,  -19.8 ], [ -13.9,  18.6,   3.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -799.1,  394.0,   52.0 ], [  -21.0,  -22.0,  -12.0 ], [ -16.4,   7.7,  14.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -804.5,  342.9,   51.1 ], [   18.4,  -46.2,   24.5 ], [  10.1,   7.0,  19.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -779.4,  314.9,   94.5 ], [   43.8,  -11.6,   23.2 ], [   0.2,  17.5,  15.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -725.7,  321.9,   94.8 ], [   50.5,    2.4,   -4.1 ], [   0.4,  17.6,  15.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -673.6,  342.2,   84.9 ], [   40.1,  -23.1,    1.4 ], [   8.5,  15.6,  15.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -654.8,  296.3,   92.0 ], [    7.3,  -43.2,    5.1 ], [   9.2,   4.0,  21.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -638.4,  213.8,   96.3 ], [  -29.6,  -31.8,    0.0 ], [  -8.5,   7.9,  20.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -671.5,  206.0,   94.0 ], [  -29.5,   25.2,    0.1 ], [  11.3,  13.1,  15.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -693.1,  254.6,   92.0 ], [   -5.8,   31.8,    6.0 ], [  13.5,  -1.1,  19.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -692.7,  282.1,   99.6 ], [  -11.7,   28.5,    0.0 ], [  12.9,   5.2,  18.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -714.5,  303.8,  105.3 ], [  -27.8,  -10.4,    3.1 ], [  -3.9,  15.5,  17.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -732.9,  276.2,  100.4 ], [  -10.5,  -24.2,   -5.4 ], [  -7.0,  -1.9,  22.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -739.6,  242.4,   94.9 ], [    6.2,  -21.6,    0.0 ], [   0.2,  -3.9,  15.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -729.9,  208.4,   92.9 ], [   31.1,  -37.0,    1.3 ], [  -1.0,   0.2,  16.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -671.4,  183.6,   95.2 ], [   26.2,    4.6,  -18.3 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -676.5,  223.4,   56.9 ], [  -64.0,   41.8,   -9.8 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -680.8,  291.9,   29.6 ], [  125.6,   70.8,   -6.6 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -714.1,  347.3,   15.3 ], [  -19.5,   82.9,  -38.4 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -801.9,  404.6,   -7.1 ], [  -58.9,  -30.3,    2.3 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -808.7,  365.8,  -13.7 ], [   44.6,  -53.0,  -15.8 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -688.0,  338.4,  -11.7 ], [   63.8,  -48.8,    8.7 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -673.0,  259.5,   -3.2 ], [   81.0,   -7.1,    7.5 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -674.4,  198.7,   27.6 ], [ -166.7,   -5.5,   78.5 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -751.2,  330.4,    5.6 ], [ -146.4,  -62.9,  102.7 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -771.4,  224.3,   -0.5 ], [ -206.6,    3.0,   -5.4 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -809.2,  353.0,   16.2 ], [ -189.7,   36.4,   -2.1 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -838.2,  311.9,   41.3 ], [   13.3,  -30.3,   21.4 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -817.1,  288.0,   42.4 ], [   34.4,  -12.2,    0.7 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -752.4,  294.0,   78.8 ], [   68.3,    9.1,    2.0 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -741.4,  241.8,   54.9 ], [  -60.0,  -10.6,  -17.5 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -802.8,  226.0,   43.4 ], [  -39.0,   36.4,   14.8 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -833.1,  266.0,   68.3 ], [  -22.1,   29.7,   29.3 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -822.7,  279.5,   92.0 ], [   26.6,    9.7,    0.3 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -799.3,  280.5,   92.0 ], [   19.6,   -7.3,    0.0 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -788.6,  264.4,   92.0 ], [   -4.8,  -26.5,    0.0 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -814.9,  246.1,   92.0 ], [  -26.1,   -9.0,    1.3 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -826.2,  223.0,   92.0 ], [   12.3,  -25.3,    0.0 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -802.8,  203.4,   92.0 ], [   45.0,    8.6,    0.0 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -783.7,  237.4,   92.1 ], [   32.0,   -1.8,  -16.2 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -754.4,  192.4,   70.5 ], [   45.0,  -18.7,   -9.8 ], [   0.8,   1.6,  14.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -680.8,  179.6,   31.6 ], [    3.9,  -10.6,  -67.3 ], [  17.7,  -3.0,  -2.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -737.1,  190.0,   35.6 ], [  -37.2,   15.7,   15.3 ], [  -5.2,  -0.1, -14.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -781.9,  194.0,   48.1 ], [  -20.9,   -6.2,   -4.1 ], [  -2.1,   1.4, -13.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -802.3,  178.1,   41.9 ], [  -11.3,  -27.2,    3.0 ], [  -6.9,   3.9, -11.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -799.1,  145.8,   39.0 ], [   22.4,  -22.6,  -18.4 ], [  -8.7,   0.3, -10.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -731.2,  157.3,    2.0 ], [   50.1,  -43.0,   -3.7 ], [  -8.7,   0.3, -10.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -669.4,  112.8,    4.4 ], [  136.3,  170.2,  -47.5 ], [  -8.7,   0.3, -10.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -750.7,  184.1,   17.7 ], [  -76.4,  -20.3,  -34.7 ], [  -8.7,   0.3, -10.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -840.7,  197.3,   -8.0 ], [  -65.7,  -54.0,  -27.9 ], [  -8.7,   0.3, -10.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -770.0,  153.9,  -18.3 ], [  146.6,   96.2,   29.9 ], [  -8.7,   0.3, -10.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -687.8,  179.2,  -17.8 ], [   93.0,    6.9,   16.7 ], [   7.4,   3.5, -14.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -660.0,  168.8,  104.6 ], [  -61.1,  -12.3,   52.7 ], [   3.8,   4.2,   6.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -732.5,  164.5,   92.4 ], [  -75.7,  -13.6,   -3.4 ], [  -9.6,   4.3,  12.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -785.8,  124.6,   91.4 ], [  -10.2,  -35.8,   -2.2 ], [ -10.1,  -0.0,  14.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -759.0,   91.2,   83.6 ], [  119.5,   -1.5,    4.3 ], [   3.8,  -6.3,  16.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -640.8,  136.6,   98.6 ], [   46.6,   50.1,    7.6 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -579.7,  151.6,  100.3 ], [   19.2,  -49.7,    4.5 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -626.3,   99.7,  102.9 ], [  -45.2,   -3.5,    0.2 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -677.5,   87.9,  102.1 ], [   -1.1,  -44.0,   15.6 ], [  -3.8,   1.0,  15.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -658.6,   70.8,  123.2 ], [   64.5,  -18.3,    7.6 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -663.8,   52.8,  121.6 ], [  -54.4,  -14.3,  -27.1 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -714.9,   78.7,   84.4 ], [  -26.0,  -24.4,  -65.4 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -658.4,   32.5,   83.9 ], [    2.1,  116.0,   -7.2 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -604.6,  113.3,   63.5 ], [  -28.8,   78.0,  -53.1 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -692.9,  147.1,   50.5 ], [  -30.7,   48.1,  -22.5 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -760.4,  156.3,   47.1 ], [   12.4,  -68.3,   10.1 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -714.6,  117.3,   46.3 ], [   59.1,  -31.9,   -3.2 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -642.1,   94.7,   41.5 ], [   86.4,    3.3,  -36.5 ], [   3.1,  -9.3,  16.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -636.1,  134.8,   16.4 ], [  117.5,   19.2,  -27.5 ], [  11.1,  10.0,  -3.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -578.2,   88.0,   25.2 ], [    2.5,  -59.3,   17.4 ], [  11.1,  10.0,  -3.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -631.3,   39.1,   34.6 ], [ -128.7,  -21.4,  -13.2 ], [  10.5,  -8.1,   2.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -704.7,   57.7,    8.4 ], [ -125.7,  -34.5,   10.5 ], [  -6.2, -24.3,   4.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -784.5,  117.6,   16.9 ], [   39.8,   42.0,  -36.3 ], [ -20.5,  10.4,  -2.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -699.8,  123.1,  -24.3 ], [   55.4, -153.8,  -40.3 ], [  -0.9,  12.6, -23.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -800.5,  108.1,  -31.7 ], [  -87.4,   47.7,   13.7 ], [  -0.9,  12.6, -23.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -802.6,   73.8,  -45.8 ], [   60.7,  -32.1,   -7.2 ], [  -0.9,  12.6, -23.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -700.5,   58.6,  -44.7 ], [   56.4,   29.0,   19.9 ], [  -0.9,  12.6, -23.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -650.8,  100.6,  -19.7 ], [   52.4,   82.2,   48.0 ], [  -0.9,  12.6, -23.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -659.1,  149.9,  -29.6 ], [   34.0,   79.4,   15.2 ], [  -9.4,   5.8, -25.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -609.5,  156.9,  -23.7 ], [   15.6,  -53.6,  -14.7 ], [  -0.9,  12.6, -23.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -609.7,  110.7,  -23.2 ], [   40.8,  -48.1,   -0.9 ], [  -0.9,  12.6, -23.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -646.5,   26.1,  -11.2 ], [ -162.6,  -77.4,   15.8 ], [ -11.9,   7.1, -24.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -611.3,   41.4,   96.1 ], [   57.3,   45.8,   -2.4 ], [  -3.4,  11.3,  14.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -577.6,   74.9,   90.6 ], [   55.8,  -10.0,   15.4 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -584.9,   34.8,  107.9 ], [  -13.4,  -36.1,   14.6 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -599.8,   -4.2,  106.4 ], [  -10.5,  -58.0,  -59.3 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -560.7,   -5.8,  103.5 ], [   22.8,   77.0,   53.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -539.5,   62.7,  103.5 ], [   25.8,   91.3,    0.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -509.9,  125.7,  103.5 ], [   40.9,   31.1,    0.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -480.3,  136.3,  103.5 ], [   27.9,   -1.5,    0.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -463.8,  126.6,  103.5 ], [    2.2,  -41.6,    0.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -502.4,   82.3,  103.5 ], [  -23.6,  -27.2,    0.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -509.3,   56.8,  103.5 ], [   23.0,  -22.9,    0.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -475.5,   64.7,  103.5 ], [   44.1,   -1.9,    0.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -430.2,   49.3,  103.5 ], [   14.5,  -56.1,    1.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -469.9,    5.3,  104.6 ], [  -58.9,  -25.6,  -18.2 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -530.2,    8.8,   71.2 ], [  -89.4,    9.0,  -28.7 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -549.8,   51.3,   74.5 ], [   56.6,   60.5,   38.3 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -531.3,  109.4,   71.3 ], [   86.6,   49.6,    3.5 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -456.5,  140.5,   51.6 ], [   45.4,   11.9,   -8.3 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -423.0,   81.0,   50.8 ], [  -44.5, -103.3,   -0.2 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -549.8,   34.3,   52.0 ], [  -12.3, -134.8,  -12.9 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -465.3,  -17.5,   49.3 ], [  -46.3, -108.1,  -19.6 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -578.6,  -16.5,   42.3 ], [  -82.7,   25.5,  -15.4 ], [   0.7,   7.9,  15.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -536.6,   25.2,   15.6 ], [   37.7,   57.9,  -21.8 ], [ -12.9,  22.1,   9.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -529.0,   93.9,   11.8 ], [   61.5,   67.5,   -4.9 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -468.2,  137.5,   16.6 ], [   50.6,   13.7,   -2.5 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -398.8,  135.9,   16.1 ], [   54.5,  -33.9,  -12.1 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -378.8,   78.8,   14.5 ], [   -8.1,  -48.9,   36.1 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -398.0,   36.2,   45.8 ], [  -34.0,  -45.8,   24.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -442.2,  -17.6,   57.5 ], [  -86.5,  -40.8,   33.2 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -533.5,  -18.1,   72.8 ], [  -62.7,  -13.5,   25.8 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -559.1,  -59.0,   75.1 ], [   21.7,  -55.5,    8.3 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -500.7,  -84.9,   81.0 ], [   63.7,   28.7,    0.2 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -421.1,  -26.6,   49.0 ], [   38.3,   45.9,  -36.3 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -382.0,    4.7,    3.4 ], [    8.9,   68.8,  -40.9 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -389.1,   52.2,  -22.1 ], [  -29.8,   81.2,  -22.4 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -412.0,  118.0,  -36.8 ], [  -99.4,  115.4,   42.3 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -500.7,  120.5,  -22.3 ], [  -52.5, -122.9,   19.4 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -430.5,    6.2,  -21.6 ], [ -110.0, -102.9,   20.2 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -561.6,   93.9,  -37.5 ], [ -136.9,  -93.8,   13.7 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -478.9,   -3.9,  -57.8 ], [   64.0, -137.5,   21.4 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -431.2,  -73.1,   -4.7 ], [  -33.2,   -2.1,   83.8 ], [   4.6,   5.2,  13.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -498.1,  -26.5,    1.1 ], [ -106.3,   89.8,  -35.3 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -612.2,   36.1,   28.2 ], [  -42.0,  -52.5,   15.5 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -599.6,  -31.3,   28.4 ], [   13.4,  -54.2,   -7.2 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -551.1,  -68.3,   48.8 ], [   70.0,  -22.3,    2.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -489.3,  -77.5,   38.1 ], [   74.3,   17.0,  -23.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -463.6,  -26.2,   12.8 ], [  -33.3,   65.7,   20.7 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -582.6,   39.8,  -15.3 ], [  -97.0,  -93.3,  -56.8 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -535.1,  -48.0,   -7.6 ], [  163.7,  -58.3, -145.0 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -451.0, -130.2,   37.3 ], [  -10.4,   16.5,   63.1 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -404.7,  -90.3,   45.3 ], [   28.5,   23.8,   -5.2 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -373.4,  -43.1,   39.4 ], [    5.9,   71.9,   -6.8 ], [  -5.5,  -4.4,  16.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -350.7,   43.1,   -5.5 ], [    0.3,  -23.0,  -85.3 ], [  -5.6,  21.1,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -326.2,  -52.1,    3.9 ], [  120.1, -136.4,   45.1 ], [   1.5,  -3.4, -18.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -415.4, -151.9,   68.8 ], [  -13.6,  -50.5,  -41.7 ], [  23.7,  17.3, -26.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -353.6, -118.4,   -8.3 ], [   36.0,   49.6,  -67.8 ], [  -3.8,  12.4,  10.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -394.0,  -54.9,  -43.9 ], [  -58.5,   62.1,   18.6 ], [  -2.8,   8.1,  15.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -374.2,   70.3,  -64.1 ], [  -51.8,  111.2,  -42.8 ], [  -9.0,   5.3,  10.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -480.4,   88.9,  -81.2 ], [  -77.0,  -72.5,  -24.5 ], [ -13.1,   5.3,  16.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -447.7,   23.2,  -62.7 ], [   98.2,  -53.9,  -35.4 ], [  -5.7,   6.5,  20.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -382.9, -144.7,  -29.6 ], [ -142.9,  -41.3,  110.6 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -481.8,  -39.4,  -48.9 ], [ -146.6,  -39.2,   12.2 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -453.6, -166.5,  -38.8 ], [  150.0,   -6.0,  -58.5 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -534.6,    2.0,  -60.7 ], [ -321.9,   11.5,   28.9 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -620.8,   88.6,  -65.8 ], [ -173.3, -112.9,  -24.5 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -601.0,  -61.2,  -12.3 ], [   87.0,  -77.0,   21.4 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -524.3,  -72.5,  -58.3 ], [  -55.4, -200.4,  -26.2 ], [  22.5, -13.1,   2.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -516.7, -125.7,   44.6 ], [   99.1,  -71.0,   10.1 ], [  16.0,  21.6,  12.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -511.5, -154.2,   13.7 ], [  -22.7,   29.2,  -59.8 ], [  26.8,  14.4, -11.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -507.9, -150.6,  -52.2 ], [   77.8,   65.8,  -26.7 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -501.9,  -69.2,  -84.5 ], [  124.8,  -47.2,  -52.7 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -435.4, -182.4,  -85.3 ], [  133.2,  -88.9,   11.7 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -364.9, -190.7,  -12.5 ], [  -54.9,  -42.0,   69.0 ], [ -15.5,  16.5,  12.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -340.3, -163.8,   48.1 ], [  128.9,   72.9,  -11.0 ], [ -13.7,  20.3,   3.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -317.0, -143.9,  -26.8 ], [  -88.4,  -30.9,  -19.8 ], [ -10.8,  17.1,  13.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -358.5, -142.3,  -98.1 ], [  -66.1,   25.4,   -1.3 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -360.3,   29.8, -104.3 ], [  133.2,   -2.9,   37.9 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -315.6,  -96.6,  -32.7 ], [   59.8,  120.9,  165.9 ], [ -19.3,  -5.2,  23.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -314.2,   51.5,  -89.1 ], [   14.3,  127.7,   -5.6 ], [  14.7,   5.3,  21.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -333.7,   97.1,   47.0 ], [   52.0,   38.5,   28.6 ], [ -14.0,  -6.2,  18.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -304.7,  120.3,  -32.0 ], [   62.8,  -40.8,  -60.1 ], [  -4.5,  10.9, -24.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -235.3,  -46.6,   17.4 ], [  156.6,  -68.6,   59.2 ], [ -10.9, -12.7, -22.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -160.6,    8.9,   41.9 ], [   96.2,   60.7,  -14.4 ], [  12.5,  -5.7, -10.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -233.6,  108.7,   14.1 ], [ -196.1,  -54.6,  -35.6 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -327.1,   -7.9,   -1.4 ], [   30.2,  -91.1,   80.6 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -223.1,    8.6,   56.8 ], [   64.4,  127.0,   90.6 ], [  -2.2,   6.9,  19.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -253.9,   87.4,  105.1 ], [   -5.8,   39.6,  111.4 ], [  12.8, -17.2,   8.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -243.1,  128.5,  112.8 ], [   95.0,   13.8,    1.4 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -198.4,  111.3,  103.5 ], [   29.7,  -22.5,    0.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -179.8,   86.8,  103.5 ], [   15.5,  -40.2,   -2.2 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -181.0,   44.3,  103.4 ], [  -23.3,  -31.2,    0.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -212.6,   28.5,  103.5 ], [  -45.5,   -9.5,    0.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -268.3,   32.4,  103.8 ], [  -47.7,   23.2,   -5.7 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -303.2,   68.7,   92.2 ], [  -16.0,   43.3,   -4.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -264.4,  113.1,   92.2 ], [   96.4,   13.4,  -16.3 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -194.8,   79.9,   66.6 ], [   66.4,  101.3,   -2.5 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -259.9,  145.4,   69.1 ], [  -50.2,   11.7,    0.7 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -332.4,  128.5,   76.0 ], [  -49.0,  -52.4,   -1.3 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -350.8,   61.9,   80.8 ], [   -2.8,  -49.4,    0.5 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -310.5,    4.1,   73.4 ], [  145.8,  -25.2,   -6.4 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -271.1,  -43.8,   94.4 ], [   16.1,  -48.3,    3.8 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -270.9,  -77.4,  103.4 ], [   39.7,  -14.8,    1.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -220.7,  -87.8,   97.4 ], [   44.9,   15.0,    0.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -173.4,  -67.3,   82.3 ], [  -11.8,   54.9,  -11.4 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -229.9,  -30.6,   62.7 ], [  -49.8,    8.3,   -5.9 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -291.7,  -26.7,   39.1 ], [  -43.9,    3.8,    0.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -323.6,  -28.8,   45.9 ], [  -29.1,   -6.0,   15.7 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -340.4,  -46.9,   56.0 ], [   -7.7,  -28.6,   -0.3 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -341.7,  -83.9,   64.3 ], [    5.2,  -25.3,    4.6 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -327.6, -149.5,   76.4 ], [   26.3,  -22.6,    3.3 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -277.6, -101.4,   42.9 ], [   45.1,   -4.2,    0.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -231.0, -108.2,   27.8 ], [   42.4,   14.1,    0.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -180.1, -109.9,   40.7 ], [   59.1,  -19.4,   39.6 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -174.7, -116.6,   80.3 ], [  -37.4,   27.5,   45.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -300.7,  -86.1,   54.3 ], [  -63.8,  -23.1,   55.9 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -289.0, -163.3,  110.5 ], [  -99.3,  -43.3,   98.2 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -323.0,  -11.3,   56.1 ], [   39.1,   49.2,   -8.9 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -277.5,    6.4,   49.7 ], [   95.7,  -13.0,    0.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -191.4,  -10.9,   80.5 ], [   62.8,   -7.7,    9.5 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -133.3,   24.3,   81.0 ], [   42.2,   67.0,    0.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -125.4,   67.1,   93.5 ], [  -20.5,   21.2,   -1.4 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -181.3,   58.0,   64.5 ], [  -62.1,  -31.8,  -29.8 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -295.9,   96.8,   39.6 ], [ -106.3, -114.7,  -51.1 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -170.5,   62.7,  -13.2 ], [  -26.9, -123.6,   24.0 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -274.3,   42.9,   28.4 ], [  -12.6, -108.5,  -11.4 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -142.9,  -22.5,   18.3 ], [   11.7,    9.5,   57.3 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -150.9,  -36.1,   84.2 ], [   -5.3,  -48.9,   32.2 ], [   3.8,  13.8,  18.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -135.3, -105.3,   94.8 ], [   -3.2,  -32.6,   55.9 ], [   1.8,  19.5,  18.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -138.4, -135.9,  107.8 ], [    4.8,  -33.7,   -0.7 ], [  18.7,  -8.9,  23.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -150.8, -112.6,   40.6 ], [    1.4,  115.3,    8.1 ], [  11.8, -10.5, -33.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -176.8,   19.3,  -24.6 ], [   67.2,  241.3,  -20.6 ], [ -14.3, -11.5, -16.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -276.7,   70.9,  -69.1 ], [  -51.9, -162.2,   10.1 ], [  -5.0, -14.4, -22.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -221.2,    2.3,  -47.3 ], [ -165.7, -117.1,   -0.9 ], [  -5.0, -14.4, -22.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -283.7,  -70.8,  -59.1 ], [   66.2,  -75.0,   17.6 ], [  -5.0, -14.4, -22.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -200.9,  -32.0,  -51.7 ], [   94.9,   36.1,   50.7 ], [  -5.0, -14.4, -22.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -114.9,   21.0,  -43.5 ], [   88.8,  141.0,  123.3 ], [  18.3,  -9.8, -13.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -154.4,  135.8,  -62.6 ], [  -78.3,   -1.0,   -2.3 ], [  27.2,  11.0,  -8.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -274.8,  136.5,    7.9 ], [   22.1,   45.9,   70.8 ], [   2.5,  12.9,   1.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -141.6,  114.9,   31.3 ], [   60.3,   66.5,   26.5 ], [   2.5,  12.9,   1.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -108.4,  101.2,   30.1 ], [   11.3,  -64.6,  -17.8 ], [   2.5,  12.9,   1.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -117.5,   36.3,   28.1 ], [  -11.8,  -88.5,    4.4 ], [  18.4,  -4.9,   3.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -123.0,  -16.9,   76.9 ], [   -5.4,  -40.5,   20.4 ], [  20.0,  -0.3,  -3.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -83.5,   -94.8,  124.4 ], [   -5.3,  -37.1,  -69.1 ], [  22.6,   7.8,   7.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -114.6,   -3.8,  -22.0 ], [  -48.4,  -89.3,  -74.4 ], [  22.6,   7.8,   7.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -106.1,  -89.8,   24.0 ], [  -40.7,  -98.9,  -72.1 ], [  18.0,  -7.3,  -2.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -134.6,  -44.8,  -44.7 ], [  -45.5,  -59.2, -129.4 ], [   9.8,  -0.4,  -3.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -152.2, -122.4,   -0.9 ], [ -138.7,  -24.0,  -10.7 ], [  11.1,   3.6, -30.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -215.0,  -66.5,  -51.9 ], [  -95.4,  -77.8,    7.5 ], [  10.1, -10.6,  -4.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -281.5, -141.6,  -97.1 ], [  -33.1,  -74.4,   -0.8 ], [  27.9, -15.1,  -0.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -142.9, -169.8,  -24.7 ], [   57.1,   22.5,   62.4 ], [  27.9, -15.1,  -0.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -114.9,  -94.6,   56.0 ], [  -22.7,  -27.9,  103.3 ], [  27.9, -15.1,  -0.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -165.8, -210.2,   55.7 ], [  -82.8,    8.9,  -63.6 ], [  27.9, -15.1,  -0.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -221.8, -171.3,  -25.0 ], [ -110.6,  -77.1,  -27.5 ], [  27.9, -15.1,  -0.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -212.4, -201.2,   82.9 ], [  136.9,    5.4,   26.7 ], [  -3.5, -13.5, -38.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -187.1, -132.6,   25.2 ], [  -87.2,   10.4, -111.3 ], [  27.9, -15.1,  -0.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -324.0, -192.7,  -37.0 ], [  -25.1,  -36.3,   21.2 ], [  22.6,   7.8,   7.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -280.2, -208.5,   37.4 ], [  -89.0,  -10.6,  119.1 ], [  22.6,   7.8,   7.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -271.9, -187.9,  119.4 ], [   42.7,   34.8,    6.6 ], [ -11.8,  21.0, -20.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -252.2, -147.1,  107.1 ], [   22.7,   45.1,   -4.0 ], [  -9.3,  23.7, -27.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -216.6, -117.8,   95.7 ], [   40.3,  -19.2,   -0.5 ], [  -1.7,  24.8, -16.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -210.3, -154.5,  112.4 ], [  -20.6,  -33.9,   26.2 ], [  22.6,   7.8,   7.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -241.1, -217.8,  103.3 ], [  -99.3,  -50.4,  -38.8 ], [  22.6,   7.8,   7.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -262.0, -235.1,   20.0 ], [   83.3,   -4.4, -126.2 ], [  13.3,  13.8, -20.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -341.3, -228.9,  -31.5 ], [   21.9,  106.1,  -95.2 ], [  22.6,   7.8,   7.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -223.0, -253.5,  -46.0 ], [    3.4,   60.9,  -83.8 ], [  22.6,   7.8,   7.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -309.6, -169.8,  -81.1 ], [   21.9,  106.1,  -95.2 ], [  32.4,   2.2,  -3.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -209.5, -219.1,  -37.8 ], [   53.1,  -46.5,  -77.1 ], [  -3.6,  10.4, -32.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -173.6, -172.2,  -85.4 ], [  -26.9,   37.0,  -15.3 ], [  -3.6,  10.4, -32.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -291.3,  -84.3, -106.0 ], [   97.0,  135.7,   42.8 ], [   9.6, -12.2,  -2.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -120.9, -148.9,  -48.4 ], [   39.7,   69.5,  -18.4 ], [  14.5, -12.8,  -8.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -144.1,  -35.1,  -76.7 ], [  -46.2,    4.3,   -3.9 ], [  14.5, -12.8,  -8.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -190.9, -106.2, -108.7 ], [  -85.3,   10.9,  -25.6 ], [  14.5, -12.8,  -8.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -285.7,  -73.5, -145.8 ], [  -88.8,  -66.7,   -0.9 ], [  14.5, -12.8,  -8.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -321.8, -178.8, -111.7 ], [  -31.8,  -89.2,   89.1 ], [  14.5, -12.8,  -8.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -371.6, -176.4, -124.4 ], [   19.2,   94.5,   -4.5 ], [  14.5, -12.8,  -8.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -321.8,  -83.5, -118.8 ], [  -75.9,  112.2,  -24.8 ], [  12.8,   5.4, -20.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -395.5, -163.2, -123.3 ], [  -53.4,  -89.2,    1.1 ], [  -9.5,  15.8, -17.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -473.4, -148.9, -123.5 ], [  -56.1,   65.2,  -46.9 ], [   6.3,  12.3,  -1.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -429.7, -106.4, -120.5 ], [  163.6,  -61.7,  -10.2 ], [  14.5, -12.8,  -8.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -402.5,  -38.3, -145.2 ], [  -67.7,   81.2,    4.5 ], [   0.3,  -4.4, -20.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -461.9,  -56.9, -150.0 ], [  -80.1,  -50.6,    3.7 ], [  -7.9,   7.5, -31.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -522.7, -134.3, -113.6 ], [  -48.3,    6.5,   89.2 ], [ -18.3,   2.2, -15.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -492.6,  -30.9, -110.5 ], [   -4.7,   62.6,   70.1 ], [ -12.5,  29.0, -34.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -562.4,  -79.2,  -83.1 ], [  -57.3,   18.4,   43.9 ], [ -12.5,  29.0, -34.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -607.0,   13.5,  -78.8 ], [   37.3,  101.7,   -7.8 ], [ -12.5,  29.0, -34.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -511.1,   40.2, -112.7 ], [  103.5,  -93.3,  -48.2 ], [ -12.5,  29.0, -34.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -523.2,   -9.5, -122.5 ], [ -147.0,    5.3,   22.4 ], [ -10.3,   8.3, -38.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -588.7,  -24.1,  -95.8 ], [   34.0,  -42.6,    1.7 ], [  -9.8,   6.3, -34.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -532.3,  -72.3, -120.3 ], [  111.5,  -25.5,  -41.1 ], [  -7.6,  14.7, -41.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -460.9,  -12.4, -123.4 ], [  176.7,   55.6,   45.0 ], [  -2.5,  21.5, -39.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -427.4,   65.3,  -95.7 ], [  -43.6,   32.5,  -16.0 ], [  -3.0,  20.9, -39.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -471.4,   54.7, -118.0 ], [  -79.1,  -33.9,    8.9 ], [  -1.6,  26.7, -39.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -580.4,  127.5,  -99.5 ], [   11.0,   31.0,   79.8 ], [ -12.5,  29.0, -34.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -556.8,  144.6,  -36.3 ], [   82.2,  -21.4,    6.3 ], [   0.7,  30.6, -37.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -519.6,  126.7, -112.7 ], [  103.3,  -10.8,  -25.1 ], [ -12.5,  29.0, -34.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -425.9,  145.3,  -95.2 ], [    0.1,    5.4,   86.1 ], [ -12.5,  29.0, -34.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -512.4,  124.3,  -46.7 ], [  -26.4,   -6.8,  -45.5 ], [ -12.5,  29.0, -34.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -476.9,   99.1,  -97.7 ], [   61.4,   -4.2,  -24.7 ], [ -12.5,  29.0, -34.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -371.4,  119.0, -116.3 ], [   47.5,   67.0,   21.4 ], [ -12.5,  29.0, -34.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -341.6,  157.9,  -39.4 ], [ -130.4,  -15.2,  -17.3 ], [   8.9,  30.9, -15.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -401.4,  155.5,  -49.2 ], [  -44.0,   17.8,  118.3 ], [  -0.2,  12.9,  -1.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -326.7,  144.8,  -18.4 ], [   68.8,  -24.0,  -65.2 ], [  -0.2,  12.3,   5.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -264.8,  135.6,  -54.6 ], [   15.7,   44.4,   54.4 ], [  -0.4,  40.8, -23.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -311.0,  161.6,   22.3 ], [   30.3,  -19.6,   79.0 ], [  16.7,  32.8,  -4.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -216.7,  146.0,   34.6 ], [   48.1,  -13.4,    2.2 ], [  11.5,  25.6,  -6.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -127.2,   99.8,   74.2 ], [   48.1,  -23.7,    1.3 ], [  11.6,  25.0,  -8.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -31.5,   35.8,   58.0 ], [    9.6,  -94.2,   34.0 ], [  27.4,   2.1,   3.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -93.2,   47.9,   78.7 ], [  -40.4,    4.3,   -4.7 ], [   0.2,  -1.0,   7.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -83.6,   41.8,   21.7 ], [  -25.4,   95.3,  -36.3 ], [  -7.2,   3.8,  -0.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -94.9,  120.0,  -16.7 ], [   54.7,   18.0,   -1.3 ], [  14.0,  24.9,   0.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [   -6.0,    8.7,   44.4 ], [  -86.7,  -73.7,   -3.6 ], [  18.9, -22.7,   9.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -69.5,   67.1,   -5.3 ], [  -28.2,   -6.4,  -92.2 ], [  30.3, -21.3, -22.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -80.8,  -39.4,  -67.4 ], [  124.6,  -54.3,   39.8 ], [  15.3,  46.5,  -7.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -47.5,   28.9,  -23.7 ], [   59.8,  -55.9,   69.3 ], [  -5.5,  27.3,   4.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -37.5,  -11.9,  -20.6 ], [  -26.3,  -38.4,   68.2 ], [  26.3,  16.0,  18.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -36.8,  -10.9,   27.5 ], [   -1.6,    7.4,   38.0 ], [  26.8,  -1.3,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -42.7,   -8.0,   62.7 ], [   -6.5,    1.7,    7.1 ], [  26.8,  -1.3,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -63.3,    1.2,   96.3 ], [  -26.0,   27.2,    0.0 ], [  26.8,  -1.3,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -78.0,   27.6,  100.9 ], [   37.3,   13.8,    0.2 ], [  26.8,  -1.3,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -31.5,   35.2,  105.2 ], [   15.6,  -22.2,    3.0 ], [ -11.0, -18.6,  11.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -34.5,   -1.2,  104.8 ], [  -10.9,  -26.5,  -24.4 ], [ -22.8, -11.4,   9.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   -5.7,    4.1,   69.8 ], [   22.6,   26.5,   -0.5 ], [   9.2, -17.5,  -4.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   -8.0,   34.9,   79.8 ], [   -9.0,   30.2,   23.8 ], [  26.8,  -1.3,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -24.4,   59.8,   98.7 ], [  -18.3,   17.3,   14.1 ], [  26.8,  -1.3,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -52.8,   83.2,  117.0 ], [  -19.4,   32.6,   10.0 ], [  15.1,  12.7,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -33.1,   94.8,  102.9 ], [   22.1,   -6.2,  -30.4 ], [  29.5, -10.6,  27.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -19.6,   67.8,   71.0 ], [  -12.0,   23.6,  -71.1 ], [  25.3, -10.2,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -69.8,  126.8,   91.5 ], [  -61.8,   12.7,    9.1 ], [  12.0,  20.5,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -121.3,  169.5,   92.0 ], [  -11.1,   37.0,    0.0 ], [  26.8,  -1.3,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -114.5,  225.5,   92.0 ], [   69.1,   44.9,    0.0 ], [  18.5, -18.6,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -65.2,  213.4,   92.0 ], [   33.2,  -32.2,    0.0 ], [  -7.1, -21.8,  -7.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -41.7,  168.2,   81.0 ], [    8.2,  -39.5,   -1.8 ], [ -23.8, -15.5, -17.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -32.2,  130.3,   62.3 ], [   24.3,  -54.0,  -19.6 ], [ -16.3, -19.5, -15.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [   13.3,   68.9,   25.9 ], [   74.4,   51.7,  -56.5 ], [ -11.9, -10.0, -12.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -52.2,  119.9,  -39.1 ], [  -64.1,  -49.6,   36.3 ], [  13.5, -16.8,  -8.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [    0.6,   40.9,   -6.3 ], [  -35.1,  -73.3,  -52.2 ], [  12.2,  -5.0, -17.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -93.6,   68.7,  -44.6 ], [  -75.0,   91.8,  -77.6 ], [ -13.9, -13.8,   0.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -10.3,   30.2,  -41.9 ], [   25.0,  -66.2,  -30.8 ], [ -13.9, -13.8,   0.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -50.4,  -34.5,  -84.6 ], [  -46.0,  102.9,  -35.5 ], [  10.9,   6.4,  20.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -41.4,   32.4,  -58.9 ], [  -91.3,   85.2,  -56.1 ], [  10.9,   5.5,  21.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -114.3,  -19.2, -124.3 ], [  -81.0,    8.4,    9.1 ], [  -0.8,  13.4,  22.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -129.5,   59.3,  -86.4 ], [  -83.6,   13.0,   50.8 ], [  12.5,   5.5,  25.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -193.4,  -50.2, -121.3 ], [ -158.0,   10.7,   45.1 ], [   4.6,  -1.5,  20.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -260.7,    9.9,  -99.5 ], [  -72.1,  -15.5,  -24.1 ], [   0.6,   9.0,  18.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -339.4,  -58.3, -138.9 ], [  -98.2,    0.4,  -23.0 ], [   0.8,   2.8,  18.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -392.7,    8.1, -127.6 ], [   58.0,   12.9,  -44.3 ], [  -5.4,  14.0,  19.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -318.6,  -12.7, -117.3 ], [   82.3,   72.0,   33.3 ], [  -0.3,  15.2,  43.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -400.2,   75.3, -134.7 ], [  185.9,  159.1,   45.2 ], [  -5.0,   9.4,  20.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -297.1,   32.2, -114.4 ], [  170.7,  159.0,   13.0 ], [  -0.4,   8.7,  10.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -183.7,   12.9,  -89.7 ], [   13.0,  150.4,   -5.3 ], [ -11.8,  19.4,  17.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -336.8,   87.5, -103.7 ], [  -17.5,   86.1,   36.1 ], [ -14.2,  -2.5,  17.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -251.9,   87.5,  -90.8 ], [   85.4, -108.3,   36.1 ], [   2.3,   9.9,  20.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -181.5,  101.0,  -68.4 ], [   88.5,  -70.2,   49.5 ], [  -1.5,  -0.5,  19.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -170.3,   54.6, -111.8 ], [   73.7,  -24.1,  -48.1 ], [  -1.5,  -6.8,  18.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -123.2,   76.7,  -98.7 ], [   73.8,   14.7,   83.7 ], [ -11.8,  19.4,  17.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -53.7,  112.3,  -68.1 ], [ -112.2,   97.9,  113.5 ], [ -13.9, -13.8,   0.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -42.1,  136.8,   15.4 ], [  139.6,   -1.8,  110.3 ], [ -13.9, -13.8,   0.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [    8.2,   95.1,   34.3 ], [  -24.8,   -1.9,   92.7 ], [  -9.4, -20.2,  -1.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   16.9,  122.1,   92.6 ], [   26.3,    5.6,   42.1 ], [ -11.6, -12.4,   8.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [   25.3,  149.5,   92.3 ], [  -11.1,   35.9,    0.0 ], [ -13.9, -13.8,   0.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   -9.3,  181.4,   93.4 ], [  -17.1,   39.4,    0.0 ], [ -13.9, -13.8,   0.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   -3.3,  218.3,   94.2 ], [   39.1,   16.2,    0.0 ], [ -22.7,  23.9,  -9.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [   33.3,  192.2,   83.1 ], [   42.5,  -69.9,  -17.1 ], [  26.6,  16.2,  -2.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [   29.0,  113.8,   73.3 ], [   13.2,   -7.9,  -43.2 ], [  34.4,  14.7,   7.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [   53.5,  137.4,   54.4 ], [   22.9,   17.8,    6.3 ], [  21.0,  -5.7,  -0.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [   57.6,  178.1,   62.9 ], [   -9.4,   32.2,   -3.1 ], [  33.4,   7.1,  -5.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [   40.1,  195.2,   59.6 ], [   -3.4,   25.8,   -0.9 ], [  33.4,   7.1,  -5.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [   57.3,  231.0,   78.9 ], [  -15.5,   11.5,   19.4 ], [  33.4,   7.1,  -5.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [   34.3,  245.8,   93.8 ], [  -46.9,    3.2,    0.0 ], [  33.4,   7.1,  -5.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [    5.8,  255.4,   92.6 ], [    2.3,    2.4,   34.8 ], [  33.4,   7.1,  -5.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [   37.8,  263.4,  105.5 ], [   25.3,    1.4,    0.0 ], [   6.1, -20.3,  -7.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [   72.9,  262.0,   95.3 ], [   27.1,  -36.0,  -29.3 ], [  -9.4, -12.7,  -4.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [   74.3,  240.0,   41.9 ], [   65.3,  -11.4,  -10.1 ], [   0.1, -20.3,  -5.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [  104.2,  247.5,   91.2 ], [   23.8,   44.0,    5.5 ], [  15.6, -21.3,  -5.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  105.2,  286.3,   78.8 ], [  -59.1,   14.7,   15.4 ], [  13.8,  21.2,   3.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [   79.0,  257.0,   53.9 ], [  -35.6,   -5.0,  -14.3 ], [  -3.0,  25.6,  -5.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [   20.0,  252.7,   43.9 ], [  -73.7,  -16.0,   -8.6 ], [  -4.0,  16.6,   0.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -18.8,  195.3,   54.9 ], [  -24.5,  -54.2,  -37.4 ], [ -11.3,   5.8,  -5.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -85.5,  167.3,   15.4 ], [  -18.5,  -23.9,  -52.2 ], [ -11.3,   5.8,  -5.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [   -9.9,  178.4,   10.9 ], [   70.5,  -60.8,   38.3 ], [   2.7,   5.0, -50.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [   40.8,  131.0,   22.3 ], [   93.0,   82.1,  -75.6 ], [  -4.7,  -1.4, -18.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [   15.5,  223.0,   15.3 ], [   66.3,   73.5,  -18.1 ], [   1.4,  -5.8, -16.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [   84.5,  187.6,    6.7 ], [   63.2,  -22.9,   74.1 ], [  21.7, -16.5, -16.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [   81.8,  195.0,   99.0 ], [   85.2,  -17.5,   -8.3 ], [   7.8,   6.1, -19.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [  144.3,  215.8,   88.4 ], [   56.6,   35.6,  -14.8 ], [ -14.9,  12.9, -58.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  178.4,  287.4,   86.2 ], [    6.5,   74.9,   -2.8 ], [ -30.2,   6.7, -37.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [  160.5,  350.9,   83.5 ], [  -55.4,   54.5,    2.6 ], [ -25.7, -27.0, -17.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [   79.5,  372.5,   92.0 ], [  -94.7,  -10.0,   12.8 ], [   5.4, -34.2, -10.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [   56.2,  296.4,   44.3 ], [    5.1,  -77.1,  -40.6 ], [  26.1,  12.8, -21.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [   86.5,  241.9,   24.5 ], [   53.0,  -52.3,    0.2 ], [   9.7,  27.8, -33.7 ], [   0.0,  0.0,   0.5 ] ],
                [ [  152.1,  205.8,   56.0 ], [   63.6,   -8.3,   10.0 ], [   6.7,  -4.5, -23.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  198.4,  215.1,   56.0 ], [   38.2,   22.5,   -1.4 ], [   5.4,  -1.7, -23.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  231.6,  243.6,   55.7 ], [   23.7,   41.9,   -1.2 ], [   3.5,   3.0, -23.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  254.2,  308.6,   53.7 ], [  -36.3,   58.9,    0.0 ], [   6.3,   0.4, -23.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  201.6,  339.1,   45.3 ], [  -53.5,   14.4,  -12.4 ], [  10.7,  -1.9, -25.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  225.6,  364.9,   17.7 ], [   84.0,  -17.1,  -36.3 ], [  10.0, -10.7,  10.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [  262.2,  310.8,   24.6 ], [   19.9,  -60.5,    3.1 ], [   0.9,  -7.8,  22.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  266.7,  240.6,   17.5 ], [  -54.8,  -92.3,    1.0 ], [  -0.9,  -6.1,  34.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  191.5,  210.2,   17.5 ], [  -54.2,  -18.8,   19.4 ], [  -3.5,   0.2,  39.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [  126.5,  239.4,   20.1 ], [  -85.5,   -4.1,  -17.8 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   27.7,  256.9,  -12.0 ], [   25.6,   34.7,   43.2 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   83.2,  288.3,   -1.0 ], [   33.2,   -4.8,   -3.7 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  213.4,  266.0,    3.1 ], [   68.3,  112.6,   23.2 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  188.2,  299.9,   26.4 ], [  -48.4,  -18.8,    4.4 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  130.2,  290.8,   30.5 ], [  -44.6,   71.5,   28.9 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  136.7,  339.4,   47.8 ], [ -100.8,   75.5,   26.6 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   47.8,  248.5,   21.6 ], [  -98.1,   56.5,    2.3 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   59.3,  327.6,   28.1 ], [   34.8,  103.2,  -11.3 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  158.5,  381.3,   13.0 ], [  145.5,   -5.8,  -42.4 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  240.7,  304.7,   -6.3 ], [   49.3,  -67.7,  -27.1 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  240.4,  234.9,  -17.2 ], [  -74.1,  -88.5,   16.1 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  173.0,  289.9,  -12.5 ], [   56.5,  158.1,    0.1 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  109.5,  348.1,   -9.7 ], [  -74.1,  -88.5,   16.1 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   88.4,  320.9,   -6.1 ], [  -71.1,  -68.4,   21.4 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [   61.8,  280.2,  -37.6 ], [   69.1, -114.8,  -49.0 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  132.9,  264.7,  -28.9 ], [  -44.7, -165.9,  -57.3 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  147.2,  179.7,   61.3 ], [   91.2,  -36.6,   30.4 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  238.2,  177.2,   65.0 ], [   18.3,  -82.8,  -18.1 ], [   3.7,   9.1,  31.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  197.8,  161.4,   50.3 ], [   67.6,   35.6, -126.7 ], [ -20.7,  -0.9,  -7.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [  254.2,  183.8,   49.3 ], [  -11.8,   23.5,   62.1 ], [   5.6,  -2.7, -14.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  277.7,  254.8,   52.0 ], [   -1.1,   48.5,  -21.4 ], [  16.6,  -4.4,   4.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [  291.4,  209.5,   13.1 ], [  -39.3,  -47.2,   -6.6 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  227.2,  207.9,    6.3 ], [  -93.7,  -69.7,   -8.9 ], [  20.7, -14.6, -61.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [  186.4,  164.4,   16.9 ], [  -71.7,  -61.2,  -10.0 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  118.4,  159.7,    1.3 ], [  -74.4,   26.4,  -38.5 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [   42.7,  245.0,  -45.3 ], [   77.7,  116.5,   19.3 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  159.9,  180.5,  -31.4 ], [  106.2,   76.6,   20.2 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  134.1,  292.3,  -47.1 ], [   34.5,   97.6,   13.8 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  193.3,  275.1,  -46.3 ], [   23.8,  -61.7,   26.1 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  190.9,  187.0,  -52.2 ], [   66.5,  -39.3,    3.6 ], [  -2.7,   2.9, -13.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [  254.2,  247.3,  -47.6 ], [  -23.5,   92.5,   19.8 ], [  -4.0,   2.0, -25.9 ], [   0.0,  0.0,   0.5 ] ],
                [ [  228.4,  321.3,  -41.2 ], [   52.7,  188.2,   -7.7 ], [  -0.6,  -6.1, -55.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  176.6,  339.5,  -38.2 ], [  -78.7,   16.0,   21.5 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  119.0,  367.3,  -29.7 ], [ -127.0,   18.7,    4.7 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [   91.4,  252.3,  -63.2 ], [   23.1, -112.2,  118.6 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  175.1,  162.4,  -53.9 ], [  -54.8,  -66.8,   32.7 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [   70.8,  203.5,  -65.8 ], [ -102.4,  -41.9,  -13.5 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [   52.3,  140.2,  -39.6 ], [ -180.5,  -24.1,   75.0 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [    6.8,  106.5,  -47.4 ], [ -171.3,   31.8,   -4.1 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [   27.9,  191.6,  -30.8 ], [  -96.9,   52.7,  -49.3 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -17.9,  115.6,  -73.1 ], [ -104.1,   42.3,   14.1 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [    1.9,  223.3,  -37.4 ], [   44.0,   82.0,   -9.1 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [   33.2,  282.6,  -58.5 ], [  -23.0,   67.1,  -26.7 ], [  35.2,  -9.9, -56.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [   67.8,  358.2,  -24.3 ], [   -2.3,    0.3,   54.6 ], [  36.1,  33.0,  13.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [   40.7,  358.7,   43.4 ], [  -22.6,   47.2,   59.8 ], [  58.9,  36.1, -18.4 ], [   0.0,  0.0,   0.5 ] ],
                [ [   29.2,  413.0,   41.5 ], [  -60.5,   10.9,   19.0 ], [   8.9,  33.6,  19.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -26.5,  361.8,   43.1 ], [  -56.3,   17.2,    1.6 ], [  14.7,  37.0,  13.5 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -23.4,  393.0,   43.0 ], [   -7.7,   31.4,   -0.9 ], [  36.2,  38.3, -10.2 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -36.6,  415.6,   41.8 ], [  -25.5,   17.7,    1.0 ], [  14.4,  49.6,  -1.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -70.0,  421.5,   45.4 ], [  -30.8,   -9.5,    1.4 ], [  -6.6,  55.6,   1.8 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -90.3,  401.6,   44.5 ], [  -15.1,  -33.4,   -0.7 ], [ -47.8,   9.1,  27.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -91.8,  361.2,   44.5 ], [   35.1,  -38.1,  -18.4 ], [ -24.9, -37.8,  24.3 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -38.8,  367.1,   17.8 ], [   33.3,   56.6,  -27.6 ], [  26.1, -52.9,   2.6 ], [   0.0,  0.0,   0.5 ] ],
                [ [  -55.2,  436.5,   11.5 ], [  -50.1,   54.5,   -0.1 ], [  37.8,  40.8,  -3.0 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -121.4,  460.3,   17.1 ], [  -86.8,   60.9,   -1.8 ], [  20.1,  34.0,  -5.1 ], [   0.0,  0.0,   0.5 ] ],
                [ [ -164.5,  536.1,  -17.3 ], [   -9.9,   55.2,   -8.5 ], [  54.3,   6.4, -16.0 ], [   0.0,  0.0,   0.5 ] ],
                ])
        } ),
        'Mouse 1' : ScaffoldPackage(MeshType_1d_path1, {
            'scaffoldSettings' : {
                'D2 derivatives': True,
                'Coordinate dimensions' : 3,
                'Length' : 1.0,
                'Number of elements' : 45
                },
            'meshEdits' : exnodeStringFromNodeValues(
                [ Node.VALUE_LABEL_VALUE, Node.VALUE_LABEL_D_DS1, Node.VALUE_LABEL_D_DS2, Node.VALUE_LABEL_D2_DS1DS2  ], [
                [ [  -2.3, 18.5,  -4.4 ], [ -4.2, -0.8,   3.7 ], [  0.0,  5.0,  0.0 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -8.6, 16.3,  -0.4 ], [ -7.1, -2.7,   1.6 ], [  0.0,  5.0,  0.0 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -18.3, 12.6,  -1.5 ], [ -6.4, -1.7,  -3.8 ], [  0.0,  5.0,  0.0 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -15.6, 13.7,  -6.1 ], [  7.0,  2.1,  -1.8 ], [  0.0,  5.0,  0.0 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -9.3, 14.8,  -4.9 ], [  4.7,  0.7,   1.8 ], [  0.0,  5.0,  0.0 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -3.9, 15.7,  -3.0 ], [  4.3,  0.7,   2.0 ], [  0.0,  5.0,  0.0 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -3.4, 13.4,  -2.8 ], [ -4.1, -0.7,  -1.7 ], [  0.6, -2.0,  0.3 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -7.6, 12.4,  -4.6 ], [ -3.7, -0.8,  -0.9 ], [  0.0, -2.1,  0.1 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -11.6, 11.6,  -5.7 ], [ -4.2, -0.7,  -0.2 ], [  0.0, -1.9,  0.1 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -16.5, 11.7,  -3.9 ], [ -1.0,  0.2,   5.8 ], [  0.3, -1.4, -0.2 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -12.5, 11.7,  -1.4 ], [  3.6,  0.1,   0.6 ], [ -0.1, -1.4, -0.7 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -6.8, 11.8,  -0.6 ], [  2.9,  0.0,   0.7 ], [ -0.7, -1.2, -0.9 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -6.4,  9.8,  -1.6 ], [ -2.9, -0.3,  -1.4 ], [ -0.9,  1.6,  0.4 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -9.5,  9.5,  -2.9 ], [ -4.6,  0.0,  -1.8 ], [ -0.5,  1.7,  0.7 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -14.3,  9.4,  -4.6 ], [ -3.4,  0.1,  -1.6 ], [ -0.1,  1.6,  0.5 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -19.0,  9.4,  -2.9 ], [  0.3,  0.2,   6.7 ], [  0.0,  1.8,  0.0 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -14.5,  9.7,   0.2 ], [  3.6, -1.2,   1.0 ], [  0.7,  2.2,  0.9 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -12.6,  7.7,   0.7 ], [  0.6, -2.7,   0.2 ], [  1.7,  0.8,  1.6 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -13.1,  3.8,   0.3 ], [ -4.0, -3.6,  -1.5 ], [  1.0, -1.1,  1.5 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -15.2,  5.1,  -0.8 ], [  6.0,  6.9,   1.8 ], [ -0.9, -0.4,  0.3 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -17.3,  6.9,  -1.0 ], [ -2.5,  0.0,  -0.4 ], [ -1.6,  0.0,  1.7 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -19.9,  6.8,  -2.5 ], [ -1.5, -1.1,  -3.4 ], [ -1.8, -0.7,  0.4 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -17.2,  6.3,  -5.1 ], [  4.0,  0.8,  -1.3 ], [ -0.4, -1.3, -1.5 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -12.2,  7.8,  -6.8 ], [  4.8,  1.7,  -0.3 ], [  0.1, -0.6, -1.6 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -7.9,  9.6,  -6.5 ], [  3.7,  1.7,   0.7 ], [  0.5, -0.6, -1.6 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -3.8, 10.3,  -5.5 ], [  3.8, -2.7,  -0.1 ], [ -1.3, -0.3, -3.1 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -5.3,  7.6,  -6.4 ], [ -3.5, -1.0,  -1.3 ], [ -0.4,  1.1, -1.5 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -9.0,  6.4,  -7.3 ], [ -3.2, -1.3,   1.9 ], [ -0.9,  0.5, -1.1 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -11.6,  4.0,  -2.0 ], [  5.6, -0.2,   4.3 ], [ -1.8, -0.3,  1.2 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -5.9,  5.0,  -3.1 ], [  4.1,  1.2,  -1.6 ], [  1.0, -0.3,  1.9 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -2.5,  6.0,  -3.8 ], [  3.6,  0.7,   3.2 ], [ -1.7, -1.2,  1.2 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -4.1,  3.2,  -0.4 ], [ -3.5, -1.7,   2.6 ], [ -1.0,  0.4, -1.1 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -9.7,  1.7,   2.3 ], [ -7.9, -1.0,   1.0 ], [ -0.6,  1.0, -1.6 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -19.0,  0.6,  -0.4 ], [  0.2,  3.7,  -6.8 ], [  2.6,  0.8,  0.0 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -13.9,  2.3,  -5.8 ], [  4.4,  0.6,  -1.1 ], [  0.5, -0.1,  2.4 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -7.7,  1.2,  -4.6 ], [  3.9, -3.4,   1.5 ], [ -0.5,  0.0,  2.2 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -4.8, -4.0,  -1.3 ], [ -4.2, -3.3,   3.1 ], [ -1.5,  2.5,  2.3 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -10.9, -6.1,  -0.6 ], [ -5.3, -1.2,  -0.9 ], [ -0.5,  1.9,  1.3 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -19.9, -6.4,  -5.5 ], [ -0.3,  1.7, -10.5 ], [ -0.9,  2.0,  0.7 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -10.7, -3.2,  -8.8 ], [  7.8,  0.4,   0.1 ], [ -0.2,  2.1,  0.2 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -1.2, -1.9,  -7.3 ], [  0.8,  8.1,   2.5 ], [ -3.8,  0.1, -0.6 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -6.3,  0.5,  -8.1 ], [ -9.8, -1.2,   0.5 ], [  0.2, -1.9, -0.5 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -16.0, -0.7,  -7.4 ], [ -7.6,  1.2,   1.5 ], [ -0.1, -2.0, -0.9 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -20.5,  2.3,  -6.1 ], [  3.5,  7.2,  -2.9 ], [ -2.3,  0.2, -0.2 ], [ 0.0, 0.0, 0.5 ] ],
                [ [ -11.4,  2.6, -10.1 ], [ 10.4,  1.5,  -0.2 ], [ -0.2,  1.3, -2.2 ], [ 0.0, 0.0, 0.5 ] ],
                [ [  -3.8,  4.2,  -7.3 ], [  3.5,  0.9,   2.7 ], [  0.2,  1.0, -2.6 ], [ 0.0, 0.0, 0.5 ] ] ] )
            } )
        }

    @staticmethod
    def getName():
        return '3D Small Intestine 1'

    @staticmethod
    def getParameterSetNames():
        return [
            'Default',
            'Cattle 1',
            'Mouse 1']

    @classmethod
    def getDefaultOptions(cls, parameterSetName='Default'):
        if 'Cattle 1' in parameterSetName:
            centralPathOption = cls.centralPathDefaultScaffoldPackages['Cattle 1']
        else:
            centralPathOption = cls.centralPathDefaultScaffoldPackages['Mouse 1']
        options = {
            'Central path': copy.deepcopy(centralPathOption),
            'Number of segments': 100,
            'Number of elements around': 8,
            'Number of elements along segment': 4,
            'Number of elements through wall': 1,
            'Duodenum length': 25.0,
            'Jejunum length': 240.0,
            'Ileum length': 10.0,
            'Duodenum inner radius': 0.6,
            'Duodenum-jejunum inner radius': 1.0,
            'Jejunum-ileum inner radius': 1.0,
            'Ileum inner radius': 1.0,
            'Wall thickness': 0.1,
            'Use cross derivatives': False,
            'Use linear through wall': True,
            'Refine': False,
            'Refine number of elements around': 1,
            'Refine number of elements along': 1,
            'Refine number of elements through wall': 1
            }
        if 'Cattle 1' in parameterSetName:
            options['Number of segments'] = 300
            options['Duodenum length'] = 1000.0
            options['Jejunum length'] = 14700.0
            options['Ileum length'] = 26500.0
            options['Duodenum inner radius'] = 10.0
            options['Duodenum-jejunum inner radius'] = 10.0
            options['Jejunum-ileum inner radius'] = 20.0
            options['Ileum inner radius'] = 10.0
            options['Wall thickness'] = 2.0

        return options

    @staticmethod
    def getOrderedOptionNames():
        return [
            'Central path',
            'Number of segments',
            'Number of elements around',
            'Number of elements along segment',
            'Number of elements through wall',
            'Duodenum length',
            'Jejunum length',
            'Ileum length',
            'Duodenum inner radius',
            'Duodenum-jejunum inner radius',
            'Jejunum-ileum inner radius',
            'Ileum inner radius',
            'Wall thickness',
            'Use cross derivatives',
            'Use linear through wall',
            'Refine',
            'Refine number of elements around',
            'Refine number of elements along',
            'Refine number of elements through wall' ]

    @classmethod
    def getOptionValidScaffoldTypes(cls, optionName):
        if optionName == 'Central path':
            return [ MeshType_1d_path1 ]
        return []

    @classmethod
    def getOptionScaffoldTypeParameterSetNames(cls, optionName, scaffoldType):
        if optionName == 'Central path':
            return list(cls.centralPathDefaultScaffoldPackages.keys())
        assert scaffoldType in cls.getOptionValidScaffoldTypes(optionName), \
            cls.__name__ + '.getOptionScaffoldTypeParameterSetNames.  ' + \
            'Invalid option \'' + optionName + '\' scaffold type ' + scaffoldType.getName()
        return scaffoldType.getParameterSetNames()

    @classmethod
    def getOptionScaffoldPackage(cls, optionName, scaffoldType, parameterSetName=None):
        '''
        :param parameterSetName:  Name of valid parameter set for option Scaffold, or None for default.
        :return: ScaffoldPackage.
        '''
        if parameterSetName:
            assert parameterSetName in cls.getOptionScaffoldTypeParameterSetNames(optionName, scaffoldType), \
                'Invalid parameter set ' + str(parameterSetName) + ' for scaffold ' + str(scaffoldType.getName()) + \
                ' in option ' + str(optionName) + ' of scaffold ' + cls.getName()
        if optionName == 'Central path':
            if not parameterSetName:
                parameterSetName = list(cls.centralPathDefaultScaffoldPackages.keys())[0]
            return copy.deepcopy(cls.centralPathDefaultScaffoldPackages[parameterSetName])
        assert False, cls.__name__ + '.getOptionScaffoldPackage:  Option ' + optionName + ' is not a scaffold'

    @classmethod
    def checkOptions(cls, options):
        if not options['Central path'].getScaffoldType() in cls.getOptionValidScaffoldTypes('Central path'):
            options['Central path'] = cls.getOptionScaffoldPackage('Central path', MeshType_1d_path1)
        for key in [
            'Number of segments',
            'Number of elements around',
            'Number of elements along segment',
            'Number of elements through wall',
            'Refine number of elements around',
            'Refine number of elements along',
            'Refine number of elements through wall']:
            if options[key] < 1:
                options[key] = 1
        for key in [
            'Duodenum length',
            'Jejunum length',
            'Ileum length',
            'Duodenum inner radius',
            'Duodenum-jejunum inner radius',
            'Jejunum-ileum inner radius',
            'Ileum inner radius',
            'Wall thickness']:
            if options[key] < 0.0:
                options[key] = 0.0

    @classmethod
    def generateBaseMesh(cls, region, options):
        """
        Generate the base tricubic Hermite mesh.
        :param region: Zinc region to define model in. Must be empty.
        :param options: Dict containing options. See getDefaultOptions().
        :return: annotationGroups
        """
        centralPath = options['Central path']
        segmentCount = options['Number of segments']
        elementsCountAround = options['Number of elements around']
        elementsCountAlongSegment = options['Number of elements along segment']
        elementsCountThroughWall = options['Number of elements through wall']
        duodenumLength = options['Duodenum length']
        jejunumLength = options['Jejunum length']
        duodenumInnerRadius = options['Duodenum inner radius']
        duodenumJejunumInnerRadius = options['Duodenum-jejunum inner radius']
        jejunumIleumInnerRadius = options['Jejunum-ileum inner radius']
        ileumInnerRadius = options['Ileum inner radius']
        wallThickness = options['Wall thickness']
        useCrossDerivatives = options['Use cross derivatives']
        useCubicHermiteThroughWall = not(options['Use linear through wall'])
        elementsCountAlong = int(elementsCountAlongSegment*segmentCount)
        startPhase = 0.0

        firstNodeIdentifier = 1
        firstElementIdentifier = 1

        # Central path
        tmpRegion = region.createRegion()
        centralPath.generate(tmpRegion)
        cx, cd1, cd2, cd12 = extractPathParametersFromRegion(tmpRegion,
                                                             [Node.VALUE_LABEL_VALUE, Node.VALUE_LABEL_D_DS1,
                                                              Node.VALUE_LABEL_D_DS2, Node.VALUE_LABEL_D2_DS1DS2])
        # for i in range(len(cx)):
        #     print(i+1, '[', cx[i], ',', cd1[i], ',', cd2[i],',', cd12[i], '],')

        del tmpRegion

        # find arclength of colon
        length = 0.0
        elementsCountIn = len(cx) - 1
        sd1 = interp.smoothCubicHermiteDerivativesLine(cx, cd1, fixAllDirections = True,
            magnitudeScalingMode = interp.DerivativeScalingMode.HARMONIC_MEAN)
        for e in range(elementsCountIn):
            arcLength = interp.getCubicHermiteArcLength(cx[e], sd1[e], cx[e + 1], sd1[e + 1])
            # print(e+1, arcLength)
            length += arcLength
        segmentLength = length / segmentCount
        elementAlongLength = length / elementsCountAlong
        # print('Length = ', length)

        # Sample central path
        sx, sd1, se, sxi, ssf = interp.sampleCubicHermiteCurves(cx, cd1, elementsCountAlongSegment*segmentCount)
        sd2, sd12 = interp.interpolateSampleCubicHermite(cd2, cd12, se, sxi, ssf)

        # Generate variation of radius & tc width along length
        lengthList = [0.0, duodenumLength, duodenumLength + jejunumLength, length]
        innerRadiusList = [duodenumInnerRadius, duodenumJejunumInnerRadius, jejunumIleumInnerRadius, ileumInnerRadius]
        innerRadiusSegmentList, dInnerRadiusSegmentList = interp.sampleParameterAlongLine(lengthList, innerRadiusList,
                                                                                          segmentCount)

        # Create annotation groups for small intestine sections
        elementsAlongDuodenum = round(duodenumLength / elementAlongLength)
        elementsAlongJejunum = round(jejunumLength / elementAlongLength)
        elementsAlongIleum = elementsCountAlong - elementsAlongDuodenum - elementsAlongJejunum
        elementsCountAlongGroups = [elementsAlongDuodenum, elementsAlongJejunum, elementsAlongIleum]

        smallintestineGroup = AnnotationGroup(region, get_smallintestine_term("small intestine"))
        duodenumGroup = AnnotationGroup(region, get_smallintestine_term("duodenum"))
        jejunumGroup = AnnotationGroup(region, get_smallintestine_term("jejunum"))
        ileumGroup = AnnotationGroup(region, get_smallintestine_term("ileum"))

        annotationGroupAlong = [[smallintestineGroup, duodenumGroup],
                                [smallintestineGroup, jejunumGroup],
                                [smallintestineGroup, ileumGroup]]

        annotationGroupsAlong = []
        for i in range(len(elementsCountAlongGroups)):
            elementsCount = elementsCountAlongGroups[i]
            for n in range(elementsCount):
                annotationGroupsAlong.append(annotationGroupAlong[i])

        annotationGroupsAround = []
        for i in range(elementsCountAround):
            annotationGroupsAround.append([ ])

        annotationGroupsThroughWall = []
        for i in range(elementsCountThroughWall):
            annotationGroupsThroughWall.append([ ])

        xExtrude = []
        d1Extrude = []
        d2Extrude = []
        d3UnitExtrude = []
        relativeThicknessList = []

        # Create object
        smallIntestineSegmentTubeMeshInnerPoints = CylindricalSegmentTubeMeshInnerPoints(
            elementsCountAround, elementsCountAlongSegment, segmentLength,
            wallThickness, innerRadiusSegmentList, dInnerRadiusSegmentList, startPhase)

        for nSegment in range(segmentCount):
            # Create inner points
            xInner, d1Inner, d2Inner, transitElementList, segmentAxis, radiusAlongSegmentList = \
               smallIntestineSegmentTubeMeshInnerPoints.getCylindricalSegmentTubeMeshInnerPoints(nSegment)

            # Project reference point for warping onto central path
            start = nSegment*elementsCountAlongSegment
            end = (nSegment + 1)*elementsCountAlongSegment + 1
            sxRefList, sd1RefList, sd2ProjectedListRef, zRefList = \
                tubemesh.getPlaneProjectionOnCentralPath(xInner, elementsCountAround, elementsCountAlongSegment,
                                                         segmentLength, sx[start:end], sd1[start:end], sd2[start:end],
                                                         sd12[start:end])

            # Warp segment points
            xWarpedList, d1WarpedList, d2WarpedList, d3WarpedUnitList = tubemesh.warpSegmentPoints(
                xInner, d1Inner, d2Inner, segmentAxis, sxRefList, sd1RefList, sd2ProjectedListRef,
                elementsCountAround, elementsCountAlongSegment, zRefList, radiusAlongSegmentList,
                closedProximalEnd=False)

            # Store points along length
            xExtrude = xExtrude + (xWarpedList if nSegment == 0 else xWarpedList[elementsCountAround:])
            d1Extrude = d1Extrude + (d1WarpedList if nSegment == 0 else d1WarpedList[elementsCountAround:])

            # Smooth d2 for nodes between segments and recalculate d3
            if nSegment == 0:
                d2Extrude = d2Extrude + (d2WarpedList[:-elementsCountAround])
                d3UnitExtrude = d3UnitExtrude + (d3WarpedUnitList[:-elementsCountAround])
            else:
                xSecondFace = xWarpedList[elementsCountAround:elementsCountAround*2]
                d2SecondFace = d2WarpedList[elementsCountAround:elementsCountAround*2]
                for n1 in range(elementsCountAround):
                    nx = [xLastTwoFaces[n1], xLastTwoFaces[n1 + elementsCountAround], xSecondFace[n1]]
                    nd2 = [d2LastTwoFaces[n1], d2LastTwoFaces[n1 + elementsCountAround], d2SecondFace[n1]]
                    d2 = interp.smoothCubicHermiteDerivativesLine(nx, nd2, fixStartDerivative = True,
                                                                  fixEndDerivative = True)[1]
                    d2Extrude.append(d2)
                    d3Unit = vector.normalise(vector.crossproduct3(vector.normalise(d1LastTwoFaces[n1 + elementsCountAround]),
                                                                   vector.normalise(d2)))
                    d3UnitExtrude.append(d3Unit)
                d2Extrude = d2Extrude + \
                            (d2WarpedList[elementsCountAround:-elementsCountAround] if nSegment < segmentCount - 1 else
                             d2WarpedList[elementsCountAround:])
                d3UnitExtrude = d3UnitExtrude + \
                                (d3WarpedUnitList[elementsCountAround:-elementsCountAround] if nSegment < segmentCount - 1 else
                                 d3WarpedUnitList[elementsCountAround:])
            xLastTwoFaces = xWarpedList[-elementsCountAround*2:]
            d1LastTwoFaces = d1WarpedList[-elementsCountAround*2:]
            d2LastTwoFaces = d2WarpedList[-elementsCountAround*2:]

        # Create coordinates and derivatives
        xList, d1List, d2List, d3List, curvatureList = tubemesh.getCoordinatesFromInner(xExtrude, d1Extrude,
            d2Extrude, d3UnitExtrude, [wallThickness]*(elementsCountAlong+1), relativeThicknessList,
            elementsCountAround, elementsCountAlong, elementsCountThroughWall, transitElementList)

        flatWidthList, xiList = smallIntestineSegmentTubeMeshInnerPoints.getFlatWidthAndXiList()

        # Create flat coordinates
        xFlat, d1Flat, d2Flat = tubemesh.createFlatCoordinates(
            xiList, flatWidthList, length, wallThickness, relativeThicknessList, elementsCountAround,
            elementsCountAlong, elementsCountThroughWall, transitElementList)

        xOrgan = d1Organ = d2Organ = []

        # Create nodes and elements
        nextNodeIdentifier, nextElementIdentifier, annotationGroups = tubemesh.createNodesAndElements(
            region, xList, d1List, d2List, d3List, xFlat, d1Flat, d2Flat, xOrgan, d1Organ, d2Organ, None,
            elementsCountAround, elementsCountAlong, elementsCountThroughWall,
            annotationGroupsAround, annotationGroupsAlong, annotationGroupsThroughWall,
            firstNodeIdentifier, firstElementIdentifier, useCubicHermiteThroughWall, useCrossDerivatives,
            closedProximalEnd=False)

        return annotationGroups

    @classmethod
    def refineMesh(cls, meshrefinement, options):
        """
        Refine source mesh into separate region, with change of basis.
        :param meshrefinement: MeshRefinement, which knows source and target region.
        :param options: Dict containing options. See getDefaultOptions().
        """
        refineElementsCountAround = options['Refine number of elements around']
        refineElementsCountAlong = options['Refine number of elements along']
        refineElementsCountThroughWall = options['Refine number of elements through wall']

        meshrefinement.refineAllElementsCubeStandard3d(refineElementsCountAround, refineElementsCountAlong,
                                                       refineElementsCountThroughWall)
        return
