# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['zentables']

package_data = \
{'': ['*'], 'zentables': ['templates/*']}

install_requires = \
['Jinja2>=3.0.1,<4.0.0', 'click>=8.1.3,<9.0.0', 'pandas>=1.4.0,<2.0.0']

setup_kwargs = {
    'name': 'zentables',
    'version': '0.2.4',
    'description': 'Stress-free descriptive tables in Python.',
    'long_description': '# ZenTables - Stress-Free Descriptive Tables in Python\n\n`ZenTables` transforms your `pandas` DataFrames into beautiful, publishable tables in one line of code, which you can then transfer into Google Docs and other word processors with one click. Supercharge your workflow when you are writing papers and reports.\n\n```python\nimport zentables as zen\n\ndf.zen.pretty()\n```\n\n![Formatting tables in one line](https://raw.githubusercontent.com/thepolicylab/ZenTables/main/docs/images/image1.png)\n\n## Features\n\n* Beautiful tables in one line\n* Google Docs/Word ready in one click\n* Descriptive statistics at varying levels of aggregation\n* Control table aesthetics globally\n* and many more to come....\n\n## Installation\n\nVia `pip` from PyPI:\n\n```sh\npip install zentables\n```\n\nVia `pip` from GitHub directly\n\n```sh\npip install -U git+https://github.com/thepolicylab/ZenTables\n```\n\n## How to use `ZenTables`\n\n### 1. How to format any `DataFrame`\n\nFirst, import the package alongside `pandas`:\n\n```python\nimport pandas as pd\nimport zentables as zen\n```\n\nThen, to format any `DataFrame`, simply use:\n\n```python\ndf.zen.pretty()\n```\n\nAnd this is the result:\n\n![zen.pretty() result](https://raw.githubusercontent.com/thepolicylab/ZenTables/main/docs/images/image2.png)\n\nClick on the `Copy Table` button to transfer the table to Google Docs and Word. Formatting will be preserved.\n\nResults in Google Docs (Tested on Chrome, Firefox, and Safari):\n\n![Results in Google Docs](https://raw.githubusercontent.com/thepolicylab/ZenTables/main/docs/images/image3.png)\n\nResults in Microsoft Word:\n\n![Results in Word](https://raw.githubusercontent.com/thepolicylab/ZenTables/main/docs/images/image4.png)\n\n### 2. How to control my tables\' appearance?\n\n`ZenTables` provides two ways to control the aesthetics of the tables. You can use global settings to control the font and font size of the tables via:\n\n```python\nzen.set_options(font_family="Times New Roman, serif", font_size=12)\n```\n\n**Note:** When `font_size` is specified as an `int`, it will be interpreted as points (`pt`). All other CSS units are accepted as a `str`.\n\nOr you can override any global options by specifying `font_family` and `font_size` in `zen.pretty()` method:\n\n```python\ndf.zen.pretty(font_family="Times New Roman, serif", font_size="12pt")\n```\n\nBoth will result in a table that looks like this\n\n![Result change fonts](https://raw.githubusercontent.com/thepolicylab/ZenTables/main/docs/images/image5.png)\n\nWe are working on adding more customization options in a future release.\n\n### 3. How to create common descriptive tables using `ZenTables`?\n\n#### 3.1. Frequency Tables\n\nUse `df.zen.freq_tables()` to create simple frequency tables:\n\n```python\nfreq_table = df.zen.freq_table(\n    index=["Segment", "Region"],\n    columns=["Category"],\n    values="Order ID",\n    props="index",\n    totals=True,\n    subtotals=True,\n    totals_names="Total"\n    subtotals_names="Subtotal",\n)\nfreq_table.zen.pretty() # You can also chain the methods\n```\n\nUse `props` to control whether to add percentages of counts. When `props` is not set (the default), no percentages will be added. You can also specify `props` to calculate percentages over `"index"` (rows), `"columns"`, or `"all"` (over the totals of the immediate top level).\n\nUse `totals` and `subtotals` parameters to specify whether totals and subtotals will be added. Note that when `props` is not `None`, both `totals` and `subtotals` will be `True`, and when `subtotals` is set to `True`, this will also override `totals` settings to `True`.\n\nAdditionally, you can control the names of the total and subtotal categories using `totals_names` and `subtotals_names` parameters.\n\n![Result freq_table()](https://raw.githubusercontent.com/thepolicylab/ZenTables/main/docs/images/image6.png)\n\n#### 3.2. Mean and standard deviation tables\n\nUse `df.zen.mean_sd_table()` to create descriptives with n, mean, and standard deviation:\n\n```python\nmean_sd_table = df.zen.mean_sd_table(\n    index=["Segment", "Region"],\n    columns=["Category"],\n    values="Sales",\n    margins=True,\n    margins_name="All",\n    submargins=True,\n    submargins_name="All Regions",\n)\nmean_sd_table.zen.pretty() # You can also chain the methods\n```\n\nSimilar to `freq_tables`, you can use `margins` and `submargins` parameters to specify whether aggregations at the top and intermediate levels will be added. Additionally, you can control the names of the total and subtotal categories using `margins_names` and `submargins_names` parameters.\n\n![Result mean_sd_table()](https://raw.githubusercontent.com/thepolicylab/ZenTables/main/docs/images/image7.png)\n\n#### 3.3 Other descriptive statistics tables\n\nFor all other types of tables, `ZenTables` provides its own `df.zen.pivot_table()` method:\n\n```python\nmean_median_table = df.zen.pivot_table(\n    index=["Segment", "Region"],\n    columns=["Category"],\n    values="Sales",\n    aggfunc=["count", "mean", "median"],\n    margins=True,\n    margins_name="All",\n    submargins=True,\n    submargins_name="All Regions",\n).rename( # rename columns\n    columns={\n        "count": "n",\n        "mean": "Mean",\n        "median": "Median",\n    }\n)\nmean_median_table.zen.pretty().format(precision=1) # Specify level of precision\n```\n\nThere are two differences between this `pivot_table()` method and the `pandas` `pivot_table` method. First, like `mean_sd_table()`, it provides `submargins` and `submargins_names` for creating intermediate-level aggregations. Second, results are grouped by `values`, `columns`, and `aggfuncs`, instead of `aggfuncs`, `values`, and `columns`. This provides more readability than what the `pandas` version provides.\n\n![Result pivot_table()](https://raw.githubusercontent.com/thepolicylab/ZenTables/main/docs/images/image8.png)\n\n### 4. Tips and tricks\n\n1. `df.zen.pretty()` returns a subclass of `pandas` `Styler`, which means you can chain all other methods after `df.style`. `format()` in the previous section is an example. For more formatting options, please see [this page in `pandas` documentation](https://pandas.pydata.org/pandas-docs/stable/user_guide/style.html)\n\n2. All other methods in `ZenTables` returns a regular `DataFrame` that can be modified further.\n\n3. The names of the index and columns are by default hidden. You can get them back by doing this:\n\n```python\ndf.zen.pretty().show_index_names().show_column_names()\n```\n\n4. You can also disable the `Copy Table` button like this:\n\n```python\ndf.zen.pretty().hide_copy_button()\n```\n\n## TODO\n\n- [ ] More tests on compatibility with `Styler` in `pandas`\n- [ ] More options for customization\n- [ ] A theming system\n- [ ] More to come...\n\n## Contributing\n\nContributions are welcome, and they are greatly appreciated! If you have a new idea for a simple table that we should add, please submit an issue.\n\n## Contributors\n\nPrincipally written by Paul Xu at [The Policy Lab](https://thepolicylab.brown.edu). Other contributors:\n  * Kevin H. Wilson\n  * Edward Huh\n\n## Special thanks\n\n* All the members of [The Policy Lab](https://thepolicylab.brown.edu) at Brown University for their feedback\n* The [`sidetable` package](https://github.com/chris1610/sidetable) for ideas and inspiration.\n',
    'author': 'Paul Xu',
    'author_email': 'yang_xu@brown.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/thepolicylab/ZenTables.git',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0.0',
}


setup(**setup_kwargs)
