"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
function deployNewFunc(stack) {
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    return new lib_1.IotToLambda(stack, 'test-iot-lambda-integration', props);
}
function useExistingFunc(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.PYTHON_3_6,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const props = {
        existingLambdaObj: new lambda.Function(stack, 'MyExistingFunction', lambdaFunctionProps),
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    return new lib_1.IotToLambda(stack, 'test-iot-lambda-integration', props);
}
test('snapshot test IotToLambda default params', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check lambda function properties for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testiotlambdaintegrationLambdaFunctionServiceRole27C3EE41",
                "Arn"
            ]
        },
        Runtime: "nodejs10.x"
    });
});
test('check lambda function permission for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::Permission', {
        Action: "lambda:InvokeFunction",
        FunctionName: {
            "Fn::GetAtt": [
                "testiotlambdaintegrationLambdaFunctionC5329DBA",
                "Arn"
            ]
        },
        Principal: "iot.amazonaws.com",
        SourceArn: {
            "Fn::GetAtt": [
                "testiotlambdaintegrationIotTopic18B6A735",
                "Arn"
            ]
        }
    });
});
test('check iot lambda function role for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/lambda/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "LambdaFunctionServiceRolePolicy"
            }
        ]
    });
});
test('check iot topic rule properties for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Lambda: {
                        FunctionArn: {
                            "Fn::GetAtt": [
                                "testiotlambdaintegrationLambdaFunctionC5329DBA",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'connectedcar/dtc/#'"
        }
    });
});
test('check lambda function properties for deploy: false', () => {
    const stack = new cdk.Stack();
    useExistingFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "MyExistingFunctionServiceRoleF9E14BFD",
                "Arn"
            ]
        },
        Runtime: "python3.6"
    });
});
test('check lambda function permissions for deploy: false', () => {
    const stack = new cdk.Stack();
    useExistingFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::Permission', {
        Action: "lambda:InvokeFunction",
        FunctionName: {
            "Fn::GetAtt": [
                "MyExistingFunction4D772515",
                "Arn"
            ]
        },
        Principal: "iot.amazonaws.com",
        SourceArn: {
            "Fn::GetAtt": [
                "testiotlambdaintegrationIotTopic18B6A735",
                "Arn"
            ]
        }
    });
});
test('check iot lambda function role for deploy: false', () => {
    const stack = new cdk.Stack();
    useExistingFunc(stack);
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        ManagedPolicyArns: [
            {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                    ]
                ]
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.iotTopicRule !== null);
    expect(construct.lambdaFunction !== null);
});
test('check exception for Missing existingObj from props for deploy = false', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    try {
        new lib_1.IotToLambda(stack, 'test-iot-lambda-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check deploy = true and no prop', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    try {
        new lib_1.IotToLambda(stack, 'test-iot-lambda-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW90LWxhbWJkYS50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaW90LWxhbWJkYS50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCw0Q0FBNkM7QUFDN0MsZ0NBQXVEO0FBQ3ZELDhDQUE4QztBQUM5QyxxQ0FBcUM7QUFDckMsZ0NBQThCO0FBRTlCLFNBQVMsYUFBYSxDQUFDLEtBQWdCO0lBQ3JDLE1BQU0sS0FBSyxHQUFxQjtRQUM5QixtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsaUJBQWlCLEVBQUU7WUFDZixnQkFBZ0IsRUFBRTtnQkFDZCxZQUFZLEVBQUUsS0FBSztnQkFDbkIsV0FBVyxFQUFFLHFFQUFxRTtnQkFDbEYsR0FBRyxFQUFFLG9DQUFvQztnQkFDekMsT0FBTyxFQUFFLEVBQUU7YUFDZDtTQUNKO0tBQ0YsQ0FBQztJQUVGLE9BQU8sSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSw2QkFBNkIsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUN0RSxDQUFDO0FBRUQsU0FBUyxlQUFlLENBQUMsS0FBZ0I7SUFFdkMsTUFBTSxtQkFBbUIsR0FBeUI7UUFDOUMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtRQUNsQyxPQUFPLEVBQUUsZUFBZTtRQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztLQUNyRCxDQUFDO0lBRUYsTUFBTSxLQUFLLEdBQXFCO1FBQzVCLGlCQUFpQixFQUFFLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsb0JBQW9CLEVBQUUsbUJBQW1CLENBQUM7UUFDeEYsaUJBQWlCLEVBQUU7WUFDakIsZ0JBQWdCLEVBQUU7Z0JBQ2QsWUFBWSxFQUFFLEtBQUs7Z0JBQ25CLFdBQVcsRUFBRSxxRUFBcUU7Z0JBQ2xGLEdBQUcsRUFBRSxvQ0FBb0M7Z0JBQ3pDLE9BQU8sRUFBRSxFQUFFO2FBQ2Q7U0FDRjtLQUNKLENBQUM7SUFFRixPQUFPLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsNkJBQTZCLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDdEUsQ0FBQztBQUVELElBQUksQ0FBQywwQ0FBMEMsRUFBRSxHQUFHLEVBQUU7SUFDcEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3JCLE1BQU0sQ0FBQyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsZUFBZSxFQUFFLENBQUM7QUFDL0QsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsbURBQW1ELEVBQUUsR0FBRyxFQUFFO0lBQzdELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVyQixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHVCQUF1QixFQUFFO1FBQ2xELE9BQU8sRUFBRSxlQUFlO1FBQ3hCLElBQUksRUFBRTtZQUNKLFlBQVksRUFBRTtnQkFDWiwyREFBMkQ7Z0JBQzNELEtBQUs7YUFDTjtTQUNGO1FBQ0QsT0FBTyxFQUFFLFlBQVk7S0FDeEIsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsbURBQW1ELEVBQUUsR0FBRyxFQUFFO0lBQzNELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVyQixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHlCQUF5QixFQUFFO1FBQ3BELE1BQU0sRUFBRSx1QkFBdUI7UUFDL0IsWUFBWSxFQUFFO1lBQ1osWUFBWSxFQUFFO2dCQUNaLGdEQUFnRDtnQkFDaEQsS0FBSzthQUNOO1NBQ0Y7UUFDRCxTQUFTLEVBQUUsbUJBQW1CO1FBQzlCLFNBQVMsRUFBRTtZQUNULFlBQVksRUFBRTtnQkFDWiwwQ0FBMEM7Z0JBQzFDLEtBQUs7YUFDTjtTQUNGO0tBQ0osQ0FBQyxDQUFDO0FBQ1AsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsaURBQWlELEVBQUUsR0FBRyxFQUFFO0lBQzNELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVyQixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO1FBQzdDLHdCQUF3QixFQUFFO1lBQ3hCLFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxNQUFNLEVBQUUsZ0JBQWdCO29CQUN4QixNQUFNLEVBQUUsT0FBTztvQkFDZixTQUFTLEVBQUU7d0JBQ1QsT0FBTyxFQUFFLHNCQUFzQjtxQkFDaEM7aUJBQ0Y7YUFDRjtZQUNELE9BQU8sRUFBRSxZQUFZO1NBQ3RCO1FBQ0QsUUFBUSxFQUFFO1lBQ1I7Z0JBQ0UsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxNQUFNLEVBQUU7Z0NBQ04scUJBQXFCO2dDQUNyQixzQkFBc0I7Z0NBQ3RCLG1CQUFtQjs2QkFDcEI7NEJBQ0QsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFO2dDQUNSLFVBQVUsRUFBRTtvQ0FDVixFQUFFO29DQUNGO3dDQUNFLE1BQU07d0NBQ047NENBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5Q0FDdEI7d0NBQ0QsUUFBUTt3Q0FDUjs0Q0FDRSxHQUFHLEVBQUUsYUFBYTt5Q0FDbkI7d0NBQ0QsR0FBRzt3Q0FDSDs0Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3lDQUN0Qjt3Q0FDRCwwQkFBMEI7cUNBQzNCO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNELE9BQU8sRUFBRSxZQUFZO2lCQUN0QjtnQkFDRCxVQUFVLEVBQUUsaUNBQWlDO2FBQzlDO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxrREFBa0QsRUFBRSxHQUFHLEVBQUU7SUFDMUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFOUIsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRXJCLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMscUJBQXFCLEVBQUU7UUFDaEQsZ0JBQWdCLEVBQUU7WUFDaEIsT0FBTyxFQUFFO2dCQUNQO29CQUNFLE1BQU0sRUFBRTt3QkFDTixXQUFXLEVBQUU7NEJBQ1gsWUFBWSxFQUFFO2dDQUNaLGdEQUFnRDtnQ0FDaEQsS0FBSzs2QkFDTjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1lBQ0QsV0FBVyxFQUFFLHFFQUFxRTtZQUNsRixZQUFZLEVBQUUsS0FBSztZQUNuQixHQUFHLEVBQUUsb0NBQW9DO1NBQzFDO0tBQ0osQ0FBQyxDQUFDO0FBQ1AsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsb0RBQW9ELEVBQUUsR0FBRyxFQUFFO0lBQzVELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLGVBQWUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUV2QixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHVCQUF1QixFQUFFO1FBQ2xELE9BQU8sRUFBRSxlQUFlO1FBQ3hCLElBQUksRUFBRTtZQUNKLFlBQVksRUFBRTtnQkFDWix1Q0FBdUM7Z0JBQ3ZDLEtBQUs7YUFDTjtTQUNGO1FBQ0QsT0FBTyxFQUFFLFdBQVc7S0FDdkIsQ0FBQyxDQUFDO0FBQ1AsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMscURBQXFELEVBQUUsR0FBRyxFQUFFO0lBQy9ELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLGVBQWUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUV2QixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHlCQUF5QixFQUFFO1FBQ3BELE1BQU0sRUFBRSx1QkFBdUI7UUFDL0IsWUFBWSxFQUFFO1lBQ1osWUFBWSxFQUFFO2dCQUNaLDRCQUE0QjtnQkFDNUIsS0FBSzthQUNOO1NBQ0Y7UUFDRCxTQUFTLEVBQUUsbUJBQW1CO1FBQzlCLFNBQVMsRUFBRTtZQUNULFlBQVksRUFBRTtnQkFDWiwwQ0FBMEM7Z0JBQzFDLEtBQUs7YUFDTjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ1AsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsa0RBQWtELEVBQUUsR0FBRyxFQUFFO0lBQzVELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLGVBQWUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUV2QixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO1FBQzdDLHdCQUF3QixFQUFFO1lBQ3hCLFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxNQUFNLEVBQUUsZ0JBQWdCO29CQUN4QixNQUFNLEVBQUUsT0FBTztvQkFDZixTQUFTLEVBQUU7d0JBQ1QsT0FBTyxFQUFFLHNCQUFzQjtxQkFDaEM7aUJBQ0Y7YUFDRjtZQUNELE9BQU8sRUFBRSxZQUFZO1NBQ3RCO1FBQ0QsaUJBQWlCLEVBQUU7WUFDakI7Z0JBQ0UsVUFBVSxFQUFFO29CQUNWLEVBQUU7b0JBQ0Y7d0JBQ0UsTUFBTTt3QkFDTjs0QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3lCQUN0Qjt3QkFDRCwyREFBMkQ7cUJBQzVEO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRTtJQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLFNBQVMsR0FBZ0IsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRXBELE1BQU0sQ0FBQyxTQUFTLENBQUMsWUFBWSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3hDLE1BQU0sQ0FBQyxTQUFTLENBQUMsY0FBYyxLQUFLLElBQUksQ0FBQyxDQUFDO0FBQzlDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHVFQUF1RSxFQUFFLEdBQUcsRUFBRTtJQUNqRixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLEtBQUssR0FBcUI7UUFDOUIsaUJBQWlCLEVBQUU7WUFDakIsZ0JBQWdCLEVBQUU7Z0JBQ2QsWUFBWSxFQUFFLEtBQUs7Z0JBQ25CLFdBQVcsRUFBRSxxRUFBcUU7Z0JBQ2xGLEdBQUcsRUFBRSxvQ0FBb0M7Z0JBQ3pDLE9BQU8sRUFBRSxFQUFFO2FBQ2Q7U0FDRjtLQUNGLENBQUM7SUFFRixJQUFJO1FBQ0YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSw2QkFBNkIsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUM5RDtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsQ0FBQztLQUNqQztBQUNILENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtJQUMzQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLEtBQUssR0FBcUI7UUFDOUIsaUJBQWlCLEVBQUU7WUFDakIsZ0JBQWdCLEVBQUU7Z0JBQ2QsWUFBWSxFQUFFLEtBQUs7Z0JBQ25CLFdBQVcsRUFBRSxxRUFBcUU7Z0JBQ2xGLEdBQUcsRUFBRSxvQ0FBb0M7Z0JBQ3pDLE9BQU8sRUFBRSxFQUFFO2FBQ2Q7U0FDRjtLQUNGLENBQUM7SUFFRixJQUFJO1FBQ0YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSw2QkFBNkIsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUM5RDtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsQ0FBQztLQUNqQztBQUNILENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IDIwMTkgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgeyBTeW50aFV0aWxzIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCB7IElvdFRvTGFtYmRhLCBJb3RUb0xhbWJkYVByb3BzIH0gZnJvbSBcIi4uL2xpYlwiO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCJAYXdzLWNkay9jb3JlXCI7XG5pbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcblxuZnVuY3Rpb24gZGVwbG95TmV3RnVuYyhzdGFjazogY2RrLlN0YWNrKSB7XG4gIGNvbnN0IHByb3BzOiBJb3RUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1gsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGlvdFRvcGljUnVsZVByb3BzOiB7XG4gICAgICAgIHRvcGljUnVsZVBheWxvYWQ6IHtcbiAgICAgICAgICAgIHJ1bGVEaXNhYmxlZDogZmFsc2UsXG4gICAgICAgICAgICBkZXNjcmlwdGlvbjogXCJQcm9jZXNzaW5nIG9mIERUQyBtZXNzYWdlcyBmcm9tIHRoZSBBV1MgQ29ubmVjdGVkIFZlaGljbGUgU29sdXRpb24uXCIsXG4gICAgICAgICAgICBzcWw6IFwiU0VMRUNUICogRlJPTSAnY29ubmVjdGVkY2FyL2R0Yy8jJ1wiLFxuICAgICAgICAgICAgYWN0aW9uczogW11cbiAgICAgICAgfVxuICAgIH1cbiAgfTtcblxuICByZXR1cm4gbmV3IElvdFRvTGFtYmRhKHN0YWNrLCAndGVzdC1pb3QtbGFtYmRhLWludGVncmF0aW9uJywgcHJvcHMpO1xufVxuXG5mdW5jdGlvbiB1c2VFeGlzdGluZ0Z1bmMoc3RhY2s6IGNkay5TdGFjaykge1xuXG4gIGNvbnN0IGxhbWJkYUZ1bmN0aW9uUHJvcHM6IGxhbWJkYS5GdW5jdGlvblByb3BzID0ge1xuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfNixcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApXG4gIH07XG5cbiAgY29uc3QgcHJvcHM6IElvdFRvTGFtYmRhUHJvcHMgPSB7XG4gICAgICBleGlzdGluZ0xhbWJkYU9iajogbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ015RXhpc3RpbmdGdW5jdGlvbicsIGxhbWJkYUZ1bmN0aW9uUHJvcHMpLFxuICAgICAgaW90VG9waWNSdWxlUHJvcHM6IHtcbiAgICAgICAgdG9waWNSdWxlUGF5bG9hZDoge1xuICAgICAgICAgICAgcnVsZURpc2FibGVkOiBmYWxzZSxcbiAgICAgICAgICAgIGRlc2NyaXB0aW9uOiBcIlByb2Nlc3Npbmcgb2YgRFRDIG1lc3NhZ2VzIGZyb20gdGhlIEFXUyBDb25uZWN0ZWQgVmVoaWNsZSBTb2x1dGlvbi5cIixcbiAgICAgICAgICAgIHNxbDogXCJTRUxFQ1QgKiBGUk9NICdjb25uZWN0ZWRjYXIvZHRjLyMnXCIsXG4gICAgICAgICAgICBhY3Rpb25zOiBbXVxuICAgICAgICB9XG4gICAgICB9XG4gIH07XG5cbiAgcmV0dXJuIG5ldyBJb3RUb0xhbWJkYShzdGFjaywgJ3Rlc3QtaW90LWxhbWJkYS1pbnRlZ3JhdGlvbicsIHByb3BzKTtcbn1cblxudGVzdCgnc25hcHNob3QgdGVzdCBJb3RUb0xhbWJkYSBkZWZhdWx0IHBhcmFtcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGRlcGxveU5ld0Z1bmMoc3RhY2spO1xuICBleHBlY3QoU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSkudG9NYXRjaFNuYXBzaG90KCk7XG59KTtcblxudGVzdCgnY2hlY2sgbGFtYmRhIGZ1bmN0aW9uIHByb3BlcnRpZXMgZm9yIGRlcGxveTogdHJ1ZScsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgZGVwbG95TmV3RnVuYyhzdGFjayk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywge1xuICAgICAgSGFuZGxlcjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgICBSb2xlOiB7XG4gICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgXCJ0ZXN0aW90bGFtYmRhaW50ZWdyYXRpb25MYW1iZGFGdW5jdGlvblNlcnZpY2VSb2xlMjdDM0VFNDFcIixcbiAgICAgICAgICBcIkFyblwiXG4gICAgICAgIF1cbiAgICAgIH0sXG4gICAgICBSdW50aW1lOiBcIm5vZGVqczEwLnhcIlxuICB9KTtcbn0pO1xuXG50ZXN0KCdjaGVjayBsYW1iZGEgZnVuY3Rpb24gcGVybWlzc2lvbiBmb3IgZGVwbG95OiB0cnVlJywgKCkgPT4ge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgZGVwbG95TmV3RnVuYyhzdGFjayk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkxhbWJkYTo6UGVybWlzc2lvbicsIHtcbiAgICAgICAgQWN0aW9uOiBcImxhbWJkYTpJbnZva2VGdW5jdGlvblwiLFxuICAgICAgICBGdW5jdGlvbk5hbWU6IHtcbiAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgXCJ0ZXN0aW90bGFtYmRhaW50ZWdyYXRpb25MYW1iZGFGdW5jdGlvbkM1MzI5REJBXCIsXG4gICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBQcmluY2lwYWw6IFwiaW90LmFtYXpvbmF3cy5jb21cIixcbiAgICAgICAgU291cmNlQXJuOiB7XG4gICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgIFwidGVzdGlvdGxhbWJkYWludGVncmF0aW9uSW90VG9waWMxOEI2QTczNVwiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgIH0pO1xufSk7XG5cbnRlc3QoJ2NoZWNrIGlvdCBsYW1iZGEgZnVuY3Rpb24gcm9sZSBmb3IgZGVwbG95OiB0cnVlJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICBkZXBsb3lOZXdGdW5jKHN0YWNrKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6Um9sZScsIHtcbiAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBcInN0czpBc3N1bWVSb2xlXCIsXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUHJpbmNpcGFsOiB7XG4gICAgICAgICAgICBTZXJ2aWNlOiBcImxhbWJkYS5hbWF6b25hd3MuY29tXCJcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgIH0sXG4gICAgUG9saWNpZXM6IFtcbiAgICAgIHtcbiAgICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICAgXCJsb2dzOkNyZWF0ZUxvZ0dyb3VwXCIsXG4gICAgICAgICAgICAgICAgXCJsb2dzOkNyZWF0ZUxvZ1N0cmVhbVwiLFxuICAgICAgICAgICAgICAgIFwibG9nczpQdXRMb2dFdmVudHNcIlxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgIFwiYXJuOlwiLFxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgUmVmOiBcIkFXUzo6UGFydGl0aW9uXCJcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgXCI6bG9nczpcIixcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlJlZ2lvblwiXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIFwiOlwiLFxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgUmVmOiBcIkFXUzo6QWNjb3VudElkXCJcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgXCI6bG9nLWdyb3VwOi9hd3MvbGFtYmRhLypcIlxuICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgfSxcbiAgICAgICAgUG9saWN5TmFtZTogXCJMYW1iZGFGdW5jdGlvblNlcnZpY2VSb2xlUG9saWN5XCJcbiAgICAgIH1cbiAgICBdXG4gIH0pO1xufSk7XG5cbnRlc3QoJ2NoZWNrIGlvdCB0b3BpYyBydWxlIHByb3BlcnRpZXMgZm9yIGRlcGxveTogdHJ1ZScsICgpID0+IHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIGRlcGxveU5ld0Z1bmMoc3RhY2spO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJb1Q6OlRvcGljUnVsZScsIHtcbiAgICAgICAgVG9waWNSdWxlUGF5bG9hZDoge1xuICAgICAgICAgIEFjdGlvbnM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgTGFtYmRhOiB7XG4gICAgICAgICAgICAgICAgRnVuY3Rpb25Bcm46IHtcbiAgICAgICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgICAgIFwidGVzdGlvdGxhbWJkYWludGVncmF0aW9uTGFtYmRhRnVuY3Rpb25DNTMyOURCQVwiLFxuICAgICAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBEZXNjcmlwdGlvbjogXCJQcm9jZXNzaW5nIG9mIERUQyBtZXNzYWdlcyBmcm9tIHRoZSBBV1MgQ29ubmVjdGVkIFZlaGljbGUgU29sdXRpb24uXCIsXG4gICAgICAgICAgUnVsZURpc2FibGVkOiBmYWxzZSxcbiAgICAgICAgICBTcWw6IFwiU0VMRUNUICogRlJPTSAnY29ubmVjdGVkY2FyL2R0Yy8jJ1wiXG4gICAgICAgIH1cbiAgICB9KTtcbn0pO1xuXG50ZXN0KCdjaGVjayBsYW1iZGEgZnVuY3Rpb24gcHJvcGVydGllcyBmb3IgZGVwbG95OiBmYWxzZScsICgpID0+IHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIHVzZUV4aXN0aW5nRnVuYyhzdGFjayk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCB7XG4gICAgICAgIEhhbmRsZXI6IFwiaW5kZXguaGFuZGxlclwiLFxuICAgICAgICBSb2xlOiB7XG4gICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgIFwiTXlFeGlzdGluZ0Z1bmN0aW9uU2VydmljZVJvbGVGOUUxNEJGRFwiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgIF1cbiAgICAgICAgfSxcbiAgICAgICAgUnVudGltZTogXCJweXRob24zLjZcIlxuICAgIH0pO1xufSk7XG5cbnRlc3QoJ2NoZWNrIGxhbWJkYSBmdW5jdGlvbiBwZXJtaXNzaW9ucyBmb3IgZGVwbG95OiBmYWxzZScsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgdXNlRXhpc3RpbmdGdW5jKHN0YWNrKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkxhbWJkYTo6UGVybWlzc2lvbicsIHtcbiAgICAgIEFjdGlvbjogXCJsYW1iZGE6SW52b2tlRnVuY3Rpb25cIixcbiAgICAgIEZ1bmN0aW9uTmFtZToge1xuICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgIFwiTXlFeGlzdGluZ0Z1bmN0aW9uNEQ3NzI1MTVcIixcbiAgICAgICAgICBcIkFyblwiXG4gICAgICAgIF1cbiAgICAgIH0sXG4gICAgICBQcmluY2lwYWw6IFwiaW90LmFtYXpvbmF3cy5jb21cIixcbiAgICAgIFNvdXJjZUFybjoge1xuICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgIFwidGVzdGlvdGxhbWJkYWludGVncmF0aW9uSW90VG9waWMxOEI2QTczNVwiLFxuICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgXVxuICAgICAgfVxuICAgIH0pO1xufSk7XG5cbnRlc3QoJ2NoZWNrIGlvdCBsYW1iZGEgZnVuY3Rpb24gcm9sZSBmb3IgZGVwbG95OiBmYWxzZScsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgdXNlRXhpc3RpbmdGdW5jKHN0YWNrKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6Um9sZScsIHtcbiAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBcInN0czpBc3N1bWVSb2xlXCIsXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUHJpbmNpcGFsOiB7XG4gICAgICAgICAgICBTZXJ2aWNlOiBcImxhbWJkYS5hbWF6b25hd3MuY29tXCJcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgIH0sXG4gICAgTWFuYWdlZFBvbGljeUFybnM6IFtcbiAgICAgIHtcbiAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgXCJcIixcbiAgICAgICAgICBbXG4gICAgICAgICAgICBcImFybjpcIixcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgUmVmOiBcIkFXUzo6UGFydGl0aW9uXCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBcIjppYW06OmF3czpwb2xpY3kvc2VydmljZS1yb2xlL0FXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZVwiXG4gICAgICAgICAgXVxuICAgICAgICBdXG4gICAgICB9XG4gICAgXVxuICB9KTtcbn0pO1xuXG50ZXN0KCdjaGVjayBwcm9wZXJ0aWVzJywgKCkgPT4ge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgY29uc3QgY29uc3RydWN0OiBJb3RUb0xhbWJkYSA9IGRlcGxveU5ld0Z1bmMoc3RhY2spO1xuXG4gICAgZXhwZWN0KGNvbnN0cnVjdC5pb3RUb3BpY1J1bGUgIT09IG51bGwpO1xuICAgIGV4cGVjdChjb25zdHJ1Y3QubGFtYmRhRnVuY3Rpb24gIT09IG51bGwpO1xufSk7XG5cbnRlc3QoJ2NoZWNrIGV4Y2VwdGlvbiBmb3IgTWlzc2luZyBleGlzdGluZ09iaiBmcm9tIHByb3BzIGZvciBkZXBsb3kgPSBmYWxzZScsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgcHJvcHM6IElvdFRvTGFtYmRhUHJvcHMgPSB7XG4gICAgaW90VG9waWNSdWxlUHJvcHM6IHtcbiAgICAgIHRvcGljUnVsZVBheWxvYWQ6IHtcbiAgICAgICAgICBydWxlRGlzYWJsZWQ6IGZhbHNlLFxuICAgICAgICAgIGRlc2NyaXB0aW9uOiBcIlByb2Nlc3Npbmcgb2YgRFRDIG1lc3NhZ2VzIGZyb20gdGhlIEFXUyBDb25uZWN0ZWQgVmVoaWNsZSBTb2x1dGlvbi5cIixcbiAgICAgICAgICBzcWw6IFwiU0VMRUNUICogRlJPTSAnY29ubmVjdGVkY2FyL2R0Yy8jJ1wiLFxuICAgICAgICAgIGFjdGlvbnM6IFtdXG4gICAgICB9XG4gICAgfVxuICB9O1xuXG4gIHRyeSB7XG4gICAgbmV3IElvdFRvTGFtYmRhKHN0YWNrLCAndGVzdC1pb3QtbGFtYmRhLWludGVncmF0aW9uJywgcHJvcHMpO1xuICB9IGNhdGNoIChlKSB7XG4gICAgZXhwZWN0KGUpLnRvQmVJbnN0YW5jZU9mKEVycm9yKTtcbiAgfVxufSk7XG5cbnRlc3QoJ2NoZWNrIGRlcGxveSA9IHRydWUgYW5kIG5vIHByb3AnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGNvbnN0IHByb3BzOiBJb3RUb0xhbWJkYVByb3BzID0ge1xuICAgIGlvdFRvcGljUnVsZVByb3BzOiB7XG4gICAgICB0b3BpY1J1bGVQYXlsb2FkOiB7XG4gICAgICAgICAgcnVsZURpc2FibGVkOiBmYWxzZSxcbiAgICAgICAgICBkZXNjcmlwdGlvbjogXCJQcm9jZXNzaW5nIG9mIERUQyBtZXNzYWdlcyBmcm9tIHRoZSBBV1MgQ29ubmVjdGVkIFZlaGljbGUgU29sdXRpb24uXCIsXG4gICAgICAgICAgc3FsOiBcIlNFTEVDVCAqIEZST00gJ2Nvbm5lY3RlZGNhci9kdGMvIydcIixcbiAgICAgICAgICBhY3Rpb25zOiBbXVxuICAgICAgfVxuICAgIH1cbiAgfTtcblxuICB0cnkge1xuICAgIG5ldyBJb3RUb0xhbWJkYShzdGFjaywgJ3Rlc3QtaW90LWxhbWJkYS1pbnRlZ3JhdGlvbicsIHByb3BzKTtcbiAgfSBjYXRjaCAoZSkge1xuICAgIGV4cGVjdChlKS50b0JlSW5zdGFuY2VPZihFcnJvcik7XG4gIH1cbn0pOyJdfQ==