"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_rds_1 = require("monocdk/aws-rds");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * Non-Aurora RDS DB instances and Aurora clusters are configured to export all possible log types to CloudWatch
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    var _a, _b;
    if (node instanceof aws_rds_1.CfnDBInstance) {
        const dbType = String(nag_rules_1.NagRules.resolveIfPrimitive(node, node.engine));
        const exports = (_a = monocdk_1.Stack.of(node).resolve(node.enableCloudwatchLogsExports)) !== null && _a !== void 0 ? _a : [];
        const needed = [];
        if (!dbType.includes('aurora')) {
            if (dbType.includes('mariadb') || dbType.includes('mysql')) {
                needed.push('audit', 'error', 'general', 'slowquery');
            }
            else if (dbType.includes('postgres')) {
                needed.push('postgresql', 'upgrade');
            }
            else if (dbType.includes('oracle')) {
                needed.push('audit', 'alert', 'listener', 'oemagent', 'trace');
            }
            else if (dbType.includes('sqlserver')) {
                needed.push('agent', 'error');
            }
            const findings = needed
                .filter((log) => !exports.includes(log))
                .map((log) => `LogExport::${log}`);
            return findings.length ? findings : nag_rules_1.NagRuleCompliance.COMPLIANT;
        }
        else {
            return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
        }
    }
    else if (node instanceof aws_rds_1.CfnDBCluster) {
        const engine = nag_rules_1.NagRules.resolveIfPrimitive(node, node.engine).toLowerCase();
        const needed = [];
        const exports = (_b = monocdk_1.Stack.of(node).resolve(node.enableCloudwatchLogsExports)) !== null && _b !== void 0 ? _b : [];
        if (engine.toLowerCase() === 'aurora' ||
            engine.toLowerCase() === 'aurora-mysql') {
            needed.push('audit', 'error', 'general', 'slowquery');
        }
        else if (engine.toLowerCase() === 'aurora-postgresql') {
            needed.push('postgresql');
        }
        const findings = needed
            .filter((log) => !exports.includes(log))
            .map((log) => `LogExport::${log}`);
        return findings.length ? findings : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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