"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_events_1 = require("monocdk/aws-events");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * EventBridge event bus policies do not allow for open access
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_events_1.CfnEventBusPolicy) {
        if (monocdk_1.Stack.of(node).resolve(node.principal) === '*') {
            const condition = monocdk_1.Stack.of(node).resolve(node.condition);
            if (condition === undefined ||
                condition.key === undefined ||
                condition.type === undefined ||
                condition.value === undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        const resolvedStatement = monocdk_1.Stack.of(node).resolve(node.statement);
        if (resolvedStatement !== undefined) {
            const condition = monocdk_1.Stack.of(node).resolve(resolvedStatement === null || resolvedStatement === void 0 ? void 0 : resolvedStatement.Condition);
            if ((resolvedStatement === null || resolvedStatement === void 0 ? void 0 : resolvedStatement.Effect) === 'Allow' &&
                checkMatchingPrincipal(resolvedStatement === null || resolvedStatement === void 0 ? void 0 : resolvedStatement.Principal) === true &&
                (condition === undefined || JSON.stringify(condition) === '{}')) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the event bus policy applies to all principals
 * @param node The CfnEventBusPolicy to check
 * @param principal The principals in the event bus policy
 * @returns Whether the CfnEventBusPolicy applies to all principals
 */
function checkMatchingPrincipal(principals) {
    if (principals === '*') {
        return true;
    }
    const awsPrincipal = principals.AWS;
    if (Array.isArray(awsPrincipal)) {
        for (const account of awsPrincipal) {
            if (account === '*') {
                return true;
            }
        }
    }
    else if (awsPrincipal === '*') {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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