"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ec2_1 = require("monocdk/aws-ec2");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
const BLOCKED_PORTS = [20, 21, 3389, 3309, 3306, 4333];
/**
 * EC2 instances have all common TCP ports restricted for ingress IPv4 traffic
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/default-custom-security-groups.html
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = monocdk_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = monocdk_1.Stack.of(node).resolve(rule);
                for (const portNum of BLOCKED_PORTS) {
                    if (!testPort(node, resolvedRule, portNum)) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        for (const portNum of BLOCKED_PORTS) {
            if (!testPort(node, node, portNum)) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to identify if the given port number is unrestricted
 * @param node the CfnResource to check
 * @param rule the CfnSecurityGroupIngress rule to check
 * @param portNum the number of the port to check
 */
function testPort(node, rule, portNum) {
    //Does this rule apply to TCP traffic?
    const ipProtocol = nag_rules_1.NagRules.resolveIfPrimitive(node, rule.ipProtocol);
    const cidrIp = nag_rules_1.NagRules.resolveIfPrimitive(node, rule.cidrIp);
    const fromPort = nag_rules_1.NagRules.resolveIfPrimitive(node, rule.fromPort);
    const toPort = nag_rules_1.NagRules.resolveIfPrimitive(node, rule.toPort);
    if (ipProtocol === 'tcp') {
        //Does this rule allow all IPv4 addresses (unrestricted access)?
        if (cidrIp != undefined && cidrIp.includes('/0')) {
            //Is a port range specified?
            if (fromPort != undefined && toPort != undefined) {
                if ((fromPort <= portNum && toPort >= portNum) ||
                    fromPort == -1 ||
                    toPort == -1) {
                    return false;
                }
            }
            else {
                if (fromPort == portNum) {
                    return false;
                }
            }
        }
    }
    //Are all ports allowed?
    if (ipProtocol === '-1') {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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