"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cognito_1 = require("monocdk/aws-cognito");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * Cognito user pools have password policies that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cognito_1.CfnUserPool) {
        const policies = monocdk_1.Stack.of(node).resolve(node.policies);
        if (policies == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const passwordPolicy = monocdk_1.Stack.of(node).resolve(policies.passwordPolicy);
        if (passwordPolicy == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const minimumLength = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.minimumLength);
        if (minimumLength == undefined || minimumLength < 8) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requireUppercase = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.requireUppercase);
        if (requireUppercase !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requireNumbers = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.requireNumbers);
        if (requireNumbers !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requireSymbols = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.requireSymbols);
        if (requireSymbols !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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