"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cloudfront_1 = require("monocdk/aws-cloudfront");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 *  CloudFront distributions use a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS viewer connections
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cloudfront_1.CfnDistribution) {
        const distributionConfig = monocdk_1.Stack.of(node).resolve(node.distributionConfig);
        const viewerCertificate = monocdk_1.Stack.of(node).resolve(distributionConfig.viewerCertificate);
        if (viewerCertificate === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const minimumProtocolVersion = monocdk_1.Stack.of(node).resolve(viewerCertificate.minimumProtocolVersion);
        const sslSupportMethod = monocdk_1.Stack.of(node).resolve(viewerCertificate.sslSupportMethod);
        const cloudFrontDefaultCertificate = monocdk_1.Stack.of(node).resolve(viewerCertificate.cloudFrontDefaultCertificate);
        const outdatedProtocols = ['SSLv3', 'TLSv1', 'TLSv1_2016'];
        if (cloudFrontDefaultCertificate === true ||
            sslSupportMethod === undefined ||
            sslSupportMethod.toLowerCase() === 'vip' ||
            minimumProtocolVersion === undefined ||
            outdatedProtocols
                .map((x) => x.toLowerCase())
                .includes(minimumProtocolVersion.toLowerCase())) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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