"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const fs_1 = require("fs");
const path_1 = require("path");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("./nag-rules");
const nag_suppression_helper_1 = require("./utils/nag-suppression-helper");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
const SUPPRESSION_ID = 'CdkNagSuppression';
/**
 * The level of the message that the rule applies.
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel["WARN"] = "Warning";
    NagMessageLevel["ERROR"] = "Error";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule packs.
 */
class NagPack {
    constructor(props) {
        this.reportStacks = new Array();
        this.packName = '';
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
        this.logIgnores =
            props == undefined || props.logIgnores == undefined
                ? false
                : props.logIgnores;
        this.reports =
            props == undefined || props.reports == undefined ? true : props.reports;
    }
    get readPackName() {
        return this.packName;
    }
    get readReportStacks() {
        return this.reportStacks;
    }
    /**
     * Create a rule to be used in the NagPack.
     * @param params The @IApplyRule interface with rule details.
     */
    applyRule(params) {
        if (this.packName === '') {
            throw Error('The NagPack does not have a pack name, therefore the rule could not be applied. Set a packName in the NagPack constructor.');
        }
        const allIgnores = nag_suppression_helper_1.NagSuppressionHelper.getSuppressions(params.node);
        const ruleSuffix = params.ruleSuffixOverride
            ? params.ruleSuffixOverride
            : params.rule.name;
        const ruleId = `${this.packName}-${ruleSuffix}`;
        try {
            const ruleCompliance = params.rule(params.node);
            if (this.reports === true) {
                this.initializeStackReport(params);
                if (ruleCompliance === nag_rules_1.NagRuleCompliance.COMPLIANT) {
                    this.writeToStackComplianceReport(params, ruleId, ruleCompliance);
                }
            }
            if (this.isNonCompliant(ruleCompliance)) {
                const findings = this.asFindings(ruleCompliance);
                for (const findingId of findings) {
                    const suppressionReason = this.ignoreRule(allIgnores, ruleId, findingId);
                    if (this.reports === true) {
                        this.writeToStackComplianceReport(params, ruleId, nag_rules_1.NagRuleCompliance.NON_COMPLIANT, suppressionReason);
                    }
                    if (suppressionReason) {
                        if (this.logIgnores === true) {
                            const message = this.createMessage(SUPPRESSION_ID, findingId, `${ruleId} was triggered but suppressed.`, `Provided reason: "${suppressionReason}"`);
                            monocdk_1.Annotations.of(params.node).addInfo(message);
                        }
                    }
                    else {
                        const message = this.createMessage(ruleId, findingId, params.info, params.explanation);
                        if (params.level == NagMessageLevel.ERROR) {
                            monocdk_1.Annotations.of(params.node).addError(message);
                        }
                        else if (params.level == NagMessageLevel.WARN) {
                            monocdk_1.Annotations.of(params.node).addWarning(message);
                        }
                    }
                }
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allIgnores, VALIDATION_FAILURE_ID, '');
            if (this.reports === true) {
                this.writeToStackComplianceReport(params, ruleId, 'UNKNOWN', reason);
            }
            if (reason) {
                if (this.logIgnores === true) {
                    const message = this.createMessage(SUPPRESSION_ID, '', `${VALIDATION_FAILURE_ID} was triggered but suppressed.`, reason);
                    monocdk_1.Annotations.of(params.node).addInfo(message);
                }
            }
            else {
                const information = `'${ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, '', information, error.message);
                monocdk_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored.
     * @param ignores The ignores listed in cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @param findingId The id of the finding that is being checked.
     * @returns The reason the rule was ignored, or an empty string.
     */
    ignoreRule(ignores, ruleId, findingId) {
        for (let ignore of ignores) {
            if (nag_suppression_helper_1.NagSuppressionHelper.doesApply(ignore, ruleId, findingId)) {
                if (!ignore.appliesTo) {
                    // the rule is not granular so it always applies
                    return ignore.reason;
                }
                else {
                    return `[${findingId}] ${ignore.reason}`;
                }
            }
        }
        return '';
    }
    /**
     * The message to output to the console when a rule is triggered.
     * @param ruleId The id of the rule.
     * @param findingId The id of the finding.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     */
    createMessage(ruleId, findingId, info, explanation) {
        let message = findingId
            ? `${ruleId}[${findingId}]: ${info}`
            : `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}\n` : `${message}\n`;
    }
    /**
     * Write a line to the rule pack's compliance report for the resource's Stack
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    writeToStackComplianceReport(params, ruleId, compliance, explanation = '') {
        var _b;
        const line = this.createComplianceReportLine(params, ruleId, compliance, explanation);
        const outDir = (_b = monocdk_1.App.of(params.node)) === null || _b === void 0 ? void 0 : _b.outdir;
        const stackName = params.node.stack.nested
            ? monocdk_1.Names.uniqueId(params.node.stack)
            : params.node.stack.stackName;
        const fileName = `${this.packName}-${stackName}-NagReport.csv`;
        const filePath = path_1.join(outDir ? outDir : '', fileName);
        fs_1.appendFileSync(filePath, line);
    }
    /**
     * Initialize the report for the rule pack's compliance report for the resource's Stack if it doesn't exist
     * @param params
     */
    initializeStackReport(params) {
        var _b;
        const stackName = params.node.stack.nested
            ? monocdk_1.Names.uniqueId(params.node.stack)
            : params.node.stack.stackName;
        const fileName = `${this.packName}-${stackName}-NagReport.csv`;
        if (!this.reportStacks.includes(fileName)) {
            const outDir = (_b = monocdk_1.App.of(params.node)) === null || _b === void 0 ? void 0 : _b.outdir;
            const filePath = path_1.join(outDir ? outDir : '', fileName);
            this.reportStacks.push(fileName);
            fs_1.writeFileSync(filePath, 'Rule ID,Resource ID,Compliance,Exception Reason,Rule Level,Rule Info\n');
        }
    }
    /**
     * Helper function to create a line for the compliance report
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    createComplianceReportLine(params, ruleId, compliance, explanation = '') {
        //| Rule ID | Resource ID | Compliance | Exception Reason | Rule Level | Rule Info
        const line = Array();
        line.push(ruleId);
        line.push(params.node.node.path);
        if ((compliance === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            compliance === 'UNKNOWN') &&
            explanation !== '') {
            line.push('Suppressed');
            line.push(explanation);
        }
        else {
            line.push(compliance);
            line.push('N/A');
        }
        line.push(params.level);
        line.push(params.info);
        return line.map((i) => '"' + i.replace(/"/g, '""') + '"').join(',') + '\n';
    }
    isNonCompliant(ruleResult) {
        return (ruleResult === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            Array.isArray(ruleResult));
    }
    asFindings(ruleResult) {
        if (Array.isArray(ruleResult)) {
            return ruleResult;
        }
        else {
            return [''];
        }
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "monocdk-nag.NagPack", version: "1.12.38" };
//# sourceMappingURL=data:application/json;base64,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