# SPDX-FileCopyrightText: 2019 Sebastian Wagner
#
# SPDX-License-Identifier: AGPL-3.0-or-later

# -*- coding: utf-8 -*-
import unittest
import json
import requests_mock
from download_mapping import create_parser, main, generate_regex_from_human, IDENTIFIER_FROM_HUMAN, URL
from intelmq.lib.test import skip_internet


class TestDownloadMapping(unittest.TestCase):
    @skip_internet()
    def test_download(self):
        parser = create_parser()
        args = parser.parse_args("")
        result = json.loads(main(args))
        self.assertGreater(len(result), 100)
        self.assertIsInstance(result, list)

    @skip_internet()
    def test_download_identifier_expression(self):
        parser = create_parser()
        args = parser.parse_args('-e "^foobar$" -i foobar'.split(' '))
        result = json.loads(main(args))
        self.assertGreater(len(result), 100)
        self.assertIsInstance(result, list)

    def test_download_ignore_adware(self):
        parser = create_parser()
        args = parser.parse_args(["--mwnmp-ignore-adware"])
        with requests_mock.Mocker() as mocker:
            mocker.get(URL, text='"^foobar$",foobar,adware,barfoo')
            result = json.loads(main(args))
        self.assertEqual(len(result), 0)
        self.assertIsInstance(result, list)

    def test_download_not_ignore_not_adware(self):
        parser = create_parser()
        args = parser.parse_args(["--mwnmp-ignore-adware"])
        with requests_mock.Mocker() as mocker:
            mocker.get(URL, text='"^foobar$",foobar,not adware,barfoo')
            result = json.loads(main(args))
        self.assertEqual(len(result), 1)
        self.assertIsInstance(result, list)

    def test_download_add_default(self):
        " Test --add-default parameter. "
        parser = create_parser()
        args = parser.parse_args(["--add-default"])
        with requests_mock.Mocker() as mocker:
            mocker.get(URL, text='')
            result = json.loads(main(args))
        self.assertEqual(result,
                         [{'if': {'classification.taxonomy': 'malicious-code',
                                  'malware.name': '.*'},
                           'rulename': 'default',
                           'then': {'classification.identifier': '{msg[malware.name]}'}}]
                         )

    def test_download_add_default_constant(self):
        " Test --add-default parameter with an argument"
        parser = create_parser()
        args = parser.parse_args(["--add-default", "constant"])
        with requests_mock.Mocker() as mocker:
            mocker.get(URL, text='')
            result = json.loads(main(args))
        self.assertEqual(result,
                         [{'if': {'classification.taxonomy': 'malicious-code',
                                  'malware.name': '.*'},
                           'rulename': 'default',
                           'then': {'classification.identifier': 'constant'}}]
                         )

    maxDiff = None


class TestParser(unittest.TestCase):
    def test_parser_default(self):
        parser = create_parser()
        args = parser.parse_args(["--add-default"])
        self.assertEqual(args.add_default, '{msg[malware.name]}')

    def test_parser_default_argument(self):
        parser = create_parser()
        args = parser.parse_args(["--add-default", "constant"])
        self.assertEqual(args.add_default, "constant")

    def test_parser_no_default(self):
        parser = create_parser()
        args = parser.parse_args()
        self.assertEqual(args.add_default, None)


class TestDownloadMappingHelpers(unittest.TestCase):
    def test_generate_regex_from_human(self):
        self.assertEqual("^(foo[-_ ]?bar|bar[-_ ]?foo)$",
                         generate_regex_from_human("foo bar", "bar foo"))
        self.assertEqual("^(arid[-_ ]?viper)$",
                         generate_regex_from_human("AridViper"))
        self.assertEqual("^(apt[-_ ]?10)$",
                         generate_regex_from_human("APT 10"))
        self.assertEqual("^(foo[-_ ]?bar)$",
                         generate_regex_from_human("foo-bar"))
        self.assertEqual("^(cyber[-_ ]?fighters[-_ ]?of[-_ ]?izz[-_ ]?ad[-_ ]?din[-_ ]?al[-_ ]?qassam)$",
                         generate_regex_from_human("Cyber fighters of Izz Ad-Din Al Qassam"))

    def test_identifier_from_human(self):
        self.assertEqual("stealth-falcon",
                         IDENTIFIER_FROM_HUMAN.sub("-", "stealth falcon"))
